/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* keyb_lnx.c - Linux keyboard routines */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <termios.h>

#include "os.h"
#include "chat.h"
#include "config.h"
#include "output.h"
#include "modem.h"
#include "userbase.h"
#include "timeslic.h"

static struct termios oldtty;

static int waiting = 0;
static int Fin;

int KbdInit(void)
{
    char dev[20];
    struct termios tty;

    if (ttynum != 0)
    {
        sprintf(dev, "/dev/tty%d", ttynum);
        Fin = open(dev, O_RDONLY);
        if (Fin == -1) return 0;
        dup2(Fin, STDIN_FILENO);
    }
    Fin = STDIN_FILENO;
    waiting = 0;

    if (tcgetattr(Fin, &tty) == -1) return 0;
    memcpy(&oldtty, &tty, sizeof(tty));

    tty.c_iflag &= ~(ECHO | INLCR | ICRNL);
    tty.c_iflag |= IXON;
    tty.c_lflag = ISIG | NOFLSH;
    tty.c_cc[VSUSP] = 255; /* Ignore Ctrl-Z */

    if (tcsetattr(Fin, TCSANOW, &tty) == -1) return 0;

    return 1;
}

void KbdDeInit(void)
{
    tcsetattr(Fin, TCSANOW, &oldtty);
    if (ttynum != 0) dup2(open("/dev/ttyd", O_RDONLY), STDIN_FILENO);
}

int kbhit_nowait(void)
{
    int numchar;

    if (waiting) return 1;
    if (ioctl(Fin, FIONREAD, &numchar) < 0) return 0;

    return numchar > 0;
}

int kbhit(void)
{
    static fd_set fdset;
    struct timeval wait;

    if (waiting) return 1;

    wait.tv_sec = 0;
    wait.tv_usec = 200000;

    FD_ZERO(&fdset);
    FD_SET(Fin, &fdset);
    FD_SET(hCom, &fdset);

    return select(hCom+1, &fdset, NULL, NULL, &wait) >= 1 && FD_ISSET(Fin, &fdset);
}

int local_kbhit(void)
{
    static fd_set fdset;
    struct timeval wait;

    if (waiting) return 1;

    wait.tv_sec = 0;
    wait.tv_usec = 1000;

    FD_ZERO(&fdset);
    FD_SET(Fin, &fdset);

    return select(Fin+1, &fdset, NULL, NULL, &wait) >= 1;
}

int char_in(void)
{
    unsigned char c;

    while (!local_kbhit()) ;

    while (read(Fin, &c, 1) == -1)
    {
        if (errno == EINTR) continue;

        printf("getch() - Error\n");
        exit(1);
    }

    return c;
}

int getch(void)
{
    char alt_table[] = "qwertyuiopasdfghjklzxcvbnm";

    int ch, num;

    inactivity = 0; told_inactivity = 0;
    if (waiting != 0)
    {
        ch = waiting;
        waiting = 0;
        return ch;
    }

    ch = char_in();
    if (ch == 127) return 8; /* Backspace */

    if (ch == 27)
    {
        /* Scan code? */
        if (!local_kbhit()) return ch;

        /* Yep.. */
        ch = getch();

        if (ch == '[')
        {
            if (!local_kbhit()) return 27;
            ch = getch();

            if (ch >= '0' && ch <= '9')
            {
                num = 0;
                while (ch >= '0' && ch <= '9')
                {
                    num = num*10+ch-'0';
                    if (!local_kbhit()) return 27;
                    ch = getch();
                }

                switch (num)
                {
                    case 1:
                        /* Home */
                        waiting = 'G';
                        return 0;
                    case 2:
                        /* Insert */
                        waiting = 'R';
                        return 0;
                    case 3:
                        /* Delete */
                        waiting = 'S';
                        return 0;
                    case 4:
                        /* End */
                        waiting = 'O';
                        return 0;
                    case 5:
                        /* PgUp */
                        waiting = 'I';
                        return 0;
                    case 6:
                        /* PgDn */
                        waiting = 'Q';
                        return 0;
                }

                return 27;
            }

            switch (ch)
            {
                case 'A':
                    /* Cursor up */
                    waiting = 'H';
                    return 0;
                case 'B':
                    /* Down */
                    waiting = 'P';
                    return 0;
                case 'C':
                    /* Right */
                    waiting = 'M';
                    return 0;
                case 'D':
                    /* Left */
                    waiting = 'K';
                    return 0;
            }
        }
        else
        {
            if (ch == 'l')
            {
                /* Alt-L = Line chat */
                line_chat();
                waiting = 255;
                return 0;
            }
            if (ch == 'c')
            {
                /* Alt-C = Split screen chat */
                if (user.Emulation == USER_EMULATION_ASCII)
                    line_chat();
                else
                    split_screen_chat();
                waiting = 255;
                return 0;
            }
            if (ch == 'h')
            {
                /* Alt-H = Hang up */
                carrier = 0;
                waiting = 254;
                return 0;
            }

            if (ch == 'j')
            {
                /* Alt-J = shell */
                system(shell);
                waiting = 255;
                return 0;
            }

            if (ch == '1')
            {
                /* Alt-1 = Less time */
                if (time_left < 5) time_left = 0; else time_left -= 5;
                user.TodayMinutes += 5;
                waiting = 254;
                return 0;
            }

            if (ch == '2')
            {
                /* Alt-2 = More time */
                time_left += 5;
                if (user.TodayMinutes > 5) user.TodayMinutes -= 5;
                waiting = 254;
                return 0;
            }

            if (ch >= 'a' && ch <= 'z')
            {
                /* Alt-key */
                num = (int) (strchr(alt_table,ch)-alt_table);
                if (num < 10) num += 16;
                else if (num < 19) num += 30-10;
                else num += 44-19;
                waiting = num;
                return 0;
            }
            waiting = ch;
            return 27;
        }
    }

    if (ch == 2)
    {
        KbdDeInit();
        exit(1);
    }

    return local_in[(unsigned char) ch];
}
