/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* files.c - File handling */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <limits.h>

#include "os.h"
#include "memory.h"
#include "files.h"
#include "output.h"
#include "logfile.h"
#include "timeslic.h"

char *_fgets(char *buf, size_t n, int handle)
{
    char *strp;
    int readed,left;
    unsigned long fpos;

    fpos = FileSeek(handle,0,SEEK_CUR);
    readed = FileRead(handle,buf,n);
    if (readed <= 0) return NULL;

    if (*buf == 26) return NULL; /* EOF */

    strp = buf; left = readed;
    while (left != 0)
    {
        left--;
        if (*strp == '\r' || *strp == '\n' || *strp == 26)
        {
            if (*strp == '\r' && left > 0 && *(strp+1) == '\n') left--;
            *strp = '\0';
            break;
        }
        strp++;
    }

    if (left != 0) FileSeek(handle,fpos+readed-left,SEEK_SET);

    return buf;
}

int copyfile(char *src, char *dest)
{
    FILE *Fsrc,*Fdst;
    char *buf;
    size_t readed;

    if ((Fsrc = fopen(src,"rb")) == NULL) return 0;
    if ((Fdst = fopen(dest,"w+b")) == NULL) { fclose(Fsrc); return 0; }
    chmod(dest, 0660);
    buf = (char *) _malloc(4096);

    do {
        readed = fread(buf,1,4096,Fsrc);
        fwrite(buf,1,readed,Fdst);
    } while (readed == 4096);

    _free(buf);
    fclose(Fsrc); fclose(Fdst);
    return 1;
}

int FileLock(int handle)
{
    int tries;

    tries = 0;
    while (lock(handle,0,1) != 0)
    {
#ifdef __linux__
        if (errno != EWOULDBLOCK)
        {
#ifdef __OUTPUT_H
            output("\rFATAL ERROR while locking file!!!\r");
            write_log("FATAL ERROR while locking file!!!");
#endif
            return 0;
        }
#endif

        tries++;
#ifdef __OUTPUT_H
        if (tries == 20)
        {
            output("\rPlease wait, trying to lock file (takes max. 20 sec)...\r");
        }
        if (tries == 60)
        {
            output("\rOK, I give up. Something has to be wrong..\r\n");
            write_log("Can't lock file!\n");
            return 0;
        }
        usleep(500000);
#endif
        give_timeslice();
    }

    return 1;
}

#ifndef __NO_DRIVES__
void change_dir(char *path)
{
#ifndef __EMX__
    unsigned drives;
#endif

    if (path[0] == '\0') return;
    if (path[1] == ':')
    {
#ifdef __EMX__
        _chdrive(path[0]);
#else
        _dos_setdrive(path[0]-'A',&drives);
#endif
        chdir(path+2);
    }
    else
    {
        chdir(path);
    }
}
#endif

#ifdef __linux__
int exec(char *path, char *params)
{
    char tmp[1024];

    sprintf(tmp,"%s %s",path,params);
    return system(tmp);
}
#endif

#ifdef __FILES_CASE_SENSITIVE__
char *search_ign_file(char *fname)
{
    DIR *dirp;
    struct dirent *direntp;
    char *strp;

    /* Open directory */
    strp = strrchr(fname,SLASH);
    if (strp != NULL)
    {
        *strp = '\0';
        dirp = opendir(fname);
        *strp++ = SLASH;
    }
    else
    {
        strp = fname;
        dirp = opendir(".");
    }

    if (dirp == NULL) return NULL;

    /* Read directory */
    for (;;)
    {
        direntp = readdir(dirp);
        if (direntp == NULL) break;

        if (stricmp(direntp->d_name, strp) == 0)
        {
            /* Found file! */
            strcpy(strp, direntp->d_name);
            closedir(dirp);
            return fname;
        }
    }
    closedir(dirp);

    return NULL;
}

int FileOpenIgn(char *fname, int flags, int share)
{
    char str[MAX_PATH_LEN];

    share = share;
    strcpy(str,fname);
    if (search_ign_file(str) == NULL) return -1;

    return FileOpen(str,flags,0);
}
#endif

int mkpath(char *path)
{
    char str[256];
#ifndef __NO_DRIVES__
    unsigned drive;
#endif
    int pos,slen,spos;
    int start;
    struct stat statbuf;

    slen = strlen(path);

    if (path[slen-1] == SLASH) slen--;

    start = 0;
#ifndef __NO_DRIVES__
    if (path[1] == ':')
    {
        drive = toupper(path[0]);
        start = 2;
    }
    else
    {
#ifdef __EMX__
        drive = _getdrive();
#else
        _dos_getdrive(&drive); drive += 'A'-1;
#endif
    }

    if (path[start] == SLASH)
    {
        str[0] = (char) drive;
        str[1] = ':';
        str[2] = SLASH;
        str[3] = '\0';
        start++;
    } else str[0] = '\0';
#else
    if (path[0] == SLASH)
    {
        str[0] = SLASH;
        start++;
    }
    str[start] = '\0';
#endif

    spos = strlen(str);
    do
    {
        for (pos=start; pos<slen; pos++)
        {
            if (path[pos] == SLASH) break;

            str[spos] = path[pos];
            spos++;
        }
        str[spos] = 0;

        start = pos+1;
        if (stat(str, &statbuf) != 0)
        {
#ifdef __linux__
            if (mkdir(str, 0770) != 0) return 0;
            chmod(str, 0770); /* Why doesn't Linux give group-write access just with mkdir() ?? */
#elif defined (__EMX__)
            if (mkdir(str, 0) != 0) return 0;
#else
            if (mkdir(str) != 0) return 0;
#endif
        }

        str[spos] = SLASH; str[spos+1] = 0; spos++;
    } while (slen >= start);

    return 1;
}

int insert_space(int handle, long pos, long size, long fsize)
{
    char tmpbuf[128];
    char *buffer;

    unsigned bufsize;
    unsigned long movesz,lpos;

    bufsize = 16384;
    movesz = fsize - pos;
    if (bufsize > movesz) bufsize = (unsigned) movesz;
    if (bufsize == 0) return 0;

    if ((bufsize <= sizeof(tmpbuf)) || ((buffer = (char *) _malloc(bufsize)) == NULL))
    {
        bufsize = sizeof(tmpbuf);
        buffer = tmpbuf;
    }

    for (lpos=1; lpos<=movesz/bufsize; lpos++)
    {
        FileSeek(handle,fsize-bufsize*lpos,SEEK_SET);
        if (FileRead(handle,buffer,bufsize) != (int) bufsize)
        {
            if (buffer != tmpbuf) _free(buffer);
            return 0;
        }
        FileSeek(handle,fsize-bufsize*lpos+size,SEEK_SET);
        if (FileWrite(handle,buffer,bufsize) != (int) bufsize)
        {
            if (buffer != tmpbuf) _free(buffer);
            return 0;
        }
    }

    if (movesz % bufsize > 0)
    {
        FileSeek(handle,pos,SEEK_SET);
        if (FileRead(handle,buffer,(unsigned) (movesz % bufsize)) != (int) (movesz % bufsize))
        {
            if (buffer != tmpbuf) _free(buffer);
            return 0;
        }
        FileSeek(handle,pos+size,SEEK_SET);
        if (FileWrite(handle,buffer,(unsigned) (movesz % bufsize)) != (int) (movesz % bufsize))
        {
            if (buffer != tmpbuf) _free(buffer);
            return 0;
        }
    }

    if (buffer != tmpbuf) _free(buffer);
    return 1;
}

FILE_BUF *FileBufOpen(char *fname, int mode, int share_mode)
{
    FILE_BUF *file;
    int handle;
    struct stat statbuf;

    if (stat(fname, &statbuf) != 0) return NULL;

    handle = FileOpen(fname, mode, share_mode);
    if (handle == -1) return NULL;

    file = (FILE_BUF *) _malloc(sizeof(FILE_BUF));
    if (file == NULL)
    {
        close(handle);
        return NULL;
    }

    file->handle = handle;
    file->last_mtime = statbuf.st_mtime;

    file->bufstart = 1;
    file->bufend = 0;

    file->seekpos = 0;
    file->filesize = lseek(handle, 0, SEEK_END);
    lseek(handle, 0, SEEK_SET);

    return file;
}

FILE_BUF *FileBufCreate(char *fname, int mode)
{
    FILE_BUF *file;
    int handle;

    handle = FileCreate(fname, mode);
    if (handle == -1) return NULL;

    file = (FILE_BUF *) _malloc(sizeof(FILE_BUF));
    if (file == NULL)
    {
        close(handle);
        return NULL;
    }

    file->handle = handle;
    file->last_mtime = 0;

    file->bufstart = 1;
    file->bufend = 0;

    file->seekpos = 0;
    file->filesize = lseek(handle, 0, SEEK_END);
    lseek(handle, 0, SEEK_SET);

    return file;
}

void FileBufClose(FILE_BUF *file)
{
    if (file == NULL) return;

    close(file->handle);
    _free(file);
}

int FileBufRead(FILE_BUF *file, void **buffer, int bufsize)
{
    int readed,pos;
    struct stat statbuf;
    long fpos;

    if (fstat(file->handle, &statbuf) != 0) return 0;

    if (statbuf.st_mtime != file->last_mtime)
    {
        /* File modified, dump buffer */
        file->last_mtime = statbuf.st_mtime;
        file->bufstart = 1;
        file->bufend = 0;
        file->filesize = lseek(file->handle, 0, SEEK_END);
    }

    if (bufsize > (int) sizeof(file->buffer))
    {
        printf("ReadBufFile() : Buffer too small! This should NEVER happen!!\n");
        write_log("ReadBufFile() : Buffer too small! This should NEVER happen!!");
        return 0;
    }

    if (file->seekpos >= (long) file->filesize) return 0;

    if (file->bufstart > file->seekpos || (file->bufend < file->seekpos+bufsize-1 && (unsigned long) file->bufend != file->filesize-1))
    {
        if (file->bufstart > file->seekpos && file->buffering)
        {
            /* Probably reading backwards.. */
            fpos = file->seekpos-sizeof(file->buffer)+bufsize+256;
            if (fpos > file->seekpos) fpos = file->seekpos;
            if (fpos < 0) fpos = 0;
        }
        else
        {
            /* Probably seeking forward */
            fpos = file->seekpos;
        }
        FileSeek(file->handle, fpos, SEEK_SET);
        readed = FileRead(file->handle, file->buffer, file->buffering ? sizeof(file->buffer) : bufsize);
        if (readed < 1) return -1;

        file->bufstart = fpos;
        file->bufend = file->bufstart+readed-1;

        pos = 0;
        if (readed > bufsize) readed = bufsize;
        file->seekpos += bufsize;
    }
    else
    {
        pos = (int) (file->seekpos - file->bufstart);
        file->seekpos += bufsize;
        readed = bufsize;
    }

    *buffer = (void *) (file->buffer+pos);
    return readed;
}

int FileBufWrite(FILE_BUF *file, void *buffer, int bufsize)
{
    int readed;
    struct stat statbuf;

    if (fstat(file->handle, &statbuf) != 0) return 0;

    file->bufstart = 1;
    file->bufend = 0;
    file->last_mtime = statbuf.st_mtime;

    FileSeek(file->handle, file->seekpos, SEEK_SET);
    readed = FileWrite(file->handle, buffer, bufsize);
    file->filesize = lseek(file->handle, 0, SEEK_END);
    if (readed > 0) file->seekpos += readed;

    return readed;
}

long FileBufSeek(FILE_BUF *file, long filepos, int type)
{
    struct stat statbuf;

    if (fstat(file->handle, &statbuf) != 0) return 0xffffffff;

    if (statbuf.st_mtime != file->last_mtime)
    {
        /* File modified, dump buffer */
        file->last_mtime = statbuf.st_mtime;
        file->bufstart = 1;
        file->bufend = 0;
        file->filesize = lseek(file->handle, 0, SEEK_END);
    }

    switch (type)
    {
        case SEEK_SET:
            file->seekpos = filepos;
            break;
        case SEEK_CUR:
            file->seekpos += filepos;
            break;
        case SEEK_END:
            file->seekpos = file->filesize+filepos;
            break;
    }

    return file->seekpos;
}

int erase_path(char *path)
{
    DIR *dirp;
    struct dirent *direntp;
    struct stat statbuf;
    char tmp[256];

    dirp = opendir(path);
    if (dirp == NULL) return 0;

    for (;;)
    {
        direntp = readdir(dirp);
        if (direntp == NULL) break;

        sprintf(tmp, "%s"SSLASH"%s", path, direntp->d_name);
        if (stat(tmp, &statbuf) != 0) continue;
        if (ISDIR(statbuf)) continue;

        remove(tmp);
    }

    closedir(dirp);

    return 1;
}
