#define const 

/* ripwraps.c: RIPscrip API
 *
 * Eric P. Scott, San Francisco State University, December 1993
 * Copyright 1993 by Eric P. Scott.  All rights reserved.
 *
 * This software is "freeware" but is not in the public domain.
 *
 * The author grants you a nonexclusive royalty-free license to copy,
 * redistribute, and use this software for any purpose provided that
 * source distributions retain this entire copyright and license.
 * You may not sublicense or distribute modified versions to anyone else
 * except as part of another product or program, and modified versions
 * must be so labeled.
 *
 * You are encouraged to give proper credit to the author in binary
 * distributions and associated documentation.
 *
 * This software is provided "as is" with no warranties of any kind,
 * including the warranties of merchantability and fitness for a
 * particular purpose.
 *
 * RIPscrip is a trademark of TeleGrafix Communications, Inc.
 *
 * This software is not supported or endorsed by TeleGrafix.
 *
 */

/*
 * rip_* functions correspond to commands described in _RIPscrip
 * Graphics Protocol Specification_, Revision 1.54, by TeleGrafix
 * Communications, Inc.
 *
 * ripv_* functions are equivalent to their rip_* counterparts,
 * but take an argument vector in place of an inline list.
 *
 * ripio_* functions are specific to this library.
 */
 
/* Revision History
 *
 * 1.0  2/ 8/94  EPS  Initial Release
 * 1.1  4/11/94  EPS  Fixed rip_button()
 *
 */

#ifndef lint
/*static char sccsid[]="@(#)ripwraps.c	1.1 (SFSU) 4/11/94";*/
#endif

#ifdef NX_COMPILER_RELEASE_3_0
#include <ansi/stdio.h>
#ifdef __STRICT_BSD__
#include <bsd/strings.h>
#else
#include <ansi/string.h>
#endif
#ifdef __STRICT_ANSI__
#include <ansi/stdarg.h>
#define STDARG_RELIGION 1
#else
#include <bsd/varargs.h>
#define VARARGS_RELIGION 1
#endif
#else	/* NX_COMPILER_RELEASE_3_0 */
#include <stdio.h>
#ifdef __STRICT_BSD__
#include <strings.h>
#else
#include <string.h>
#endif
#ifdef __STRICT_ANSI__
#include <stdarg.h>
#define STDARG_RELIGION 1
#else
#include <varargs.h>
#define VARARGS_RELIGION 1
#endif
#endif	/* NX_COMPILER_RELEASE_3_0 */


/* ----- infrastructure ----- */

static int (*ripio_putc_func)(int, void *);
static void *ripio_putc_arg;
unsigned int ripio_cmd_chars;
static unsigned int ripio_width=80;
static int ripio_autoflush=1;

/* [internal] output a single character */
int ripio_putchar(int c)
{
    return(ripio_putc_func ? (*ripio_putc_func)(c, ripio_putc_arg) :
	putchar(c));
}

/* [internal] output a C string; no newline added */
int ripio_putstr(const char *string)
{
    register const char *s;
    register int r;

    if (ripio_putc_func) {
	r=0;
	for (s=string;*s;) r=(*ripio_putc_func)((int)*s++, ripio_putc_arg);
	return (r==EOF) ? r : 0;
    }
    else return(fputs(string, stdout));
}

/* allow user to specify an alternative character output function */
void ripio_set_putc(int (*func)(int, void *), void *arg)
{
    ripio_putc_func=func;
    ripio_putc_arg=arg;
}

/* return current alternative character output function pointer */
int (*ripio_get_putc_func(void))(int, void *)
{
    return ripio_putc_func;
}

/* return current alternative character output function argument */
void *ripio_get_putc_arg(void)
{
    return ripio_putc_arg;
}

/* set screen width (for command wrapping) */
void ripio_set_width(unsigned int width)
{
    if (width<ripio_width&&ripio_cmd_chars+2>=width) {
	(void)ripio_putchar('\n');
	ripio_cmd_chars=0;
    }
    ripio_width=width;
}

/* obtain current screen width setting */
unsigned int ripio_get_width(void)
{
    return ripio_width;
}

/* set autoflush flag */
void ripio_set_autoflush(int autoflush)
{
    ripio_autoflush=autoflush;
    if (autoflush&&ripio_cmd_chars>0) {
	(void)ripio_putchar('\n');
	ripio_cmd_chars=0;
    }
}

/* obtain current autoflush setting */
int ripio_get_autoflush(void)
{
    return ripio_autoflush;
}

/* [internal] begin a RIPscrip command */
void ripio_command(int baggage)
{
    if (ripio_cmd_chars==0) {
	(void)ripio_putstr("!|");
	ripio_cmd_chars=2;
    }
    else if (ripio_cmd_chars+4+baggage>=ripio_width) {
	(void)ripio_putstr("\n!|");
	ripio_cmd_chars=2;
    }
    else {
	(void)ripio_putchar('|');
	ripio_cmd_chars++;
    }
}

/* end RIPscrip command line */
void ripio_flush(void)
{
    if (ripio_cmd_chars>0) {
	(void)ripio_putchar('\n');
	ripio_cmd_chars=0;
    }
}

static void ripio_megadig(unsigned int value, int digits)
{
    if (value==0) do (void)ripio_putchar('0'); while (--digits>0);
    else {
	if (digits>1) ripio_megadig(value/36, digits-1); /* recursive! */
	(void)ripio_putchar("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"[value%36]);
    }
}

/* [internal] output an unsigned int in base 36 */
void ripio_meganum(unsigned int value, int digits)
{
    if (digits>0) {
	if (ripio_cmd_chars+2+digits>=ripio_width) {
	    (void)ripio_putstr("\\\n");
	    ripio_cmd_chars=0;
	}
	ripio_megadig(value, digits);
	ripio_cmd_chars+=digits;
    }
}

/* [internal] output a text string */
void ripio_text(const char *string)
{
    register const char *s;
    register int quotable, length;

    if ((!string)||!*string) return;
    quotable=0, length=0;
    for (s=string;*s;length++) switch (*s++) {
    case '!':
    case '|':
    case '\\':
	quotable++;
	break;
    default:
	break;
    }
    if (ripio_cmd_chars+2+length+quotable>=ripio_width) {
	(void)ripio_putstr("\\\n");
	ripio_cmd_chars=0;
    }
    if (quotable==0&&length+2<ripio_width) {
	(void)ripio_putstr(string);
	ripio_cmd_chars+=length;
    }
    else for (s=string;*s;) {
	if (ripio_cmd_chars+4>=ripio_width) {
	    (void)ripio_putstr("\\\n");
	    ripio_cmd_chars=0;
	}
	switch (*s) {
	case '!':
	case '|':
	case '\\':
	    (void)ripio_putchar('\\');
	    ripio_cmd_chars++;
	    break;
	default:
	    break;
	}
	(void)ripio_putchar((int)*s++);
	ripio_cmd_chars++;
    }
}

/* Query RIPscrip version number */
void ripio_query(void)
{
    (void)ripio_putstr("\033[!");
}

/* Disables all RIPscrip processing */
void ripio_disable(void)
{
    (void)ripio_putstr("\033[1!");
}

/* Enables RIPscrip processing */
void ripio_enable(void)
{
    (void)ripio_putstr("\033[2!");
}

/* ----- RIPscrip 1.54 functions ----- */

/* Define the size and location of the Text Window */
/* cf: Borland window(), _wscroll */
void rip_text_window(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int wrap, unsigned int size)
{
    ripio_command(10);
    (void)ripio_putchar('w');
    ripio_cmd_chars++;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(wrap, 1);
    ripio_meganum(size, 1);
    if (ripio_autoflush) ripio_flush();
}

/* Define the size & location of the Graphics Window */
/* cf: Borland setviewport() */
void rip_viewport(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1)
{
    ripio_command(8);
    (void)ripio_putchar('v');
    ripio_cmd_chars++;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Clear Graphics/Text Windows & reset to full screen */
/* cf: Borland setgraphmode() */
void rip_reset_windows(void)
{
    ripio_command(0);
    (void)ripio_putchar('*');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* Clears Text Window to current background color */
/* cf: Borland clrscr() */
void rip_erase_window(void)
{
    ripio_command(0);
    (void)ripio_putchar('e');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* Clear Graphics Window to current background color */
/* cf: Borland clearviewport() */
void rip_erase_view(void)
{
    ripio_command(0);
    (void)ripio_putchar('E');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* Move text cursor to row & column in Text Window */
/* cf: Borland gotoxy() */
void rip_gotoxy(unsigned int x, unsigned int y)
{
    ripio_command(4);
    (void)ripio_putchar('g');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Move cursor to upper-left corner of Text Window */
void rip_home(void)
{
    ripio_command(0);
    (void)ripio_putchar('H');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* Erase current line from cursor to end of line */
/* cf: Borland clreol() */
void rip_erase_eol(void)
{
    ripio_command(0);
    (void)ripio_putchar('>');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* Set current Drawing Color for graphics */
/* cf: Borland setcolor() */
void rip_color(unsigned int color)
{
    ripio_command(2);
    (void)ripio_putchar('c');
    ripio_cmd_chars++;
    ripio_meganum(color, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Set 16-color Palette from master 64-color palette */
/* cf: Borland setallpalette() */
void rip_set_palette(unsigned int c1, unsigned int c2, unsigned int c3,
    unsigned int c4, unsigned int c5, unsigned int c6, unsigned int c7,
    unsigned int c8, unsigned int c9, unsigned int c10, unsigned int c11,
    unsigned int c12, unsigned int c13, unsigned int c14,unsigned int c15,
    unsigned int c16)
{
    ripio_command(32);
    (void)ripio_putchar('Q');
    ripio_cmd_chars++;
    ripio_meganum(c1, 2);
    ripio_meganum(c2, 2);
    ripio_meganum(c3, 2);
    ripio_meganum(c4, 2);
    ripio_meganum(c5, 2);
    ripio_meganum(c6, 2);
    ripio_meganum(c7, 2);
    ripio_meganum(c8, 2);
    ripio_meganum(c9, 2);
    ripio_meganum(c10, 2);
    ripio_meganum(c11, 2);
    ripio_meganum(c12, 2);
    ripio_meganum(c13, 2);
    ripio_meganum(c14, 2);
    ripio_meganum(c15, 2);
    ripio_meganum(c16, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Set color of 16-color Palette from Master Palette */
/* cf: Borland setpalette() */
void rip_one_palette(unsigned int color, unsigned int value)
{
    ripio_command(4);
    (void)ripio_putchar('a');
    ripio_cmd_chars++;
    ripio_meganum(color, 2);
    ripio_meganum(value, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Set drawing mode for graphics primitives */
/* cf: Borland setwritemode() */
void rip_write_mode(unsigned int mode)
{
    ripio_command(2);
    (void)ripio_putchar('W');
    ripio_cmd_chars++;
    ripio_meganum(mode, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Move the current drawing position to (X,Y) */
/* cf: Borland moveto() */
void rip_move(unsigned int x, unsigned int y)
{
    ripio_command(4);
    (void)ripio_putchar('m');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw text in current font/color at current spot */
/* cf: Borland outtext() */
void rip_text(const char *text_string)
{
    ripio_command(strlen(text_string));
    (void)ripio_putchar('T');
    ripio_cmd_chars++;
    ripio_text(text_string);
    if (ripio_autoflush) ripio_flush();
}

/* Draw text in current font/color at specific spot */
/* cf: Borland outtextxy() */
void rip_text_xy(unsigned int x, unsigned int y, const char *text_string)
{
    ripio_command(4+strlen(text_string));
    (void)ripio_putchar('@');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_text(text_string);
    if (ripio_autoflush) ripio_flush();
}

/* Select current font style, orientation and size */
/* cf: Borland settextstyle() */
void rip_font_style(unsigned int font, unsigned int direction,
    unsigned int size, unsigned int res)
{
    ripio_command(8);
    (void)ripio_putchar('Y');
    ripio_cmd_chars++;
    ripio_meganum(font, 2);
    ripio_meganum(direction, 2);
    ripio_meganum(size, 2);
    ripio_meganum(res, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draws one pixel using current drawing color */
/* cf: Borland putpixel() */
void rip_pixel(unsigned int x, unsigned int y)
{
    ripio_command(4);
    (void)ripio_putchar('X');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw a line in the current color/line style */
/* cf: Borland line() */
void rip_line(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1)
{
    ripio_command(8);
    (void)ripio_putchar('L');
    ripio_cmd_chars++;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw a rectangle in current color/line style */
/* cf: Borland rectangle() */
void rip_rectangle(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1)
{
    ripio_command(8);
    (void)ripio_putchar('R');
    ripio_cmd_chars++;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw filled rectangle with fill color/pattern */
/* cf: Borland bar() */
void rip_bar(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1)
{
    ripio_command(8);
    (void)ripio_putchar('B');
    ripio_cmd_chars++;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw circle in current color and line thickness */
/* cf: Borland circle() */
void rip_circle(unsigned int x_center, unsigned int y_center,
    unsigned int radius)
{
    ripio_command(6);
    (void)ripio_putchar('C');
    ripio_cmd_chars++;
    ripio_meganum(x_center, 2);
    ripio_meganum(y_center, 2);
    ripio_meganum(radius, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw elliptical arc in current color/line style */
/* cf: Borland ellipse() */
void rip_oval(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int end_ang, unsigned int x_rad, unsigned int y_rad)
{
    ripio_command(12);
    (void)ripio_putchar('O');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(st_ang, 2);
    ripio_meganum(end_ang, 2);
    ripio_meganum(x_rad, 2);
    ripio_meganum(y_rad, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw filled ellipse using current color/pattern */
/* cf: Borland fillellipse() */
void rip_filled_oval(unsigned int x_center, unsigned int y_center,
    unsigned int x_rad, unsigned int y_rad)
{
    ripio_command(8);
    (void)ripio_putchar('o');
    ripio_cmd_chars++;
    ripio_meganum(x_center, 2);
    ripio_meganum(y_center, 2);
    ripio_meganum(x_rad, 2);
    ripio_meganum(y_rad, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw circular arc in current color/line thickness */
/* cf: Borland arc() */
void rip_arc(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int end_ang, unsigned int radius)
{
    ripio_command(10);
    (void)ripio_putchar('A');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(st_ang, 2);
    ripio_meganum(end_ang, 2);
    ripio_meganum(radius, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw an elliptical arc */
/* (what's the difference between this and rip_oval()?) */
void rip_oval_arc(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int e_ang, unsigned int radx, unsigned int rady)
{
    ripio_command(12);
    (void)ripio_putchar('V');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(st_ang, 2);
    ripio_meganum(e_ang, 2);
    ripio_meganum(radx, 2);
    ripio_meganum(rady, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draws a circular pie slice */
/* cf: Borland pieslice() */
void rip_pie_slice(unsigned int x, unsigned int y, unsigned int start_ang,
    unsigned int end_ang, unsigned int radius)
{
    ripio_command(10);
    (void)ripio_putchar('I');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(start_ang, 2);
    ripio_meganum(end_ang, 2);
    ripio_meganum(radius, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draws an elliptical pie slice */
/* cf: Borland sector() */
void rip_oval_pie_slice(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int e_ang, unsigned int radx, unsigned int rady)
{
    ripio_command(12);
    (void)ripio_putchar('i');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(st_ang, 2);
    ripio_meganum(e_ang, 2);
    ripio_meganum(radx, 2);
    ripio_meganum(rady, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw a bezier curve */
void rip_bezier(unsigned int x1, unsigned int y1, unsigned int x2,
    unsigned int y2, unsigned int x3, unsigned int y3, unsigned int x4,
    unsigned int y4, unsigned int cnt)
{
    ripio_command(18);
    (void)ripio_putchar('Z');
    ripio_cmd_chars++;
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(x2, 2);
    ripio_meganum(y2, 2);
    ripio_meganum(x3, 2);
    ripio_meganum(y3, 2);
    ripio_meganum(x4, 2);
    ripio_meganum(y4, 2);
    ripio_meganum(cnt, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Draw polygon in current color/line-style */
#ifdef STDARG_RELIGION
void rip_polygon(unsigned int npoints, ...)
#else
void rip_polygon(va_alist)
va_dcl
#endif
{
    va_list ap;
#ifndef STDARG_RELIGION
    register unsigned int npoints;

    va_start(ap);
    npoints=va_arg(ap, unsigned int);
#else

    va_start(ap, npoints);
#endif
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('P');
    ripio_cmd_chars++;
    ripio_meganum(npoints, 2);
    while (npoints>0) {
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* xn */
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* yn */
	--npoints;
    }
    va_end(ap);
    if (ripio_autoflush) ripio_flush();
}

/* Draw filled polygon in current color/fill pattern */
/* cf: Borland fillpoly() */
#ifdef STDARG_RELIGION
void rip_fill_polygon(unsigned int npoints, ...)
#else
void rip_fill_polygon(va_alist)
va_dcl
#endif
{
    va_list ap;
#ifndef STDARG_RELIGION
    register unsigned int npoints;

    va_start(ap);
    npoints=va_arg(ap, unsigned int);
#else

    va_start(ap, npoints);
#endif
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('p');
    ripio_cmd_chars++;
    ripio_meganum(npoints, 2);
    while (npoints>0) {
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* xn */
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* yn */
	--npoints;
    }
    va_end(ap);
    if (ripio_autoflush) ripio_flush();
}

/* Draw a Poly-Line (multi-faceted line) */
/* cf: Borland drawpoly() */
#ifdef STDARG_RELIGION
void rip_polyline(unsigned int npoints, ...)
#else
void rip_polyline(va_alist)
va_dcl
#endif
{
    va_list ap;
#ifndef STDARG_RELIGION
    register unsigned int npoints;

    va_start(ap);
    npoints=va_arg(ap, unsigned int);
#else

    va_start(ap, npoints);
#endif
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('l');
    ripio_cmd_chars++;
    ripio_meganum(npoints, 2);
    while (npoints>0) {
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* xn */
	ripio_meganum(va_arg(ap, unsigned int), 2);	/* yn */
	--npoints;
    }
    va_end(ap);
    if (ripio_autoflush) ripio_flush();
}

/* Flood fill screen area with current fill settings */
/* cf: Borland floodfill() */
void rip_fill(unsigned int x, unsigned int y, unsigned int border)
{
    ripio_command(6);
    (void)ripio_putchar('F');
    ripio_cmd_chars++;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(border, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Defines a line style and thickness */
/* cf: Borland setlinestyle() */
void rip_line_style(unsigned int style, unsigned int user_pat,
    unsigned int thick)
{
    ripio_command(8);
    (void)ripio_putchar('=');
    ripio_cmd_chars++;
    ripio_meganum(style, 2);
    ripio_meganum(user_pat, 4);
    ripio_meganum(thick, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Set current fill style (predefined) & fill color */
/* cf: Borland setfillstyle() */
void rip_fill_style(unsigned int pattern, unsigned int color)
{
    ripio_command(4);
    (void)ripio_putchar('S');
    ripio_cmd_chars++;
    ripio_meganum(pattern, 2);
    ripio_meganum(color, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Set user-definable (custom) fill pattern/color */
/* cf: Borland setfillpattern() */
void rip_fill_pattern(unsigned int c1, unsigned int c2, unsigned int c3,
    unsigned int c4, unsigned int c5, unsigned int c6, unsigned int c7,
    unsigned int c8, unsigned int col)
{
    ripio_command(18);
    (void)ripio_putchar('s');
    ripio_cmd_chars++;
    ripio_meganum(c1, 2);
    ripio_meganum(c2, 2);
    ripio_meganum(c3, 2);
    ripio_meganum(c4, 2);
    ripio_meganum(c5, 2);
    ripio_meganum(c6, 2);
    ripio_meganum(c7, 2);
    ripio_meganum(c8, 2);
    ripio_meganum(col, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Defines a rectangular hot mouse region */
void rip_mouse(unsigned int num, unsigned int x0, unsigned int y0,
    unsigned int x1, unsigned int y1, unsigned int clk, unsigned int clr, 
    unsigned int res, const char *text)
{
    ripio_command(18+strlen(text));
    (void)ripio_putstr("1M");
    ripio_cmd_chars+=2;
    ripio_meganum(num, 2);
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(clk, 1);
    ripio_meganum(clr, 1);
    ripio_meganum(res, 5);
    ripio_text(text);
    if (ripio_autoflush) ripio_flush();
}

/* Destroys all previously defined hot mouse regions */
void rip_kill_mouse_fields(void)
{
    ripio_command(1);
    (void)ripio_putstr("1K");
    ripio_cmd_chars+=2;
    if (ripio_autoflush) ripio_flush();
}

/* Define a rectangular text region */
void rip_begin_text(unsigned int x1, unsigned int y1, unsigned int x2,
    unsigned int y2, unsigned int res)
{
    ripio_command(11);
    (void)ripio_putstr("1T");
    ripio_cmd_chars+=2;
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(x2, 2);
    ripio_meganum(y2, 2);
    ripio_meganum(res, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Display a line of text in rectangular text region */
void rip_region_text(unsigned int justify, const char *text_string)
{
    ripio_command(1);
    (void)ripio_putstr("1t");
    ripio_cmd_chars+=2;
    ripio_meganum(justify, 1);
    ripio_text(text_string);
    if (ripio_autoflush) ripio_flush();
}

/* End a rectangular text region */
void rip_end_text(void)
{
    ripio_command(1);
    (void)ripio_putstr("1E");
    ripio_cmd_chars+=2;
    if (ripio_autoflush) ripio_flush();
}

/* Copy rectangular image to clipboard (as icon) */
void rip_get_image(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int res)
{
    ripio_command(10);
    (void)ripio_putstr("1C");
    ripio_cmd_chars+=2;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(res, 1);
    if (ripio_autoflush) ripio_flush();
}

/* Pastes the clipboard contents onto the screen */
void rip_put_image(unsigned int x, unsigned int y, unsigned int mode,
    unsigned int res)
{
    ripio_command(8);
    (void)ripio_putstr("1P");
    ripio_cmd_chars+=2;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(mode, 2);
    ripio_meganum(res, 1);
    if (ripio_autoflush) ripio_flush();
}

/* Write contents of the clipboard (icon) to disk */
void rip_write_icon(unsigned int res, const char *filename)
{
    ripio_command(2+strlen(filename));
    (void)ripio_putstr("1W");
    ripio_cmd_chars+=2;
    ripio_meganum(res, 1);
    ripio_text(filename);
    if (ripio_autoflush) ripio_flush();
}


/* Loads and displays a disk-based icon to screen */
void rip_load_icon(unsigned int x, unsigned int y, unsigned int mode,
    unsigned int clipboard, unsigned int res, const char *filename)
{
    ripio_command(10+strlen(filename));
    (void)ripio_putstr("1I");
    ripio_cmd_chars+=2;
    ripio_meganum(x, 2);
    ripio_meganum(y, 2);
    ripio_meganum(mode, 2);
    ripio_meganum(clipboard, 1);
    ripio_meganum(res, 2);	/* should be set to 36 for v1.54 */
    ripio_text(filename);
    if (ripio_autoflush) ripio_flush();
}

/* Button style definition */
void rip_button_style(unsigned int wid, unsigned int hgt, unsigned int orient,
    unsigned int flags, unsigned int bevsize, unsigned int dfore,
    unsigned int dback, unsigned int bright, unsigned int dark,
    unsigned int surface, unsigned int grp_no, unsigned int flags2,
    unsigned int uline_col, unsigned int corner_col, unsigned int res)
{
    ripio_command(37);
    (void)ripio_putstr("1B");
    ripio_cmd_chars+=2;
    ripio_meganum(wid, 2);
    ripio_meganum(hgt, 2);
    ripio_meganum(orient, 2);
    ripio_meganum(flags, 4);
    ripio_meganum(bevsize, 2);
    ripio_meganum(dfore, 2);
    ripio_meganum(dback, 2);
    ripio_meganum(bright, 2);
    ripio_meganum(dark, 2);
    ripio_meganum(surface, 2);
    ripio_meganum(grp_no, 2);
    ripio_meganum(flags2, 2);
    ripio_meganum(uline_col, 2);
    ripio_meganum(corner_col, 2);
    ripio_meganum(res, 6);
    if (ripio_autoflush) ripio_flush();
}

/* Define a Mouse Button */
void rip_button(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int hotkey, unsigned int flags,
    unsigned int res, const char *text)
{
    ripio_command(13);
    (void)ripio_putstr("1U");
    ripio_cmd_chars+=2;
    ripio_meganum(x0, 2);
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);
    ripio_meganum(y1, 2);
    ripio_meganum(hotkey, 2);
    ripio_meganum(flags, 1);
    ripio_meganum(res, 1);
    ripio_text(text);
    if (ripio_autoflush) ripio_flush();
}

/* Define a text variable */
void rip_define(unsigned int flags, unsigned int res, const char *text)
{
    ripio_command(6+strlen(text));
    (void)ripio_putstr("1D");
    ripio_cmd_chars+=2;
    ripio_meganum(flags, 3);
    ripio_meganum(res, 2);
    ripio_text(text);
    if (ripio_autoflush) ripio_flush();
}

/* Query the contents of a text variable */
void rip_query(unsigned int mode, unsigned int res, const char *text)
{
    ripio_command(5+strlen(text));
    (void)ripio_putstr("1\033");
    ripio_cmd_chars+=2;
    ripio_meganum(mode, 1);
    ripio_meganum(res, 3);
    ripio_text(text);
    if (ripio_autoflush) ripio_flush();
}

/* Copy screen region up/down */
void rip_copy_region(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int res, unsigned int dest_line)
{
    ripio_command(13);
    (void)ripio_putstr("1G");
    ripio_cmd_chars+=2;
    ripio_meganum(x0, 2);	/* must be divisible by 8 */
    ripio_meganum(y0, 2);
    ripio_meganum(x1, 2);	/* must be divisible by 8 */
    ripio_meganum(y1, 2);
    ripio_meganum(res, 2);
    ripio_meganum(dest_line, 2);
    if (ripio_autoflush) ripio_flush();
}

/* Playback local .RIP file */
void rip_read_scene(unsigned int res, const char *filename)
{
    ripio_command(9+strlen(filename));
    (void)ripio_putstr("1R");
    ripio_cmd_chars+=2;
    ripio_meganum(res, 8);
    ripio_text(filename);
    ripio_flush();	/* mandatory */
}

/* Query existing information on a particular file */
void rip_file_query(unsigned int mode, unsigned int res, const char *filename)
{
    ripio_command(7+strlen(filename));
    (void)ripio_putstr("1F");
    ripio_cmd_chars+=2;
    ripio_meganum(mode, 2);
    ripio_meganum(res, 4);
    ripio_text(filename);
    if (ripio_autoflush) ripio_flush();
}

/* Enter block transfer mode with host */
void rip_enter_block_mode(unsigned int mode, unsigned int proto,
    unsigned int file_type, unsigned int res, const char *filename)
{
    ripio_command(9+strlen(filename));
    (void)ripio_putstr("9\033");
    ripio_cmd_chars+=2;
    ripio_meganum(mode, 1);
    ripio_meganum(proto, 1);
    ripio_meganum(file_type, 2);
    ripio_meganum(res, 4);
    ripio_text(filename);
    ripio_flush();	/* mandatory */
}

/* End of RIPscrip Scene */
void rip_no_more(void)
{
    ripio_command(0);
    (void)ripio_putchar('#');
    ripio_cmd_chars++;
    if (ripio_autoflush) ripio_flush();
}

/* ----- vector interfaces ----- */

/* Set 16-color Palette from master 64-color palette */
/* cf: Borland setallpalette() */
void ripv_set_palette(unsigned char colors[16])
{
    register unsigned char *c;
    register int i;

    ripio_command(32);
    (void)ripio_putchar('Q');
    ripio_cmd_chars++;
    c=colors; i=15; do ripio_meganum((unsigned int)*c++, 2); while (--i>=0);
    if (ripio_autoflush) ripio_flush();
}

/* Draw polygon in current color/line-style */
void ripv_polygon(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;
    
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('P');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
    if (ripio_autoflush) ripio_flush();
}

/* Draw filled polygon in current color/fill pattern */
/* cf: Borland fillpoly() */
void ripv_fill_polygon(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;
    
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('p');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
    if (ripio_autoflush) ripio_flush();
}

/* Draw a Poly-Line (multi-faceted line) */
/* cf: Borland drawpoly() */
void ripv_polyline(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;

    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('l');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
    if (ripio_autoflush) ripio_flush();
}

/* Set user-definable (custom) fill pattern/color */
/* cf: Borland setfillpattern() */
void ripv_fill_pattern(unsigned char upattern[8], unsigned int color)
{
    register unsigned char *u;
    register int i;

    ripio_command(18);
    (void)ripio_putchar('s');
    ripio_cmd_chars++;
    u=upattern; i=7; do ripio_meganum((unsigned int)*u++, 2); while (--i>=0);
    ripio_meganum(color, 2);
    if (ripio_autoflush) ripio_flush();
}
