// Filename:   user.h
// Contents:   the user interface object
// Author:     Greg Shaw
// Created:        5/30/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _USER_H_
#define _USER_H_

// Object: user
// Purpose:    encapsulate all of the user funtions into one object.
// Attributes:
//     fname, lname - the user's name
//         login_name - the user's logon
//         city - city where user from
//         state - two character state
//         uterm - terminal type
//     last_logon - the last time the user logged on (in seconds)
//     logins - the number of times the user has logged on
//         downloads - number of downloads by the user
//         uploads - the number of uploads by user
//         priv_msgs - the number of private messages by user (unused)
//     pub_msgs - the  number of public messages by user
//         credited - credited time (for chat or uploads)
//     flags - user access flags
//     tmpflags - temporary flags (if authorized by sysop)
//     acl - the access level of user
//     tmpacl - the temporary access level of user
//     timelimit - the timelimit for this user
//     temptl - temporary timelimit (if authorized by sysop)
//     timeused - the amount of time used on last call
//     msgflags - flags for when user has seen message (true/false)
// Methods:
//     get - load the user object with user info, or, get user info from user
//     list - list the users in the user list
//     change_acl - change the user's access level
//     change_tl - change user's timelimit
//     change_flags - change user's flags
//     u_flags - return user's flags
//     u_acl - return user's access level
//     u_timelimit - return user's timelimit
// Author: Greg Shaw
// Created:    7/13/93

class User: public language     // inherited from user output function
{
	// login name
	char   login_name[MAX_LOGIN_LENGTH];
	// user's alias (for chatting)
	char   alias[MAX_ALIAS_LENGTH];
	char   city[21];            // city
	char   state[21];           // state
	char   uterm[MAX_TERM_LENGTH];           // terminal type
	char   editor[15];          // favorite editor
	// chosen language
	char   lang[MAX_LANGUAGE_FILENAME];
	CardRec    *card;           // card information for user
	time_t last_login;          // last login
	time_t login_time;          // login time
	time_t anniversary;         // date of first logon (for anniversary calculations)
	time_t expiration;          // when his call back time period has expired
	time_t validated;           // when user was validated (for account expiration)
	time_t mail_check;          // time of last mail check
	int    color_capable;       // has color?
	int    lines;               // lines on screen
	int    cols;                // columns on screen
	int    card_color;          // color (ord) of card
	int    logins;              // number of logins
	int    downloads;           // number of downloads
	int    uploads;             // number of uploads
	int    priv_msgs;           // private messages
	int    pub_msgs;            // private messages
	int    credited;            // credited time
	int    total_timeused;	    // total timeused
	int    c_fore;		    // outgoing chat foreground
	int    c_back;		    // outgoing chat background
	off_t  mail_size;   	    // size of mail file on last check
	unsigned long  flags;       // flags
	unsigned long  tmpflags;    // temp flags
	unsigned long  callernumber;// caller number
	unsigned int   acl;         // access level
	unsigned int   tmpacl;      // temp access level
	unsigned int   timelimit;   // time limit
	unsigned int   tmptl;       // temporary time limit
	unsigned int   timeused;    // time used during last call
public:
	// public variables
	unsigned int   start_kused;   // number of kbytes downloaded last call
	unsigned int   kused;   // number of kbytes downloaded last call
	unsigned int	total_kused;	    // total K downloaded
	char   fname[MAX_FIRSTNAME];       // user's first name
	char   lname[MAX_LASTNAME];       // user's last

	// methods
	User();                 // constructor
	~User();                // destructor

	time_t	account_expiration(void);	// when account expires next
	int    append(void);    // append user to userlog
	void callingfrom(char *stor)
	{
		sprintf(stor,"%s, %s",city, state);	
	}
	unsigned long caller(void) { return(callernumber); };	// caller number 
	// can user access this item?
	int	can_access(int acl, int flags, char am, char fm);
	// change access level - perm true for perm.
	int    change_acl(unsigned int nacl, char perm);
	// change alias name (for door exec)
	void   change_alias(char *new_alias);
	// change # of downloads (for door execute)
	void   change_dl(int dls) { downloads = dls; };
	// change editor (for door execute)
	void   change_editor(char *new_editor);
	// change flags - perm true for perm.
	int    change_flags(unsigned long fl, char perm, char or);
	// change # of lines	 (for door execute)
	void   change_lines(int nlines) { lines = nlines; };
	// change time of logon	 (for door usage)
	void   change_login(time_t now) { login_time = now; };
	// change terminal type (for door exec)
	void   change_terminal(char *new_term);
	// change timelimit - perm true for perm.
	int    change_tl(unsigned int tl, char perm);
	// change # of uploads (for door execute)
	void   change_ul(int nlines) { uploads = nlines; };
	int	chat_back(void) { return(c_back); };
	int	chat_fore(void) { return(c_fore); };
	int	check_access(void);	// check time expired, etc.
	int	check_acl(void);	// check for acl change key file
	int	check_for_ansi(void);	// check for ansi capability
	int    check_card(void);// check card color vs. access level
	int    check_info(void);// ask user about term, lines, cols and such
	int     check_line(void);       // check tty line for security

	int    col(void) { return(cols); };
	int    credit() { return(credited); };
	
	int    delete_user(char *name); // delete user with login name
	int    display_info(int logon); // display user information

	int    edit_chat_colors(void);
	char   *editorname(void) { return(editor); };
	int	exists(char *username);	// does this user exist in the userlog?
	int    export_soft(void); // export important variables to environment
	int    export_hard(int num); // export important variables to environment

	int	forward_email(void);	// create .forward in user's directory

	int    get_editor(void);   // get user's editor
	int    getinfo(void);   // get user's setup and information
	int    get(char *name, int check); // get user - load user record or create new
	int    has_color(void) { return(color_capable); };
	// increment credited minutes
	void   inc_credit(int num)
	{
		credited += num;
	};
	// increment number of downloads
	void   inc_downloads(int num)
	{
		downloads += num;
	};
	// increment total k used 
	void   inc_tot_kused(int num)
	{
		total_kused += num;
	};
	// increment private messages sent
	void	inc_priv_messages(int num)
	{
		priv_msgs += num;
	}
	// increment public messages sent
	void	inc_pub_messages(int num)
	{
		pub_msgs += num;
	}
	// increment number of uploads
	void   inc_uploads(int num)
	{
		uploads += num;
	};
	int    inactive_delete(void);
	int    inactive_list(void);

	int    lin(void) { return(lines); };
	int    list(int search,int sysop); // generate a list of users
	char   *logname(void) { return(login_name); };

	int    mailavail(void); // is mail available?

	int    prevtimeused() { return(timeused); };
	
	int    save(char *name);// save the current user to the user file
	char	*stringornum(int num);	// return string or number
	int    sysop_edit(void);// sysop's editor for users

	char   *terminal(void) { return(uterm); };
	int	timeleft(void);	
	int	tot_kused(void) { return(total_kused); };

	unsigned int u_acl(void)   
	{ 
		if (tmpacl != acl)  
			return tmpacl; 
		else 
			return acl;
	};
	int    u_downloads(void){ return(downloads); };
	unsigned long u_flags(void)    
	{ 
		if (tmpflags != flags)  
			return tmpflags; 
		else 
			return flags; 
	};
	int    u_logins(void) { return(logins); };
	unsigned int u_timelimit(void)     
	{ 
		if (tmptl!=timelimit) 
			return tmptl; 
		else 
			return timelimit;
	};
	int    u_uploads(void) { return(uploads); };
	int    update_recent_callers(void); // display last caller information
	int    update_callers(void); // display last caller information
	CardRec    *usercard(void) { return(card); };
	char   *user_alias(void) { return(alias); };
	char 	*user_lang(void) { return(lang); };
	time_t user_logon(void) { return(login_time);};
	time_t last_logon(void) { return(last_login);};
};


#endif                          // _USER_H_






