// Filename:   moncon.C
// Contents:   the methods for the monitor connection object
// Author: Greg Shaw
// Created:    8/26/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _MONCON_C_
#define _MONCON_C_

#include "bbshdr.h"


// Method: constructor
// Purpose:    initialize all variables and attempt to connect to sysop process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

moncon::moncon()
{

	cstart = cend = 0;          // setup buffer pointers
	costart = coend = 0;
	watch_on = 0;               // not watching
	mon_connected = 0;
	last_attempt = 0L;          // last attempt = 0;
	locked_out = 0;		// user keyboard not locked out
	outmsg[0] = 0;		// no outgoing message
	lastmsg[0] = 0;		// no previous outgoing message
};


// Method: connected
// Purpose:    return true if monitor connected
// Input:  none
// Output: internal object status is updated
// Author: Greg Shaw
// Created:    8/26/93

int moncon::connected(void)
{
	return(mon_connected);
};


// Method: connect_mon
// Purpose:    try to connect to the sysop monitor program
// Input:  none
// Output: internal object status is updated
// Author: Greg Shaw
// Created:    8/26/93

int moncon::connect_mon(void)
{
	char   watchname[30];
	char   msg[30];
	int    msgid;
	int    tmp;
	time_t now;
	char ttystr[50];
	char *u;
	char sendstr[255];
	char hostname[50];


	time(&now);             // set for next attempt
	// space tries to 10 seconds minimum
	if (abs(now - last_attempt) > MONITOR_CONNECT_DELAY)
	{
		time(&last_attempt);
		if (strcpy(watchname,watchhost()), watchname == NULL)
			return(1);
			// try to open socket
		if (ipc_obj.open_sock(watchname,MONITOR_PORT) == 0)
		{
			last_attempt = 0;
			// Ok.  We're connected.  Let's see what the monitor has for us.
			if (ipc_obj.receive(msg,1) >= 0)
			{
				if (sscanf(msg,"%d %*s",&msgid) != 1)
				{
				switch(msgid)
				{
				case 1:	// poll by server
					if (*outmsg == '\0')
					{
						// send back 'nothing for you' message
						ipc_obj.send(POLL_NO_MESSAGE,1);
					}
					else	// send back message
					{
						ipc_obj.send(POLL_MESSAGE_AVAIL,1);
						// check for valid tty
						if (isatty(fileno(stdout)))
						{
							// get ttyname
							strcpy(ttystr,ttyname(fileno(stdout)));
							if (u = strchr(ttystr,'y'), u != NULL)
							{                   // chop off /dev/tty
								// chop after y
								strcpy(sendstr,++u);
								// copy back
								strcpy(ttystr,sendstr);
							}
							gethostname(hostname,49);
							sprintf(sendstr,"10 %s %s %s %d %ld",user.logname(),hostname,ttystr,user.u_timelimit(),user.user_logon());
							ipc_obj.send(sendstr,1);
							ipc_obj.send(outmsg,1);
						}
						outmsg[0] = 0;
					}
					break;
				case 2:	// return message to server
					// shouldn't happen
					ap_log("Invalid message 02 from monitor.");
					break;
				case 3:	// log user off
					sscanf(msg,"%*d %d",&tmp);
					if (tmp)	// nicely?
					{
						ap_log("Got logoff from SysOp Monitor");
						// temporarily change timlimit to 2 minutes
						// to give the user time to log out nicely
						user.change_tl(2,0);
					}
					else	// with line noise
					{
						ap_log("Got hangup from SysOp Monitor");
						// be nasty.  send fake line noise
						for (tmp = 0; tmp<(rand()%255); tmp++) 
							putchar(rand()%255);
						// bye-bye!
						raise(SIGHUP);
					}
					break;
				case 4:	// lock out keyboard and enter monitoring
					locked_out = 1;
					watch_on = 1;
					break;
				case 5:	// turn on interactive monitoring
					watch_on = 1;
					break;
				case 6:	// turn off interactive monitoring
					locked_out = 0;
					watch_on = 0;
					break;
				case 10:	// update message to monitor	
					// shouldn't happen
					ap_log("Invalid message 10 from monitor.");
					break;
				case 11:	// message to monitor
					// shouldn't happen
					ap_log("Invalid message 11 from monitor.");
					break;
				default:
					ap_log("Invalid message from monitor.");
					break;
				}

				}
			}
		}
	}
	// don't close socket if watching
	if (!watch_on)
		ipc_obj.close_sock(0);
	return(0);
};


// Method: get_char
// Purpose:    attempt to get a character from the monitor process
// Input:  none
// Output: a character, should one be found
// Author: Greg Shaw
// Created:    8/27/93

char moncon::get_char(void)
{
	char   msg[255];            // message from monitor
	unsigned int    x;

	if (watch_on)
	{
		if (cstart != cend)     // empty buffer first
		{
			x = cstart++;
			cstart %= CBUF_MAX;
			return(cbuf[cstart++]);
		}
		if (msg_avail(1,990000))
		{
			// look for command string
			if (ipc_obj.receive(msg,1) == -1)
			{                   // receive failed - socket closed
				mon_connected = 0;
				locked_out = 0;
				watch_on = 0;
			} else
			{
				if (strstr(msg,COMMAND_MSG) != NULL)
				{               // command message
					// 5th byte
					commands[coend++] = msg[4];
					coend %= COMMANDS_MAX;
				}
				else
				{
					for (x=0; x<strlen(msg); x++)
					{
						cbuf[cend++] = msg[x];
						cend %= CBUF_MAX;
						// full?
						if (cstart == cend)
							// disregard
							break;
						//rest of message
					}
					x = cstart++;
					cstart %= CBUF_MAX;
					return(cbuf[x]);
				}
			}
		}
	}
	return(0);
}



// Method: get_command
// Purpose:    attempt to get a command from the monitor process
// Input:  none
// Output: a character, should one be found
// Author: Greg Shaw
// Created:    8/27/93

int moncon::get_command(void)
{
	char   msg[255];            // message from monitor
	unsigned int    x;

	if (watch_on)
	{
		if (costart != coend)   // empty buffer first
		{
			x = costart++;
			costart %= COMMANDS_MAX;
			return(commands[x]);
		}
		if (msg_avail(0,0))
		{
			// look for command string
			if (ipc_obj.receive(msg,1) == -1)
			{                   // receive failed - socket closed
				mon_connected = 0;
				locked_out = 0;		// user keyboard not locked out
				watch_on = 0;
			} else
			{
				if (strstr(msg,COMMAND_MSG) != NULL)
				{               // command message
					return(msg[4]);
				}
				else
				{
					for (x=0; x<strlen(msg); x++)
					{
						cbuf[cend++] = msg[x];
						cend %= CBUF_MAX;
						// full?
						if (cstart == cend)
							// disregard
							break;
						//rest of message
					}
				}
			}
		}
	}
	return(0);
};


// Function:	send_message
// Purpose:	send an update message to the central monitor
// Author:	Gregory Shaw
// Input:	msg - the message to send to the monitor (status)
// Output:	none
// Created:	7/31/95

int moncon::send_message(char *msg)    // send message to monitor master
{
	// message too long?
	if (strlen(msg) >= MONITOR_MESSAGE_MAX - 5)
		msg[MONITOR_MESSAGE_MAX-5] = 0;
	if (!strcmp(msg,lastmsg))	// different?
	{
		return(0);	// no, don't repeat (we already did that)
	}
	strcpy(lastmsg,msg);	// save for later check
	sprintf(outmsg,"10 %s",msg);	// queue message
	return(0);
};


// Method: send_monitor
// Purpose:    send a string to the monitor process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

int moncon::send_monitor(char *str)
{
	if (watch_on)
		if (send(str,1) != 0)     // error?
	{                           // yes - close socket
		watch_on = 0;
		locked_out = 0;		// user keyboard not locked out
		mon_connected = 0;
	}
	return(0);
};


// Method: watching
// Purpose:    return true if sysop monitor is watching this BBS process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

int moncon::watching(void)
{
	return(watch_on);
};

#endif                          // _MONCON_C_
