// Filename:   errlogd.C
// Contents:   the error logger daemon
// Author:     Greg Shaw
// Created:        7/11/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _ERRLOGD_C_
#define _ERRLOGD_C_

#include "bbshdr.h"

#undef DEBUG

#define MAX_ERROR_MESSAGE	512*1024	// largest error message

// declared global so that the 'getout' function can close the socket
bbsipc ipcobj;

// Function:   getout
// Purpose:    get out of the program.  Quick and dirty
// Author: Greg Shaw
// Created:    8/16/93

void getout(int sig)
{
	char   tmpstr[255];         // temporary string
	static int inalready = 0;   // if already in, don't do again
	time_t now;                 // current time
	struct tm *timeinfo;        // time information (broken down)
	char   filestr[255];        // string to output to file
	FILE   *outfile;            // file to append to

	if (sig != SIGUSR1 && !inalready)
	{
		inalready++;
		ipcobj.close_sock(1);   // close for exit
		ipcobj.close_sock(0);   // close for exit
		sprintf(tmpstr,"errlogd: got signal %d.  Exiting.",sig);
		if (outfile = fopen(ERR_LOG_PATH,"a"), outfile != NULL)
		{
			time(&now);         // get current time
			timeinfo = localtime(&now);
			// get formatted time
			strftime(filestr,50,"%D %T: ",timeinfo);
			// add message
			strcat(filestr,tmpstr);
			// append to file
			fprintf(outfile,"%s\n",filestr);
			fclose(outfile);    // close file so that buffers cleared
		}
		exit(0);                // exit to OS
	}
	else // sigusr used for debugging locked daemon
	{
		abort();
	}
}


// Function:   main
// Purpose:        the main calling routine for the error logger daemon
// Input:      socket connections will connect with program.
// Output:     all data sent to this daemon will be logged to a generic file
// Author:     Greg Shaw
// Created:        7/11/93

main()                          // no arguments
{
	struct timeval waittime;
	FILE   *outfile;            // file to append to
	char   msg[MAX_ERROR_MESSAGE];
	time_t now;                 // current time
	int    ret,x;               // return from ipc
	struct tm *timeinfo;        // time information (broken down)
	char   filestr[255];        // string to output to file

#ifdef DEBUG
printf("Init...\r\n");
fflush(stdout);
#endif
	for (x=1; x<16; x++)
		if (x != SIGABRT) // sigabrt used for debugging
			signal(x,&getout);
	// create socket
	if(ipcobj.open_sock (NULL, ERRLOG_PORT) != 0)
	{
		printf("Unable to open server socket.\r\n");
		return (0);
	}
	if (outfile = fopen(ERR_LOG_PATH,"a"), outfile != NULL)
	{
		time(&now);             // get current time
		// get formatted time
		timeinfo = localtime(&now);
		strftime(filestr,50,"%D %T: ",timeinfo);
		// add message
		strcat(filestr,"errlogd (re) started");
		// append to file
		fprintf(outfile,"%s\n",filestr);
		fclose(outfile);        // close file so that buffers cleared
	}
	else
	{
		fprintf(stderr,"errlogd: I am unable to write to %s",ERR_LOG_PATH);
		fflush(stderr);
		exit(0);
	}
	ipcobj.do_connect(1);       // wait for connection
	waittime.tv_usec = 100;     // 100msec
	waittime.tv_sec = 0;
	while(1)                    // loop forever
	{
#ifdef DEBUG
printf("Got connect ... waiting for message.\r\n");
fflush(stdout);
#endif
		if (ret = ipcobj.msg_avail(1,990000), ret < 0)
		{
#ifdef DEBUG
printf("Nothing available.  Closing socket\r\n");
fflush(stdout);
#endif
			// close for exit
			ipcobj.close_sock(1);
#ifdef DEBUG
printf("Disconnect.\r\n");
fflush(stdout);
#endif
			// wait for connection
			ipcobj.do_connect(1);
		}
		else if (ret > 0)
		{
#ifdef DEBUG
printf("Something available.\r\n");
fflush(stdout);
#endif
			// connection closed
			if (ipcobj.receive(msg,1)< 0)
			{
				// close for exit
				ipcobj.close_sock(1);
#ifdef DEBUG
printf("Disconnect.\r\n");
fflush(stdout);
#endif
				// wait for connection
				ipcobj.do_connect(1);
			}
			else                // append to output file
			{
				if (outfile = fopen(ERR_LOG_PATH,"a"), outfile != NULL)
				{
					time(&now); // get current time
					timeinfo = localtime(&now);
					// get formatted time
					strftime(filestr,50,"%D %T: ",timeinfo);
					// add message
					strcat(filestr,msg);
					// append to file
					fprintf(outfile,"%s\n",filestr);
					// close file so that buffers cleared
					fclose(outfile);
#ifdef DEBUG
printf("appended %s.\r\n",filestr);
fflush(stdout);
#endif
				}

#ifdef DEBUG
else
{
	printf("Unable to open file.\r\n");
	fflush(stdout);
}
#endif
			}
		}
		//     select(0,NULL,NULL,NULL,&waittime);// delay
	}
	ipcobj.close_sock(0);       // close for exit
}


#endif







