#include <curses.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#include "inc.h"
#include "status.h"
#include "tokens.h"

WINDOW *msg_w = NULL, *list_w = NULL;

extern char token[1024];

void message(char *, ...);

int list_start, list_lines;
int msg_start, msg_lines;

int first_node;

char *user[MAX_NODES], *speed[MAX_NODES];
char *msg_buf;

char *things[] = {
  "initializing",
  "waiting for call",
  "logging in",
  "online",
  "chatting",
  "running door",
  "paging sysop",
  "shutting down",
  "downloading",
  "uploading"
};

char *reasons[] = {
  "logoff",
  "initialization error",
  "fatal error",
  "terminated",
  "timeout on input",
  "time is up",
  "hangup",
  "segmentation fault",
  "bad login",
  "too many nodes",
  "bad connect",
  "unknown"
};

struct node_t {
  int node, fd, restart, frequency;
  char cmd_line[256];
  pid_t pid;
  struct node_t *next;
} *node_list = NULL, *last_node = NULL;

void error(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  fprintf(stderr, "bbserv: ");
  vfprintf(stderr, fmt, args);
  
  exit(1);
}

struct node_t *add_node()
{
  struct node_t *tmp;

  tmp = (struct node_t *)malloc(sizeof(struct node_t));
  tmp -> next = NULL;

  if(last_node)
    last_node -> next = tmp;

  if(node_list == NULL)
    node_list = tmp;

  last_node = tmp;
  return tmp;
}

void kill_info(int node)
{
  if(user[node])
    free(user[node]);

  if(speed[node])
    free(speed[node]);
  
  user[node] = NULL;
  speed[node] = NULL;
}

void read_config()
{
  FILE *cfg;
  char buf[256];
  struct node_t *node;
  int type;

  cfg = fopen("data/nodes", "r");
  if(cfg == NULL)
    return;
  
  for(;;) {
    type = read_token(cfg);
    if(type == TOKEN_EOF)
      break;

    if(type == TOKEN_COMMENT || type == TOKEN_EOL)
      continue;
    
    if(type != TOKEN_NUM)
      error("error in node list");

    if(fgets(buf, 255, cfg) == NULL)
      error("unexpected EOF in node list");

    if(strlen(buf) <= 1)
      error("error in node list");

    if(buf[strlen(buf) - 1] == '\n') // could have hit EOF
      buf[strlen(buf) - 1] = 0;
    
    node = add_node();
    strcpy(node -> cmd_line, buf);
    node -> node = atoi(token);
  }

  fclose(cfg);
}

void run_node(struct node_t *node)
{
  char buf[256], node_num[5];
  char *args[128];
  int t, f, fd[2];
  
  sprintf(node_num, "%d", node -> node);

  strcpy(buf, node -> cmd_line);
  args[0] = "./os";
  args[1] = "-node";
  args[2] = node_num;

  args[3] = strtok(buf, " \t");

  if(node -> frequency == 0)
    message("Starting node %d", node -> node);
  else
    message("Restarting node %d", node -> node);

  if(isanode(node -> node)) {
    message("Node %d already exists", node -> node);
    node -> pid = 1;
    return;
  }

  t = 4;
  if(args[3]) {
    for(;;t++) {
      args[t] = strtok(NULL, " \t");
      if(args[t] == NULL)
	break;
    }
  }

  if(pipe(fd)) {
    message("Could not create pipe");
    return;
  }

  node -> fd = fd[0];

  f = fork();

  if(f == 0) {
    close(fd[0]);
    dup2(fd[1], 2);
    setsid();
    execv("./bb", args);
    raise(SIGKILL);
  }
  else {
    close(fd[1]);
    node -> pid = f;
  }
}

void get_user(int node)
{
  if(user[node])
    free(user[node]);

  if(!isanode(node) || status_pool -> status[node].busy == BUSY_INIT 
     || status_pool -> status[node].busy == BUSY_WAITING 
     || status_pool -> status[node].busy == BUSY_LOGIN
     || status_pool -> status[node].busy == BUSY_SHUTDOWN) {
    user[node] = NULL;
    return;
  }
  
  if(inc(node, INC_GET, "user"))
    return;

  user[node] = (char *)malloc(64);
  strncpy(user[node], msg_buf, 40);
}

void get_speed(int node)
{
  if(speed[node])
    free(speed[node]);

  if(!isanode(node) || status_pool -> status[node].busy == BUSY_INIT
     || status_pool -> status[node].busy == BUSY_WAITING
     || status_pool -> status[node].busy == BUSY_SHUTDOWN) {
    speed[node] = NULL;
    return;
  }

  if(inc(node, INC_GET, "baudrate"))
    return;

  speed[node] = (char *)malloc(16);
  strncpy(speed[node], msg_buf, 16);
}

void redraw()
{
  int t, y;
  struct status_t *s;

  if(list_w == NULL)
    return;

  mvwprintw(list_w, 0, 0, "Nodes: %d\n", status_pool -> nodes);
  
  y = 1;

  for(t = 0;t < MAX_NODES && y < list_lines;t++) {
    if(user[t] == NULL)
      get_user(t);

    if(speed[t] == NULL)
      get_speed(t);

    if(isanode(t)) {
      s = &(status_pool -> status[t]);
      
      mvwprintw(list_w, y, 0, "%3d: %s", t, things[s -> busy]);
      wclrtoeol(list_w);
      
      if(speed[t])
	mvwprintw(list_w, y, 24, "%6s", speed[t]);

      if(user[t])
	mvwprintw(list_w, y, 32, "%s", user[t]);
      
      y++;
    }
  }
  
  for(;y < list_lines;y++) {
    wmove(list_w, y, 0);
    wclrtoeol(list_w);
  }

  wrefresh(list_w);
}

void message(char *fmt, ...)
{
  va_list args;

  if(msg_w == NULL)
    return;

  va_start(args, fmt);
  scroll(msg_w);
  wmove(msg_w, msg_lines - 1, 0);
  vwprintw(msg_w, fmt, args);
  wclrtoeol(msg_w);
  wrefresh(msg_w);
  va_end(args);
}

void do_errormsg(struct node_t *node)
{
  char buf[256], msg[256];
  int t;
  
  for(;;) {
    for(t = 0;t < 256;t++) {
      if(read(node -> fd, buf + t, 1) <= 0)
	return;
    
      if(buf[t] == '\n')
	break;
    }
  
    if(t == 0)
      return;

    if(buf[t] == '\n')
      buf[t] = 0;

    sprintf(msg, "%d>%s", node -> node, buf);
    message(msg);
  }
}

void check_nodes()
{
  struct node_t *tmp;
  int status, options, exitcode, sig, t;
  pid_t pid;

  options = WNOHANG;

  tmp = node_list;
  while(tmp) {
    pid = tmp -> pid;
    if(pid == 1) {
      tmp = tmp -> next;
      continue;
    }

    if(wait4(pid, &status, options, NULL) == pid) {
      kill_info(tmp -> node);
      
      if(WIFEXITED(status)) {
	exitcode = WEXITSTATUS(status);

	if(exitcode < 0 || exitcode > 10)
	  exitcode = 11;

	message("Node %d has exited (%s)", tmp -> node, reasons[exitcode]);
	close(tmp -> fd);
	tmp -> fd = -1;
	tmp -> pid = 1;

	if(exitcode == 0) {
	  tmp -> frequency = 0;
	  tmp -> restart = 0;
	  run_node(tmp);
	}
	else if(exitcode != 1 && exitcode != 3)
	  run_node(tmp);
	else if(exitcode != 1)
	  tmp -> frequency = 0; // And _stay_ dead.

	if(exitcode == 1) { // init error
	  if(tmp -> frequency == 0) {
	    tmp -> frequency = tmp -> restart = 1;
	    message("Restarting node %d in 1 minute", tmp -> node);
	  }
	  else {
	    if(tmp -> frequency < 16)
	      tmp -> frequency <<= 1;
	    tmp -> restart = tmp -> frequency;
	    message("Restarting node %d in %d minutes", tmp -> node, 
		    tmp -> frequency);
	  }
	}
      } 
      else if(WIFSIGNALED(status)) {
	sig = WTERMSIG(status);
	message("Node %d was killed by signal %d", tmp -> node);
      }
    }

    tmp = tmp -> next;
  }

  for(t = 0;t < MAX_NODES;t++)
    if(!isanode(t))
      kill_info(t);
}

void recalc()
{
  list_start = 0;
  list_lines = LINES >> 1;
  
  msg_start = list_start + list_lines;
  msg_lines = LINES - msg_start;

  if(list_w)
    delwin(list_w);
  
  if(msg_w)
    delwin(msg_w);

  if(msg_lines == 0 || list_lines == 0) {
    msg_w  = NULL;
    list_w = NULL;
    return;
  }

  list_w = newwin(list_lines, COLS, list_start, 0);
  wclear(list_w);
  leaveok(list_w, TRUE);

  msg_w = newwin(msg_lines, COLS, msg_start, 0);
  wclear(msg_w);
  scrollok(msg_w, TRUE);
  leaveok(msg_w, TRUE);
}

int main()
{
  struct node_t *tmp;
  int t, ret, max_fd, seconds = 0;
  fd_set ifd;
  struct timeval tv;
  read_config();

  if(init_pool(1))
    error("error initalizing shared memory pool");

  if(initscr() == NULL)
    error("curses problem");

  for(t = 0;t < MAX_NODES;t++) {
    user[t] = NULL;
    speed[t] = NULL;
  }

  msg_buf = (char *)malloc(1024);

  recalc();

  signal(SIGWINCH, (void (*)(int))recalc);
  
  tmp = node_list;
  while(tmp) {
    tmp -> frequency = 0;
    run_node(tmp);
    tmp = tmp -> next;
  }
  
  for(;;) {
    redraw();
    check_nodes();

    FD_ZERO(&ifd);
    max_fd = 0;

    tv.tv_sec = 0;
    tv.tv_usec = 500000;

    tmp = node_list;
    while(tmp) {
      if(tmp -> fd != -1) {
        FD_SET(tmp -> fd, &ifd);
        if(tmp -> fd > max_fd)
  	  max_fd = tmp -> fd;
      }
      tmp = tmp -> next; 
    }

    if(max_fd) {
      for(;;) {
	ret = select(max_fd + 1, &ifd, NULL, NULL, &tv);
	if(ret == 0)
	  break;
      
	tmp = node_list;
	while(tmp) {
	  if(FD_ISSET(tmp -> fd, &ifd)) {
	    FD_CLR(tmp -> fd, &ifd);
	    do_errormsg(tmp);
	  }
	  tmp = tmp -> next;
	}
      }
    }
    else
      usleep(500000);

    seconds++;
    if(seconds == 120) {
      seconds = 0;
      tmp = node_list;
      while(tmp) {
	if(tmp -> frequency) {
	  tmp -> restart--;
	  if(tmp -> restart == 0)
	    run_node(tmp);
	}
	tmp = tmp -> next;
      }
    }
      
  }

  endwin();
  return 42;
}
