#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <fnmatch.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include "areas.h"
#include "os.h"
#include "doors.h"
#include "lock.h"
#include "modem.h"
#include "output.h"
#include "tokens.h"
#include "varlist.h"

extern char token[1024], *out_buf;
extern varlist list;
extern int ansi;
extern char no_key, yes_key, mark_key, quit_key, cont_key, unmark_key;
extern vmodem *modem;

volatile int sort_changed = 1, areanum;
int areas = 0, num_files;
int show_desc = 1, forever, area_read = -1;
static int line_num, lines, rows;

struct area_t area[MAXAREAS];
struct file_t *first_file = NULL;
struct mark_t *first_mark = NULL;

void mark_files();

static int cmp_filesizes(const void *file1, const void *file2)
{
  return (*(struct file_t **)file1) -> size < 
         (*(struct file_t **)file2) -> size;
}

static int cmp_filenames(const void *file1, const void *file2)
{
  return strcmp((*(struct file_t **)file1) -> list_name, 
                (*(struct file_t **)file2) -> list_name);
}

static int cmp_filedates(const void *file1, const void *file2)
{
  return (*(struct file_t **)file1) -> date < 
         (*(struct file_t **)file2) -> date;
}

void sort_filelist()
{
  struct file_t **tmp_list, *file, *last_file;
  char *method;
  int t, n;
  int (*sort_func)(const void *, const void *);
  
  if(first_file == NULL)
    return;

  method = string("filesort");
  if(!strcmp(method, "none"))
    return;
  
  tmp_list = (struct file_t **)malloc(sizeof(struct file_t) * num_files);
  if(tmp_list == NULL)
    fatal_error("no memory for sort list");
  
  t = 0;
  file = first_file;
  
  while(file) {
    tmp_list[t++] = file;
    file = file -> next;
  }

  if(!strcmp(method, "bysize"))
    sort_func = cmp_filesizes;
  else if(!strcmp(method, "bydate"))
    sort_func = cmp_filedates;
  else
    sort_func = cmp_filenames;

  qsort(tmp_list, t, sizeof(struct file_t *), sort_func);  

  for(n = 0;n < t;n++) {
    if(n == 0)
      last_file = first_file = tmp_list[n];
    else
      last_file -> next = tmp_list[n];
    last_file = tmp_list[n];
    last_file -> next = NULL;  
  }

  free(tmp_list);
}

int is_marked(struct file_t *file)
{
  struct mark_t *mark = first_mark;
   
  while(mark) {
    if(!strcmp(file -> name, mark -> name))
      return 1;
    mark = mark -> next;
  }
   
  return 0;
}

void unmark_file(struct file_t *file) 
{
  struct mark_t *mark = first_mark, *prev = NULL;
   
  while(mark) {
    if(!strcmp(file -> name, mark -> name))
      break;
    prev = mark;
    mark = mark -> next;
  }
  
  if(!mark)
    return;
   
  if(prev)
    prev -> next = mark -> next;
  else
    first_mark = mark -> next;

  list["markedsize"] -> i -= file -> size;
  list["marked"] -> i--;
   
  free(mark -> name);
  free(mark);
  file -> marked = 0;
}

void mark_file(struct file_t *file) 
{
  struct mark_t *mark, *tmp;
   
  mark = (struct mark_t *)malloc(sizeof(struct mark_t));
  mark -> name = (char *)malloc(strlen(file -> name) + 1);
  strcpy(mark -> name, file -> name);
  mark -> next = NULL;
  mark -> size = file -> size;
   
  if(!first_mark)
    first_mark = mark;
  else {
    for(tmp = first_mark;tmp -> next;tmp = tmp -> next);
    tmp -> next = mark;
  }
   
  list["markedsize"] -> i += file -> size;
  list["marked"] -> i++;
  file -> marked = 1;
}

void kill_marks()
{
  struct mark_t *mark = first_mark, *next;
  struct file_t *file = first_file;
   
  while(mark) {
    next = mark -> next;
    free(mark -> name);
    free(mark);
    mark = next;
  }

  while(file) {
    file -> marked = 0;
    file = file -> next;
  }
   
  // Obscure bugs? Who, me?
  *list["marked"] = 0; 
  *list["markedsize"] = 0;
   
  first_mark = NULL;
}

int get_size(char *file) 
{
  struct stat st;
   
  if(stat(file, &st))
    return 0;
  else
    return st.st_size;
}

void get_info(struct file_t *file)
{
  struct stat st;
  
  if(stat(file -> name, &st))
    file -> size = 0;
  else {
    file -> size = st.st_size;
    file -> date = st.st_mtime;
  }
}

void kill_filelist()
{
  struct file_t *tmp = first_file, *next;
   
  while(tmp) {
    next = tmp -> next;
    free(tmp -> name);
    if(area[areanum].type == AREA_SUNSITE)
      free(tmp -> desc_txt);
    free(tmp);
    tmp = next;
  }

  num_files = 0;
  first_file = NULL;
}

void init_areas()
{
  int t, type;
  FILE *arealist;
   
  dprintf(">loading area list: ");

  for(t = 0;t < MAXAREAS;t++)
    area[t].name = NULL;
   
  arealist = fopen("/osbbs/data/areas", "r");
  if(!arealist) {
    perror("fopen");
    init_error("error opening /osbbs/data/areas");
  }
      
  for(;;) {
    type = read_line(arealist);
    if(type == TOKEN_EOF)
      break;
        
    if(type != TOKEN_TEXT)
      init_error("error in arealist: area tag expected");
    area[areas].tag = (char *)malloc(strlen(token) + 1);
    strcpy(area[areas].tag, token);
      
    type = read_line(arealist);
    if(type != TOKEN_TEXT)
      init_error("error in arealist: area name expected");
  
    area[areas].name = (char *)malloc(strlen(token) + 1);
    strcpy(area[areas].name, token);
     
    type = read_line(arealist);
    if(type != TOKEN_TEXT)
      init_error("error in arealist: index expected");
    
    area[areas].index = (char *)malloc(strlen(token) + 1);
    strcpy(area[areas].index, token);
     
    type = read_line(arealist);
    if(type != TOKEN_TEXT)
      init_error("error in arealist: format expected");
     
    if(!strcmp(token, "native"))
      area[areas].type = AREA_NATIVE;
    else if(!strcmp(token, "sunsite"))
      area[areas].type = AREA_SUNSITE;
    else if(!strcmp(token, "unix"))
      area[areas].type = AREA_UNIX;
    else
      init_error("error in arealist: unknown format \"%s\", token");
     
    areas++;
    if(areas == MAXAREAS)
      init_error("too many areas");
  }   
   
  dprintf("%d areas\n", areas);
  fclose(arealist);

  list.add_sys("areaname", areas ? area[0].name : "none");
  list.add_sys("areanum", 0);
  
  if(area[0].type == AREA_NATIVE)
    list.add_sys("areatype", "native");
  else if(area[0].type == AREA_SUNSITE)
    list.add_sys("areatype", "sunsite");
  else 
    list.add_sys("areatype", "unix");

  list.add_sys("areatag", areas ? area[0].tag : "none");
  list.add_sys("marked", 0);
  list.add_sys("markedsize", 0);
}

void read_native_list(FILE *index)
{
  int type, dir_length;
  struct file_t *file, *last_file;
  char dir[256], *tmp;
   
  kill_filelist(); 
  type = read_token(index);   
  
  strncpy(dir, area[areanum].index, 256);
  tmp = strrchr(dir, '/');
  if(tmp)
    *(tmp + 1) = 0;
  
  dir_length = strlen(dir); 
     
  for(;;) { 
    if(type == TOKEN_EOF)
      break;

    last_file = file;
    file = (struct file_t *)malloc(sizeof(struct file_t));
    if(!first_file)
      first_file = file;
    else
      last_file -> next = file;
     
    file -> next = NULL;
    
    if(*token != '/') {
      file -> name = (char *)malloc(strlen(token) + dir_length + 1);
      sprintf(file -> name, "%s%s", dir, token);
      file -> list_name = file -> name + dir_length;
    } 
    else {
      file -> name = (char *)malloc(strlen(token) + 1);
      sprintf(file -> name, "%s", token);
      file -> list_name = strrchr(file -> name, '/') + 1;
    }
     
    dprintf(">rnl: f->n=%s\n", file -> name);
     
    //file -> size = get_size(file -> name);
    get_info(file); 
     
    skip_line(index);
    file -> desc = ftell(index);

    file -> marked = is_marked(file);
    
    for(;;) {
      type = read_token(index);
      if(type == TOKEN_TEXT && *token == '*')
	skip_line(index);
      else
	break;
    }
    num_files++;
  }
}

void read_sunsite_list(FILE *index)
{
  int type, t, c, dir_length;
  struct file_t *file, *last_file;
  char dir[500], ftype;
    
  kill_filelist();

  strcpy(dir, area[areanum].index);
  *(strrchr(dir, '/') + 1) = 0;
  
  dir_length = strlen(dir);
   
  for(t = 0;t < 2;t++)
    do { 
    } while(read_token(index) != TOKEN_EOL);
   
  for(;;) {
    type = read_token(index);
    if(type == TOKEN_EOF)
      break;

    ftype = token[strlen(token) - 1];
     
    if(ftype == '/' || ftype == '@') {
      skip_line(index);
      continue;
    }
     
    file = (struct file_t *)malloc(sizeof(struct file_t));
    file -> next = NULL;
    
    file -> name = (char *)malloc(strlen(token) + dir_length + 1);
    sprintf(file -> name, "%s%s", dir, token);
    file -> list_name = file -> name + dir_length;

    //file -> size = get_size(file -> name);
    get_info(file);
    if(!file -> size) {
      free(file -> name);
      free(file);
      skip_line(index);
      continue;
    }
    
    if(!first_file) {
      first_file = file;
      last_file = first_file;
    }
    else
      last_file -> next = file;
    
    last_file = file;
     
    do {
    } while((c = fgetc(index)) == ' ');

    if(c != '\n' && c != EOF) {
      token[0] = c;
      fgets(token + 1, 1024, index);
    }
    else
      strcpy(token, "(No description)\n");
     
    file -> desc_txt = (char *)malloc(strlen(token) + 1);
    strcpy(file -> desc_txt, token);

    file -> marked = is_marked(file);
     
    num_files++;
  }
}

void read_unix_list() 
{
  DIR *dir;
  char *dirname;
  struct dirent *entry;
  struct file_t *file, *last_file = NULL;
  struct stat st;
  
  num_files = 0;
  dirname = area[areanum].index;
  
  dir = opendir(dirname);
  if(!dir) {
    dprintf(">error opening dir %s\n", dirname);
    return;
  }

// Wheee! Look, no gotos!
  while(entry = readdir(dir)) {
    file = (struct file_t *)malloc(sizeof(struct file_t));
    if(first_file == NULL)
      first_file = file;
    file -> next = NULL;
    file -> name = (char *)malloc(strlen(entry -> d_name) 
                                  + strlen(dirname) + 2); //  '/' + '\0'
    sprintf(file -> name, "%s/%s", dirname, entry -> d_name);
    file -> list_name = strrchr(file -> name, '/') + 1;
    file -> marked = 0;
    
    if(stat(file -> name, &st) || (st.st_mode & S_IFDIR)) {
      free(file -> name);
      if(first_file == file)
        first_file = NULL;
      free(file);
      continue;
    }
    
    if(last_file != NULL)
      last_file -> next = file;
    
    file -> size = st.st_size;
    file -> date = st.st_mtime;
    num_files++;
    last_file = file;
  }
  
  closedir(dir);
}

void mark_files(char unmark)
{
  int t, n, length, width;
  char *buf, *tmp = NULL;
  struct file_t *file;
   
  cook(unmark ? string("unmarkprompt") : string("markprompt"));
  length = ansi_length(out_buf);
  output_raw(out_buf);
  width = number("rows") - length;
  buf = (char *)malloc(width);
  kill_line(length + input_alpha(buf, width, 0));
   
  for(;;) {
    if(tmp == NULL)
      tmp = strtok(buf, " ,");
    else
      tmp = strtok(NULL, " ,");
     
    if(tmp == NULL)
      break;

    file = first_file;
     
    if(isdigit(*tmp) && (n = atoi(tmp))) {
      n--;
      if(n < num_files) {
        for(t = 0;t < n;t++)
	  file = file -> next;
	if(unmark) 
	  unmark_file(file);
	else 
	  mark_file(file);
      }
    }
    else {
      for(t = 0;t < num_files;t++) {
        if(!strcasecmp(file -> list_name, tmp) 
	   || !fnmatch(tmp, file -> list_name, 0)) {
	  if(unmark)
	    unmark_file(file);
	  else
	    mark_file(file);
        }
        file = file -> next;
      }
    }   
  }
      
  free(buf);
}

int file_prompt1()
{
  char c;
  int plen;
  
  for(;;) {
    plen = output_cooked(string("fileprompt1"));  
    flush();
    c = tolower(read_char());
    if(c == no_key)
      return 1;
    else if(c == mark_key) {
      kill_line(plen);
      mark_files(0);
    }
    else if(c == unmark_key) {
      kill_line(plen);
      mark_files(1);
    }
    else 
      return 0;
  }
}

int print_file(struct file_t *file, FILE *index)
{
  int c;
   
  if(file -> marked)
    cook(string("markedfmt"));
  else 
    cook(string("filefmt"));
  
  output_raw(out_buf);
  output_cooked("\n");
  line_num++;
   
  if(area[areanum].type == AREA_SUNSITE) {
    output_raw(file -> desc_txt); 
    line_num++;
  }
  
  if(area[areanum].type == AREA_NATIVE) {
    fseek(index, file -> desc, SEEK_SET);
    for(;;) {
      c = fgetc(index);
      if(c != '*') {
	ungetc(c, index);
	break;
      }
      fgets(token, rows, index);
      output_cooked(token);
      line_num++;
      if(line_num >= lines - 2) {
        if(!file -> next)
	  return 0;
	else if(file_prompt1())
	  return 1;
	else {
	  output_cooked("\\c");
	  line_num = 0;
	}
      }
       
    }
  }
   
  if(line_num >= lines - 2) { 
    if(!file -> next)
      return 0;
    else if(file_prompt1())
      return 1;
    else {
      output_cooked("\\c");
      line_num = 0;
    }
  }
   
  return 0;
}

void file_prompt2()
{
  char c;
  int plen;

  for(;;) {
    plen = output_cooked(string("fileprompt2"));
    flush();
    c = tolower(read_char());
    if(c == mark_key) {
      kill_line(plen);
      mark_files(0);
    }
    else if(c == unmark_key) {
      kill_line(plen);
      mark_files(1);
    }
    else
      break;
  }
}

int list_files(int since)
{
  FILE *index;
  variable *vname, *vsize, *vnum, *vdate;
  struct file_t *file;
  int n = 1, quit = 0, type;

  lines = number("lines");
  rows = number("rows");
  type = area[areanum].type;

  if(type != AREA_UNIX) { 
    index = fopen(area[areanum].index, "r");
    if(!index)
      return 1;
  }

  if(area_read != areanum) {
    if(type == AREA_NATIVE)
      read_native_list(index);
    else if(type == AREA_SUNSITE)
      read_sunsite_list(index);
    else if(type == AREA_UNIX)
      read_unix_list();
    if(num_files == 0) {
      if(type != AREA_UNIX)     
        fclose(index);
      return 1;
    }
    sort_changed = 1;
  }

  if(since) {
    file = first_file;
    while(file) {
      if(file -> date > since)
        break;
      file = file -> next;
    }
    if(file == NULL)
      return 1;
  }

  if(sort_changed) {
    sort_changed = 0;
    sort_filelist();
  }

  area_read = areanum;
  file = first_file;
   
  vname = list["filename"];
  vsize = list["filesize"];
  vnum  = list["filenum"];
  vdate = list["filedate"];
  
  if(ansi)
    output_cooked("\\c");

  line_num = 0;
   
  for(n = 1;file;n++) {
    if(file -> date < since) {
      file = file -> next;
      continue;
    }
    
    *vname = file -> list_name;
    *vsize = file -> size;
    *vnum = n;
    *vdate = format_date(file -> date);
    
    if(print_file(file, index)) {
      quit = 1;
      break;
    }
     
    file = file -> next;
  }

  if(!quit && line_num)
    file_prompt2();
   
  if(type != AREA_UNIX)
    fclose(index);
  return 0;
}
