/* extract.c - extract data from vt_info
 *
 * $Id: extract.c,v 1.1.1.1 1999/12/02 20:00:58 ivarch Exp $
 */

#include "Internal.h"


/* Put the size of "vt" into "rows" and "cols". Returns nonzero on error.
 */
int vt100_get_size (vt_info vt, int * rows, int * cols) {
  if ((!vt) || (!rows) || (!cols)) return (1);

  *rows = vt->rows;
  *cols = vt->cols;

  return (0);
}


/* Put the cursor position of "vt" into "x" and "y". Returns nonzero on error.
 *
 * Note that "x" may be greater than vt->cols - if so, it just means that the
 * line has been filled, and that any more characters will be added on the
 * next line.
 */
int vt100_get_pos (vt_info vt, int * x, int * y) {
  if ((!vt) || (!x) || (!y)) return (1);

  *x = vt->x;
  *y = vt->y;

  return (0);
}


/* Put the new cursor position - that we're about to move to - of "vt" into
 * "x" and "y". Returns nonzero on error.
 *
 * Note that "x" may be greater than vt->cols - if so, it just means that the
 * line has been filled, and that any more characters will be added on the
 * next line.
 *
 * This function should only be used inside a VT_ACTION_GOTO callback, to see
 * where the cursor's about to go.
 */
int vt100_get_newpos (vt_info vt, int * x, int * y) {
  if ((!vt) || (!x) || (!y)) return (1);

  *x = vt->new_x;
  *y = vt->new_y;

  return (0);
}


/* Get up to "len" bytes of text from "vt", starting at (x,y) (origin is at
 * (1,1) at the top left). Stops at the end of a line.
 *
 * Returns the number of bytes read, or negative on error.
 *
 * Out-of-range (x,y) is an error, but outsized "len" is not - the output is
 * just truncated.
 *
 * Note that "len"+1 bytes should be available in "buf", because the string
 * is null terminated.
 */
int vt100_get_text (vt_info vt, unsigned char * buf, int x, int y, int len) {
  int end, num;

  if ((!vt) || (!buf)) return (-1);
  if ((x < 1) || (y < 1) || (x > vt->cols) || (y > vt->rows)) return (-1);

  end = x + len;
  if (end > vt->cols) end = vt->cols;

  for (num = 0; x <= end; x ++) {
    buf[num ++] = vt->text[((y-1) * (vt->cols)) + (x-1)];
  }

  buf[num] = 0;

  return (num);
}


/* Get up to "len" short integers of attributes from "vt", starting at (x,y)
 * (origin is at (1,1) at the top left). Stops at the end of a line.
 *
 * Returns the number of shorts read, or negative on error.
 *
 * Out-of-range (x,y) is an error, but outsized "len" is not - the output is
 * just truncated.
 *
 * Note that unlike vt100_get_text, the output is not null terminated, and
 * remember that the return value is a count of values read, NOT the number
 * of bytes.
 */
int vt100_get_attr (vt_info vt, unsigned short * buf, int x, int y, int len) {
  int end, num;

  if ((!vt) || (!buf)) return (-1);
  if ((x < 1) || (y < 1) || (x > vt->cols) || (y > vt->rows)) return (-1);

  end = x + len;
  if (end > vt->cols) end = vt->cols;

  for (num = 0; x < end; x ++) {
    buf[num ++] = vt->attrs[((y-1) * (vt->cols)) + (x-1)];
  }

  return (num);
}


/* Return a pointer to "vt"'s xterm title string, which should be null
 * terminated.
 *
 * Returns 0 on error.
 */
char * vt100_get_xtitle (vt_info vt) {
  if (!vt) return (0);
  return (vt->xtitle);
}


/* Return the current clear type, as used by vt100__clear(). 12 means the
 * whole screen is to be cleared.
 *
 * This should only be called inside a VT_ACTION_CLEAR callback.
 */
int vt100_get_clrtype (vt_info vt) {
  return (vt->par[0]);
}

/* EOF */
