import os
import traceback
import Global
from Utils import *
from xml.dom.minidom import parse, parseString

class DropFileTypes:
    """
    Doors generally expect to see a 'drop-file' with the user name and other handy info.
    """
    NoDrop = "none"
    #Droplet = 0 # our own teeny format
    DoorSys = "door.sys" # used by GAP and many others 
    CallInfoBBS = "callinfo.bbs" # Used by ???
    DORINFO = "dorinfo" # Used by RBBS
    
class ExecutionTypes:
    """
    How are we wrapping the door?
    Note that DOS programs can communicate with serial ports by (A) using the serial port's IRQ, or by
    (B) using interrupt 14H.  Method (B) can be handled by FOSSIL drivers; this is a more recent, and
    generally better, way to access the modem.  However, old doors like FoodFite use method (A), so in
    order to run them, you need some sort of virtual modem (like Com/IP on Windows XP, or ViP on Windows 9x).  
    """
    Pipes = "pipes" # Run using stdin/stdout/stderr piping
    COM = "com" # Run using a com-port-to-IP redirector (works poorly, no freeware one available)
    FOSSIL = "fossil" # Run a process, relying on FOSSIL to communicate
    Plugin = "plugin" # Run a Python function

class DoorClass:
    "A game, such as Operation Overkill...or a BBS function, like the message board"
    def __init__(self):
        self.Name = None #req'd
        self.ID = None # internally assigned
        # Set the nickname if you're running several instances of one door (e.g. an OO tournament, and
        # a standard OO instance)
        self.Nickname = None
        self.Version = None
        self.DropFileType = DropFileTypes.NoDrop
        self.Dir = None  
        self.DropFileDir = None
        self.RunCommand = None #req'd
        self.ExecutionType = ExecutionTypes.Pipes
        self.NoUserFlag = 0 # TRUE if this door can run without a user record
        self.NodeCount = 1 # Number of simultaneous users (0 for unlimited)
        self.PluginModuleName = None
        self.PluginFunctionName = None
        self.SkipStatReporting = 0 # TRUE if this is an "internal" door, not reported in user stats.
    def GetDir(self):
        Dir = self.Dir
        if Dir==None:
            Dir = ""
        Dir = Dir.replace("/", "\\")
        return Dir
    def GetCommand(self):
        return self.RunCommand
    def IsValid(self):
        "Cleans and validates data"
        if not self.Name:
            raise ValueError, "Nameless door"
        if not self.ExecutionType:
            raise ValueError, "No execution type for door '%s'"%self
        self.ExecutionType = self.ExecutionType.lower()
        if self.ExecutionType == ExecutionTypes.Plugin:
            if not self.PluginModuleName:
                raise ValueError, "Plugin door '%s' has no module defined."%self
            if not self.PluginFunctionName:
                self.PluginFunctionName = self.PluginModuleName # default: same as module name
        else:
            if not self.RunCommand:
                raise ValueError, "Door %s has no command defined."%self
        self.ForceToInteger("NodeCount", 1)
        self.ForceToInteger("SkipStatReporting", 0)
        return 1
    def ForceToInteger(self, Attribute, Default):
        Value = getattr(self, Attribute)
        if Value == None:
            setattr(self, Attribute, Default)
        else:
            try:
                setattr(self, Attribute, int(Value))
            except:
                raise ValueError, "Door %s has invalid %s '%s'"%(self, Attribute, Value)
            
    def __str__(self):
        if self.Version and self.Nickname:
            return "%s %s (%s)"%(self.Name, self.Version, self.Nickname)
        if self.Nickname:
            return "%s (%s)"%(self.Name, self.Nickname)
        return self.Name
    
class DoorDatabaseClass:
    def __init__(self):
        self.Doors = []
        self.NextDoorID = 1
        self.LoginDoor = None # where we go if there's no user
        self.DefaultDoor = None # the "main menu", or the only door
    def Load(self):
        try:
            self.LoadInternal()
        except:
            print "Error loading door database.  Traceback follows:"
            traceback.print_exc()
    def GetByName(self, DoorName):
        DoorName = DoorName.strip().lower()
        # First, try to find a matching nickname:
        #print "--SEARCH"
        for Door in self.Doors:
            #print "'%s' '%s' '%s'"%(Door.Nickname, Door.Name, DoorName)
            if Door.Nickname and Door.Nickname.strip().lower() == DoorName:
                return Door
        # Now just go for a name match:
        for Door in self.Doors:
            if Door.Name.strip().lower() == DoorName:
                return Door
        #print "--NO MATCH!"
        return None
    def LoadInternal(self):
        DOM = parse("Doors.xml")
        # Structure:
        # root <dumple> node contains a list of <door> nodes
        Dumple = DOM.getElementsByTagName("Dumple")[0]
        for DoorNode in Dumple.getElementsByTagName("Door"):
            Door = self.LoadDoor(DoorNode)
            if Door:
                Door.ID = self.NextDoorID
                self.NextDoorID += 1
                self.Doors.append(Door)
        #############################################
        # Default door and login door must exist:
        # Check to see if it's specified in a tag, first:
        Nodes = DOM.getElementsByTagName("LoginDoor")
        if Nodes:
            Name = self.GetXMLText(Nodes[0])
            if Name:
                self.LoginDoor = self.GetByName(Name)
        Nodes = DOM.getElementsByTagName("DefaultDoor")
        if Nodes:
            Name = self.GetXMLText(Nodes[0])
            if Name:
                self.DefaultDoor = self.GetByName(Name)
        if not self.Doors:
            raise ValueError, "** Warning: No doors are defined.  This won't be an exciting place to visit."
        # If login and default door aren't set, use the first from the list:
        if not self.LoginDoor:
            for Door in self.Doors:
                if Door.NoUserFlag:
                    self.LoginDoor = Door
                    break
            if not self.LoginDoor: # %%% fallback: hard-coded ones
                raise ValueError,"No login door is available; users cannot connect!  Set at least one door with NoUserFlag set"
        if not self.DefaultDoor:
            self.DefaultDoor = self.Doors[0]
        
    def LoadDoor(self, Node):        
        # Structure:
        # Sub-tags for each data field
        Door = DoorClass()
        self.LoadDoorTag(Node, Door, "Name", "Name")
        self.LoadDoorTag(Node, Door, "Version", "Version")
        self.LoadDoorTag(Node, Door, "DropFileType", "DropFileType")
        self.LoadDoorTag(Node, Door, "Nickname", "Nickname")
        self.LoadDoorTag(Node, Door, "Type", "ExecutionType")
        self.LoadDoorTag(Node, Door, "DropFile", "DropFileType")
        self.LoadDoorTag(Node, Door, "NoUser", "NoUserFlag")
        self.LoadDoorTag(Node, Door, "Dir", "Dir")
        self.LoadDoorTag(Node, Door, "Command", "RunCommand")
        self.LoadDoorTag(Node, Door, "Module", "PluginModuleName")
        self.LoadDoorTag(Node, Door, "Function", "PluginFunctionName")
        self.LoadDoorTag(Node, Door, "NodeCount", "NodeCount", 1)
        self.LoadDoorTag(Node, Door, "SkipStats", "SkipStatReporting", 0)
        if Door.IsValid():
            return Door
    def LoadDoorTag(self, Node, Door, TagName, AttributeName, Default = None, TranslationDict = None):
        Tags = Node.getElementsByTagName(TagName)
        if not Tags:
            Value = Default
        else:
            Value = GetXMLText(Tags[0])
            if TranslationDict:
                Value = TranslationDict.get(Value.lower(), None)
        setattr(Door, AttributeName, Value)
    def GetXMLText(self, Node):
        rc = ""
        for SubNode in Node.childNodes:
            if SubNode.nodeType == SubNode.TEXT_NODE:
                rc = rc + SubNode.data
        return rc
    def GetDoorList(self):
        return self.Doors
    
Global.DoorDatabase = DoorDatabaseClass()
Global.DoorDatabase.Load()
