/***************************************************************************
 *   Copyright (C) 2004-2009 by Michael Griffin                            *
 *   mrmisticismo@hotmail.com                                              *
 *                                                                         *
 *   Purpose: Working on IPC Node Chat                                     *
 *                                                                         *
 *   Node Messaging Modeled after Day Dream particially.                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

// Enthral SVN: $Id: console.cpp 152 2010-04-10 10:53:59Z mercyful $
// Source: $HeadURL: http://svn.enthralbbs.com/trunk/src/console.cpp $
// $LastChangedDate: 2010-04-10 06:53:59 -0400 (Sat, 10 Apr 2010) $
// $LastChangedRevision: 152 $
// $LastChangedBy: mercyful $

#include <errno.h>
#include <fcntl.h>
#include <signal.h>

#include <cstdio>
#include <cstdlib>

//#include <stdlib.h>
//#include <string.h>

#include <sys/types.h>  //mkfifo gcc 3.x
#include <sys/stat.h>   //mkfifo gcc 3.x

#include <sys/socket.h>
#include <sys/un.h>

//#include <unistd.h>
//#include <syslog.h>
//#include <sys/ioctl.h>
#include <termios.h>
//#include <sys/wait.h>

#include "console.h"
#include "struct.h"

static int conin = 0;
static int conout = 0;
static int conon = 0;
static int dummyfd = 0;

int sockfd;
int serhandle;
struct List *olms;

static struct termios oldtty;
static struct sockaddr_un sock;

static void create_internode_socket(void) {

    char socket_name[4096];
    if ((sockfd = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) {
        perror("/tmp/enthral ***cannot create communication socket, check permissions!");
    }
    snprintf(socket_name, sizeof socket_name, "%s/enthral_sock%d", ENTHRALTMP, NODE_NUM);
    unlink(socket_name);

    strncpy(sock.sun_path, socket_name, sizeof sock.sun_path);
    sock.sun_path[sizeof sock.sun_path - 1] = 0;
    sock.sun_family = AF_UNIX;
    if (bind(sockfd, (struct sockaddr *) &sock, sizeof sock) < 0) {
        perror("/tmp/enthral ***cannot bind communication socket, check permissions!");
        close(sockfd);
    }
}

void clear_nodes(void) {

    char buff[512];

    tcsetattr(0, TCSANOW, &oldtty);

    finalize_console();
    snprintf(buff, sizeof(buff), "%s/enthral%dr", ENTHRALTMP, NODE_NUM);
    unlink(buff);
    close(sockfd);
    snprintf(buff, sizeof(buff), "%s/enthral_sock%d", ENTHRALTMP, NODE_NUM);
    unlink(buff);
    snprintf(buff, sizeof(buff), "%s/enthral%dw", ENTHRALTMP, NODE_NUM);
    unlink(buff);
    snprintf(buff, sizeof(buff), "%s/nodeinfo%d.data", ENTHRALTMP, NODE_NUM);
    unlink(buff);

}

int init_nodes(void) {

    serhandle = open(ttyname(0), O_RDWR);

    // Create communication fifos for Snoop
    create_internode_socket();
    if (init_console() == -1) {
        perror("/tmp/enthral ***communication socket(s) failed to init, check permissions!");
    }

    atexit(clear_nodes);
    return 1;
}

/* mode = 0 -> blocking, 1 -> non-blocking */
int set_blocking_mode(int fd, int mode) {

    int fl;
    if ((fl = fcntl(fd, F_GETFL)) == -1)
        return -1;
    fl &= ~O_NONBLOCK;
    return fcntl(fd, F_SETFL, fl | (mode ? O_NONBLOCK : 0));
}

ssize_t safe_read(int fd, void *buf, size_t buflen) {

    char *p;
    ssize_t bread, n;

    bread = 0;
    p = (char *) buf;
    while (buflen) {
        if ((n = read(fd, p, buflen)) == -1) {
            if (errno == EINTR)
                continue;
            return -1;
        }
        p += n;
        bread += n;
        buflen -= n;
    }

    return bread;
}

ssize_t safe_write(int fd, const void *buf, size_t buflen) {

    char *p;
    ssize_t bwrite, n;

    bwrite = 0;
    p = (char *) buf;
    while (buflen) {
        if ((n = write(fd, p, buflen)) == -1) {
            if (errno == EINTR)
                continue;
            return -1;
        }
        p += n;
        bwrite += n;
        buflen -= n;
    }

    return bwrite;
}


int console_active(void) {
    return conon != 0;
}

int init_console(void) {

    char buffer[4096];
    struct sigaction sigact;
    sigset_t sigset;

    /* Writing to pipe with no readers on the other side raises
     * SIGPIPE and its default action is to terminate the program.
     */
    sigact.sa_handler = SIG_IGN;
    sigemptyset(&sigset);
    sigact.sa_mask = sigset;
    sigact.sa_flags = SA_RESTART;
    if (sigaction(SIGPIPE, &sigact, NULL) == -1)
        abort();

    snprintf(buffer, sizeof buffer, "%s/enthral%dw", ENTHRALTMP, NODE_NUM);
    unlink(buffer);
    if (mkfifo(buffer, 0777) == -1) {
        //syslog(LOG_ERR, "cannot mkfifo(\"%.200s\"): %m", buffer);
        fputs("Cannot create communication FIFO\r\n", stderr);
        exit(1);
    }

    /* We must first open the writing FIFO as O_RDONLY, if we're
     * ever going to get it opened as O_WRONLY. Without this trick
     * open() would always return ENXIO.
     *
     * We can still know whether there are no readers on the other
     * side of pipe, since safe_write() will return EPIPE on such case.
     */
    if ((dummyfd = open(buffer, O_RDONLY | O_NONBLOCK)) == -1 ||
        (conout = open(buffer, O_WRONLY | O_NONBLOCK)) == -1)
        abort();

    close(dummyfd);
    set_blocking_mode(conout, 0);

    snprintf(buffer, sizeof(buffer), "%s/enthral%dr", ENTHRALTMP, NODE_NUM);
    unlink(buffer);
    if (mkfifo(buffer, 0777) == -1) {
        perror("/tmp/enthral ***cannot create communication FIFO socket (Nodes), check permissions!");
        exit(1);
    }

    /* Opening a FIFO for O_RDONLY in non blocking mode should
     * work on all systems.
     */
    if ((conin = open(buffer, O_RDONLY | O_NONBLOCK)) == -1 ||
        (dummyfd = open(buffer, O_WRONLY | O_NONBLOCK)) == -1)
        abort();

    set_blocking_mode(conin, 0);

    return 0;
}

void finalize_console(void) {
    close(conin);
    close(conout);
    close(dummyfd);
    conon = 0;
}

void open_console(void) {

    if (conon != 2)
        conon = 1;
}

void close_console(void) {

    if (conon != 2)
        conon = 0;
}

int console_select_input(int maxfd, fd_set *set) {

    FD_SET(conin, set);
    return maxfd < conin ? conin : maxfd;
}

int console_pending_input(fd_set *set) {
    return FD_ISSET(conin, set);
}

int console_getc(void) {

    unsigned char ch;

    switch (read(conin, &ch, 1)) {
    case 0:
        return EOF;
    case -1:
        if (errno == EPIPE) {
            conon = 0;
            return EOF;
        }
        return -1;
    default:
        return ch;
    }
}

int console_putsn(void *str, size_t n) {
    int writecnt = 0;

    // Write to console first for Telnet Users Connection
    writecnt = write(1,(char*)str,n); //- - Normal Output for Connection.

    // Safe Write to Sockets for Snooping IPC Connections locally.
    // - Local connection For snoop sysop utils.
    writecnt = safe_write(conout, str, n);
    if (writecnt == -1 && errno == EPIPE) {
        return 0;
    }
    else{
        return writecnt;
    }
}
