<?php
/*
 * ----------------------------------------------------------------------------
 * VADV-PHP_Dat.inc.php                                              07/30/2006
 * This script will read the vadv-php.dat file for the specified user and store
 * its information into an array. This data file is not created by VADV, but is
 * a VADV-PHP settings file that stores various user preferences.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* VADV-PHP_Dat.inc.php :: GetUserSettings()
*
* This function will get the requested entry information from vadv-php.dat.
*
* @param integer $userNo
* @return array
*/
function GetUserSettings($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    global $CONFIG;

    // Defaults
    $default['hidedeleted']     = 0;
    $default['hidesentdeleted'] = 0;
    $default['timezone']        = (integer) date("O");
    $default['htmlencode']      = 0;
    $default['tagline']         = 0;
    $default['newmessagescan']  = 0;    // Reverse 0 = true, 1 = false
    $default['password']        = '';

    if (($userNo <= 0) || ($userNo == $CONFIG['guestacct'])) {
        return $default;
    }

    $fileString = ReadUserSettings();

    $user   = array();
    $offset = 100 * ($userNo - 1);
    $user['hidedeleted']     = GetInteger($fileString, 1 + $offset);
    $user['hidesentdeleted'] = GetInteger($fileString, 3 + $offset);
    $user['timezone']        = GetInteger($fileString, 5 + $offset);
    $user['htmlencode']      = GetInteger($fileString, 7 + $offset);
    $user['tagline']         = GetInteger($fileString, 9 + $offset);
    $user['newmessagescan']  = GetInteger($fileString, 11 + $offset);
    $user['password']        = GetString($fileString, 13 + $offset, 32);

    // If any of the settings are not present, then set them to the defaults.
    if ($user['hidedeleted'] === '') {
        $user['hidedeleted'] = $default['hidedeleted'];
    }
    if ($user['hidesentdeleted'] === '') {
        $user['hidesentdeleted'] = $default['hidesentdeleted'];
    }
    if (($user['timezone'] === '') || ($user['timezone'] == 0)) {
        $user['timezone'] = (integer) date("O");
    } elseif ($user['timezone'] == 1) {
        $user['timezone'] = 0;
    }
    if ($user['htmlencode'] === '') {
        $user['htmlencode'] = $default['htmlencode'];
    }
    if ($user['tagline'] === '') {
        $user['tagline'] = $default['tagline'];
    }
    if ($user['newmessagescan'] === '') {
        $user['newmessagescan'] = $default['newmessagescan'];
    }
    if ($user['password'] === '') {
        $user['password'] = $default['password'];
    }

    // Until we can add the sent db to vcleandb...
    $user['hidesentdeleted'] = 0;

    return $user;                           // Return the user information.
}


/**
* Va-Userfile.inc.php :: PutUserSettings()
*
* This function will put new information into the vadv-php.dat file string.
*
* @param string $fileString
* @param integer $userNo
* @param array $configVariable
* @return string
*/
function PutUserSettings($fileString, $userNo, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($userNo)) {
        $userNo     = (integer) $userNo;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $total = intval(strlen($fileString) / 82);
    if ($userNo > $total) {
        $fileString = $fileString . str_pad('', 82, chr(0));
    }

    $offset = 100 * ($userNo - 1);
    if (isset($configVariable['hidedeleted']))     $fileString = PutInteger($fileString, $configVariable['hidedeleted'], 1 + $offset);
    if (isset($configVariable['hidesentdeleted'])) $fileString = PutInteger($fileString, $configVariable['hidesentdeleted'], 3 + $offset);
    if (isset($configVariable['timezone']))        $fileString = PutInteger($fileString, $configVariable['timezone'], 5 + $offset);
    if (isset($configVariable['htmlencode']))      $fileString = PutInteger($fileString, $configVariable['htmlencode'], 7 + $offset);
    if (isset($configVariable['tagline']))         $fileString = PutInteger($fileString, $configVariable['tagline'], 9 + $offset);
    if (isset($configVariable['newmessagescan']))  $fileString = PutInteger($fileString, $configVariable['newmessagescan'], 11 + $offset);
    if (isset($configVariable['password']))        $fileString = PutString($fileString, $configVariable['password'], 13 + $offset, 32);

    return $fileString;                     // Return the new file string.
}


/**
* VADV-PHP_Dat.inc.php :: ReadUserSettings()
*
* This function will read the vadv-php.dat file.
*
* @return string
*/
function ReadUserSettings() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/VADV-PHP.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* VADV-PHP_Dat.inc.php :: WriteUserSettings()
*
* This function writes the new file string into vadv-php.dat.
*
* @param string $fileString
* @return mixed
*/
function WriteUserSettings($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/VADV-PHP.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}

?>