<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Userfile.inc.php                                               07/16/2006
 * This script will read the userfile.dat file for the specified user and store
 * its information into an array.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2005 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Userfile.inc.php :: AddUserEmailInfo()
*
* This function will add a user's email routing info into handles.uip. It will
* attempt to keep the file in numerical order. Multiple entries that exist for
* $userNo will be removed. This function is mainly used for creating new user
* entries and not to modify existing ones.
*
* @param integer $userNo
* @param array $configVariable
* @return mixed
*/
function AddUserEmailInfo($userNo, $configVariable) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    global $VA_SYSPATHS;                    // Include global variable.
    global $includeDir;
    include_once($includeDir . '/va-networks.inc.php');
    $vnetinfo = GetVnetInfo();
    
    $filename = $VA_SYSPATHS['main'] . '/HANDLES.UIP';

    if (is_file($filename)) {
        $file = file($filename);            // Read the file.
    } else {
        $file[0] = 'system-daemon,SYSTEM-DAEMON,' . $vnetinfo['systemid'] . ',1';
    }
    
    // Search for an existing entry.
    $found = 0;
    for ($x = 0; $x < count($file); $x++) {
        if (trim($file[$x]) != '') {
            $line = explode(',', $file[$x]);
            if (($line[1] == $userNo) && (trim($line[1]) != '')) {
                if ($found == 0) {
                    $file[$x] = $configVariable['handle'] . ',' . $configVariable['userno'] . ',' .
                                $configVariable['netnode'] . ',' . $configVariable['netno'];
                    $found = 1;
                } else {
                    for ($y = $x; $y < count($file) - 1; $y++) {
                        $file[$y] = $file[$y + 1];  // Move all elements down one.
                    }
                    unset($file[$y]);               // Remove the last entry.
                    $x--;                           // Remain at the same entry.
                }
            }
        }
    }

    if ($found == 0) {
        $file[count($file)] = $configVariable['handle'] . ',' . $configVariable['userno'] . ',' .
                              $configVariable['netnode'] . ',' . $configVariable['netno'];
    }
    
    $fp = fopen($filename, 'w');            // Open the file.
    flock($fp, 2);                          // Lock the file.
    for ($x = 0; $x < count($file); $x++) {
        $status = fwrite($fp, trim($file[$x]) . "\r\n");   // Write to the file.
    }
    flock($fp, 3);                          // Unlock the file.
    fclose($fp);                            // Close the file.
    
    return $status;
}


/**
* Va-Userfile.inc.php :: CalcAge()
*
* This function will calculate the age in years of the passed month, day and
* year date.
*
* @param integer $month
* @param integer $day
* @param integer $year
* @return integer
*/
function CalcAge($month, $day, $year) {
    if (!is_int($month)) {
        $month = (integer) $month;
    }
    if (!is_int($day)) {
        $day   = (integer) $day;
    }
    if (!is_int($year)) {
        $year  = (integer) $year;
    }

    // Calcuate the day numbers.
    $today = CalcDayNumber(date('m'), date('d'), date('Y'));
    $date  = CalcDayNumber($month, $day, $year);

    $age      = $today - $date;             // Age in days.
    $leapyear = intval(($age / 365) / 4);   // Cacluate number of leap year days.

    return intval(($age - $leapyear) / 365);
}


/**
* Va-Userfile.inc.php :: CalcDayNumber()
*
* This function will determine the day number of the specified date.
* Algorithm from: http://alcor.concordia.ca/~gpkatch/gdate-algorithm.html
*
* @param integer $month
* @param integer $day
* @param integer $year
* @return integer
*/
function CalcDayNumber($month, $day, $year) {
    if (!is_int($month)) {
        $month = (integer) $month;
    }
    if (!is_int($day)) {
        $day   = (integer) $day;
    }
    if (!is_int($year)) {
        $year  = (integer) $year;
    }

    $month = ($month + 9) % 12;
    $year  = $year - intval($month / 10);

    return (365 * $year) +
           intval($year / 4) -
           intval($year / 100) +
           intval($year / 400) +
           intval((($month * 306) + 5) / 10) +
           $day - 1;
}


/**
* Va-Userfile.inc.php :: CompressUserfile()
*
* This function will compress the userfile to its smallest state without
* disrupting user numbers (i.e. removes old accounts from the end).
*
* @return boolean
*/
function CompressUserfile() {
    global $VA_SYSPATHS;                    // Include global variable.

    $fileString = ReadUserfile();           // Read the userfile.
    $length     = strlen($fileString);      // Determine the length of the file string.
    $usercount  = floor($length / 1048);    // Get the number of users.

    $offset     = 1048;                     // Set the file offset.
    $position   = $length - 1048;           // Set the starting position (last acct).
    $found      = FALSE;                    // Pointer

    for ($x = $usercount; $x > 1; $x--) {
        $seclevel = GetInteger($fileString, 374 + $position);
        if ($seclevel > 0) {
            $found = TRUE;
            break;
        }
        $position = $position - $offset;
    }

    if ($x < $usercount) {
        $length = $x * 1048;
        $fileString = substr($fileString, 0, $length);
        $status = WriteUserfile($fileString);
        MakeUserfile_IDX();
    } else {
        MakeUserfile_IDX();
        $status = TRUE;
    }

    return $status;                         // Return the status of the operation.
}


/**
* Va-Userfile.inc.php :: DeleteUser()
*
* This function will take care of all the chores involved with deleting a
* user from the system.
*
* @param integer $userno
*/
function DeleteUser($userno) {
    if (!is_int($userno)) {
        $userno = (integer) $userno;
    }

    global $includeDir;                     // Include include directory.
    global $VA_SYSPATHS;                    // System paths.
    global $USER;                           // Current user information.
    if ($USER['securitylevel'] < 255) {     // If not a sysop, then do nothing.
        return;
    }
    if ($userno == 1) {                     // Will not delete user one.
        return;
    }

    include_once($includeDir . 'va-database.inc.php');  // Include database functions.
    include_once($includeDir . 'va-vote.inc.php');      // Include voting functions.
    
    $user = GetUser($userno);               // Get the user information.
    $user['flags']         = '';            // Set flags to none.
    $user['accflags']      = '';            // Set access flags to none.
    $user['securitylevel'] = 0;             // Set the sl to zero.
    $user['maxtime']       = 1;             // Set the maxtime to one.

    $useridx = GetUser_IDX($userno);        // Get the user information.
    $useridx['handle'] = '!-*<DELETED>*-!'; // Set the handle to deleted status.

    $datString = ReadUserfile();            // Read the userfile.dat.
    $idxString = ReadUserfile_IDX();        // Read the userfile.idx.

    $datString = PutUser($datString, $userno, $user);           // Insert the
    $idxString = PutUser_IDX($idxString, $userno, $useridx);    // new information.

    WriteUserfile($datString);              // Write the new userfile.dat.
    WriteUserfile_IDX($idxString);          // Write the new userfile.idx.

    // Delete any mail addressed to this account.
    $db     = SetEmailInfo($userno);
    $msgcnt = $db['msgcnt']['msgno'];
    $file   = $VA_SYSPATHS['db'] . '/' . $db['filename'] . '.DAT';
    $fileString = ReadCfgFile($file);
    $pos = 0;
    for ($x = 1; (($x <= $db['msgno']) && ($msgcnt > 0)); $x++) {
        $db      = SetEmailInfo($userno);
        $msgInfo = GetMessageInfo($db, $x, $fileString);
        if ($msgInfo['touserno'] == $userno) {
            DeleteMessage($db, $x - $pos);
            $pos++;
        }
    }

    // Delete any sent email from this account.
    $db     = SetSentEmailInfo($userno);
    $msgcnt = $db['msgcnt']['msgno'];
    $file   = $VA_SYSPATHS['db'] . '/' . $db['filename'] . '.DAT';
    $fileString = ReadCfgFile($file);
    $pos = 0;
    for ($x = 1; (($x <= $db['msgno']) && ($msgcnt > 0)); $x++) {
        $db      = SetSentEmailInfo($userno);
        $msgInfo = GetMessageInfo($db, $x, $fileString);
        if ($msgInfo['fromuserno'] == $userno) {
            DeleteMessage($db, $x - $pos);
            $pos++;
        }
    }

    // Delete any oneliners addressed to this account.
    $db     = SetOnelinerInfo($userno);
    $msgcnt = $db['msgcnt']['msgno'];
    $file   = $VA_SYSPATHS['db'] . '/' . $db['filename'] . '.DAT';
    $fileString = ReadCfgFile($file);
    $pos = 0;
    for ($x = 1; (($x <= $db['msgno']) && ($msgcnt > 0)); $x++) {
        $db      = SetOnelinerInfo($userno);
        $msgInfo = GetMessageInfo($db, $x, $fileString);
        if ($msgInfo['touserno'] == $userno) {
            DeleteMessage($db, $x - $pos);
            $pos++;
        }
    }

    // Remove votes from any voting booths.
    $file = $VA_SYSPATHS['data'] . '/VOTE';
    for ($x = 1; is_file($file . $x . '.DAT'); $x++) {
        PutUserVote($userno, $x, 0);
    }
    
    // Remove any multi-mail lists.
    $file = $VA_SYSPATHS['data'] . '/MM' . $userno . '.';
    for ($x = 1; $x <= 10; $x++) {
        if (is_file($file . $x)) {
            unlink($file);
        }
    }

    // File Cleanup
    $file = $VA_SYSPATHS['data'] . '/AB' . $userno . '.DAT';
    if (is_file($file)) {
        unlink($file);
    }

    $file = $VA_SYSPATHS['data'] . '/IRC' . $userno . '.DAT';
    if (is_file($file)) {
        unlink($file);
    }
}


/**
* Va-Userfile.inc.php :: FindHandle()
*
* This function will search the userfile.dat file for the $handle and return
* the user number.
*
* @param string $handle
* @return integer
*/
function FindHandle($handle) {
    if (!is_string($handle)) {
        $handle = (string) $handle;
    }

    // Check to see if the variable was passed, if not then return zero.
    if (trim($handle) == '') {
        return 0;
    }

    $fileString = ReadUserfile();           // Read the userfile.
    $handle = trim(strtolower($handle));    // Change the handle to uppercase.

    $position = 1;                          // Set the starting position.
    $seclvl   = 0;

    for ($x = 1; $position < strlen($fileString); $x++) {
        $string = GetString($fileString, $position, 40);
        $seclvl = GetInteger($fileString, $position + 373);
        $string = strtolower($string);          // Change the string to uppercase.
        $result = strcmp($handle, $string);     // Compare the handle to the string.
        if (($result == 0) && ($seclvl > 0)) {
            $x++;
            break;                              
        }
        $seclvl   = 0;                          // Reset the security level.
        $position = $position + 1048;           // Goto the next user record.
    }
    
    if ($seclvl == 0) {
        return 0;
    }

    return $x - 1;                             // Return the user number.
}


/**
* Va-Userfile.inc.php :: FindNewAcct()
*
* This function will find and return the first blank (deleted) user account
* number or an account with zero total calls that is over a week old or an
* account that is over the set idle limit.
*
* @return integer
*/
function FindNewAcct() {
    global $CONFIG;                         // Include global variable.

    $fileString = ReadUserfile();           // Read the userfile.
    $length     = strlen($fileString);      // Determine the length of the file string.

    $offset     = 1048;                     // Set the file offset.
    $position   = 0;                        // Set the starting position.

    for ($x = 1; $position < $length; $x++) {
        $seclevel   = GetInteger($fileString, 374 + $position);
        $laston     = GetString($fileString, 376 + $position, 6);
        $totalcalls = GetInteger($fileString, 386 + $position);
        
        // Determine age of the account.
        $month = substr($laston, 2, 2);
        $day   = substr($laston, 4, 2);
        $year  = substr($laston, 0, 2);
        if (($year > date("y")) || ($year > 70)) {
            $year = '19' . $year;
        } else {
            $year = '20' . $year;
        }
        $last  = CalcDayNumber($month, $day, $year);
        $today = CalcDayNumber(date('m'), date('d'), date('Y'));
        $age   = $today - $last;

        if (($seclevel == 0) ||
           (($totalcalls == 0) && ($age > 7)) ||
           ((($CONFIG['purgetime'] > 0) && ($age > $CONFIG['purgetime'])) &&
            (($CONFIG['purgeminlevel'] > 0) &&
             ($seclevel >= $CONFIG['purgeminlevel']) &&
             ($seclevel <= $CONFIG['purgemaxlevel'])))) {
           break;
       }

        $position = $position + $offset;    // Goto the next user record.
    }

    if ($position > $length) {
        $x++;
    }
    
    return $x;                              // Return the user number.
}


/**
* Va-Userfile.inc.php :: FindRealName()
*
* This function will search the userfile.dat file for the real $name and return
* the user number.
*
* @param string $name
* @return integer
*/
function FindRealName($name) {
    if (!is_string($name)) {
        $name = (string) $name;
    }

    // Check to see if the variable was passed, if not then return zero.
    if (trim($name) == '') {
        return 0;
    }

    $fileString = ReadUserfile();           // Read the userfile.
    $name       = trim(strtolower($name));  // Change the name to uppercase.

    $position   = 41;                       // Set the starting position.
    $seclvl     = 0;

    for ($x = 1; $position < strlen($fileString); $x++) {
        $string = GetString($fileString, $position, 40);
        $seclvl = GetInteger($fileString, $position + 333);
        $string = strtolower($string);      // Change the string to uppercase.
        $result = strcmp($name, $string);   // Compare the name to the string.
        if (($result == 0) && ($seclvl > 0)) {
            $x++;
            break;                         
        }
        $seclvl   = 0;                      // Reset the security level.
        $position = $position + 1048;       // Goto the next user record.
    }
    
    if ($seclvl == 0) {
        return 0;
    }

    return $x - 1;                          // Return the user number.
}


/**
* Va-Userfile.inc.php :: GetUser()
*
* This function will retrieve the information stored in the userfile.dat file
* for the specified user number. This file contains all of the user's data.
*
* @param integer $userNo
* @return array
*/
function GetUser($userNo, $fileString = '') {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $USER;                           // Include global variable.

    if ($fileString == '') {
        $fileString = ReadUserFile();
        if ($fileString === FALSE) {
            trigger_error(VA_Error(3));
        }
    }

    // Create a default blank user.
    $default['userno']        = $userNo;
    $default['handle']        = '';
    $default['realname']      = '';
    $default['address']       = '';
    $default['city']          = '';
    $default['state']         = '';
    $default['zipcode']       = '';
    $default['phone1']        = '';
    $default['phone2']        = '';
    $default['password']      = '';
    $default['mailfwdflag']   = 0;
    $default['mailfwdhandle'] = '';
    $default['mailfwduserno'] = 0;
    $default['mailfwdnode']   = 0;
    $default['mailfwdnetno']  = 0;
    $default['lngdistcost']   = 0;
    $default['lngdistacc']    = 0;
    $default['sortfilelst']   = 0;
    $default['pagebreak']     = 24;
    $default['bdaymonth']     = 0;
    $default['bdayday']       = 0;
    $default['bdayyear']      = 0;
    $default['age']           = 0;
    $default['accflags']      = 0;
    $default['flags']         = '';
    $default['securitylevel'] = 0;
    $default['laston']        = date("ymd");
    $default['maxtime']       = 0;
    $default['timecounter']   = 0;
    $default['totalcalls']    = 0;
    $default['totaltime']     = 0;
    $default['videomode']     = 1;
    $default['ratioexempt']   = 0;
    $default['verifybday']    = 0;
    $default['textcolor']     = chr(7);
    $default['basecolor']     = chr(0);
    $default['promptcolor']   = chr(2);
    $default['inputcolor']    = chr(5);
    $default['boxcolor']      = chr(5);
    $default['mnuselcolor']   = chr(1);
    $default['headercolor']   = chr(2);
    $default['extra1']        = '';
    $default['extra2']        = '';
    $default['extra3']        = '';
    $default['extra4']        = '';
    $default['extra5']        = '';
    $default['extra6']        = '';
    $default['extra7']        = '';
    $default['extra8']        = '';
    $default['macro1']        = '';
    $default['macro2']        = '';
    $default['uplfiles']      = 0;
    $default['uplkb']         = 0;
    $default['dlfiles']       = 0;
    $default['dlkb']          = 0;
    $default['totalposts']    = 0;
    $default['totalemail']    = 0;
    $default['postsread']     = 0;
    $default['timebank']      = 0;
    $default['credits']       = 0;
    $default['comment']       = '';
    $default['protocol']      = 'Z';
    $default['testid1']       = '';
    $default['testid2']       = '';
    $default['offsym']        = '';
    $default['expdate']       = '';
    $default['reserved']      = '';
    $default['lastcalltime']  = time();
    $default['liststyle']     = 2;
    $default['extra']         = '';

    // If the userno is less than zero or greater than the last account, return the default information.
    $lastuser = intval(strlen($fileString) / 1048);
    if (($userNo <= 0) || ($userNo > $lastuser)) {
        return $default;
    }

    $user   = array();
    $offset = 1048 * ($userNo - 1);
    $user['userno']        = $userNo;
    $user['handle']        = GetString($fileString, 1 + $offset, 40);
    $user['realname']      = GetString($fileString, 41 + $offset, 40);
    $user['address']       = GetString($fileString, 81 + $offset, 40);
    $user['city']          = GetString($fileString, 121 + $offset, 32);
    $user['state']         = GetString($fileString, 153 + $offset, 3);
    $user['zipcode']       = GetString($fileString, 156 + $offset, 10);
    $user['phone1']        = GetString($fileString, 166 + $offset, 12);
    $user['phone2']        = GetString($fileString, 178 + $offset, 12);
    // User password is encrypted with md5 for security purposes.
    $user['password']      = md5(strtoupper(GetString($fileString, 190 + $offset, 8)));
    $user['mailfwdflag']   = GetInteger($fileString, 198 + $offset);
    $user['mailfwdhandle'] = GetString($fileString, 200 + $offset, 40);
    $user['mailfwduserno'] = GetLongInt($fileString, 240 + $offset);
    $user['mailfwdnode']   = GetString($fileString, 244 + $offset, 80);
    $user['mailfwdnetno']  = GetInteger($fileString, 324 + $offset);
    $user['lngdistcost']   = GetSingle($fileString, 326 + $offset, 2);
    $user['lngdistacc']    = GetSingle($fileString, 330 + $offset, 2);
    $user['sortfilelst']   = GetInteger($fileString, 334 + $offset);
    $user['pagebreak']     = GetInteger($fileString, 336 + $offset);
    $user['bdaymonth']     = GetInteger($fileString, 338 + $offset);
    $user['bdayday']       = GetInteger($fileString, 340 + $offset);
    $user['bdayyear']      = GetInteger($fileString, 342 + $offset);
    $user['age']           = CalcAge($user['bdaymonth'], $user['bdayday'], $user['bdayyear']);
    $user['flags']         = str_replace(' ', '', GetString($fileString, 344 + $offset, 26));
    $user['accflags']      = str_replace(' ', '', GetAccFlags(GetLongInt($fileString, 370 + $offset)));
    $user['securitylevel'] = GetInteger($fileString, 374 + $offset);
    $user['laston']        = GetString($fileString, 376 + $offset, 6);
    $user['maxtime']       = GetInteger($fileString, 382 + $offset);
    $user['timecounter']   = GetInteger($fileString, 384 + $offset);
    $user['totalcalls']    = GetLongInt($fileString, 386 + $offset);
    $user['totaltime']     = GetLongInt($fileString, 390 + $offset);
    $user['videomode']     = GetInteger($fileString, 394 + $offset);
    $user['ratioexempt']   = GetInteger($fileString, 396 + $offset);
    $user['verifybday']    = GetInteger($fileString, 398 + $offset);
    $user['textcolor']     = GetChar($fileString, 400 + $offset);
    $user['basecolor']     = GetChar($fileString, 401 + $offset);
    $user['promptcolor']   = GetChar($fileString, 402 + $offset, 1);
    $user['inputcolor']    = GetChar($fileString, 403 + $offset, 1);
    $user['boxcolor']      = GetChar($fileString, 404 + $offset, 1);
    $user['mnuselcolor']   = GetChar($fileString, 405 + $offset, 1);
    $user['headercolor']   = GetChar($fileString, 406 + $offset, 1);
    $user['extra1']        = GetString($fileString, 407 + $offset, 12);
    $user['extra2']        = GetString($fileString, 419 + $offset, 12);
    $user['extra3']        = GetString($fileString, 431 + $offset, 12);
    $user['extra4']        = GetString($fileString, 443 + $offset, 12);
    $user['extra5']        = GetString($fileString, 455 + $offset, 12);
    $user['extra6']        = GetString($fileString, 467 + $offset, 12);
    $user['extra7']        = GetString($fileString, 479 + $offset, 12);
    $user['extra8']        = GetString($fileString, 491 + $offset, 12);
    $user['macro1']        = GetString($fileString, 503 + $offset, 160);
    $user['macro2']        = GetString($fileString, 663 + $offset, 160);
    $user['uplfiles']      = GetLongInt($fileString, 823 + $offset);
    $user['uplkb']         = GetLongInt($fileString, 827 + $offset);
    $user['dlfiles']       = GetLongInt($fileString, 831 + $offset);
    $user['dlkb']          = GetLongInt($fileString, 835 + $offset);
    $user['totalposts']    = GetLongInt($fileString, 839 + $offset);
    $user['totalemail']    = GetLongInt($fileString, 843 + $offset);
    $user['postsread']     = GetLongInt($fileString, 847 + $offset);
    $user['timebank']      = GetInteger($fileString, 851 + $offset);
    $user['credits']       = GetLongInt($fileString, 853 + $offset);
    $user['comment']       = GetString($fileString, 857 + $offset, 16);
    $user['protocol']      = GetString($fileString, 873 + $offset, 1);
    $user['testid1']       = GetString($fileString, 874 + $offset, 3);
    $user['testid2']       = GetString($fileString, 877 + $offset, 4);
    $user['offsym']        = GetString($fileString, 881 + $offset, 22);
    $user['expdate']       = GetString($fileString, 903 + $offset, 6);
    $user['reserved']      = GetString($fileString, 909 + $offset, 6);
    $user['lastcalltime']  = GetLongInt($fileString, 915 + $offset);
    $user['liststyle']     = GetInteger($fileString, 919 + $offset);
    $user['extra']         = GetString($fileString, 921 + $offset, 128);

    // If a deleted account, return the default information (unless you are the sysop).
    if (($user['securitylevel'] == 0) && ($USER['securitylevel'] < 255)) {
        return $default;
    }
    
    return $user;                           // Return the user information.
}


/**
* Va-Userfile.inc.php :: GetUser_IDX()
*
* This function will get the requested entry information from userfile.idx.
*
* @param integer $userNo
* @return array
*/
function GetUser_IDX($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    if ($userNo <= 0) {
        return 0;
    }

    $fileString = ReadUserFile_IDX();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $user   = array();
    $offset = 82 * ($userNo - 1);
    $user['realname'] = GetString($fileString, 1 + $offset, 40);
    $user['handle']   = GetString($fileString, 41 + $offset, 40);
    $user['userno']   = GetInteger($fileString, 81 + $offset);

    return $user;                           // Return the user information.
}


/**
* Va-Userfile.inc.php :: GetUserCount()
*
* This function simply returns the number of user accounts in the dat file.
*
* @return integer
*/
function GetUserCount() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.DAT';
    if (!is_file($file)) {
        return 0;
    }

    $userCount = intval(filesize($file) / 1048);

    return $userCount;
}


/**
* Va-Userfile.inc.php :: GetUserEmailInfo()
*
* This function will search handles.uip and get the user's email routing
* information.
*
* @param integer $userNo
* @return array
*/
function GetUserEmailInfo($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    $user['handle']  = '';
    $user['userno']  = 0;
    $user['netnode'] = '';
    $user['netno']   = 0;

    global $VA_SYSPATHS;                    // Include global variable.
    $filename = $VA_SYSPATHS['main'] . '/HANDLES.UIP';
    if (!is_file($file)) {
        return $user;
    }
    
    $file = file($filename);                // Read the file into an array.

    $found = 0;
    for ($x = 0; (($found != 1) && ($x < count($file))); $x++) {
        $line = explode(',', $file[$x]);
        if ($line[1] == $userNo) {
            $user['handle']  = $line[0];
            $user['userno']  = $line[1];
            $user['netnode'] = $line[2];
            $user['netno']   = $line[3];
            $found = 1;
        }
    }
    
    return $user;                           // Return the user info.
}


/**
* Va-Userfile.inc.php :: MakeUserfile_IDX()
*
* This function will create a new userfile.idx.
*
* @return mixed
*/
function MakeUserfile_IDX() {
    global $VA_SYSPATHS;

    $file = $VA_SYSPATHS['data'] . '/USERFILE.IDX';

    // If the IDX file exists, then remove it.
    if (is_file($file)) {
        unlink($file);
    }

    $fileString = '';

    for ($x = 1; $x <= GetUserCount(); $x++) {
        $user = GetUser($x);
        if ($user['securitylevel'] == 0) {  // If deleted user, change handle.
            $user['handle'] = '!-*<DELETED>*-!';
        }
        $fileString = PutUser_IDX($fileString, $x, $user);
    }
    
    $status = WriteUserfile_IDX($fileString);

    return $status;                         // Return the result status.
}


/**
* Va-Userfile.inc.php :: PutUser()
*
* This function will write updated information to the specified user's account.
*
* @param string $fileString
* @param integer $userNo
* @param array $configVariable
* @return string
*/
function PutUser($fileString, $userNo, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($userNo)) {
        $userNo     = (integer) $userNo;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $total = intval(strlen($fileString) / 1048);
    if ($userNo > $total) {
        $fileString = $fileString . str_pad('', 1048, chr(0));
    }

    $offset = 1048 * ($userNo - 1);
    if (isset($configVariable['handle']))        $fileString = PutString($fileString, $configVariable['handle'], 1 + $offset, 40);
    if (isset($configVariable['realname']))      $fileString = PutString($fileString, $configVariable['realname'], 41 + $offset, 40);
    if (isset($configVariable['address']))       $fileString = PutString($fileString, $configVariable['address'], 81 + $offset, 40);
    if (isset($configVariable['city']))          $fileString = PutString($fileString, $configVariable['city'], 121 + $offset, 32);
    if (isset($configVariable['state']))         $fileString = PutString($fileString, $configVariable['state'], 153 + $offset, 3);
    if (isset($configVariable['zipcode']))       $fileString = PutString($fileString, $configVariable['zipcode'], 156 + $offset, 10);
    if (isset($configVariable['phone1']))        $fileString = PutString($fileString, $configVariable['phone1'], 166 + $offset, 12);
    if (isset($configVariable['phone2']))        $fileString = PutString($fileString, $configVariable['phone2'], 178 + $offset, 12);
    // To change the passwords, they must be in plain text and less than 8 chars.
    // If it is encrypted with md5, then it has a length of 32 characters and will not write it.
    if ((isset($configVariable['password'])) &&
        (strlen($configVariable['password']) <= 8))
                                                 $fileString = PutString($fileString, $configVariable['password'], 190 + $offset, 8);
    if (isset($configVariable['mailfwdflag']))   $fileString = PutInteger($fileString, $configVariable['mailfwdflag'], 198 + $offset);
    if (isset($configVariable['mailfwdhandle'])) $fileString = PutString($fileString, $configVariable['mailfwdhandle'], 200 + $offset, 40);
    if (isset($configVariable['mailfwduserno'])) $fileString = PutLongInt($fileString, $configVariable['mailfwduserno'], 240 + $offset);
    if (isset($configVariable['mailfwdnode']))   $fileString = PutString($fileString, $configVariable['mailfwdnode'], 244 + $offset, 80);
    if (isset($configVariable['mailfwdnetno']))  $fileString = PutInteger($fileString, $configVariable['mailfwdnetno'], 324 + $offset);
//    if (isset($configVariable['lngdistcost']))   $fileString = PutSingle($fileString, $configVariable['lngdistcost'], 326 + $offset);
//    if (isset($configVariable['lngdistacc']))    $fileString = PutSingle($fileString, $configVariable['lngdistacc'], 330 + $offset);
    if (isset($configVariable['sortfilelst']))   $fileString = PutInteger($fileString, $configVariable['sortfilelst'], 334 + $offset);
    if (isset($configVariable['pagebreak']))     $fileString = PutInteger($fileString, $configVariable['pagebreak'], 336 + $offset);
    if (isset($configVariable['bdaymonth']))     $fileString = PutInteger($fileString, $configVariable['bdaymonth'], 338 + $offset);
    if (isset($configVariable['bdayday']))       $fileString = PutInteger($fileString, $configVariable['bdayday'], 340 + $offset);
    if (isset($configVariable['bdayyear']))      $fileString = PutInteger($fileString, $configVariable['bdayyear'], 342 + $offset);
    if (isset($configVariable['flags']))         $fileString = PutString($fileString, MakeFlags($configVariable['flags']), 344 + $offset, 26);
    if (isset($configVariable['accflags']))      $fileString = PutLongInt($fileString, MakeAccFlags($configVariable['accflags']), 370 + $offset);
    if (isset($configVariable['securitylevel'])) $fileString = PutInteger($fileString, $configVariable['securitylevel'], 374 + $offset);
    if (isset($configVariable['laston']))        $fileString = PutString($fileString, $configVariable['laston'], 376 + $offset, 6);
    if (isset($configVariable['maxtime']))       $fileString = PutInteger($fileString, $configVariable['maxtime'], 382 + $offset);
    if (isset($configVariable['timecounter']))   $fileString = PutInteger($fileString, $configVariable['timecounter'], 384 + $offset);
    if (isset($configVariable['totalcalls']))    $fileString = PutLongInt($fileString, $configVariable['totalcalls'], 386 + $offset);
    if (isset($configVariable['totaltime']))     $fileString = PutLongInt($fileString, $configVariable['totaltime'], 390 + $offset);
    if (isset($configVariable['videomode']))     $fileString = PutInteger($fileString, $configVariable['videomode'], 394 + $offset);
    if (isset($configVariable['ratioexempt']))   $fileString = PutInteger($fileString, $configVariable['ratioexempt'], 396 + $offset);
    if (isset($configVariable['verifybday']))    $fileString = PutInteger($fileString, $configVariable['verifybday'], 398 + $offset);
    if (isset($configVariable['textcolor']))     $fileString = PutChar($fileString, $configVariable['textcolor'], 400 + $offset);
    if (isset($configVariable['basecolor']))     $fileString = PutChar($fileString, $configVariable['basecolor'], 401 + $offset);
    if (isset($configVariable['promptcolor']))   $fileString = PutChar($fileString, $configVariable['promptcolor'], 402 + $offset, 1);
    if (isset($configVariable['inputcolor']))    $fileString = PutChar($fileString, $configVariable['inputcolor'], 403 + $offset, 1);
    if (isset($configVariable['boxcolor']))      $fileString = PutChar($fileString, $configVariable['boxcolor'], 404 + $offset, 1);
    if (isset($configVariable['mnuselcolor']))   $fileString = PutChar($fileString, $configVariable['mnuselcolor'], 405 + $offset, 1);
    if (isset($configVariable['headercolor']))   $fileString = PutChar($fileString, $configVariable['headercolor'], 406 + $offset, 1);
    if (isset($configVariable['extra1']))        $fileString = PutString($fileString, $configVariable['extra1'], 407 + $offset, 12);
    if (isset($configVariable['extra2']))        $fileString = PutString($fileString, $configVariable['extra2'], 419 + $offset, 12);
    if (isset($configVariable['extra3']))        $fileString = PutString($fileString, $configVariable['extra3'], 431 + $offset, 12);
    if (isset($configVariable['extra4']))        $fileString = PutString($fileString, $configVariable['extra4'], 443 + $offset, 12);
    if (isset($configVariable['extra5']))        $fileString = PutString($fileString, $configVariable['extra5'], 455 + $offset, 12);
    if (isset($configVariable['extra6']))        $fileString = PutString($fileString, $configVariable['extra6'], 467 + $offset, 12);
    if (isset($configVariable['extra7']))        $fileString = PutString($fileString, $configVariable['extra7'], 479 + $offset, 12);
    if (isset($configVariable['extra8']))        $fileString = PutString($fileString, $configVariable['extra8'], 491 + $offset, 12);
    if (isset($configVariable['macro1']))        $fileString = PutString($fileString, $configVariable['macro1'], 503 + $offset, 160);
    if (isset($configVariable['macro2']))        $fileString = PutString($fileString, $configVariable['macro2'], 663 + $offset, 160);
    if (isset($configVariable['uplfiles']))      $fileString = PutLongInt($fileString, $configVariable['uplfiles'], 823 + $offset);
    if (isset($configVariable['uplkb']))         $fileString = PutLongInt($fileString, $configVariable['uplkb'], 827 + $offset);
    if (isset($configVariable['dlfiles']))       $fileString = PutLongInt($fileString, $configVariable['dlfiles'], 831 + $offset);
    if (isset($configVariable['dlkb']))          $fileString = PutLongInt($fileString, $configVariable['dlkb'], 835 + $offset);
    if (isset($configVariable['totalposts']))    $fileString = PutLongInt($fileString, $configVariable['totalposts'], 839 + $offset);
    if (isset($configVariable['totalemail']))    $fileString = PutLongInt($fileString, $configVariable['totalemail'], 843 + $offset);
    if (isset($configVariable['postsread']))     $fileString = PutLongInt($fileString, $configVariable['postsread'], 847 + $offset);
    if (isset($configVariable['timebank']))      $fileString = PutInteger($fileString, $configVariable['timebank'], 851 + $offset);
    if (isset($configVariable['credits']))       $fileString = PutLongInt($fileString, $configVariable['credits'], 853 + $offset);
    if (isset($configVariable['comment']))       $fileString = PutString($fileString, $configVariable['comment'], 857 + $offset, 16);
    if (isset($configVariable['protocol']))      $fileString = PutString($fileString, $configVariable['protocol'], 873 + $offset, 1);
    if (isset($configVariable['testid1']))       $fileString = PutString($fileString, $configVariable['testid1'], 874 + $offset, 3);
    if (isset($configVariable['testid2']))       $fileString = PutString($fileString, $configVariable['testid2'], 877 + $offset, 4);
    if (isset($configVariable['offsym']))        $fileString = PutString($fileString, $configVariable['offsym'], 881 + $offset, 22);
    if (isset($configVariable['expdate']))       $fileString = PutString($fileString, $configVariable['expdate'], 903 + $offset, 6);
    if (isset($configVariable['reserved']))      $fileString = PutString($fileString, $configVariable['reserved'], 909 + $offset, 6);
    if (isset($configVariable['lastcalltime']))  $fileString = PutLongInt($fileString, $configVariable['lastcalltime'], 915 + $offset);
    if (isset($configVariable['liststyle']))     $fileString = PutInteger($fileString, $configVariable['liststyle'], 919 + $offset);
    if (isset($configVariable['extra']))         $fileString = PutString($fileString, $configVariable['extra'], 921 + $offset, 128);

    return $fileString;                     // Return the new file string.
}


/**
* Va-Userfile.inc.php :: PutUser_IDX()
*
* This function will put new information into the userfile.idx file string.
*
* @param string $fileString
* @param integer $userNo
* @param array $configVariable
* @return string
*/
function PutUser_IDX($fileString, $userNo, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($userNo)) {
        $userNo     = (integer) $userNo;
    }

    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $total = intval(strlen($fileString) / 82);
    if ($userNo > $total) {
        $fileString = $fileString . str_pad('', 82, chr(0));
    }

    $offset = 82 * ($userNo - 1);
    if (isset($configVariable['realname'])) $fileString = PutString($fileString, $configVariable['realname'], 1 + $offset, 40);
    if (isset($configVariable['handle']))   $fileString = PutString($fileString, $configVariable['handle'], 41 + $offset, 40);
    if (isset($configVariable['userno']))   $fileString = PutInteger($fileString, $configVariable['userno'], 81 + $offset);

    return $fileString;                     // Return the new file string.
}


/**
* Va-Userfile.inc.php :: ReadUserfile()
*
* This function will read the userfile.dat file.
*
* @return string
*/
function ReadUserfile() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Userfile.inc.php :: ReadUserfile_IDX()
*
* This function reads userfile.idx.
*
* @return string
*/
function ReadUserfile_IDX() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.IDX';
    
    if (!is_file($file)) {
        MakeUserfile_IDX();
    }

    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Userfile.inc.php :: WriteUserfile()
*
* This function will write a new userfile.dat file based on the file string
* passed to it. If the file string is not in the correct format, then your
* userfile.dat file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteUserfile($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


/**
* Va-Userfile.inc.php :: WriteUserfile_IDX()
*
* This function writes the new file string into userfile.idx.
*
* @param string $fileString
* @return mixed
*/
function WriteUserfile_IDX($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.IDX';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}

?>