<?php
/*
 * ----------------------------------------------------------------------------
 * Va-TextSect.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * bulletins that are found in TEXTSECT.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-TextSect.inc.php :: GetTextSect()
*
* Retrieve all of the bulletin section information and store it into a
* multidimensional array.
*
* Array format: $textsect[1]['setting']
*               $textsect[1]['bulletin'][1]['setting']
*               $textsect[0] = Total number of sections.
*
* @return array
*/
function GetTextSect() {
    $fileString = ReadTextSect();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    $textSect = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $textSect[$x]['desc']    = GetString($fileString, 1 + $offset, 32);
        $textSect[$x]['path']    = GetString($fileString, 33 + $offset, 32);
        $textSect[$x]['minsl']   = GetInteger($fileString, 65 + $offset);
        $textSect[$x]['accflag'] = GetString($fileString, 67 + $offset, 1);
        $offset = 67 * $x;
        
        // Get the bulletins for this section.
        $datFile       = $VA_SYSPATHS['data'] . '/TEXT' . $x . '.DAT';
        $datFileString = ReadCfgFile($datFile);
        $datOffset     = 0;
        
        for ($y = 1; $datOffset < strlen($datFileString); $y++) {
            $textSect[$x]['bulletin'][$y]['desc']     = GetString($datFileString, 1 + $datOffset, 32);
            $textSect[$x]['bulletin'][$y]['filename'] = GetString($datFileString, 33 + $datOffset, 12);
            $datOffset = 44 * $y;
        }
        
        $textSect[$x]['bulletin'][0] = $y - 1;  // The number of bulletins.
    }

    $textSect[0] = $x - 1;                  // Put the number of sections in position 0.

    return $textSect;                       // Return the text sections.

}


/**
* Va-TextSect.inc.php :: PutTextSect()
*
* This function will change the textsect.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting']. Currently this function does not
* allow you to change the actual bulletin descriptions or filenames.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutTextSect($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable) - 1;  // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 67);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 67, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['desc']))    $fileString = PutString($fileString, $configVariable[$x]['desc'], 1 + $offset, 32);
            if (isset($configVariable[$x]['path']))    $fileString = PutString($fileString, $configVariable[$x]['path'], 33 + $offset, 32);
            if (isset($configVariable[$x]['minsl']))   $fileString = PutInteger($fileString, $configVariable[$x]['minsl'], 65 + $offset);
            if (isset($configVariable[$x]['accflag'])) $fileString = PutString($fileString, $configVariable[$x]['accflag'], 67 + $offset, 1);
        }
        $offset = 67 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-TextSect.inc.php :: ReadTextSect()
*
* This function will read the textsect.cfg file.
*
* @return string
*/
function ReadTextSect() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/TEXTSECT.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-TextSect.inc.php :: WriteTextSect()
*
* This function will write a new textsect.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* textsect.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteTextSect($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/TEXTSECT.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the text sections automatically.
$VA_TEXTSECT = GetTextSect();

?>