<?php
/*
 * ----------------------------------------------------------------------------
 * Va-QuickVal.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * quick validation settings that are found in QUICKVAL.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-QuickVal.inc.php :: GetQuickVal()
*
* Retrieve all of the quick validation information and store it into a
* multidimensional array.
*
* Array format: $quickval[1]['setting']
*               $quickval[0] = Total number of configured quickvals.
*
* @return array
*/
function GetQuickVal() {
    $fileString = ReadQuickVal();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $quickval = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $quickval[$x]['seclevel']  = GetInteger($fileString, 1 + $offset);
        $quickval[$x]['timelimit'] = GetInteger($fileString, 3 + $offset);
        $quickval[$x]['flags']     = GetLongInt($fileString, 5 + $offset);
        $offset = 73 * $x;
    }

    $quickval[0] = $x - 1;                  // Put the number of configured quickvals in position 0.

    return $quickval;                       // Return the quickval information.

}


/**
* Va-QuickVal.inc.php :: PutQuickVal()
*
* This function will change the quickval.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutQuickVal($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 73);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 73, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['seclevel']))  $fileString = PutInteger($fileString, $configVariable[$x]['seclevel'], 1 + $offset);
            if (isset($configVariable[$x]['timelimit'])) $fileString = PutInteger($fileString, $configVariable[$x]['timelimit'], 3 + $offset);
            if (isset($configVariable[$x]['flags']))     $fileString = PutLongInt($fileString, $configVariable[$x]['flags'], 5 + $offset);
        }
        $offset = 73 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-QuickVal.inc.php :: ReadQuickVal()
*
* This function will read the quickval.cfg file.
*
* @return string
*/
function ReadQuickVal() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/QUICKVAL.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-QuickVal.inc.php :: WriteQuickVal()
*
* This function will write a new quickval.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* quickval.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteQuickVal($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/QUICKVAL.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the quick validation settings automatically.
$VA_QUICKVAL = GetQuickVal();

?>