/*
 WHO CALLED ProBoard PEX Version and MS-DOS EXE
 Copyright (C) 1995 by Branislav L. Slantchev

 This file is part of WHO CALLED.

 WHO CALLED is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2.

 WHO CALLED is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with WHO CALLED; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#if !defined( PB_SDK )
	#include <stdlib.h>
	#include <stdio.h>
	#include <time.h>
	#include <ctype.h>
	#include <stdarg.h>
	#include "userrec.h"
	#include "pbconfig.h"
#else
	#include <pb_sdk.h>
#endif

#include "pbinc.h"
#include "llistadt.h"
#include "date.h"
#include "usersidx.h"
#include "getopt.h"


/* just a short-hand ;-) */
typedef TimeType t_time;
typedef DateType t_date;
typedef int (*t_func)(const void *, const void *);

/* types of sort */
typedef enum { SORT_NONE, SORT_LASTDATE, SORT_MINTODAY,
			   SORT_TOTALMIN, SORT_TOTALCALLS } t_sort;

/*
 * user record for the list
*/
typedef struct{
	char   name[36];             /* user name or alias                   */
	char   city[26];             /* city last called from                */
	char   state[26];            /* state from user record               */
	byte   callsToday;           /* calls made today                     */
	t_time lastTime;             /* start time of most recent call       */
	t_date lastDate;             /* date of most recent call             */
	byte   node;                 /* node number for last call            */
	long   baud;                 /* baud rate for last call              */
	word   minToday;             /* total minutes all calls today        */
	dword  minTotal;             /* total minutes all calls              */
	dword  callsTotal;           /* total calls                          */
} UserData;

#define MAX_NOTHIRD    10        /* records per screen if two lines      */
#define MAX_THIRD       7        /* records per screen if three lines    */
#define __VERSION__    "v1.23"

#define ANSI_NAMECLR   "[0;1m"
#define ANSI_TEXTCLR   "[0;32m"
#define ANSI_DIGITCLR  "[1;33m"

#define ANSI_WAIT      "[0;1mPress [[1;33mEnter[0;1m] "\
					   "to continue...\t\xC"
#define ANSI_HEADER    "[2J[1;33mWho Called [1;31m" __VERSION__ \
 "[0;1m Copyright (C) 1995 by Silicon Creations\n[1;31m"\
 "\n"


/*
 * global variables
*/
static char systemPath[MAXPATH];      /* holds ProBoard's system path    */
static char textPath[MAXPATH];        /* holds ProBoard textfiles path   */
static t_date _dateToday;             /* today's date, ProBoard format   */
static bool _todayOnly    = FALSE;    /* TRUE if only today's callers    */
static bool _oneLine      = FALSE;    /* one-line statistics mode        */
static bool _useAlias     = FALSE;    /* TRUE if we want to use alias    */
static bool _useFile      = FALSE;    /* TRUE if we print stats to file  */
static bool _disableThird = FALSE;    /* TRUE if third line disabled     */
static bool _fileScreen   = FALSE;    /* TRUE if both to screen and file */

/*
 * local functions
*/
static int sort_none( const void *r1, const void *r2 );
static int sort_lastdate( const void *r1, const void *r2 );
static int sort_mintoday( const void *r1, const void *r2 );
static int sort_totalmin( const void *r1, const void *r2 );
static int sort_totalcalls( const void *r1, const void *r2 );
static int time_diff( int h1, int m1, int s1, int h2, int m2, int s2 );
static int compare_name( const void *r1, const void *r2 );
static int build_list( llist *list, t_func sort_function, int recToRead );
static int print_list( llist *list );
static void print_rec( UserData *rec, ... );

#pragma argsused
int
main(int argc, char *argv[])
{
	llist  *list;                /* list with all user records           */
	t_func sort_function;        /* pointer to the desired sort function */
	int  recToRead = -1;         /* records to read into the list        */

	time_t t;
	struct tm *ptm;
	int ch;

#if defined( PB_SDK )
	strcpy( systemPath, SysPath );
	strcpy( textPath, Config->txtpath );
#else
	CONFIG cfg;
	char *p = getenv( "PROBOARD" );

	if( !p ){
		printf( "Unable to locate environment PROBOARD.\n" );
		return 1;
	}
	else strcpy( systemPath, p );
	if( -1 == ReadConfig( &cfg ) ){
		printf( "Unable to load config file from '%s'\n", p );
		return 1;
	}
	else strcpy( textPath, cfg.txtpath );
#endif

	/* save today's date */
	t = time( NULL );
	ptm = localtime( &t );
	_dateToday[0] = ptm->tm_mday;
	_dateToday[1] = ptm->tm_mon + 1;
	_dateToday[2] = ptm->tm_year;

	sort_function = sort_none;

	/* read command-line */
	while( EOF != (ch = getopt( argc, argv, "s:S:aAfFdDl:L:tT1bB" )) ){
		switch( tolower(ch) ){
			case 's':
				switch( tolower(optarg[0]) ){
					case 'd': sort_function = sort_lastdate;   break;
					case 'o': sort_function = sort_mintoday;   break;
					case 'm': sort_function = sort_totalmin;   break;
					case 'c': sort_function = sort_totalcalls; break;
					default : return 1;
				}
			break;
			case 'a': _useAlias = TRUE; break;
			case 'f': _useFile = TRUE; break;
			case 'd': _disableThird = TRUE; break;
			case 'l': recToRead = atoi( optarg ); break;
			case 't': _todayOnly = TRUE; break;
			case '1': _disableThird = TRUE; _oneLine = TRUE; break;
			case 'b': _fileScreen = TRUE; break;
		}
	}
	/* adjust number of records to read */
	if( -1 == recToRead )
		recToRead = _disableThird ? MAX_NOTHIRD : MAX_THIRD;

	/* create a new list */
	if( NULL == (list = llnew()) ){
		printf( "Unable to allocate memory for the list.\n" );
		return 1;
	}

	/* build the list and print the results */
	if( -1 != build_list( list, sort_function, recToRead ) )
		print_list( list );
	else
		printf( "Error builidng the linked list.\n" );

	/* dispose of the whole list */
	lldelete( list, llfree );

#if !defined( PB_SDK )
	printf( "[0m" );
#endif
	return 0;
}

/*
 * sort by nothing: get order of records in the BINLOG file
*/
#pragma warn -par
	static int
sort_none( const void *r1, const void *r2 )
{
	return 1; /* force new records to start of list */
}
#pragma warn +par

/*
 * returns:
 *    0  : time1 == time2
 *   <0  : time1 < time2
 *   >0  : time1 > time2
*/
	static int
time_diff( int h1, int m1, int s1, int h2, int m2, int s2 )
{
	int retval = h1 - h2;
	if( !retval ) retval = m1 - m2;
	if( !retval ) retval = s1 - s2;
	return retval;
}

/*
 * compares last date and, if they are the same, last time
*/
	static int
sort_lastdate( const void *r1, const void *r2 )
{
	int retval, d1, m1, y1, d2, m2, y2;
	UserData *rec1 = CAST( r1, UserData* );
	UserData *rec2 = CAST( r2, UserData* );

	d1 = rec1->lastDate[0]; m1 = rec1->lastDate[1]; y1 = rec1->lastDate[2];
	d2 = rec2->lastDate[0]; m2 = rec2->lastDate[1]; y2 = rec2->lastDate[2];

	retval = date_diff( d2, m2, y2+1990, d1, m1, y1+1990 );
	if( !retval ){
		d1 = rec1->lastTime[0]; m1 = rec1->lastTime[1]; y1 = rec1->lastTime[2];
		d2 = rec2->lastTime[0]; m2 = rec2->lastTime[1]; y2 = rec2->lastTime[2];
		retval = time_diff( d2, m2, y2, d1, m1, y1 );
	}
	return retval;
}

/*
 * compares the minutes online today
*/
	static int
sort_mintoday( const void *r1, const void *r2 )
{
	return ((UserData *)r2)->minToday - ((UserData *)r1)->minToday;
}

/*
 * compares the total minutes spent online
*/
	static int
sort_totalmin( const void *r1, const void *r2 )
{
	return (int)(((UserData *)r2)->minTotal - ((UserData *)r1)->minTotal);
}

/*
 * compares the total number of calls
*/
	static int
sort_totalcalls( const void *r1, const void *r2 )
{
	return (int)(((UserData *)r2)->callsTotal - ((UserData *)r1)->callsTotal);
}

/*
 * compares user name from record, not case sensitive
*/
	static int
compare_name( const void *r1, const void *r2 )
{
	return stricmp( CAST(r1, UserData*)->name, CAST(r2, UserData*)->name );
}

/*
 * builds a list of user records
*/
	static int
build_list( llist *list, t_func sort_function, int recToRead )
{
	UserData  rec, *pRec;
	BINLOG    binRec;
	USER_REC  userRec;
	FILE     *fp;
	int       recNum;
	char      path[MAXPATH];

	sprintf( path, "%s\\BINLOG.PB", systemPath );
	fp = fopen( path, "rb" );
	if( !fp ){
		printf( "Unable to open the file '%s'.\n", path );
		return -1;
	}
	fseek( fp, 0, SEEK_END );

	for( ; recToRead; --recToRead ){
		if( 0L == ftell(fp) ) break; /* reached start of file */

		/* backup once, read record and backup again */
		fseek( fp, 0L - sizeof(BINLOG), SEEK_CUR );
		if( 1 != fread( &binRec, sizeof(BINLOG), 1, fp ) ) break;
		fseek( fp, 0L - sizeof(BINLOG), SEEK_CUR );

		/* see if only today and the date is smaller, quit reading */
		if( _todayOnly && date_diff(binRec.date[0], binRec.date[1],
				binRec.date[2] + 1990, _dateToday[0], _dateToday[1],
				_dateToday[2] + 1990) )
			break;

		strcpy( rec.name, binRec.name );
		strcpy( rec.city, binRec.city );
		rec.baud = binRec.baud;
		rec.node = binRec.node;
		memcpy( rec.lastDate, binRec.date, sizeof(t_date) );
		memcpy( rec.lastTime, binRec.timeIn, sizeof(t_time) );
		rec.callsToday = 1;

		/* add some info from the user record */
		if( -1 == (recNum = NumUserName( rec.name )) ){
#if 0
			printf( "Warning...unable to retrive index"
				" for '%s'. Skipping...\n", rec.name  );
#endif
			recToRead++;
			continue;
		}

#if defined( PB_SDK )
		/* PB's reading records, 0-based */
		if( -1 == ReadUser( &userRec, recNum - 1 ) ){
#else
		if( -1 == ReadUser( recNum, &userRec ) ){
#endif
			printf( "Unable to retrieve user record.\n" );
			return -1;
		}

		if( _useAlias ) strcpy( rec.name, userRec.alias );
		strcpy( rec.state, userRec.state );
		rec.minTotal = userRec.totalTimeUsed;
		rec.callsTotal = userRec.timesCalled;
		rec.minToday = userRec.timeUsed;

		/* check if we have the record already */
		pRec = llfindfirst( list, &rec, compare_name );
		if( NULL == pRec ){
			if( NULL == lladd( list, &rec, sizeof(UserData),
								LLPUT_SORT, sort_function ) ){
				printf( "Unable to allocate memory for list node.\n" );
				return -1;
			}
		}
		else{
			if( !date_diff(pRec->lastDate[0],pRec->lastDate[1],pRec->lastDate[2] + 1990,
						   rec.lastDate[0], rec.lastDate[1], rec.lastDate[2] + 1990) )
				pRec->callsToday++;
			recToRead++;
		}
	}

	fclose( fp );
	return 0;
}

/*
 * prints the list to screen or to file
*/
	static int
print_list( llist *list )
{
	int       recNum = 0, maxNum;
	char      path[MAXPATH];
	UserData *pRec;
	FILE     *fp;

	maxNum = _disableThird ? MAX_NOTHIRD : MAX_THIRD;

	if( _useFile || _fileScreen ){
		sprintf( path, "%s\\WHOCALLS.ANS", textPath );
		fp = fopen( path, "w" );
		if( !fp ){
			printf( "Unable to create file '%s'.\n", path );
			return -1;
		}
		fprintf( fp, "%s", ANSI_HEADER );
	}
	if( !_useFile ) printf( "%s", ANSI_HEADER );

	for( pRec = llfirst(list); NULL != pRec; pRec = llnext(pRec) ){

		if( _useFile || _fileScreen ) print_rec( pRec, fp );
		else print_rec( pRec );

		recNum++;
		if( recNum == maxNum ){
			if( _useFile || _fileScreen ) fprintf( fp, "%s", ANSI_WAIT );
#if defined( PB_SDK )
			if( !_useFile ) printf( "%s%s", ANSI_WAIT, ANSI_HEADER );
#endif
			recNum = 0;
		}
	}

	/* see if we need an additional prompt */
	if( recNum ){
		if( _useFile || _fileScreen ){
			fprintf( fp, "%s", ANSI_WAIT );
			fclose( fp );
		}
		if( !_useFile ){
#if defined( PB_SDK )
			for( ; recNum < maxNum; recNum++ )
				printf( "\n%c\n", _disableThird ? ' ' : '\n' );
			printf( "%s", ANSI_WAIT );
#endif
		}
	}
	return 0;
}


/*
 * prints record to screen or file
*/
	static void
print_rec( UserData *rec, ... )
{

		   char *when;
	static char  how_many[100], dateBuf[200];
	static char  line1[200], line2[200], line3[200];
	static char *times[20] = { "once", "twice", "three", "four", "five",
		"six", "seven", "eight", "nine", "ten", "eleven", "twelve",
		"thirteen", "fourteen", "fifteen", "sixteen", "seventeen",
		"eighteen", "nineteen", "twenty" };

	if( TRUE == _oneLine ){
		if( 12 >  strlen(rec->state) )
			strncat( rec->state, "             ", 12 - strlen(rec->state) );
		sprintf( line1, "%s%-25.25s %s%14.14s, %-.12s "
			"%s%02d%s/%s%02d%s/%s%02d  %02d%s:%s%02d  %ld\n",
			ANSI_NAMECLR, rec->name, ANSI_TEXTCLR, rec->city, rec->state,
			ANSI_DIGITCLR, (int)rec->lastDate[1], ANSI_TEXTCLR,
			ANSI_DIGITCLR, (int)rec->lastDate[0], ANSI_TEXTCLR,
			ANSI_DIGITCLR, (int)rec->lastDate[2],
			(int)rec->lastTime[0], ANSI_TEXTCLR,
			ANSI_DIGITCLR, (int)rec->lastTime[1],
			rec->baud );
	}
	else{
		if( rec->callsToday > 20 ) strcpy( how_many, "a lot " );
		else sprintf( how_many, "%s%s%s%s ",
			ANSI_DIGITCLR, times[rec->callsToday - 1],
			ANSI_TEXTCLR,	(rec->callsToday > 2) ? " times" : ""  );

		if( !memcmp(rec->lastDate, _dateToday, sizeof(t_date)) )
			when = "today";
		else{
			sprintf( dateBuf, "%son %s%02d%s/%s%02d%s/%s%02d%s",
				ANSI_TEXTCLR, ANSI_DIGITCLR, (int)rec->lastDate[1],
				ANSI_TEXTCLR, ANSI_DIGITCLR, (int)rec->lastDate[0],
				ANSI_TEXTCLR, ANSI_DIGITCLR, (int)rec->lastDate[2],
				ANSI_TEXTCLR );
			when = dateBuf;
		}

		sprintf( line1, "%s%s %scalled %s%s (%s%d%s minute%s).\n",
			ANSI_NAMECLR, rec->name,
			ANSI_TEXTCLR, how_many, when, ANSI_DIGITCLR, rec->minToday,
			ANSI_TEXTCLR,	(rec->minToday == 1) ? "" : "s" );
		sprintf( line2, "%s   Last call: %s, %s at %s%02d%s:%s%02d%s on "
						"node %s%d%s, %s%ld%s baud.\n",
			ANSI_TEXTCLR,	rec->city, rec->state,
			ANSI_DIGITCLR, (int)rec->lastTime[0], ANSI_TEXTCLR,
			ANSI_DIGITCLR, (int)rec->lastTime[1], ANSI_TEXTCLR,
			ANSI_DIGITCLR, rec->node, ANSI_TEXTCLR,
			ANSI_DIGITCLR, rec->baud, ANSI_TEXTCLR );
		if( !_disableThird )
			sprintf( line3, "%s   Statistic: user has called %s%ld%s time%s "
							"with %s%ld%s minute%s online.\n",
				ANSI_TEXTCLR, ANSI_DIGITCLR, rec->callsTotal,
				ANSI_TEXTCLR, (rec->callsTotal == 1) ? "" : "s",
				ANSI_DIGITCLR, rec->minTotal,
				ANSI_TEXTCLR, (rec->minTotal == 1) ? "" : "s" );
	}

	if( _useFile || _fileScreen ){
		va_list  arg;
		FILE    *fp;

		va_start( arg, rec );
		fp = va_arg( arg, FILE *);
		va_end( arg );

		if( !fp ) return;
		fprintf( fp, "%s%s%s", line1, line2, _disableThird ? "" : line3 );
	}
	if( !_useFile ){
		printf( line1 );
		if( !_oneLine )	printf( line2 );
		if( !_disableThird ) printf( line3 );
	}
}
