/*
 * This file is part of VizPak (PEX)
 *
 * Copyright (C) 1995-1997 by Branislav L. Slantchev
 * A Product of Silicon Creations, Inc.
 *
 * This file is distributed under the terms and conditions of the GNU
 * General Public License. For more information, refer to the file
 * Copying.Doc which is included in the archive.
*/
#include "vizpak.h"
#include "getopt.h"
#include "file.h"
#include "stdmac.h"
#include "profile.h"
#include "proboard.h"
#include "pblsdk.h"
#include "utils.h"

#ifndef PB_SDK
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
#else
	#include "pblibc.h"
#endif

typedef struct
{
	char copyright[80];
	char idString[7];
	long numCookies;
} COOKIE_IDX;

static void getCookie(char *text, const char *filename, const char *prog);
static int* getPalette(int *nColors, int nPalette);

static const char* COOKIE_ID      = "SC-IDX";
static const char* DefaultText    = "Silicon Creations, inc. (gargoyle)";
static       char* DefaultPalette = "0,8,7,15,7,8";

/*
*   -s x;y    - start x;y
*   -e x[;y]  - end x;y (y used only for Bounce)
*   -d 900    - delay
*   -r        - reverse
*   -c        - center
*   -t "..."  - text
*   -f name   - baked file
*   -p num    - color to use (or palette)
*   -n        - add strobe to Bounce/Marquee (fancy)
*   -k num    - skip step
*   -1        - do the effect only once and exit

   Palette used by Strobe and Glisten (Bounce & Marquee only if fancy)
*/

AbstractViz::AbstractViz(int argc, char *argv[], char *argv0,
						 FancyType isFancy, CenterType isCenter)
	:m_bounds(1,24,79,24)
	,m_nDelay(DEFAULTDELAY)
	,m_nColor(1)
	,m_nSkip(1)
	,m_flags(isFancy == forceFancy ? fancy : 0)
	,m_palette(0)
	,m_nPaletteDefs(0)
	,m_exitCode(0)
{
	int opt;

	strcpy(m_text, DefaultText);

	opterr = 0;
	while( EOF != (opt = getopt(argc, argv, "s:e:d:t:f:k:p:nrc1")) )
	{
		char *p;

		switch( opt )
		{
			case 's':
				m_bounds.a.x = atoi(optarg);
				if( 0 != (p = strchr(optarg, ';')) ) m_bounds.a.y = atoi(p+1);
				break;
			case 'e':
				m_bounds.b.x = atoi(optarg);
				if( 0 != (p = strchr(optarg, ';')) ) m_bounds.b.y = atoi(p+1);
				break;
			case 'd': m_nDelay = atoi(optarg);          break;
			case 't': strcpy(m_text, optarg);           break;
			case 'f': getCookie(m_text, optarg, argv0); break;
			case 'p': m_nColor = atoi(optarg);          break;
			case 'n': m_flags |= fancy;                 break;
			case 'r': m_flags |= reverse;               break;
			case 'c': m_flags |= center;                break;
			case '1': m_flags |= once;                  break;
			case 'k': m_nSkip = atoi(optarg);           break;
			default :
				Log(LOG_FRIEND, "vizPak: unknown option: '%c'", opt);
				exit(EXIT_FAILURE);
		}
	}

	m_nTextSize = strlen(m_text);

	// some effects are fancy by default (Strobe, Glisten)
	// others may have this capability requested (Marquee, Bounce)
	if( m_flags & fancy )
	{
		m_palette = getPalette(&m_nPaletteDefs, m_nColor);
	}

	if( m_bounds.a.x >= m_bounds.b.x || m_bounds.a.y > m_bounds.b.y ||
		m_bounds.b.x > 79 || m_bounds.a.x < 1 || m_bounds.b.y > 24)
	{
		Log(LOG_FRIEND, "vizPak: invalid coordinates specified");
		exit(EXIT_FAILURE);
	}

	if( (m_flags & center) && (isCenter == okToCenter) )
	{
		int width = m_bounds.b.x - m_bounds.a.x;
		// chop off text string or adjust starting coordinate
		if( width < m_nTextSize )
		{
			m_text[width] = EOS;
			m_nTextSize = width;
		}
		else m_bounds.a.x += (width - m_nTextSize) / 2;
	}

#ifndef PB_SDK
	_wscroll = 0;
#endif
}

AbstractViz::~AbstractViz()
{
	if( m_palette  ) delete[] m_palette;

	if( m_exitCode )
	{
		key_flush();
		key_poke(m_exitCode);
	}
}

void
AbstractViz::putns(const char *text, int n)
{
	static int cycle   = 0;
	static int skipped = m_nSkip;

	if( m_flags & fancy )
	{
		if( skipped == m_nSkip )
		{
			textattr(m_palette[cycle]);
			cycle = (cycle + 1) % m_nPaletteDefs;
		}
		skipped = abs((skipped + 1) % (m_nSkip + 1));
	}
	else textattr(m_nColor);

	for( ; n && EOS != *text; --n ) putch(*text++);
}

/****************************************************************************
 * local helper functions used by this module only
*/
void
getCookie(char *text, const char *filename, const char *progname)
{
	COOKIE_IDX  header;
	char        idxpath[MAXPATH], path[MAXPATH];
	long        ofs = 0L;
	ushort      num;
	FILE       *fp;

	// first, try the full path name
	if( file_exist(filename) ) strcpy(path, filename);
	else strcat(strcpy(path, file_dirspec(progname)), filename);

	// try finding the file in the same directory as the .PEX
	file_chext(idxpath, path, ".idx");
	if( 0 == (fp = fopen(idxpath, "rb")) )
	{
		Log(LOG_FRIEND, "vizPak: can't open baked cookie file '%s'", idxpath);
		exit(EXIT_FAILURE);
	}

	randomize();
	fread(&header, sizeof(header), 1, fp);
	if( memcmp(header.idString, COOKIE_ID, sizeof(COOKIE_ID)) )
	{
		Log(LOG_FRIEND, "vizPak: spoiled cookie file '%s'", idxpath);
		fclose(fp);
		exit(EXIT_FAILURE);
	}

	// get a random offset from the index
	num = (rand() % (ushort)header.numCookies) - 1;
	fseek(fp, (long)num * sizeof(long), SEEK_CUR);
	fread(&ofs, sizeof(long), 1, fp);
	fclose(fp);

	// read in a line
	if( 0 == (fp = fopen(path, "rt")) )
	{
		Log(LOG_FRIEND, "vizPak: can't eat baked cookies in '%s'", path);
		exit(EXIT_FAILURE);
	}

	fseek(fp, ofs, SEEK_SET);
	fgets(text, BUFFERSIZE, fp);
	fclose(fp);

	// get rid of the trailing newline
	text[strlen(text) - 1] = EOS;
}

int*
getPalette(int *nColors, int nPalette)
{
	zProfile  config;
	int       array[512], n = 0, *pColors = 0;
	char      text[1024];

	*nColors = 0;
	config.Assign( pb_getpath("SILICON.INI", PBDIR_SYS) );
	if( config.Open() )
	{
		char var[30];

		sprintf(var, "Palette%d", nPalette);
		strcpy(text, config.ReadString("Colors", var, DefaultPalette));
		config.Close();
	}
	else strcpy(text, DefaultPalette);

	char *p = strtok(text, ", ");
	while( p )
	{
		array[n++] = atoi(p);
		p = strtok(NULL, ", ");
	}

	if( 0 != (pColors = new int [n]) )
	{
		memcpy(pColors, array, n * sizeof(int));
		*nColors = n;
	}

	return pColors;
}
