//
// ProBoard RIP-library
//
// version 1.0 (c) Edwin Groothuis
//
// This RIP-library is not freeware. It's not allowed to use this code
// without mentioning me in the documentation as something like
// "This program uses the ProBoard RIP-library by Edwin Groothuis."
// and telling me you're using this library. Maybe you could give me
// a registrered copy of the program which uses this library (hint,
// hint). It's not allowed to spread this library with a modified source.
//
// Disclaimer: This library is not 100% bugfree. I've tried to find all
//             the typos, bugs etc but it's possible you still can find
//             one. If you do, please let me know so I can fix it and
//             give people using this library an update.
//
// Author:
// Edwin Groothuis                   ECA-BBS
// Johann Strausslaan 1              +31-40-550352 (14k4/V42B)
// 5583ZA Aalst-Waalre               2:284/205.1@fidonet
// The Netherlands                   115:3145/102.1@pascal-net
//
//
// 1994 mar 14 mon: First start
//



#include "..\pb_sdk.h"
#include "pb_rip.h"

static char table[36]="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

char    s0[10];
char    s1[10];
char    s2[10];
char    s3[10];
char    s4[10];
char    s5[10];
char    s6[10];
char    s7[10];
char    s8[10];
char    s9[10];
char    sa[10];
char    sb[10];
char    sc[10];
char    sd[10];
char    se[10];
char    sf[10];

char    *btom(byte b,char *s)
{
    strcpy(s,"0");s[1]=0;
    s[0]=table[b%36];
    return s;
}


char    *itom(int i,char *s)
{
    int     teller;

    strcpy(s,"00");teller=1;s[2]=0;
    while (i>0)
    {
        s[teller--]=table[i%36];
        i/=36;
    }
    return s;
}


char    *ltom(long l,char *s)
{
    int     teller;

    strcpy(s,"0000");teller=3;s[4]=0;
    while (l>0)
    {
        s[teller--]=table[l_mod(l,36)];
        l=l_div(l,36);
    }
    return s;
}



// ESC[!     Query RIPscrip version number.  RIPterm will respond with
//           RIPSCRIPxxyyvs where "xx" is equal to the major version
//           number (zero padded), "yy" is equal to the minor version
//           number (zero padded), "v" is the vendor code of the
//           terminal program (see below), and "s" is the vendor's
//           sub-version code for their software.  For v1.54, the
//           returned sequence for RIPterm (Vendor Code "1") would be
//           RIPSCRIP015410.  Another example, v1.23 with a Vendor Code
//           of "2" and a sub-revision code of "5" would return
//           RIPSCRIP012325.
//
//           Valid Vendor Codes are:
//
//                CODE   VENDOR
//                =================================================
//                  0    Generic RIPscrip terminal (vendor unknown)
//                  1    RIPterm (from TeleGrafix Communications)
//                  2    Qmodem Pro (from Mustang Software, Inc)
//
//           This ANSI sequence is often used for "Auto-Sensing" if a
//           RIPscrip terminal exists on the remote end of the
//           connection.  If a non-RIPscrip terminal receives this
//           ANSI sequence, it will ignore it.
char    *RIP_version(void)
{
    char    c;
    char    s[100];
    int     i;

    LocalDisplay(FALSE);
    printf("%c[!\n",27);
    LocalDisplay(TRUE);
    delay(500);
    if ((c=PeekChar())==0)
        return NULL;

    i=0;s[0]=0;
    while (i<14)
    {
        s[i++]=c;
        s[i]=0;
        if (i!=14)
        {
            c=PeekChar();
            if (c==0)
                return NULL;
        }
    }
    return s;
}



// ESC[2!    Enabled RIPscrip processing.  Any RIPscrip sequences will
//           be parsed and processed.
void    RIP_enable(void)
{
    LocalDisplay(FALSE);
    printf("%c[2!\n",27);
    LocalDisplay(TRUE);
}



// ESC[1!    Disables all RIPscrip processing.  Any RIPscrip sequences
//           are interpreted as raw text.
void    RIP_disable(void)
{
    LocalDisplay(FALSE);
    printf("%c[1!\n",27);
    LocalDisplay(TRUE);
}



// Function :  Define the size and location of the Text Window,
//             wraparound and fontsize (0-4)
// Arguments:  x0:2, y0:2, x1:2, y1:2, wrap:1, size:1
// Format   :  !|w <x0> <y0> <x1> <y1> <wrap> <size>
void    RIP_text_window(int x1,int y1,int x2,int y2,bool wrap,int size)
{
    LocalDisplay(FALSE);
    printf("!|w%s%s%s%s%c%c\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3),
                               wrap?'1':'0','0'+size);
    LocalDisplay(TRUE);
}



// Function :  Define the size & location of the Graphics Window
// Arguments:  x0:2, y0:2, x1:2, y1:2
// Format   :  !|v <x0> <y0> <x1> <y1>
void    RIP_viewport(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|v%s%s%s%s%\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Clear Graphics/Text Windows & reset to full screen
// Arguments:  <none>
// Format   :  !|*
void    RIP_reset_windows(void)
{
    LocalDisplay(FALSE);
    printf("!|*\n");
    LocalDisplay(TRUE);
}



// Function :  Clears Text Window to current background color
// Arguments:  <none>
// Format   :  !|e
void    RIP_erase_window(void)
{
    LocalDisplay(FALSE);
    printf("!|e\n");
    LocalDisplay(TRUE);
}



// Function :  Clear Graphics Window to current background color
// Arguments:  <none>
// Format   :  !|E
void    RIP_erase_view(void)
{
    LocalDisplay(FALSE);
    printf("!|E\n");
    LocalDisplay(TRUE);
}



// Function :  Move text cursor to row & column in Text Window
// Arguments:  x:2, y:2
// Format   :  !|g <x> <y>
void    RIP_gotoxy(int x,int y)
{
    LocalDisplay(FALSE);
    printf("!|g%s%s\n",itom(x,s0),itom(y,s1));
    LocalDisplay(TRUE);
}



// Function :  Move cursor to upper-left corner of Text Window
// Arguments:  <none>
// Format   :  !|H
void    RIP_home(void)
{
    LocalDisplay(FALSE);
    printf("!|H\n");
    LocalDisplay(TRUE);
}



// Function :  Erase current line from cursor to end of line
// Arguments:  <none>
// Format   :  !|>
void    RIP_erase_eol(void)
{
    LocalDisplay(FALSE);
    printf("!|>\n");
    LocalDisplay(TRUE);
}



// Function :  Set current Drawing Color for graphics
// Arguments:  color:2
// Format   :  !|c <color>
void    RIP_color(int c)
{
    LocalDisplay(FALSE);
    printf("!|c%s\n",itom(c,s0));
    LocalDisplay(TRUE);
}



// Function :  Set 16-color Palette from master 64-color palette
// Arguments:  c1:2, c2:2, ... c16:2
// Format   :  !|Q <c1> <c2> ... <c16>
void    RIP_set_palette(int c1,int c2,int c3,int c4,int c5,int c6,int c7,int c8,
                        int c9,int c10,int c11,int c12,int c13,int c14,int c15,int c16)
{
    LocalDisplay(FALSE);
    printf("!|Q%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n",itom(c1,s0),itom(c2,s1),
                                                   itom(c3,s2),itom(c4,s3),
                                                   itom(c5,s4),itom(c6,s5),
                                                   itom(c7,s6),itom(c8,s7),
                                                   itom(c9,s8),itom(c10,s9),
                                                   itom(c11,sa),itom(c12,sb),
                                                   itom(c13,sc),itom(c14,sd),
                                                   itom(c15,se),itom(c16,sf));
    LocalDisplay(TRUE);
}



// Function :  Set color of 16-color Palette from Master Palette
// Arguments:  color:2 value:2
// Format   :  !|a <color> <value>
void    RIP_one_palette(int color,int value)
{
    LocalDisplay(FALSE);
    printf("!|a%s%s\n",itom(color,s0),itom(value,s1));
    LocalDisplay(TRUE);
}



// Function :  Set drawing mode for graphics primitives
// Arguments:  mode:2
// Format   :  !|W <mode>
void    RIP_write_mode(int mode)
{
    LocalDisplay(FALSE);
    printf("!|W%s\n",itom(mode,s0));
    LocalDisplay(TRUE);
}



// Function :  Move the current drawing position to (X,Y)
// Arguments:  x:2, y:2
// Format   :  !|m <x> <y>
void    RIP_move(int x,int y)
{
    LocalDisplay(FALSE);
    printf("!|m%s%s\n",itom(x,s0),itom(y,s1));
    LocalDisplay(TRUE);
}



// Function :  Draw text in current font/color at current spot
// Arguments:  text-string
// Format   :  !|T <text-string>
void    RIP_text(char *s)
{
    LocalDisplay(FALSE);
    printf("!|T%s\n",s);
    LocalDisplay(TRUE);
}



// Function :  Draw text in current font/color at specific spot
// Arguments:  x:2, y:2 and text-string
// Format   :  !|@ <x> <y> <text-string>
void    RIP_text_xy(int x,int y,char *s)
{
    LocalDisplay(FALSE);
    printf("!|@%s%s%s\n",itom(x,s0),itom(y,s1),s);
    LocalDisplay(TRUE);
}



// Function :  Select current font style, orientation and size
// Arguments:  font:2, direction:2, size:2, res:2
// Format   :  !|Y <font> <direction> <size> <res>
// Notes    :  <res> always 00
void    RIP_font_style(int font,int direction,int size)
{
    LocalDisplay(FALSE);
    printf("!|Y%s%s%s00\n",itom(font,s0),itom(direction,s1),itom(size,s2));
    LocalDisplay(TRUE);
}



// Function :  Draws a one pixel using current drawing color
// Arguments:  x:2, y:2
// Format   :  !|X <x> <y>
void    RIP_pixel(int x,int y)
{
    LocalDisplay(FALSE);
    printf("!|X%s%s\n",itom(x,s0),itom(y,s1));
    LocalDisplay(TRUE);
}



// Function :  Draw a line in the current color/line style
// Arguments:  x0:2, y0:2, x1:2, y1:2
// Format   :  !|L <x0> <y0> <x1> <y1>
void    RIP_line(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|L%s%s%s%s\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Draw a rectangle in current color/line style
// Arguments:  x0:2, y0:2, x1:2, y1:2
// Format   :  !|R <x0> <y0> <x1> <y1>
void    RIP_rectangle(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|R%s%s%s%s\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Draw filled rectangle with fill color/pattern
// Arguments:  x0:2, y0:2, x1:2, y1:2
// Format   :  !|B <x0> <y0> <x1> <y1>
void    RIP_bar(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|B%s%s%s%s\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Draw circle in current color and line thickness
// Arguments:  x_center:2, y_center:2, radius:2
// Format   :  !|C <x_center> <y_center> <radius>
void    RIP_circle(int x,int y,int radius)
{
    LocalDisplay(FALSE);
    printf("!|C%s%s%s\n",itom(x,s0),itom(y,s1),itom(radius,s2));
    LocalDisplay(TRUE);
}



// Function :  Draw elliptical arc in current color/line style
// Arguments:  x:2, y:2, st_ang:2, end_ang:2, x_rad:2, y_rad:2
// Format   :  !|O <x> <y> <st_ang> <end_ang> <x_rad> <y_rad>
void    RIP_oval(int x,int y,int start_angle,int end_angle,
                 int radius_x,int radius_y)
{
    LocalDisplay(FALSE);
    printf("!|O%s%s%s%s%s%s\n",itom(x,s0),itom(y,s1),
                               itom(start_angle,s2),itom(end_angle,s3),
                               itom(radius_x,s4),itom(radius_y,s5));
    LocalDisplay(TRUE);
}



// Function :  Draw filled ellipse using current color/pattern
// Arguments:  x_center:2, y_center:2, x_rad:2, y_rad:2
// Format   :  !|o <x_center> <y_center> <x_rad> <y_rad>
void    RIP_filled_oval(int x,int y,int radius_x,int radius_y)
{
    LocalDisplay(FALSE);
    printf("!|o%s%s%s%s\n",itom(x,s0),itom(y,s1),
                           itom(radius_x,s2),itom(radius_y,s3));
    LocalDisplay(TRUE);
}



// Function :  Draw circular arc in current color/line thickness
// Arguments:  x:2, y:2, start_ang:2, end_ang:2, radius:2
// Format   :  !|A <x> <y> <start_ang> <end_ang> <radius>
void    RIP_arc(int x,int y,int start_angle,int end_angle,int radius)
{
    LocalDisplay(FALSE);
    printf("!|A%s%s%s%s%s\n",itom(x,s0),itom(y,s1),
                             itom(start_angle,s2),itom(end_angle,s3),
                             itom(radius,s4));
    LocalDisplay(TRUE);
}



// Function :  Draw an elliptical arc
// Arguments:  x:2, y:2, st_ang:2, e_ang:2, radx:2 rady:2
// Format   :  !|V <x> <y> <st_ang> <e_ang> <radx> <rady>
void    RIP_oval_arc(int x,int y,int start_angle,int end_angle,
                     int radius_x,int radius_y)
{
    LocalDisplay(FALSE);
    printf("!|V%s%s%s%s%s%s\n",itom(x,s0),itom(y,s1),
                               itom(start_angle,s2),itom(end_angle,s3),
                               itom(radius_x,s4),itom(radius_y,s5));
    LocalDisplay(TRUE);
}



// Function :  Draws a circular pie slice
// Arguments:  x:2, y:2, start_ang:2, end_ang:2, radius:2
// Format   :  !|I <x> <y> <start_ang> <end_ang> <radius>
void    RIP_pie_slice(int x,int y,int start_angle,int end_angle,int radius)
{
    LocalDisplay(FALSE);
    printf("!|I%s%s%s%s%s\n",itom(x,s0),itom(y,s1),
                             itom(start_angle,s2),itom(end_angle,s3),
                             itom(radius,s4));
    LocalDisplay(TRUE);
}



// Function :  Draws an elliptical pie slice
// Arguments:  x:2, y:2, st_ang:2, e_ang:2, radx:2 rady:2
// Format   :  !|i <x> <y> <st_ang> <e_ang> <radx> <rady>
void    RIP_oval_pie_slice(int x,int y,int start_angle,int end_angle,
                           int radius_x,int radius_y)
{
    LocalDisplay(FALSE);
    printf("!|i%s%s%s%s%s%s\n",itom(x,s0),itom(y,s1),
                               itom(start_angle,s2),itom(end_angle,s3),
                               itom(radius_x,s4),itom(radius_y,s5));
    LocalDisplay(TRUE);
}



// Function :  Draw a bezier curve
// Arguments:  x1:2 y1:2 x2:2 y2:2 x3:2 y3:2 x4:2 y4:2 cnt:2
// Format   :  !|Z <x1> <y1> <x2> <y2> <x3> <y3> <x4> <y4> <cnt>
void    RIP_bezier(int x1,int y1,int x2,int y2,
                   int x3,int y3,int x4,int y4,int seg)
{
    LocalDisplay(FALSE);
    printf("!|Z%s%s%s%s%s%s%s%s%s\n",itom(x1,s0),itom(y1,s1),
                                     itom(x2,s2),itom(y2,s3),
                                     itom(x3,s4),itom(y3,s5),
                                     itom(x4,s6),itom(y4,s7),
                                     itom(seg,s8));
    LocalDisplay(TRUE);
}



// Function :  Draw polygon in current color/line-style
// Arguments:  npoints:2, x1:2, y1:2, ... xn:2, yn:2
// Format   :  !|P <npoints> <x1> <y1> ... <xn> <yn>
void    RIP_polygon(void)
{
    // NYI
}



// Function :  Draw filled polygon in current color/fill pattern
// Arguments:  npoints:2, x1:2, y1:2, ... xn:2, yn:2
// Format   :  !|p <npoints> <x1> <y1> ... <xn> <yn>
void    RIP_fill_polygon(void)
{
    // NYI
}



// Function :  Draw a Poly-Line (multi-faceted line)
// Arguments:  npoints:2, x1:2, y1:2, ... xn:2, yn:2
// Format   :  !|l <npoints> <x1> <y1> ... <xn> <yn>
void    RIP_polyline(void)
{
    // NYI
}



// Function :  Flood fill screen area with current fill settings
// Arguments:  x:2, y:2, border:2
// Format   :  !|F <x> <y> <border>
void    RIP_fill(int x,int y,int bordercolor)
{
    LocalDisplay(FALSE);
    printf("!|F%s%s%s\n",itom(x,s0),itom(y,s1),itom(bordercolor,s2));
    LocalDisplay(TRUE);
}



// Function :  Defines a line style and thickness
// Arguments:  style:2, user_pat:4, thick:2
// Format   :  !|= <style> <user_pat> <thick>
void    RIP_line_style(int style,long userpattern,int thickness)
{
    LocalDisplay(FALSE);
    printf("!|=%s%s%s\n",itom(style,s0),ltom(userpattern,s1),itom(thickness,s2));
    LocalDisplay(TRUE);
}



// Function :  Set current fill style (predefined) & fill color
// Arguments:  pattern:2, color:2
// Format   :  !|S <pattern> <color>
void    RIP_fill_style(int pattern,int color)
{
    LocalDisplay(FALSE);
    printf("!|S%s%s\n",itom(pattern,s0),itom(color,s1));
    LocalDisplay(TRUE);
}



// Function :  Set user-definable (custom) fill pattern/color
// Arguments:  c1:2 c2:2 c3:2 c4:2 c5:2 c6:2 c7:2 c8:2 col:2
// Format   :  !|s <c1> <c2> <c3> <c4> <c5> <c6> <c7> <c8> <col>
void    RIP_fill_pattern(int c1,int c2,int c3,int c4,
                         int c5,int c6,int c7,int c8,int color)
{
    LocalDisplay(FALSE);
    printf("!|s%s%s%s%s%s%s%s%s\n",itom(c1,s0),itom(c2,s1),
                                   itom(c3,s2),itom(c4,s3),
                                   itom(c5,s4),itom(c6,s5),
                                   itom(c7,s6),itom(c8,s7),
                                   itom(color,s8));
    LocalDisplay(TRUE);
}



// Function :  Defines a rectangular hot mouse region
// Arguments:  num:2 x0:2 y0:2 x1:2 y1:2 clk:1 clr:1 res:5 text
// Format   :  !|1M <num> <x0><y0><x1><y1> <clk><clr><res><text>
// Notes    :  <num> always 00
//             <res> always 00000
void    RIP_mouse(int x1,int y1,int x2,int y2,
                  int invert,int clr,char *text)
{
    LocalDisplay(FALSE);
    printf("!|1M00%s%s%s%s%c%c00000%s\n",itom(x1,s0),itom(y1,s1),
                                         itom(x2,s2),itom(y2,s3),
                                         invert?'1':'0',clr?'1':'0',text);
    LocalDisplay(TRUE);
}



// Function :  Destroys all previously defined hot mouse regions
// Arguments:  <none>
// Format   :  !|1K
void    RIP_kill_mouse_fields(void)
{
    LocalDisplay(FALSE);
    printf("!|1K\n");
    LocalDisplay(TRUE);
}



// Function :  Define a rectangular text region
// Arguments:  x1:2, y1:2, x2:2, y2:2, res:2
// Format   :  !|1T <x1> <y1> <x2> <y2> <res>
// Notes    :  <res> always 00
void    RIP_begin_text(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|1T%s%s%s%s00\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Display a line of text in rectangular text region
// Arguments:  justify:1 and text-string
// Format   :  !|1t <justify> <text-string>
void    RIP_region_text(bool justify,char *text)
{
    LocalDisplay(FALSE);
    printf("!|1t%c%s\n",justify?'1':'0',text);
    LocalDisplay(TRUE);
}



// Function :  End a rectangular text region
// Arguments:  <none>
// Format   :  !|1E
void    RIP_end_text(void)
{
    LocalDisplay(FALSE);
    printf("!|1E\n");
    LocalDisplay(TRUE);
}



// Function :  Copy rectangular image to clipboard (as icon)
// Arguments:  x0:2, y0:2, x1:2, y1:2, res:1
// Format   :  !|1C <x0> <y0> <x1> <y1> <res>
// Notes    :  <res> always 0
void    RIP_get_image(int x1,int y1,int x2,int y2)
{
    LocalDisplay(FALSE);
    printf("!|1C%s%s%s%s0\n",itom(x1,s0),itom(y1,s1),itom(x2,s2),itom(y2,s3));
    LocalDisplay(TRUE);
}



// Function :  Pastes the clipboard contents onto the screen
// Arguments:  x:2, y:2, mode:2, res:1
// Format   :  !|1P <x> <y> <mode> <res>
// Notes    :  <res> always 0
void    RIP_put_image(int x,int y,int mode)
{
    LocalDisplay(FALSE);
    printf("!|1P%s%s%s0\n",itom(x,s0),itom(y,s1),itom(mode,s2));
    LocalDisplay(TRUE);
}



// Function :  Write contents of the clipboard (icon) to disk
// Arguments:  res:1, filename
// Format   :  !|1W <res> <filename>
// Notes    :  <res> always 0
void    RIP_write_icon(char *filename)
{
    LocalDisplay(FALSE);
    printf("!|0%s\n",filename);
    LocalDisplay(TRUE);
}



// Function :  Loads and displays a disk-based icon to screen
// Arguments:  x:2, y:2, mode:2, clipboard:1, res:2, filename
// Format   :  !|1I <x> <y> <mode> <clipboard> <res> <filename>
// Notes    :  <res> always 10
void    RIP_load_icon(int x,int y,int mode,bool clipboard,char *filename)
{
    LocalDisplay(FALSE);
    printf("!|1I%s%s%s%c10%s\n",itom(x,s0),itom(y,s1),itom(mode,s2),
                                clipboard?'1':'0',filename);
    LocalDisplay(TRUE);
}



// Function :  Button style definition
// Arguments:  wid:2 hgt:2 orient:2 flags:4 size:2 dfore:2 dback:2
//             bright:2 dark:2 surface:2 grp_no:2 flags2:2 uline_col:2
//             corner_col:2 res:6
// Format   :  !|1B <wid> <hgt> <orient> <flags> <bevsize> <dfore>
//             <dback> <bright> <dark> <surface> <grp_no> <flags2>
//             <uline_col> <corner_col> <res>
// Notes    :  <res> always 000000
void    RIP_button_style(int width,int height,int orient,long flags,
                         int size,int dfore,int dback,int bright,
                         int dark,int surface,int group_no,int flags2,
                         int uline_col,int corner_col)
{
    LocalDisplay(FALSE);
    printf("!|1B%s%s%s%s%s%s%s%s%s%s%s%s%s%s000000\n",
                        itom(width,s0),itom(height,s1),itom(orient,s2),
                        ltom(flags,s3),itom(size,s4),itom(dfore,s5),
                        itom(dback,s6),itom(bright,s7),itom(dark,s8),
                        itom(surface,s9),itom(group_no,sa),itom(flags2,sb),
                        itom(uline_col,sc),itom(corner_col,sd));
    LocalDisplay(TRUE);
}



// Function :  Define a Mouse Button
// Arguments:  x0:2 y0:2 x1:2 y1:2 hotkey:2 flags:1 res:1 text
// Format   :  !|1U <x0> <y0> <x1> <y1> <hotkey> <flags> <res> <text>
// Notes    :  <res> always 0
void    RIP_button(int x1,int y1,int x2,int y2,int hotkey,int flags,char *text)
{
    LocalDisplay(FALSE);
    printf("!|1U%s%s%s%s%s%s0%s\n",itom(x1,s0),itom(y1,s1),
                                   itom(x2,s2),itom(y2,s3),
                                   itom(hotkey,s4),btom(flags,s5),text);
    LocalDisplay(TRUE);
}



// Function :  Define a text variable
// Arguments:  flags:3 res:2 ...text
// Format   :  !|1D <flags> <res> <text>
// Notes    :  <res> always 00
void    RIP_define(int flags,char *text)
{
    LocalDisplay(FALSE);
    printf("!|1D00%c00%s\n",flags+'0',text);
    LocalDisplay(TRUE);
}



// Function :  Query the contents of a text variable
// Arguments:  mode:1 res:3 ...text
// Format   :  !|1<escape> <mode> <res> <text>
// Notes    :  <res> always 000
char    *RIP_query(int mode,char *text,bool ret,int delaytime,char *s)
{
    char    c;
    int     i;

    LocalDisplay(FALSE);
    printf("!|1%c%c000%s\n",27,mode+'0',text);
    LocalDisplay(TRUE);

    if (s==NULL)
        return s;

    if (ret)
    {
        delay(delaytime);
        if ((c=getchar())==0)
        {
            s=NULL;
            return s;
        }

        i=0;s[0]=0;
        while (i<14)
        {
            s[i++]=c;
            s[i]=0;
            if (i!=14)
            {
                c=PeekChar();
                if (c==0)
                {
                    delay(100);
                    c=PeekChar();
                }
            }
        }
        return s;
    } else {
        s=NULL;
        return NULL;
    }
}



// Function :  Copy screen region up/down
// Arguments:  x0:2 y0:2 x1:2 y1:2 res:2 dest_line:2
// Format   :  !|1G <x0> <y0> <x1> <y1> <res> <dest_line>
// Notes    :  <res> always 00
void    RIP_copy_region(int x1,int y1,int x2,int y2,int newy)
{
    LocalDisplay(FALSE);
    printf("!|1G%s%s%s%s00%s\n",itom(x1,s0),itom(y1,s1),
                                itom(x2,s2),itom(y2,s3),itom(newy,s4));
    LocalDisplay(TRUE);
}



// Function :  Playback local .RIP file
// Arguments:  res:8 filename...
// Format   :  !|1R <res> <filename>
// Notes    :  <res> always 00000000
void    RIP_read_scene(char *filename)
{
    LocalDisplay(FALSE);
    printf("!|1R00000000%s\n",filename);
    LocalDisplay(TRUE);
}



// Function :  Query existing information on a particular file
// Arguments:  mode:2 res:4 filename...
// Format   :  !|1F <mode> <res> <filename>
char    *RIP_file_query(int mode,char *filename)
{
    char    s[80];
    int     i;
    char    c;


    LocalDisplay(FALSE);
    printf("!|1F%s0000%s\n",itom(mode,s0),filename);
    LocalDisplay(TRUE);
    delay(500);
    if ((c=getchar())==0)
        return NULL;

    i=0;s[0]=0;
    while (i<14)
    {
        s[i++]=c;
        s[i]=0;
        if (i!=14)
        {
            c=PeekChar();
            if (c==0)
            {
                delay(100);
                c=PeekChar();
            }
        }
    }
    return s;
}



// Function :  Enter block transfer mode with host
// Arguments:  mode:1 proto:1 file_type:2 res:4 [filename:2] <>
// Format   :  !|9<escape> <mode> <proto> <file_type> <res> [filename] <>
// Notes    :  <res> always 0000
void    RIP_enter_block_mode(bool upload,int proto,int filetype,char *filename)
{
    LocalDisplay(FALSE);
    printf("!|9%c%c%c%s0000%s\n",27,upload?'1':'0',proto+'0',
                                 filetype+'0',filename);
    LocalDisplay(TRUE);
}



// Function :  End of RIPscrip Scene
// Arguments:  <none>
// Format   :  !|#
void    RIP_no_more(void)
{
    LocalDisplay(FALSE);
    printf("!|#|#|#\n");
    LocalDisplay(TRUE);
}
