/*
 * ChkPro v1.0r4 - ProBoard Configuration Checking Utility
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, part of the PB-GNU Project
 *
 * For details, see the file "copying".
 *
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <sys/stat.h>
#include <fsys.h>
#include <fail.h>
#include <getopt.h>
#include <memdbg.h>
#include <proutil.h>

#define _VERSION_ "1.0r4"

static struct{
	bool  repair;
	bool  files;
	bool  binlog;
	bool  message;
	bool  pack_file;
    bool  pack_msg;
	bool  log;
	FILE *logfile;
	char  logpath[MAXPATH];
} opts = { FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, NULL, "\0" };

static char progname[MAXFILE+MAXEXT]; /* program's name */

/* local help and info functions */
static void _pascal print_version(void);
static void _pascal print_help(void);
static void _pascal print_license(void);

/* logging function */
static void _pascal __log_print(const char *format, va_list args);
void _pascal log(const char *format, ...);
void _pascal logexit(const char *format, ...);

/* processing functions */
extern int _pascal do_binlog(bool repair);
extern int _pascal do_filecfg(bool repair);
extern int _pascal do_message(bool repair);
extern int _pascal do_packfile(void);
extern int _pascal do_packmsg(void);

/*
 * guess what? this is main()
*/
	int
main( int argc, char *argv[] )
{
	int opt;

	atexit((atexit_t)mem_term);
	mem_init();

	strcpy(progname, strlwr(basename(*argv))); /* save program name */
	fprintf( stderr, "%s v%s Copyright (C) 1995 by Silicon Creations, Inc.\n\n",
		 progname, _VERSION_ );

	if( 1 == argc ) print_help();

	while( EOF != (opt = getopt(argc, argv, "bBfFrRmMkKpPl:L:vVhHcC"))){
		switch(opt){
			case 'b': case 'B': opts.binlog = TRUE; break;
			case 'f': case 'F': opts.files = TRUE; break;
			case 'r': case 'R': opts.repair = TRUE; break;
			case 'm': case 'M': opts.message = TRUE; break;
			case 'p': case 'P': opts.pack_file = TRUE; break;
            case 'k': case 'K': opts.pack_msg = TRUE; break;
			case 'l': case 'L':
				strcpy(opts.logpath, optarg);
				if( NULL == (opts.logfile = fopen(optarg, "a+")) ){
						logexit( "%s (fatal): could not access logfile '%s'\n",
								progname, optarg );
				}
				opts.log = TRUE;
			break;
			case 'v': case 'V': print_version(); break;
			case 'h': case 'H': print_help(); /* we exit here */
			case 'c': case 'C': print_license();  /* and here */
			default :
				logexit( "%s (fatal): -- illegal option\n", progname );
		}
	}

	/* ok, binlog checking */
	if( opts.binlog ){
		log( "Checking the binary log file.\n" );
		if( -1 == do_binlog(opts.repair) ) log("There were errors.\n");
		else log("Everything checks ok.\n");
	}
	if( opts.files ){
		log( "Checking the filebase configuration.\n" );
		if( -1 == do_filecfg(opts.repair) ) log("There were errors.\n");
		else log("Everything checks ok.\n");
	}
	if( opts.message ){
		log( "Checking the messagebase configuration.\n" );
		if( -1 == do_message(opts.repair) ) log("There were errors.\n");
		else log("Everything check ok.\n");
	}
	if( opts.pack_file ){
		log( "Packing the filebase areas.\n" );
		if( -1 == do_packfile() ) log("There were errors.\n");
		else log("Everything check ok.\n");
	}
    if( opts.pack_msg ){
		log( "Packing the messagebase areas.\n" );
		if( -1 == do_packmsg() ) log("There were errors.\n");
		else log("Everything check ok.\n");
	}

	/* see if the logfile is 0 length and remove it if it is */
	if( opts.log ){
		struct stat statbuf;

		fclose(opts.logfile);
		if( 0 == stat(opts.logpath, &statbuf) ){
			if( 0L == statbuf.st_size ) remove(opts.logpath);
		}
	}

	mem_term();
	return 0;
}

/**************************************************************************\
 ** Help module
\**************************************************************************/

	void _pascal
print_version(void)
{
	fprintf( stderr, "PB-GNU %s v%s. Compiled on %s at %s\n",
					progname, _VERSION_, __DATE__, __TIME__ );
}

	void _pascal
print_help(void)
{
	fprintf( stderr, "\nUsage: %s -[bfmrvpk] [-l file] [-c] [-h]\n"
					 "    -b        check BINLOG.PB\n"
					 "    -f        check FILECFG.PRO\n"
					 "    -m        check MESSAGES.PB\n"
					 "    -p        pack the file areas\n"
                     "    -k        pack the message areas\n"
					 "    -r        fix changes\n"
					 "    -l file   log fixes to 'file'\n"
					 "    -v        print version\n"
					 "    -c        license and copyright\n"
					 "    -h        this help screen\n\n", progname );
	exit(EXIT_SUCCESS);
}

	void _pascal
print_license(void)
{
	fprintf( stderr,
		"This is free software, distributed under the terms of the GNU\n"
		"General Public License. For details, see the file \"copying\".\n" );
	exit(EXIT_SUCCESS);
}

/**************************************************************************\
 ** Logging module
\**************************************************************************/

	void _pascal
__log_print(const char *format, va_list args)
{
	time_t timer;
	char   buf[20];

	vfprintf( stdout, format, args );

	if( TRUE == opts.log ){
		timer = time(NULL);
		strftime( buf, sizeof(buf), "%d-%b-%y %H:%M:%S", localtime(&timer) );
		fprintf( opts.logfile, "%s ", buf );
		vfprintf( opts.logfile, format, args );
	}
}

	void _pascal
log(const char *format, ...)
{
	va_list varg;

	va_start(varg, format);
	__log_print(format, varg);
	va_end(varg);
}

	void _pascal
logexit(const char *format, ...)
{
	va_list varg;

	fputc('\n', stdout);
	va_start(varg, format);
	__log_print(format, varg);
	va_end(varg);
	exit(EXIT_FAILURE);
}
