/*
 BinFix - BINLOG File Fixer
 Copyright (C) 1995 by Branislav L. Slantchev

 This file is part of BinFix.

 BinFix is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2.

 BinFix is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with BinFix; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dos.h>
#include <pblib.h>
#include <getopt.h>
#include <fsys.h>
#include <proutil.h>

#define __VERSION__ "1.01"
#define COPYRIGHT "binfix " __VERSION__ " Copyright (C) 1995 by "\
				  "Silicon Creations, Inc.\n"
#define VERSION   "PB-GNU binfix, Version " __VERSION__ \
	". Compiled on " __DATE__ " at " __TIME__ "\n"
#define COPYING "\n\
Copyright (C) 1995 by Branislav L. Slantchev                          \n\
Copyright (C) 1995 by Silicon Creations, Inc.                         \n\n\
This file is part of the PB-GNU Project.                              \n\n\
binfix is free software; you can redistribute it and/or modify        \n\
it under the terms of the GNU General Public License as published by  \n\
the Free Software Foundation; version 2.                              \n\n\
binfix is distributed in the hope that it will be useful,             \n\
but WITHOUT ANY WARRANTY; without even the implied warranty of        \n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         \n\
GNU General Public License for more details.                          \n\n\
You should have received a copy of the GNU General Public License     \n\
along with binfix; see the file COPYING.  If not, write to            \n\
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. \n"


static void do_binlog( bool );
static void fail( const char *fmt, ... );

int
main( int argc, char *argv[] )
{
	int opt;
	bool del_missing = FALSE;

	fputs( COPYRIGHT, stderr );

	if( 1 == argc ){
__helpScreen:
		fprintf( stderr, "usage: binfix -k | -[vch]\n" );
		fprintf( stderr, "      -k  remove all entries with missing users\n" );
		fprintf( stderr, "      -v  show version information\n" );
		fprintf( stderr, "      -c  show copyright information\n" );
		fprintf( stderr, "      -h  this show help screen\n" );
		return 1;
	}

	while( EOF != (opt = getopt(argc, argv, "kvhc" ) ) ){
		switch( opt ){
			case 'k': del_missing = TRUE; break;
			case 'h': goto __helpScreen;
			case 'v': fputs( VERSION, stderr ); break;
			case 'c': fputs( COPYING, stderr ); break;
			default : return 1;
		}
	}

	if( del_missing ) do_binlog( del_missing );
	return 0;
}

#pragma argsused
	static void
do_binlog( bool del_missing )
{
	FILE *fin, *fout;
	char path[MAXPATH], *p = getenv( "PROBOARD" );
	char back[MAXPATH];
	bool bad = FALSE;
    BINLOG buf;

	sprintf( path, "%s\\BINLOG.PB", p );
	chext( back, path, ".BAK" );

	if( -1 == rename( path, back ) )
		fail( "unable to backup file '%s'.\n", path );

	if( !(fin = fopen( back, "rb" )) )
		fail( "unable to open file '%s'.\n", back );
	if( !(fout = fopen( path, "wb" )) ){
		fclose( fin );
		fail( "unable to create file '%s'.\n", path );
	}

	for( ;; ){
		if( 1 != fread( &buf, sizeof(BINLOG), 1, fin ) ){
			if( feof(fin) ) break;
			bad = TRUE;
			goto __cleanup;
		}
		if( -1L == FindUserName( buf.name ) ){
			printf( "Removing '%s'...\n", buf.name );
			continue;
		}
		fwrite( &buf, sizeof(BINLOG), 1, fout );
	}

__cleanup:
	fclose( fin );
	fclose( fout );
	if( TRUE == bad ){
		unlink( path );
		rename( back, path );
		fail( "error processing BINLOG record.\n" );
	}
}


	static void
fail( const char *fmt, ... )
{
	va_list arg;

	fprintf( stderr, "%s: ", strlwr(_argv[0]) );
	va_start(arg, fmt);
	vfprintf( stderr, fmt, arg );
	va_end(arg);

	exit(1);
}