<?php

/* SVN FILE: $Id: mystic_library.php 28 2012-09-07 17:54:11Z frank $ */

/**
 * mystic_library.php : A set of functions to access mystic bbs data files
 *
 * This revision will only with with data files from Mystic 1.12 Alpha 31
 *
 * @package mystic_library
 * @author $Author: frank $
 * @author $Author: darryl $
 * @copyright $Copyright 2011 Frank Linhares/netsurge%demonic%bbs-scene.org$
 * @version $Revision: 28 $
 * @lastrevision $Date: 2016-08-23 13:54:11 -0600 (Tue, 23 Aug 2016) $
 * @modifiedby $LastChangedBy: darryl $
 * @lastmodified $LastChangedDate: 2016-08-23 13:54:11 -0600 (Tue, 23 Aug 2016) $
*/
 
/**
 * mystic_library.php
 *
 * @package mystic_library
 *
 * This library is a collection of functions that are used to access and parse various data file from Mystic BBS 1.12 A31+
 *
 * A description of each function and what they return is listed in the doc block for each function.
 *
 * Include this library in all pages that will be showing data from Mystic BBS like this:
 *
 * 		include "inc/mystic_library.php";
 *
 * Also set the path to Mystic's data directory like this:
 *
 *		$data_path = "/home/bbs/mystic/data/";
 *
 * Available functions are:
 *
 *	mystic_lastcallers(path to mystic data dir $var, number of last callers $var, "Y" to parse pipe colours or "N" to strip pipe colours "$");		
 *	mystic_chat(path to mystic data dir $var, number of nodes you are running $int);
 *	mystic_history(path to mystic data dir $var, "TODAY" to return todays stats only or "ALL" for all stats);
 *	mystic_oneliners(path to mystic data dir $var);
 *	mystic_vote(path to mystic data dir $var);
 *	mystic_userlist(path to mystic data dir $var, "Y" to parse pipe colours or "N" to strip pipe colours "$")
 *	mystic_bbslist(path to mystic data dir and file $var);
 *	mystic_fareas(path to mystic data dir $var);
 *	mystic_mareas(path to mystic data dir $var);
 *	
 * Support for mystic_php_library can be found on the official support bbs, miserybbs.com, cyberia.darktech.org
 *
**/
  
/**
 * read and parse last 10 callers from Mystic 1.10 A18+
 *
 * @uses: mystic_lastcallers(path to mystic data dir $var, number of last callers $var, "Y" to parse pipe colours or "N" to strip pipe colours "$")
 *
 * @return: an array containing the following keys identifying the last X callers; where X is set by $number. If no number is specified it will
 *		   display the the last ten callers:
 *
 * 		   user = username/handle
 *		   city = user's city
 *		   address = user's address
 *		   peer = peer ip
 *		   host = host name
 *		   date = time and date of call
 *		   node = node number
 *		   caller = caller number
 *		   email = user's email address
 *		   info = user's usernote
 *		   opt1 = user's opt1 field
 *		   opt2 = user's opt2 field
 *		   opt3 = user's opt3 field
 *		   opt4 = user's opt4 field
 *		   opt5 = user's opt5 field
 *		   opt6 = user's opt6 field
 *		   opt7 = user's opt7 field
 *		   opt8 = user's opt8 field
 *		   opt9 = user's opt9 field
 *		   opt10 = user's opt10 field
 *
 *
 * @author: Frank Linhares
**/

function mystic_lastcallers($data_path, $number = 10, $pipe = "N")
{
	$pipe = strtoupper($pipe);        

	if ($number > 10) {
		$number = 10;
	}
	 
	$fp = fopen ($data_path.'callers.dat', 'rb');
	for ($i = 0; $i < $number; $i++) {
		$data = fread ($fp, 896);
   
		// Create a data structure
		
		$data_format =
			'ldate/' .			# Get the date                                                        
			'cnewuser/' .		# Get the new user field 
      'Cpeerlen/' .		# Get the length of the peer ip field
			'A15peer/' .		# Get the peer ip (15) padded with null
			'Chostlen/' .		# Get the length of the peer host field
			'A50host/' .		# Get the peer host (30) padded with null
			'Cnode/' .	    	# Get the node number 
			'lcaller/' .		# Get the caller number       		
			'Cuserlen/' .		# Get the length of the user field
			'A30user/' .		# Get the username (30) padded with null
			'Ccitylen/' .		# Get the length of the city field
			'A25city/' .		# Get the city (25) padded with null
			'Caddresslen/' .	# Get the length of the address field
			'A30address/' .		# Get the address (30) padded with null      
			'Agender/' . 		# Get the gender (m for male f for female) 
			'Cemaillen/' .		# Get the length of the email field
			'A35email/' .		# Get the email (35) padded with null
			'Cinfolen/' .		# Get the length of the usernote field
			'A30info/' .		# Get the usernote (30) padded with null
			'Copt1len/' .		# Get the length of the opt1 field
			'A60opt1/' .		# Get the opt1 field (60) padded with null
			'Copt2len/' .		# Get the length of the opt2 field
			'A35opt2/' .		# Get the opt2 field (60) padded with null
			'Copt3len/' .		# Get the length of the opt3 field
			'A60opt3/' .		# Get the opt3 field (60) padded with null
			'Copt4len/' .		# Get the length of the opt4 field
			'A35opt4/' .		# Get the opt4 field (60) padded with null
			'Copt5len/' .		# Get the length of the opt5 field
			'A60opt5/' .		# Get the opt5 field (60) padded with null
			'Copt6len/' .		# Get the length of the opt6 field
			'A35opt6/' .		# Get the opt6 field (60) padded with null
			'Copt7len/' .		# Get the length of the opt7 field
			'A35opt7/' .		# Get the opt7 field (60) padded with null
			'Copt8len/' .		# Get the length of the opt8 field
			'A60opt8/' .		# Get the opt8 field (60) padded with null
			'Copt9len/' .		# Get the length of the opt9 field
			'A35opt9/' .		# Get the opt9 field (60) padded with null
			'Copt10len/' .		# Get the length of the opt10 field
			'A35opt10/'.		# Get the opt10 field (60) padded with null
			'@53/';		    	# jump to the 894th btye     
     
		// Unpack the data structure
		
		$lcallers[] = unpack ($data_format, $data);
		
	}  
	
	// Change date from dos to readable and strip pipe codes 
	
	foreach ( $lcallers as &$lengfix ) {
		$lengfix['user'] = substr($lengfix['user'], 0, $lengfix['userlen']);
		$lengfix['city'] = substr($lengfix['city'], 0, $lengfix['citylen']);
		$lengfix['address'] = substr($lengfix['address'], 0, $lengfix['addresslen']);
		$lengfix['host'] = substr($lengfix['host'], 0, $lengfix['hostlen']);
		$lengfix['peer'] = substr($lengfix['peer'], 0, $lengfix['peerlen']);   
		$lengfix['email'] = substr($lengfix['email'], 0, $lengfix['emaillen']);
		$lengfix['info'] = substr($lengfix['info'], 0, $lengfix['infolen']);
		$lengfix['opt1'] = substr($lengfix['opt1'], 0, $lengfix['opt1len']);
		$lengfix['opt2'] = substr($lengfix['opt2'], 0, $lengfix['opt2len']); 
		$lengfix['opt3'] = substr($lengfix['opt3'], 0, $lengfix['opt3len']);
		$lengfix['opt4'] = substr($lengfix['opt4'], 0, $lengfix['opt4len']);
		$lengfix['opt5'] = substr($lengfix['opt5'], 0, $lengfix['opt5len']); 
		$lengfix['opt6'] = substr($lengfix['opt6'], 0, $lengfix['opt6len']);
		$lengfix['opt7'] = substr($lengfix['opt7'], 0, $lengfix['opt7len']);
		$lengfix['opt8'] = substr($lengfix['opt8'], 0, $lengfix['opt8len']); 
		$lengfix['opt9'] = substr($lengfix['opt9'], 0, $lengfix['opt9len']);
		$lengfix['opt10'] = substr($lengfix['opt10'], 0, $lengfix['opt10len']);  

	}
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $lcallers as &$rfix ) {
		unset($rfix['userlen']);
		unset($rfix['citylen']);
		unset($rfix['addresslen']);
		unset($rfix['hostlen']);
		unset($rfix['peerlen']);
		unset($rfix['emaillen']);
		unset($rfix['infolen']);
		unset($rfix['opt1len']);
		unset($rfix['opt2len']);
		unset($rfix['opt3len']);
		unset($rfix['opt4len']);
		unset($rfix['opt5len']);
		unset($rfix['opt6len']);
		unset($rfix['opt7len']);
		unset($rfix['opt8len']);
		unset($rfix['opt9len']);
		unset($rfix['opt10len']);
	}			
	
	foreach ( $lcallers as &$dfix ) 
	{
		
		// change date from dos to unix and make human readable
		
		$dfix['date'] = dos2unixtime($dfix['date']);
		$dfix['date'] = date("m/d h:i a", $dfix['date']);
	}
	
	if ($pipe == "Y") {
		
		foreach ( $lcallers as &$dfix ) 
		{
			// convert pipe codes to proper colours
			
			$dfix['info'] = str_replace("|00", "<span class=\"pipe_00\">", $dfix['info']);
			$dfix['info'] = str_replace("|01", "<span class=\"pipe_01\">", $dfix['info']);
			$dfix['info'] = str_replace("|02", "<span class=\"pipe_02\">", $dfix['info']);
			$dfix['info'] = str_replace("|03", "<span class=\"pipe_03\">", $dfix['info']);
			$dfix['info'] = str_replace("|04", "<span class=\"pipe_04\">", $dfix['info']);
			$dfix['info'] = str_replace("|05", "<span class=\"pipe_05\">", $dfix['info']);
			$dfix['info'] = str_replace("|06", "<span class=\"pipe_06\">", $dfix['info']);
			$dfix['info'] = str_replace("|07", "<span class=\"pipe_07\">", $dfix['info']);
			$dfix['info'] = str_replace("|08", "<span class=\"pipe_08\">", $dfix['info']);
			$dfix['info'] = str_replace("|09", "<span class=\"pipe_09\">", $dfix['info']);
			$dfix['info'] = str_replace("|10", "<span class=\"pipe_10\">", $dfix['info']);
			$dfix['info'] = str_replace("|11", "<span class=\"pipe_11\">", $dfix['info']);
			$dfix['info'] = str_replace("|12", "<span class=\"pipe_12\">", $dfix['info']);
			$dfix['info'] = str_replace("|13", "<span class=\"pipe_13\">", $dfix['info']);
			$dfix['info'] = str_replace("|14", "<span class=\"pipe_14\">", $dfix['info']);
			$dfix['info'] = str_replace("|15", "<span class=\"pipe_15\">", $dfix['info']);
			$dfix['info'] = str_replace("|16", "", $dfix['info']);
			$dfix['info'] = str_replace("|17", "", $dfix['info']);
			$dfix['info'] = str_replace("|18", "", $dfix['info']);
			$dfix['info'] = str_replace("|19", "", $dfix['info']);
			$dfix['info'] = str_replace("|20", "", $dfix['info']);
			$dfix['info'] = str_replace("|21", "", $dfix['info']);
			$dfix['info'] = str_replace("|22", "", $dfix['info']);
			$dfix['info'] = str_replace("|23", "", $dfix['info']);
			$dfix['info'] = str_replace("|24", "", $dfix['info']);
		}
		
	}
	
	if ($pipe == "N") {
		
		foreach ( $lcallers as &$dfix ) 
		{			
			// convert pipe codes to proper colours	
			
			$dfix['info'] = str_replace("|00", "", $dfix['info']);
			$dfix['info'] = str_replace("|01", "", $dfix['info']);
			$dfix['info'] = str_replace("|02", "", $dfix['info']);
			$dfix['info'] = str_replace("|03", "", $dfix['info']);
			$dfix['info'] = str_replace("|04", "", $dfix['info']);
			$dfix['info'] = str_replace("|05", "", $dfix['info']);
			$dfix['info'] = str_replace("|06", "", $dfix['info']);
			$dfix['info'] = str_replace("|07", "", $dfix['info']);
			$dfix['info'] = str_replace("|08", "", $dfix['info']);
			$dfix['info'] = str_replace("|09", "", $dfix['info']);
			$dfix['info'] = str_replace("|10", "", $dfix['info']);
			$dfix['info'] = str_replace("|11", "", $dfix['info']);
			$dfix['info'] = str_replace("|12", "", $dfix['info']);
			$dfix['info'] = str_replace("|13", "", $dfix['info']);
			$dfix['info'] = str_replace("|14", "", $dfix['info']);
			$dfix['info'] = str_replace("|15", "", $dfix['info']);
			$dfix['info'] = str_replace("|16", "", $dfix['info']);
			$dfix['info'] = str_replace("|17", "", $dfix['info']);
			$dfix['info'] = str_replace("|18", "", $dfix['info']);
			$dfix['info'] = str_replace("|19", "", $dfix['info']);
			$dfix['info'] = str_replace("|20", "", $dfix['info']);
			$dfix['info'] = str_replace("|21", "", $dfix['info']);
			$dfix['info'] = str_replace("|22", "", $dfix['info']);
			$dfix['info'] = str_replace("|23", "", $dfix['info']);
			$dfix['info'] = str_replace("|24", "", $dfix['info']);			
		}
	}
	
	
	return $lcallers;
}

/**
 * read and parse Mystic's chat(#).dat file on Mystic 1.10 A18+
 *
 * @uses: mystic_chat(path to mystic data dir $var, number of nodes you are running $int) 
 *
 * @return: an array containing the following keys identifying user information from each node. 
 *
 * 		   active = is anyone on the node. 1 = Yes, 0 = No
 *		   name = username/handle
 *		   action = user's current action
 *		   location = user's city and state
 *		   gender = user's gender. M = Male, F = Female
 *		   age = user's age
 *		   baud = user's connecting baud rate
 *		   invisible = user's invisibility status
 *		   available = user's availability status
 *		   inchat = is the user in multi-node chat. 1 = Yes, 0 = No
 *		   room = which multi-node chat room is the user in
 *
 * @author: Frank Linhares
 **/

function mystic_chat($data_path, $nodes)
{	

	// loop through the number of nodes with node1.dat, node2.dat, etc..

  for ($i = 1; $i <= $nodes; $i++) {

    $data = file_get_contents($data_path.'chat'.$i.'.dat');

		// Create a data structure
		
		$data_format =
			'Cactive/' .     	# Get the date
			'Cnamelen/' .    	# Get the length of the name field 
			'A30name/' .	 	# Get the username (30) padded with null
			'Cactionlen/' .  	# Get the length of the action field 
			'A40action/' .	 	# Get the user's current action (40) padded with null		
			'Clocationlen/' .	# Get the length of the location field 
			'A30location/' . 	# Get the user's city and state (40) padded with null			
			'Agender/' . 	 	# Get the gender (m for male f for female)
			'Cage/' .		    # Get the users age
			'Cbaudlen/' .    	# Get the length of the baud field 
			'A6baud/' . 	 	# Get the baud rate (6) padded with null
			'cinvisible/' .  	# Check if the user is invisible
			'cavailable/' .  	# Check if the user is available		
			'cinchat/' .     	# Check if the user is in multi-node chat		
			'Croom/' ;		    # If in multi-node chat which room
		    
		// Unpack the data structure
		
		$mystic_chat[] = unpack ($data_format, $data);

	}
	
	// inject node number into array	
	
	for ($i = 0; $i < $nodes; $i++) {
		$mystic_chat[$i]['node']=$i+1;
	}
	
//	array_pop($mystic_chat);
	
	// trim and clean up strings
	
	foreach ( $mystic_chat as &$sfix ) {
		$sfix['name'] = substr($sfix['name'], 0, $sfix['namelen']);  
		$sfix['action'] = substr($sfix['action'], 0, $sfix['actionlen']); 
		$sfix['location'] = substr($sfix['location'], 0, $sfix['locationlen']);
		$sfix['baud'] = substr($sfix['baud'], 0, $sfix['baudlen']);
	}	
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $mystic_chat as &$rfix ) {
		unset($rfix['namelen']);
		unset($rfix['actionlen']);
		unset($rfix['locationlen']);
		unset($rfix['baudlen']);
	}
	
	// check if node is active, if not clear name and set action to "waiting for caller" 	
	
	foreach ( $mystic_chat as &$wfix ) {
		if ($wfix['active'] == 0) {
			$wfix['name'] = "";
			$wfix['action'] = 'waiting for caller';
		}
		
		// don't allow users who want to be invisible to be displayed.		
		
		if ($wfix['active'] == 1 AND $wfix['invisible'] == 1) {
			$wfix['name'] = "";
			$wfix['action'] = 'waiting for caller';
		}
	
	}	
	
	return $mystic_chat;
}

/**                                                            
 * read and parse Mystic's history.dat file on Mystic 1.10 A18+
 *
 * @uses: mystic_history(path to mystic data dir $var, "TODAY" to return todays stats only or "ALL" for all stats)
 * 
 * @return: an array containing the following keys identifying system stats. 
 *
 * 		   date = Date
 *		   emails = number of emails sent today
 *		   posts = number of posts today
 *		   downloads = number of downloads today
 *		   uploads = number of uploads today
 *		   dlkb = number of kilobytes downloaded today
 *		   ulkb = number of kilobytes uploaded today
 *		   newusers = number of new users today
 *		   calls = total number of calls today
 *		   telnet = total number of telnet connections today
 *		   ftp = total number of ftp connections today
 *		   nntp = total number of nntp connections today
 *		   http = total number of http connections today
 *		   pop3 = total number of pop3 connections today
 *		   smtp = total number of smtp connections today
 *
 * @author: Frank Linhares
 **/

function mystic_history($data_path, $total = "TODAY")
{	
	$total = strtoupper($total);	
	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'history.dat'); 
	$record_length = 64;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	
	$fp = fopen ($data_path.'history.dat', 'rb');		
	
	for ($i = 0; $i < $record_number; $i++) {
		$data = fread ($fp, $record_length);
   
		/* Create a data structure */
		$data_format =
			'ldate/' .       # Get the date
			'Semails/' .     # Get the number of emails sent
			'Sposts/' .      # Get the number of posts
			'Sdownloads/' .  # Get the number of downloads
			'Suploads/' .    # Get the number of uploads
			'ldlkb/' .       # Get the number of downloaded kb
			'lupkb/' .       # Get the number of uploaded kb
			'lcalls/' .      # Get the number of calls
			'snewusers/' .   # Get the number of new users 
			'Stelnet/' .	 # Get the number of telnet connections
			'Sftp/' .		 # Get the number of ftp connections
			'Spop3/' .		 # Get the number of pop3 connections
			'Ssmtp/' .		 # Get the number of smtp connections
			'Snntp/' .       # Get the number of nntp connections
			'Shttp/' .       # Get the number of http connections
			'@24/' ;	     # Jump to the 24th byte
     
		/* Unpack the data structure */
		$myshistory[] = unpack ($data_format, $data);	  	
     }  	   
     	 
	foreach ( $myshistory as &$dfix ) 
	{

	// change date from dos to unix and make human readable

	$dfix['date'] = dos2unixtime($dfix['date']);
	$dfix['date'] = date("m/d/y", $dfix['date']);
	$dfix['date'] = substr($dfix['date'], 0, 8);
			
	}

	// if only returning today's stats then only return the last entry	
	
	if ($total == "TODAY") {	
		$myshistory = end($myshistory);
	}
	
	return $myshistory;
}

/**
 * read and parse Mystic's oneliner.dat file on Mystic 1.10 A18+
 *
 * @uses: mystic_oneliners(path to mystic data dir $var, number of oneliners to return, default is 10 $int)
 * 
 * @return: an array containing the following keys identifying mystic oneliners. 
 *
 * 		   text = oneliner text
 *		   from = username the oneliner is from
 *
 * @author: Frank Linhares
**/

function mystic_oneliners($data_path, $number = 10)
{	
	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'oneliner.dat'); 
	$record_length = 111;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	
	$fp = fopen ($data_path.'oneliner.dat', 'rb');		
	
	for ($i = 0; $i < $record_number; $i++) {
		$data = fread ($fp, $record_length);
   
		/* Create a data structure */
		$data_format =
			'Ctextlen/' .   # Get the length of the text field 
			'A79text/' . 	# Get the text (79) padded with null
			'Cfromlen/' .  	# Get the length of the from field 
			'A30from/' ; 	# Get the from (30) padded with null				
     
		/* Unpack the data structure */
		$mysoneliner[] = unpack ($data_format, $data);	  	
	}  	   
     	 
	// trim and clean up strings
	
	foreach ( $mysoneliner as &$sfix ) {
		$sfix['text'] = substr($sfix['text'], 0, $sfix['textlen']);  
		$sfix['from'] = substr($sfix['from'], 0, $sfix['fromlen']); 
	}	
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $mysoneliner as &$rfix ) {
		unset($rfix['textlen']);
		unset($rfix['fromlen']);
	}
	
	// if $number is greater than 0, return that number of oneliners 	
	
	if ($number <= 9) {	
		array_splice($mysoneliner, 0, -$number);
	}	
	
	return $mysoneliner;
}

/**
 * read and parse Mystic's votes.dat file on Mystic 1.10 A18+
 *
 * @uses: mystic_vote(path to mystic data dir $var)
 * 
 * @return: an array containing the following keys identifying voteing questions and answers.
 * 	    the array returned will only contain answers and answer vote counts for valid questions
 *
 * 		   votes = total number of votes
 *		   ansnum = total number of answers
 *		   user = user name who added the question
 *		   acs = acs to see the question
 *		   acsadd = acs to add answers
 *		   forceacs = acs to for the question
 *		   question = question
 *		   ans1 = answer 1
 *		   a1votes = total votes for answer 1
 *		   ans2 = answer 2
 *		   a1votes = total votes for answer 2
 *		   ans3 = answer 3
 *		   a1votes = total votes for answer 3
 *		   ans4 = answer 4
 *		   a1votes = total votes for answer 4
 *		   ans5 = answer 5
 *		   a1votes = total votes for answer 5
 *		   ans6 = answer 6
 *		   a1votes = total votes for answer 6
 *		   ans7 = answer 7
 *		   a1votes = total votes for answer 7
 *		   ans8 = answer 8
 *		   a1votes = total votes for answer 8
 *		   ans9 = answer 9
 *		   a1votes = total votes for answer 9
 *		   ans10 = answer 10
 *		   a1votes = total votes for answer 10
 *		   ans11 = answer 11
 *		   a1votes = total votes for answer 11
 *		   ans12 = answer 12
 *		   a1votes = total votes for answer 12
 *		   ans13 = answer 13
 *		   a1votes = total votes for answer 13
 *		   ans14 = answer 14
 *		   a1votes = total votes for answer 14
 *		   ans15 = answer 15
 *		   a1votes = total votes for answer 15
 *
 * @author: Frank Linhares
 **/

function mystic_vote($data_path)
{	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'votes.dat'); 
	$record_length = 822;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	
	$fp = fopen ($data_path.'votes.dat', 'rb');		
	
	for ($i = 0; $i < $record_number; $i++) {
		$data = fread ($fp, $record_length);
   
		/* Create a data structure */
		$data_format =
			'svotes/' .      # Get the number of votes
			'Cansnum/' .     # Get the total number of answers
			'Cuserlen/' .    # Get the length of the user field 
			'A30user/' . 	 # Get the user name who added the question padded with null			
			'Cacslen/' .     # Get the length of the acs field 
			'A20acs/' . 	 # Get the acs to see the question padded with null	
			'Cacsaddlen/' .  # Get the length of the acsadd field 
			'A20acsadd/' . 	 # Get the acs to add an answer padded with null		
			'Cforceacslen/' .# Get the length of the forceacs field 
			'A20forceacs/' . # Get the acs to force the question padded with null				
			'Cquestionlen/' .# Get the length of the question field 
			'A79question/' . # Get the question padded with null	
			'Cans1len/' .    # Get the length of the ans1 field 
			'A40ans1/' . 	 # Get the 1 answer with null	
			'sa1votes/' .    # Get the number of votes for ans1
			'Cans2len/' .    # Get the length of the ans2 field 
			'A40ans2/' . 	 # Get the 2 answer with null	
			'sa2votes/' .    # Get the number of votes for ans3
			'Cans3len/' .    # Get the length of the ans3 field 
			'A40ans3/' . 	 # Get the 3 answer with null	
			'sa3votes/' .    # Get the number of votes for ans3
			'Cans4len/' .    # Get the length of the ans4 field 
			'A40ans4/' . 	 # Get the 4 answer with null	
			'sa4votes/' .    # Get the number of votes for ans4
			'Cans5len/' .    # Get the length of the ans5 field 
			'A40ans5/' . 	 # Get the 5 answer with null	
			'sa5votes/' .    # Get the number of votes for ans5
			'Cans6len/' .    # Get the length of the ans6 field 
			'A40ans6/' . 	 # Get the 6 answer with null	
			'sa6votes/' .    # Get the number of votes for ans6
			'Cans7len/' .    # Get the length of the ans7 field 
			'A40ans7/' . 	 # Get the 7 answer with null	
			'sa7votes/' .    # Get the number of votes for ans7
			'Cans8len/' .    # Get the length of the ans8 field 
			'A40ans8/' . 	 # Get the 8 answer with null	
			'sa8votes/' .    # Get the number of votes for ans8
			'Cans9len/' .    # Get the length of the ans9 field 
			'A40ans9/' . 	 # Get the 9 answer with null	
			'sa9votes/' .    # Get the number of votes for ans9
			'Cans10len/' .    # Get the length of the ans10 field 
			'A40ans10/' . 	 # Get the 10 answer with null	
			'sa10votes/' .    # Get the number of votes for ans10
			'Cans11len/' .    # Get the length of the ans11 field 
			'A40ans11/' . 	 # Get the 11 answer with null	
			'sa11votes/' .    # Get the number of votes for ans11
			'Cans12len/' .    # Get the length of the ans12 field 
			'A40ans12/' . 	 # Get the 12 answer with null	
			'sa12votes/' .    # Get the number of votes for ans12
			'Cans13len/' .    # Get the length of the ans13 field 
			'A40ans13/' . 	 # Get the 13 answer with null	
			'sa13votes/' .    # Get the number of votes for ans13
			'Cans14len/' .    # Get the length of the ans14 field 
			'A40ans14/' . 	 # Get the 14 answer with null	
			'sa14votes/' .    # Get the number of votes for ans14
			'Cans15len/' .    # Get the length of the ans15 field 
			'A40ans15/' . 	 # Get the 15 answer with null	
			'sa15votes/' ;    # Get the number of votes for ans15     

		/* Unpack the data structure */
		$mysvotes[] = unpack ($data_format, $data);	  	
	}	  	   
     	
	// trim and clean up strings
	
	foreach ( $mysvotes as &$sfix ) {
		$sfix['user'] = substr($sfix['user'], 0, $sfix['userlen']);  
		$sfix['acs'] = substr($sfix['acs'], 0, $sfix['acslen']); 
		$sfix['acsadd'] = substr($sfix['acsadd'], 0, $sfix['acsaddlen']);
		$sfix['forceacs'] = substr($sfix['forceacs'], 0, $sfix['forceacslen']);
		$sfix['question'] = substr($sfix['question'], 0, $sfix['questionlen']);
		$sfix['ans1'] = substr($sfix['ans1'], 0, $sfix['ans1len']);
		$sfix['ans2'] = substr($sfix['ans2'], 0, $sfix['ans2len']);
		$sfix['ans3'] = substr($sfix['ans3'], 0, $sfix['ans3len']);
		$sfix['ans4'] = substr($sfix['ans4'], 0, $sfix['ans4len']);
		$sfix['ans5'] = substr($sfix['ans5'], 0, $sfix['ans5len']);
		$sfix['ans6'] = substr($sfix['ans6'], 0, $sfix['ans6len']);
		$sfix['ans7'] = substr($sfix['ans7'], 0, $sfix['ans7len']);
		$sfix['ans8'] = substr($sfix['ans8'], 0, $sfix['ans8len']);
		$sfix['ans9'] = substr($sfix['ans9'], 0, $sfix['ans9len']);
		$sfix['ans10'] = substr($sfix['ans10'], 0, $sfix['ans10len']);
		$sfix['ans11'] = substr($sfix['ans11'], 0, $sfix['ans11len']);
		$sfix['ans12'] = substr($sfix['ans12'], 0, $sfix['ans12len']);
		$sfix['ans13'] = substr($sfix['ans13'], 0, $sfix['ans13len']);
		$sfix['ans14'] = substr($sfix['ans14'], 0, $sfix['ans14len']);
		$sfix['ans15'] = substr($sfix['ans15'], 0, $sfix['ans15len']);
	}	 
	
	// preform some ninja math to only return the ansers and their values for each question	
		
	foreach ( $mysvotes as &$rfix ) {		
		
		$takeys = 3 * $rfix['ansnum'];
		$tkeys = 13 + $takeys;
		
		$numKeys = count(array_keys($rfix));

		if ($tkeys < $numKeys) {
						
			$drop = $numKeys - $tkeys;			
			#array_multiple_pop(&$rfix, $drop);
			array_multiple_pop($rfix, $drop);
		}
	}
		
	foreach ( $mysvotes as &$rfix ) {
		unset($rfix['userlen']);
		unset($rfix['acslen']);
		unset($rfix['acsaddlen']);
		unset($rfix['forceacslen']);
		unset($rfix['questionlen']);
		unset($rfix['ans1len']);
		unset($rfix['ans2len']);
		unset($rfix['ans3len']);
		unset($rfix['ans4len']);
		unset($rfix['ans5len']);
		unset($rfix['ans6len']);
		unset($rfix['ans7len']);
		unset($rfix['ans8len']);
		unset($rfix['ans9len']);
		unset($rfix['ans10len']);
		unset($rfix['ans11len']);
		unset($rfix['ans12len']);
		unset($rfix['ans13len']);
		unset($rfix['ans14len']);
		unset($rfix['ans15len']);
		
	}

	
	
	return $mysvotes;
}

/**
 * read and parse Mystic users.dat file on Mystic 1.10 A18+
 * 
 * @uses: mystic_userlist(path to mystic data dir $var, "Y" to parse pipe colours or "N" to strip pipe colours "$")
 * 
 * @return: an array containing the following keys identifying users with accounts on the bbs. 
 *
 * 		   flags = account flags
 *		   handle = user's handle
 *		   realname = user's real name
 *		   password = user's password
 *		   address = user's address
 *		   city = user's city
 *		   zipcode = user's zip code
 *		   homephone = user's home phone
 *		   dataphone = user's data phone
 *		   bdate = user's birthdate
 *		   gender = user's gender
 *		   email = user's email address
 *		   opt1 = user's optional field 1
 *		   opt2 = user's optional field 2
 *		   opt3 = user's optional field 3
 *		   info = user's usernote
 *		   security = user's security level
 *		   smnu = user's start menu
 *		   firston = first time user on
 *		   laston = last time user on
 *		   calls = total number of calls
 *		   callstoday = total calls today
 *		   dls = total number of downloads
 *		   dlstoday = total number of downloads today
 *		   dlk = total number of downloads in k
 *		   dlktoday = total number of downloads today in k
 *		   uls = total number of uploads
 *		   ulsk = total number of uploads in k
 *		   posts = total number of posts for the user.
 *		   emails = total number of emails user has sent
 *		   timeleft = user's time left today
 *
 * @author: Frank Linhares
 **/

function mystic_userlist($data_path, $pipe = "N")
{	
	$pipe = strtoupper($pipe);	
	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'users.dat'); 
	$record_length = 1536;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode 
	
	$fp = fopen ($data_path.'users.dat', 'rb');		
	
	for ($i = 0; $i < $record_number; $i++) {
		$data = fread ($fp, $record_length);
   
		/* Create a data structure */
		$data_format =
		    
		    'lpidx/' .              # Get the PemIdx (byte) 
        'lflags/' .             # Get the flags (byte)
        'Chandlelen/' .         # Get the length of the handle field 
        'A30handle/' .          # Get the handle (30) padded with null
        'Crealnamelen/' .       # Get the length of the realname field 
        'A30realname/' .        # Get the real name (30) padded with null      
        'Cpasswordlen/' .       # Get the length of the password field 
        'A15password/' .        # Get the password (15) padded with null
        'Caddresslen/' .        # Get the length of the address field 
        'A30address/' .         # Get the address (30) padded with null
        'Ccitylen/' .           # Get the length of the city field 
        'A25city/' .            # Get the city (25) padded with null
        'Czipcodelen/' .        # Get the length of the zipcode field 
        'A9zipcode/' .          # Get the zipcode (9) padded with null
        'Chomephonelen/' .      # Get the length of the homephone field 
        'A15homephone/' .       # Get the home phone (15) padded with null
        'Cdataphonelen/' .      # Get the length of the dataphone field 
        'A15dataphone/' .       # Get the dataphone (15) padded with null
        'lbdate/' .             # Get the birth date
        'Agender/' .            # Get the gender (m for male f for female)
        'Cemaillen/' .          # Get the length of the email field 
        'A60email/' .           # Get the email address (60) padded with null
        'Copt1len/' .           # Get the length of the opt1 field 
        'A60opt1/' .            # Get the opt1 field (60) padded with null
        'Copt2len/' .           # Get the length of the opt2 field
        'A60opt2/' .            # Get the opt2 field (60) padded with null
        'Copt3len/' .           # Get the length of the opt3 field
        'A60opt3/'.             # Get the opt3 field (60) padded with null
        'Copt4len/' .           # Get the length of the opt4 field 
        'A60opt4/' .            # Get the opt4 field (60) padded with null
        'Copt5len/' .           # Get the length of the opt5 field
        'A60opt5/' .            # Get the opt5 field (60) padded with null
        'Copt6len/' .           # Get the length of the opt6 field
        'A60opt6/'.             # Get the opt6 field (60) padded with null
        'Copt7len/' .           # Get the length of the opt7 field
        'A60opt7/'.             # Get the opt7 field (60) padded with null
        'Copt8len/' .           # Get the length of the opt8 field 
        'A60opt8/' .            # Get the opt8 field (60) padded with null
        'Copt9len/' .           # Get the length of the opt9 field
        'A60opt9/' .            # Get the opt9 field (60) padded with null
        'Copt10len/' .          # Get the length of the opt10 field
        'A60opt10/'.            # Get the opt10 field (60) padded with null
        'Cinfolen/' .           # Get the length of the info field
        'A30info/' .            # Get the usernote (30) padded with null
        'Cthemelen/' .          # Get the length of the theme field
        'A20theme/' .           # Get the theme (20) padded with null
        '@921/'.                # jump to the 920th btye
		    'ssecurity/' .          # Get the security level
		    'Cexpireslen/' .        # Get the length of the expires field
		    'A8expires/' .          # Get the expires field (8) padded with null
		    'Cexpiresto/' .         # Get the expires to field
		    'Clastpwlen/' .         # Get the length of the LastPWChange field
		    'A8lastpw/' .           # Get the LastPWChange field (8) padded with null
		    'Csmnulen/' .           # Get the length of the smnu field
		    'A20smnu/' .            # Get the start mnu field (20) padded with null
		    'Carchivelen/' .        # Get the length of the archive field
		    'A4archive/' .          # Get the archive field (4) padded with null
		    'Cqwkfiles/' .          # Get the QWKFiles field
		    'Cdatetype/' .          # Get the datetype field
		    'Cscreensize/' .        # Get the screensize field
		    'Cscreencols/' .        # Get the screencols field
		    'Cpeeriplen/' .         # Get the length of the peerip field
		    'A20peerip/' .          # Get the peerip field (20) padded with null
		    'Cpeerhostlen/' .       # Get the length of the peerhost field
		    'A50peerhost/' .        # Get the peerhost field (50) padded with null
		    'lfirston/' .           # Get the first on date
		    'llaston/' .            # Get the last on date
		    'lcalls/' .             # Get the number of calls
		    'scallstoday/' .        # Get the # of calls today
		    'sdls/' .               # Get the # of downloads
		    'sdlstoday/' .          # Get the # of downloads today  
		    'ldlk/' .               # Get the total downloads in k
		    'ldlktoday/' .          # Get the total downloads in k today 
		    'luls/' .               # Get the total # of uploads
		    'lulk/' .               # Get the total uploads in k
		    'lposts/' .             # Get the total posts
			'lemails/' .            # Get the total sent emails
			'ltimeleft/' .          # Get the amount of time left today
			'stimebank/' .          # Get the number of mins in the time back
			'lfratings/' .          # Get the fileratings field
			'lfcomment/' .          # Get the filecomment field
			'Slastfbase/' .         # Get the last file base on
			'Slastmbase/' .         # Get the last message base on
			'Slastmgroup/' .        # Get the last message group on
			'Slastfgroup/' .        # Get the last file group on
			'@1300/'.               # jump to the 367th btye
			'Cedittype/' .          # Get the editor type
			'Cfilelist/' .          # Get the file list type
			'Csiguse/' .            # Get the signature use
			'lsigoffset/' .         # Get the signature offset 
			'Csiglength/' .         # Get the signature length
			'Chotkeys/' .           # Get the hot keys field 
			'Cmreadtype/' .         # Get the Message reader type
			'Cuselbindex/' .        # Get the lightbar index
			'Cuselbquote/' .        # Get the lightbar quote
			'Cuselbmidx/' .         # Get the lightbar index in email
			'Cusefullchat/' .       # Get the use full chat
			'@1536/';               # jump to the 367th btye

		/* Unpack the data structure */
		$users[] = unpack ($data_format, $data);
		
     }  
	   
	// trim and clean up strings
	
	foreach ( $users as &$sfix ) {
		$sfix['handle'] = substr($sfix['handle'], 0, $sfix['handlelen']);  
		$sfix['realname'] = substr($sfix['realname'], 0, $sfix['realnamelen']); 
		$sfix['password'] = substr($sfix['password'], 0, $sfix['passwordlen']);
		$sfix['address'] = substr($sfix['address'], 0, $sfix['addresslen']);
		$sfix['city'] = substr($sfix['city'], 0, $sfix['citylen']);
		$sfix['zipcode'] = substr($sfix['zipcode'], 0, $sfix['zipcodelen']);
		$sfix['homephone'] = substr($sfix['homephone'], 0, $sfix['homephonelen']);
		$sfix['dataphone'] = substr($sfix['dataphone'], 0, $sfix['dataphonelen']);
		$sfix['email'] = substr($sfix['email'], 0, $sfix['emaillen']);
		$sfix['opt1'] = substr($sfix['opt1'], 0, $sfix['opt1len']);
		$sfix['opt2'] = substr($sfix['opt2'], 0, $sfix['opt2len']);
		$sfix['opt3'] = substr($sfix['opt3'], 0, $sfix['opt3len']);
		$sfix['opt3'] = substr($sfix['opt4'], 0, $sfix['opt4len']);
		$sfix['opt3'] = substr($sfix['opt5'], 0, $sfix['opt5len']);
		$sfix['opt3'] = substr($sfix['opt6'], 0, $sfix['opt6len']);
		$sfix['opt3'] = substr($sfix['opt7'], 0, $sfix['opt7len']);
		$sfix['opt3'] = substr($sfix['opt8'], 0, $sfix['opt8len']);
		$sfix['opt3'] = substr($sfix['opt9'], 0, $sfix['opt9len']);
		$sfix['opt3'] = substr($sfix['opt10'], 0, $sfix['opt10len']);
		$sfix['info'] = substr($sfix['info'], 0, $sfix['infolen']);
		$sfix['theme'] = substr($sfix['theme'], 0, $sfix['themelen']);
		$sfix['peerhost'] = substr($sfix['peerhost'], 0, $sfix['peerhostlen']);
		$sfix['peerip'] = substr($sfix['peerip'], 0, $sfix['peeriplen']);
		$sfix['archive'] = substr($sfix['archive'], 0, $sfix['archivelen']);
		$sfix['smnu'] = substr($sfix['smnu'], 0, $sfix['smnulen']);
		$sfix['lastpw'] = substr($sfix['lastpw'], 0, $sfix['lastpwlen']);
		$sfix['expires'] = substr($sfix['expires'], 0, $sfix['expireslen']);
	}

	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $users as &$rfix ) {
		unset($rfix['handlelen']);
		unset($rfix['realnamelen']);
		unset($rfix['passwordlen']);
		unset($rfix['addresslen']);
		unset($rfix['citylen']);
		unset($rfix['zipcodelen']);
		unset($rfix['homephonelen']);
		unset($rfix['dataphonelen']);
		unset($rfix['emaillen']);		
		unset($rfix['opt1len']);
		unset($rfix['opt2len']);
		unset($rfix['opt3len']);
		unset($rfix['opt4len']);
		unset($rfix['opt5len']);
		unset($rfix['opt6len']);
		unset($rfix['opt7len']);
		unset($rfix['opt8len']);
		unset($rfix['opt9len']);
		unset($rfix['opt10len']);
		unset($rfix['infolen']);
		unset($rfix['themelen']);
		unset($rfix['peerhostlen']);
		unset($rfix['peeriplen']);
		unset($rfix['archivelen']);
		unset($rfix['smnulen']);
		unset($rfix['lastpwlen']);
		unset($rfix['expireslen']);      
	}	
	
	foreach ( $users as &$dfix ) 
	{		
		// change date from dos to unix and make human readable
		$dfix['firston'] = dos2unixtime($dfix['firston']);
		$dfix['firston'] = date("m/d/y", $dfix['firston']);
		
		$dfix['laston'] = dos2unixtime($dfix['laston']);
		$dfix['laston'] = date("m/d/y", $dfix['laston']);
		
		$dfix['bdate'] = jdtogregorian($dfix['bdate']);

	}
	
	if ($pipe == "Y") {
		
		foreach ( $users as &$dfix ) 
		{
			// convert pipe codes to proper colours
			$dfix['info'] = str_replace("|00", "<span class=\"pipe_00\">", $dfix['info']);
			$dfix['info'] = str_replace("|01", "<span class=\"pipe_01\">", $dfix['info']);
			$dfix['info'] = str_replace("|02", "<span class=\"pipe_02\">", $dfix['info']);
			$dfix['info'] = str_replace("|03", "<span class=\"pipe_03\">", $dfix['info']);
			$dfix['info'] = str_replace("|04", "<span class=\"pipe_04\">", $dfix['info']);
			$dfix['info'] = str_replace("|05", "<span class=\"pipe_05\">", $dfix['info']);
			$dfix['info'] = str_replace("|06", "<span class=\"pipe_06\">", $dfix['info']);
			$dfix['info'] = str_replace("|07", "<span class=\"pipe_07\">", $dfix['info']);
			$dfix['info'] = str_replace("|08", "<span class=\"pipe_08\">", $dfix['info']);
			$dfix['info'] = str_replace("|09", "<span class=\"pipe_09\">", $dfix['info']);
			$dfix['info'] = str_replace("|10", "<span class=\"pipe_10\">", $dfix['info']);
			$dfix['info'] = str_replace("|11", "<span class=\"pipe_11\">", $dfix['info']);
			$dfix['info'] = str_replace("|12", "<span class=\"pipe_12\">", $dfix['info']);
			$dfix['info'] = str_replace("|13", "<span class=\"pipe_13\">", $dfix['info']);
			$dfix['info'] = str_replace("|14", "<span class=\"pipe_14\">", $dfix['info']);
			$dfix['info'] = str_replace("|15", "<span class=\"pipe_15\">", $dfix['info']);
			$dfix['info'] = str_replace("|16", "", $dfix['info']);
			$dfix['info'] = str_replace("|17", "", $dfix['info']);
			$dfix['info'] = str_replace("|18", "", $dfix['info']);
			$dfix['info'] = str_replace("|19", "", $dfix['info']);
			$dfix['info'] = str_replace("|20", "", $dfix['info']);
			$dfix['info'] = str_replace("|21", "", $dfix['info']);
			$dfix['info'] = str_replace("|22", "", $dfix['info']);
			$dfix['info'] = str_replace("|23", "", $dfix['info']);
			$dfix['info'] = str_replace("|24", "", $dfix['info']);
		}
		
	}
	
	if ($pipe == "N") {
		
		foreach ( $users as &$dfix ) 
		{			
			// convert pipe codes to proper colours	
			$dfix['info'] = str_replace("|00", "", $dfix['info']);
			$dfix['info'] = str_replace("|01", "", $dfix['info']);
			$dfix['info'] = str_replace("|02", "", $dfix['info']);
			$dfix['info'] = str_replace("|03", "", $dfix['info']);
			$dfix['info'] = str_replace("|04", "", $dfix['info']);
			$dfix['info'] = str_replace("|05", "", $dfix['info']);
			$dfix['info'] = str_replace("|06", "", $dfix['info']);
			$dfix['info'] = str_replace("|07", "", $dfix['info']);
			$dfix['info'] = str_replace("|08", "", $dfix['info']);
			$dfix['info'] = str_replace("|09", "", $dfix['info']);
			$dfix['info'] = str_replace("|10", "", $dfix['info']);
			$dfix['info'] = str_replace("|11", "", $dfix['info']);
			$dfix['info'] = str_replace("|12", "", $dfix['info']);
			$dfix['info'] = str_replace("|13", "", $dfix['info']);
			$dfix['info'] = str_replace("|14", "", $dfix['info']);
			$dfix['info'] = str_replace("|15", "", $dfix['info']);
			$dfix['info'] = str_replace("|16", "", $dfix['info']);
			$dfix['info'] = str_replace("|17", "", $dfix['info']);
			$dfix['info'] = str_replace("|18", "", $dfix['info']);
			$dfix['info'] = str_replace("|19", "", $dfix['info']);
			$dfix['info'] = str_replace("|20", "", $dfix['info']);
			$dfix['info'] = str_replace("|21", "", $dfix['info']);
			$dfix['info'] = str_replace("|22", "", $dfix['info']);
			$dfix['info'] = str_replace("|23", "", $dfix['info']);
			$dfix['info'] = str_replace("|24", "", $dfix['info']);			
		}
	}	
	
	return $users;
}

/**
 * convert mystic's dos julien based time to unix time
 * 
 * @uses: dos2unixtime(dostime $var)
 * 
 * @return: unix time 
 *
 * @author: Frank Linhares
 **/

function dos2unixtime($dostime)
{
	$sec  = 2 * ($dostime & 0x1f);
	$min  = ($dostime >> 5) & 0x3f;
	$hrs  = ($dostime >> 11) & 0x1f;
	$day  = ($dostime >> 16) & 0x1f;
	$mon  = (($dostime >> 21) & 0x0f);
	$year = (($dostime >> 25) & 0x7f) + 1980;

	return mktime($hrs, $min, $sec, $mon, $day, $year);
}

/**
 * calculate a user's birthday
 * 
 * @uses: age(birthday $var)
 * 
 * @return: age
 *
 * @author: Frank Linhares
 **/

function age($birthday)
{
	return floor((time() - strtotime($birthday))/31556926);
}

/**
 * pop off multiple keys in an array
 * 
 * @uses: array_multiple_pop(array $var, number of keys $int)
 * 
 * @return: array
 *
 * @author: Frank Linhares
**/

function array_multiple_pop($array, $num)
{
    if(!is_array($array) && is_int($num))
        return false;

    while(($num--)!=false)
        array_pop($array);
    return $array;
}



/**
 * read and parse Mystic's file area lists on Mystic 1.12 A31+
 *
 * @uses: mystic_fareas(path to mystic data dir $var)
 *
 * @return: an array containing the following keys identifying user information from each file area. 
 *
 *       index = the inde number of the file area
 *       name = the name of the file area
 *       ftpname = the ftpname of the file area
 *       filename = the filename of the filearea file lists 
 *       displayfile = the display file of the file area
 *       template = the template file of the file area
 *       listacs = the list acs of the file area
 *       ftpacs = the ftp acs of the file area
 *       dlacs = the dl acs of the file area
 *       ulacs = the ul acs of the file area
 *       hatchacs = the hatch acs of the file area
 *       sysopacs = the sysop acs of the file area
 *       path = the path of the file area
 *       defscan = the default scan setting of the file area
 *       flags = the flags setting of the file area
 *       created = the creation date of the file area
 *       netaddr = the net address of the file area
 *       echotab = the echotag of the file area
 *
 * @author: Darryl Perry
 **/
function mystic_fareas($data_path)
{	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'fbases.dat'); 
	$record_length = 495;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	$fp = fopen ($data_path.'fbases.dat', 'rb');
	for ($i = 0; $i < $record_number; $i++) {

		$data = fread ($fp, $record_length);
	
		// Create a data structure
		
		$data_format =
			'nindex/' .     	# Get the date
			'Cnamelen/' .    	# Get the length of the name field 
			'A40name/' .	 	# Get the username (30) padded with null
			'Cftpnamelen/' .    	# Get the length of the name field 
			'A60ftpname/' .	 	# Get the username (30) padded with null
			'Cfilenamelen/' .    	# Get the length of the name field 
			'A40filename/' .	 	# Get the username (30) padded with null
			'Cdisplayfilelen/' .    	# Get the length of the name field 
			'A20displayfile/' .	 	# Get the username (30) padded with null
			'Ctemplatelen/' .    	# Get the length of the name field 
			'A20template/' .	 	# Get the username (30) padded with null
			'Clistacslen/' .    	# Get the length of the name field 
			'A30listacs/' .	 	# Get the username (30) padded with null
			'Cftpacslen/' .    	# Get the length of the name field 
			'A30ftpacs/' .	 	# Get the username (30) padded with null
			'Cdlacslen/' .    	# Get the length of the name field 
			'A30dlacs/' .	 	# Get the username (30) padded with null
			'Culacslen/' .    	# Get the length of the name field 
			'A30ulacs/' .	 	# Get the username (30) padded with null
			'Chatchacslen/' .    	# Get the length of the name field 
			'A30hatchacs/' .	 	# Get the username (30) padded with null
			'Csysopacslen/' .    	# Get the length of the name field 
			'A30sysopacs/' .	 	# Get the username (30) padded with null
			'Cpathlen/' .    	# Get the length of the name field 
			'A80path/' .	 	# Get the username (30) padded with null
			'Cdefscan/' .    	# Get the length of the name field 
			'lflags/' .    	# Get the length of the name field 
			'lcreated/' .    	# Get the length of the name field 
			'Cnetaddr/' .    	# Get the length of the name field 
			'Cechotaglen/' .    	# Get the length of the name field 
			'A30echotag/' ;	 	# Get the username (30) padded with null
		    
		// Unpack the data structure
		
		$mystic_fareas[] = unpack ($data_format, $data);

		}
	
	// trim and clean up strings

	foreach ( $mystic_fareas as &$sfix ) {
		$sfix['name'] = substr($sfix['name'], 0, $sfix['namelen']);  
		$sfix['ftpname'] = substr($sfix['ftpname'], 0, $sfix['ftpnamelen']); 
		$sfix['filename'] = substr($sfix['filename'], 0, $sfix['filenamelen']); 
		$sfix['displayfile'] = substr($sfix['displayfile'], 0, $sfix['displayfilelen']); 
		$sfix['template'] = substr($sfix['template'], 0, $sfix['templatelen']); 
		$sfix['listacs'] = substr($sfix['listacs'], 0, $sfix['listacslen']); 
		$sfix['ftpacs'] = substr($sfix['ftpacs'], 0, $sfix['ftpacslen']); 
		$sfix['dlacs'] = substr($sfix['dlacs'], 0, $sfix['dlacslen']); 
		$sfix['ulacs'] = substr($sfix['ulacs'], 0, $sfix['ulacslen']); 
		$sfix['hatchacs'] = substr($sfix['hatchacs'], 0, $sfix['hatchacslen']); 
		$sfix['sysopacs'] = substr($sfix['sysopacs'], 0, $sfix['sysopacslen']); 
		$sfix['path'] = substr($sfix['path'], 0, $sfix['pathlen']); 
		$sfix['echotag'] = substr($sfix['echotag'], 0, $sfix['echotaglen']); 
	}	
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $mystic_fareas as &$rfix ) {
		unset($rfix['namelen']);
		unset($rfix['ftpnamelen']);
		unset($rfix['filenamelen']);
		unset($rfix['displayfilelen']);
		unset($rfix['templagelen']);
		unset($rfix['listacslen']);
		unset($rfix['ftpacslen']);
		unset($rfix['dlacslen']);
		unset($rfix['ulacslen']);
		unset($rfix['hatchacslen']);
		unset($rfix['sysopacslen']);
		unset($rfix['pathlen']);
		unset($rfix['echotaglen']);
		unset($rfix['baudlen']);
	}
	
	// check if node is active, if not clear name and set action to "waiting for caller" 	
	
	return $mystic_fareas;
}

/**
 * read and parse Mystic's message area lists on Mystic 1.12 A31+
 *
 * @uses: mystic_mareas(path to mystic data dir $var)
 *
 * @return: an array containing the following keys identifying user information from each message area
 *
 *       name = the name of the message area
 *       qwkname = the qwkname of the message area
 *       newsname = the nntp news name of the message area
 *       filename = the filename of the message area
 *       path = the file path of the data files of the message area
 *       nettype = the nettype of the message area
 *       readtyp3 = the readtype of the message area
 *       listtype = the listtype of the message area
 *       listacs = the listacs of the message area
 *       readacs = the readacs of the message area
 *       postacs = the postacs of the message area
 *       newsacs = the newsacs of the message area
 *       sysopacs = the sysopacs of the message area
 *       sponsor = the sponsor of the message area
 *       colquote = the quote color of the message area
 *       coltext = the text color of the message area
 *       coltear = the tearline color of the message area
 *       colorigin = the origin line color of the message area
 *       colkludge = the kludge line color of the message area
 *       netaddr = the net address of the message area
 *       origin = the origin line of the message area
 *       defnscan = the default newscan setting of the message area
 *       defqscan = the default quickscan setting of the message area
 *       maxmsgs = the max messages of the message area
 *       maxage = the max age of messages of the message area
 *       header = the header file name of the message area
 *       rtemplate = the read template file name of the message area
 *       ltemplate = the list template file name of the message area
 *       index = the index number of the message area
 *       flags = the flags of the message area
 *       created = the creation date of the message area
 *       echotag = the echotag of the message area
 *       qwknetid = the qwknetid of the message area
 *       qwkconfid = the qwknet conf ID of the message area
 *
 * @author: Darryl Perry
 **/
function mystic_mareas($data_path)
{	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'mbases.dat'); 
	$record_length = 640;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	$fp = fopen ($data_path.'mbases.dat', 'rb');
	for ($i = 0; $i < $record_number; $i++) {

		$data = fread ($fp, $record_length);
	
		// Create a data structure
		
		$data_format =
			'Cnamelen/' .    	# Get the length of the name field 
			'A40name/' .	 	# Get the username (30) padded with null
			'Cqwknamelen/' .    	# Get the length of the name field 
			'A13qwkname/' .	 	# Get the username (30) padded with null
			'Cnewsnamelen/' .    	# Get the length of the name field 
			'A60newsname/' .	 	# Get the username (30) padded with null
			'Cfilenamelen/' .    	# Get the length of the name field 
			'A40filename/' .	 	# Get the username (30) padded with null
			'Cpathlen/' .    	# Get the length of the name field 
			'A80path/' .	 	# Get the username (30) padded with null
			'Cunused/' .    	# Get the length of the name field 
			'Cnettype/' .    	# Get the length of the name field 
			'Creadtype/' .    	# Get the length of the name field 
			'Clisttype/' .    	# Get the length of the name field 
			'Clistacslen/' .    	# Get the length of the name field 
			'A30listacs/' .    	# Get the length of the name field 
			'Creadacslen/' .    	# Get the length of the name field 
			'A30readacs/' .    	# Get the length of the name field 
			'Cpostacslen/' .    	# Get the length of the name field 
			'A30postacs/' .    	# Get the length of the name field 
			'Cnewsacslen/' .    	# Get the length of the name field 
			'A30newsacs/' .    	# Get the length of the name field 
			'Csysopacslen/' .    	# Get the length of the name field 
			'A30sysopacs/' .    	# Get the length of the name field 
			'Csponsorlen/' .    	# Get the length of the name field 
			'A30sponsor/' .    	# Get the length of the name field 
			'Ccolquote/' .
			'Ccoltext/' .
			'Ccoltear/' .
			'Ccolorigin/' .
			'Ccolkludge/' .
			'Cnetaddr/' .
			'Coriginlen/' .
			'A50origin/' .
			'Cdefnscan/' .
			'Cdefqscan/' .
			'nmaxmsgs/' .
			'nmaxage/' .
			'Cheaderlen/' .
			'A20header/' .
			'Crtemplatelen/' .
			'A20rtemplate/' .
			'Cltemplatelen/' .
			'A20ltemplate/' .
			'nindex/' .
			'lflags/' .
			'lcreated/' .
			'Cechotaglen/' .
			'A40echotag/' .
			'lqwknetid/' .
			'nqwkconfid/' .
			'A29res/';
		    
		// Unpack the data structure
		
		$mystic_mareas[] = unpack ($data_format, $data);

		}
	
	// trim and clean up strings

	foreach ( $mystic_mareas as &$sfix ) {
		$sfix['name'] = substr($sfix['name'], 0, $sfix['namelen']);  
		$sfix['qwkname'] = substr($sfix['qwkname'], 0, $sfix['qwknamelen']);  
		$sfix['newsname'] = substr($sfix['newsname'], 0, $sfix['newsnamelen']);  
		$sfix['filename'] = substr($sfix['filename'], 0, $sfix['filenamelen']);  
		$sfix['path'] = substr($sfix['path'], 0, $sfix['pathlen']);  
		$sfix['listacs'] = substr($sfix['listacs'], 0, $sfix['listacslen']);  
		$sfix['readacs'] = substr($sfix['readacs'], 0, $sfix['readacslen']);  
		$sfix['postacs'] = substr($sfix['postacs'], 0, $sfix['postacslen']);  
		$sfix['newsacs'] = substr($sfix['newsacs'], 0, $sfix['newsacslen']);  
		$sfix['sysopacs'] = substr($sfix['sysopacs'], 0, $sfix['sysopacslen']);  
		$sfix['sponsor'] = substr($sfix['sponsor'], 0, $sfix['sponsorlen']);  
		$sfix['origin'] = substr($sfix['origin'], 0, $sfix['originlen']);  
		$sfix['header'] = substr($sfix['header'], 0, $sfix['headerlen']);  
		$sfix['rtemplate'] = substr($sfix['rtemplate'], 0, $sfix['rtemplatelen']);  
		$sfix['ltemplate'] = substr($sfix['ltemplate'], 0, $sfix['ltemplatelen']);  
		$sfix['echotag'] = substr($sfix['echotag'], 0, $sfix['echotaglen']);  
	}	
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $mystic_mareas as &$rfix ) {
		unset($rfix['namelen']);
		unset($rfix['qwknamelen']);
		unset($rfix['newsnamelen']);
		unset($rfix['filenamelen']);
		unset($rfix['pathlen']);
		unset($rfix['listacslen']);
		unset($rfix['readacslen']);
		unset($rfix['postacslen']);
		unset($rfix['sysopacslen']);
		unset($rfix['sponsorlen']);
		unset($rfix['originlen']);
		unset($rfix['headerlen']);
		unset($rfix['rtemplatelen']);
		unset($rfix['ltemplatelen']);
		unset($rfix['echotaglen']);
	}

	foreach ( $mystic_mareas as &$dfix ) 
	{			
		// convert pipe codes to proper colours	
		$dfix['name'] = str_replace("|00", "", $dfix['name']);
		$dfix['name'] = str_replace("|01", "", $dfix['name']);
		$dfix['name'] = str_replace("|02", "", $dfix['name']);
		$dfix['name'] = str_replace("|03", "", $dfix['name']);
		$dfix['name'] = str_replace("|04", "", $dfix['name']);
		$dfix['name'] = str_replace("|05", "", $dfix['name']);
		$dfix['name'] = str_replace("|06", "", $dfix['name']);
		$dfix['name'] = str_replace("|07", "", $dfix['name']);
		$dfix['name'] = str_replace("|08", "", $dfix['name']);
		$dfix['name'] = str_replace("|09", "", $dfix['name']);
		$dfix['name'] = str_replace("|10", "", $dfix['name']);
		$dfix['name'] = str_replace("|11", "", $dfix['name']);
		$dfix['name'] = str_replace("|12", "", $dfix['name']);
		$dfix['name'] = str_replace("|13", "", $dfix['name']);
		$dfix['name'] = str_replace("|14", "", $dfix['name']);
		$dfix['name'] = str_replace("|15", "", $dfix['name']);
		$dfix['name'] = str_replace("|16", "", $dfix['name']);
		$dfix['name'] = str_replace("|17", "", $dfix['name']);
		$dfix['name'] = str_replace("|18", "", $dfix['name']);
		$dfix['name'] = str_replace("|19", "", $dfix['name']);
		$dfix['name'] = str_replace("|20", "", $dfix['name']);
		$dfix['name'] = str_replace("|21", "", $dfix['name']);
		$dfix['name'] = str_replace("|22", "", $dfix['name']);
		$dfix['name'] = str_replace("|23", "", $dfix['name']);
		$dfix['name'] = str_replace("|24", "", $dfix['name']);			
	}
	
	return $mystic_mareas;
}

/**
 * read and parse Mystic's file lists on Mystic 1.12 A31+
 *
 * @uses: mystic_dirlist(path to mystic data dir $var, including the dir filename (without the .dir) ) 
 *
 * @return: an array containing the following keys identifying user information from each node. 
 *
 *       filename = the name of the file
 *       filesize = the size of the file
 *       datetime = the timestamp of the file
 *       uploader = who uploaded the file
 *       flags = deleted flag
 *       downloads = download count of the file 
 *       descptr = the file postion of the start of the description
 *       desclins = number of lines in the file description
 *       description = the description of the file
 *
 * @author: Darryl Perry
 **/
function mystic_dirlist($data_path)
{	
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'.dir'); 
	$record_length = 121;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	$fp = fopen ($data_path.'.dir', 'rb');
	for ($i = 0; $i < $record_number; $i++) {

		$data = fread ($fp, $record_length);
	
		// Create a data structure
		
		$data_format =
			'Cfilenamelen/' .
			'A70filename/' .
			'lfilesize/' .
			'ldatetime/' .
			'Cuploaderlen/' .
			'A30uploader/' .
			'Cflags/' .
			'ldownloads/' .
			'Crating/' .
			'Ldescptr/' .
			'Cdesclines/' ;

		$file_list[] = unpack ($data_format, $data);
	}
	fclose($fp);

	foreach ( $file_list as &$sfix ) {
		$start=$sfix['descptr'];
		$endxx=$sfix['desclines'];
		$sfix['description']=mystic_file_description($data_path,$start,$endxx);
		$sfix['filename'] = substr($sfix['filename'], 0, $sfix['filenamelen']);  
		$sfix['uploader'] = substr($sfix['uploader'], 0, $sfix['uploaderlen']);  
		}

	foreach ( $file_list as &$rfix ) {
		unset($rfix['filenamelen']);
		unset($rfix['uploaderlen']);
		}

	foreach ( $file_list as &$dfix ) {
		
		// change date from dos to unix and make human readable
		
		$dfix['datetime'] = dos2unixtime($dfix['datetime']);
		$dfix['datetime'] = date("m/d/Y", $dfix['datetime']);
		}
	return $file_list;
}

function mystic_file_description($data_path,$start,$lines)
{
	$fp = fopen ($data_path.'.des', 'rb');

	fseek($fp,$start);
	for ($x=0; $x < $lines; $x++){
		$b1=ord(fread($fp,1));
		$buffer[] = fread($fp,$b1)."\n";
		}
	fclose($fp);

	return $buffer;
	
}
	
	
/**
 * read and parse Mystic's bbslist.bbi file on Mystic 1.12 A31+
 *
 * @uses: mystic_bbslist(path to mystic data dir $var, including the bbi filename (without the .bbi) ) 
 *
 * @return: an array containing the following keys identifying user information from each node. 
 *
 *       type = BBS Type (telnet or dialup)
 *       phone = the phone number
 *       telnet = the telnet address
 *       bbsname = the bbs name
 *       location = the bbs location
 *       sysop = the bbs sysop 
 *       baud = the bbs baudrate 
 *       software = the bbs software used 
 *       deleted = if the entry is deleted or not 
 *       addedby = who added the entry
 *       verified = date that the entry was verified
 *
 * @author: Darryl Perry
 **/

function mystic_bbslist($data_path)
{
	// get file size in order to determine how many records are stored
	
	$filesize = filesize($data_path.'.bbi'); 
	$record_length = 206;
	
	// divide file size by record length to determine number of records stored
	
	$record_number = $filesize / $record_length;
	
	// Open the mystic BBS data file in binary mode
	$fp = fopen ($data_path.'.bbi', 'rb');

	for ($i = 0; $i < $record_number; $i++) {

		$data = fread ($fp, $record_length);
	
		// Create a data structure
		
		$data_format =
			'Ctype/'.		    # BBS Type (telnet or dialup)
			'Cphonelen/'.		# Get the length of the phone field 
			'A15phone/'.		# Get the phone field
			'Ctelnetlen/'.	# Get the length of the telnet field
			'A40telnet/'.		# Get the telnet field
			'Cbbsnamelen/'.	# Get the length of the bbs name field
			'A30bbsname/'.	# Get the bbsname field
			'Clocationlen/'.		
			'A25location/'.		
			'Csysoplen/'.		
			'A30sysop/'.		
			'Cbaudlen/'.		
			'A6baud/'.		
			'Csoftwarelen/'.		
			'A10software/'.		
			'Cdeleted/'.		
			'Caddedbylen/'.		
			'A30addedby/'.		
			'lverified/'.		
			'A6res/';		    	# jump to the 894th btye     
     
		// Unpack the data structure
		
		$bbslist[] = unpack ($data_format, $data);
		
	}  
	
	// Change date from dos to readable and strip pipe codes 
	
	foreach ( $bbslist as &$lengfix ) {
		$lengfix['phone'] = substr($lengfix['phone'], 0, $lengfix['phonelen']);
		$lengfix['telnet'] = substr($lengfix['telnet'], 0, $lengfix['telnetlen']);
		$lengfix['bbsname'] = substr($lengfix['bbsname'], 0, $lengfix['bbsnamelen']);
		$lengfix['location'] = substr($lengfix['location'], 0, $lengfix['locationlen']);
		$lengfix['sysop'] = substr($lengfix['sysop'], 0, $lengfix['sysoplen']);   
		$lengfix['baud'] = substr($lengfix['baud'], 0, $lengfix['baudlen']);
		$lengfix['addedby'] = substr($lengfix['addedby'], 0, $lengfix['addedbylen']);
		$lengfix['software'] = substr($lengfix['software'], 0, $lengfix['softwarelen']);

	}
	
	// remove length keys now that we have used them to trim their variables	
	
	foreach ( $bbslist as &$rfix ) {
		unset($rfix['phonelen']);
		unset($rfix['telnetlen']);
		unset($rfix['bbsnamelen']);
		unset($rfix['locationlen']);
		unset($rfix['sysoplen']);
		unset($rfix['baudlen']);
		unset($rfix['softwarelen']);
		unset($rfix['addedbylen']);
	}			
	
	foreach ( $bbslist as &$dfix ) {
		$dfix['verified'] = dos2unixtime($dfix['verified']);
		$dfix['verified'] = date("m/d/y", $dfix['verified']);
	}
	
	return $bbslist;
}

?>
