/// WHTSPC.S -----------------------------------------------------------------
/// by Chris Antos
///
/// Various enhancements to the TSE display, including displaying whitespaces
/// on the cursorline.
///
/// Execute the macro to display the options menu.

/*************************************************************************
  WhtSpc       Displays whitespace, bookmarks, and more.

  Author:      Chris Antos

  Date:        06/17/97
               05/19/99  added EOL, more configuration options
               04/12/02  configure right margin file types (SW Rienstra)
                         default = all files == '' (empty string)
                         Bookmark character -> bookmark id letter (a-z)

  Modified by: Eckhard Hillmann

  Date:        Sep 2023  Version 1.04
                         added and modified to much to mention it all here
               Sep 2023  Version 1.05 (Bugfix release)
                         Bugfix: Marker on window border is not properly
                         cleared in vertical window
               Oct 2023  Version 1.06 (Bugfix release)
                         Bugfix: Used wrong color to Highlite Bookmarked Lines
               Oct 2023  Version 1.07
                         Improved detection of window section divider when
                         display is in DISPLAY_HEX mode
                         Unified handling of Highlite Bookmarked Lines with and
                         without transparency set
                         Added conditional compile switches:
                         NEW_MARGIN_DRAWING_BEHAVIOR:  (default = TRUE)
                         TRUE = *always* draw right margin
                         FALSE= draws right margin only when not 80
                         NEW_BOOKMARKED_LINE_BEHAVIOR: (default = TRUE)
                         TRUE = Bookmarked line does *not* overrule cursorline
                         FALSE= Bookmarked line overrules cursorline
                         SHOW_SELECTED_COLORS_IN_MENU: (default = TRUE)
                         TRUE = Shows color numbers in selected color
                         FALSE= Only numbers are shown, no coloring
               Dec 2023  Version 1.08
                         Display version with separator in menu [100] -> [1.00]
                         Change Menu from 'O - Transparent attribute'
                                     to   'O - Text transparency'
                         to clearify what is displayed transparent
               Feb 2024  Version 1.09
                         Menu from 'P - Cursor Line'
                              to   'P - Cursor Line Marker'
                         Buffer-/UnbufferVideo()
                         use PutAttrXY() instead of PutOemStrXY() in
                         DrawBookmarks()
                         Color_EOL() use PutAttrXY()
               Feb 2024  Version 1.10
                         Make it compile with TSE 4.50 RC18 compiler after the
                         compiler has been fixed
               Mrz 2024  Version 1.11
                         Changed initial default settings for
                           'L - Highlite Bookmarked Lines'" to 'On'
                           'U - Color EOL' to 'Off'
                         Color_EOL() (when switched to 'On') now only works in
                         text-mode again. That's what it was expected to do from
                         the beginning. Like the default setting this got
                         messed up.

  Overview:

  WHTSPC adds a few handy visual elements, such as whitespace, bookmarks,
  a cursorline indicator, and a right margin indicator.  Each of these can
  be toggled on/off from a handy menu.


  Keys:       (none)


  Usage notes:

  Execute this macro to show a configuration menu where you can change the
  settings.


  Copyright (none) - FreeWare.

*************************************************************************/


// Conditional compile switches
#define NEW_MARGIN_DRAWING_BEHAVIOR  TRUE  // FALSE= draws right margin only when not 80
#define NEW_BOOKMARKED_LINE_BEHAVIOR TRUE  // FALSE= Bookmarked line overrides cursorline
#define SHOW_SELECTED_COLORS_IN_MENU TRUE  // FALSE= Only color numbers are shown, no coloring


forward integer proc PickColor(var integer nAttr)
forward integer proc Transparent(integer fTransparent, integer attrNew, integer attrCurrent)
forward proc DrawBookmarks(integer fDrawOnBorder, integer attrBorder)
forward proc DrawLeftBorder()
forward proc DrawWhitespace(integer fDraw)
forward integer proc FMarginFile()
forward proc DrawMargin(integer col, integer DoNotCheckMarginChr)
forward proc Color_EOL()
forward proc Idle()
forward proc AfterUpdateDisplay()
forward proc GetSettings()
forward proc WriteSettings()
forward string proc OnOffStr(integer f)
forward proc ToggleVar(var integer f)
forward proc GetString(var string s)
forward proc GetColor(var integer n)
forward proc drawAscii()
forward proc GetAscii(var integer ch)
forward menu Options()
forward proc DoOptions()
forward proc WhenLoaded()
forward proc WhenPurged()
#if SHOW_SELECTED_COLORS_IN_MENU
  forward proc StartColorMenuValues()
  forward proc StopColorMenuValues()
  forward proc NonEditIdle()
#endif
forward proc Main()


string c_stSection[] = "WHTSPC"
string c_stBookmarks[] = "Bookmarks"
string c_stBookmarkAttr[] = "BookmarkAttr"
string c_stBookmarkAttrTransparent[] = "BookmarkAttrTransparent"
string c_stHiliteBookmarks[] = "HiliteBookmarks"
string c_stHiliteBookmarksAttr[] = "HiliteBookmarksAttr"
string c_stHiliteBookmarksAttrTransparent[] = "HiliteBookmarksAttrTransparent"
string c_stHiliteInBlockAttr[] = "HiliteInBlockAttr"
string c_stSidechar[] = "CursorLine"
string c_stSidecharChr[] = "CursorLineChr"
string c_stSidecharAttr[] = "CursorLineAttr"
string c_stSidecharAttrTransparent[] = "CursorLineAttrTransparent"
string c_stMargin[] = "Margin"
string c_stMarginAttr[] = "MarginAttr"
string c_stMarginChar[] = "MarginChar"
string c_stWhitespace[] = "Whitespace"
string c_stSpaceChr[] = "WhitespaceSpaceChr"
string c_stTabChr[] = "WhitespaceTabChr"
string c_stEOL[] = "EOL"
string c_stEOLChr[] = "EOLChr"
string c_stHiliteWhitespace[] = "HiliteWhitespace"
string c_stWhitespaceAttr[] = "WhitespaceAttr"
string c_stWhitespaceAttrTransparent[] = "WhitespaceAttrTransparent"
string c_stMarginExts[] = "MarginFileExtensions"
string c_stSidecharAttrBrowse[] = "CursorLineAttrBrowse"
string c_stColorEOL[] = "ColorEOL"
string c_stColorEOLmark[] = "ColorEOLmark"
string c_stVersion[] = "Version"


integer g_fBookmarks
integer g_attrBookmark
integer g_fBookmarkTransparent
integer g_fHiliteBookmarks
integer g_attrHiliteBookmarksAttr
integer g_fHiliteBookmarksTransparent
integer g_attrInBlockBookmark
integer g_fSidechar
integer g_chSidechar
integer g_attrSidechar
integer g_attrSidecharBrowse
integer g_fSidecharTransparent
integer g_fMargin
integer g_attrMargin
integer g_ChMargin = 179
integer g_fWhitespace
integer g_chSpace
integer g_chTab
integer g_fEOL
integer g_chEOL
integer g_fHiliteWhitespace
integer g_attrWhitespaceAttr
integer g_fWhitespaceAttrTransparent
integer g_fColorEOL
integer g_ColorEOLmark
integer g_colMargin = 0
integer g_fChanged
integer g_idWindows = 0
#if SHOW_SELECTED_COLORS_IN_MENU
  integer g_PutColorToMenuValues = FALSE
#endif

constant WHTSPC_VERSION  = 111

constant BOXED_Y_OFFSET  = 1
constant COLOR_WIDTH     = 2
constant XPOS_MENU_COLOR = 39


// originally: string c_stMarginExts[] = ".c.h.cpp.hpp.cxx.hxx.rc.rc2.idl.odl.s.si.ui.inc.asm.bat.btm.txt.pas.hlp.pl.vbs."
string  g_stMarginExts[255]


integer proc PickColor(var integer nAttr)
  string s[16 * COLOR_WIDTH]
  string a[16 * COLOR_WIDTH]
  integer nI, nJ
  integer nN = nAttr
  integer nCh
  integer fRet = FALSE

  nI = WhereXAbs()
  nJ = WhereYAbs() + 1

  if nI + (16 * COLOR_WIDTH + 2 - 1) > Query(ScreenCols)
    nI = Query(ScreenCols) - (16 * COLOR_WIDTH + 2 - 1)
  endif

  if nJ + (18 - 1) > Query(ScreenRows)
    nJ = Query(ScreenRows) - (18 - 1)
  endif

  PopWinOpen(nI, nJ, nI + (16 * COLOR_WIDTH + 2 - 1), nJ + (18 - 1), 1, "Select Color", Query(CurrWinBorderAttr))

  for nI = 0 to 255 by 16
    s = ""
    a = ""

    for nJ = 0 to 15
      s = s + Format(nI + nJ:COLOR_WIDTH:"0":16)
      a = a + Format("":COLOR_WIDTH:Chr(nI + nJ))
    endfor
    PutStrAttrXY(1, nI / 16 + 1, s, a)
  endfor

  nI = (nN mod 16)
  nJ = nN / 16

  loop
    // position cursor
    GotoXY((nI + 1) * COLOR_WIDTH, nJ + 1)

    // process keys
    nCh = GetKey()
    case nCh
      when <Enter>
        // ok
        nAttr = nJ * 16 + nI
        fRet = TRUE
        break

      when <Escape>
        // cancel
        break

      when <CursorUp>, <GreyCursorUp>
        if nJ > 0
          nJ = nJ - 1
        else
          nJ = 15
        endif

      when <CursorDown>, <GreyCursorDown>
        if nJ < 15
          nJ = nJ + 1
        else
          nJ = 0
        endif

      when <CursorLeft>, <GreyCursorLeft>
        if nI > 0
          nI = nI - 1
        else
          nI = 15
        endif

      when <CursorRight>, <GreyCursorRight>
        if nI < 15
          nI = nI + 1
        else
          nI = 0
        endif

      when <Home>, <GreyHome>
        nI = 0
        nJ = 0

      when <End>, <GreyEnd>
        nI = 15
        nJ = 15
    endcase
  endloop

  PopWinClose()
  return (fRet)
end


integer proc Transparent(integer fTransparent, integer attrNew, integer attrCurrent)
  return (iif(fTransparent, (attrNew & 0x0F) | (attrCurrent & 0xF0), attrNew))
end


proc DrawBookmarks(integer fDrawOnBorder, integer attrBorder)
  integer nI, nId, nLn, nLpos, nXoffs, nRow
  integer nTop, nBottom
  integer nCid = GetBufferId()
  integer nAttr
  string s[3] = ""
  string a[3] = ""
  integer nCurrWinBorderAttr  = Query(CurrWinBorderAttr)
  integer nOtherWinBorderAttr = Query(OtherWinBorderAttr)
  integer nX = 0
  integer nY = 0
  integer nHiliteAttr   = 0
  integer nLineStart    = 0
  integer nLineEnd      = 0
  integer nColStart     = 0
  integer nColEnd       = 0
  integer fInBlock      = FALSE
  integer nBlock        = 0
  integer nWindowX1     = Query(WindowX1)
  integer nWindowY1     = Query(WindowY1)
  integer nWindowCols   = Query(WindowCols)
  integer nWinXpos      = 0
  integer nCursorLine   = CurrLine()
  integer nTempColStart = 0
  integer nTempColEnd   = 0
  integer nXoff         = CurrXoffset()

  if (not g_fBookmarks) and (not g_fHiliteBookmarks)
    return()
  endif

  nTop    = nCursorLine - CurrRow() + 1
  nBottom = nTop + Query(WindowRows)
  nAttr   = Transparent(g_fBookmarkTransparent, g_attrBookmark, attrBorder)
  nBlock  = isBlockInCurrFile()

  // get block coordinates
  if nBlock
    PushPosition()
    GotoBlockBegin()
    nLineStart = CurrLine()
    nColStart  = iif(nBlock == _LINE_, 0, CurrCol() - 1)
    GotoBlockEnd()
    nLineEnd = CurrLine()

    case nBlock
    when _LINE_
      nColEnd = nWindowCols + 2

    //             stream
    when _COLUMN_, _INCLUSIVE_
      nColEnd = CurrCol()

    // word, to end of line...
    when _NONINCLUSIVE_
      nColEnd = CurrCol() - 1

    otherwise
      nColEnd = nWindowCols + 2

    endcase

    PopPosition()
    nColStart = nColStart - iif(nBlock == _LINE_, 0, nXoff)
    nColEnd   = nColEnd   - iif(nBlock == _LINE_, 0, nXoff)
  endif

  // draw bookmark indicators
  for nI = Asc('A') to Asc('Z')
    if isBookMarkSet(Chr(nI))
      GetBookMarkInfo(Chr(nI), nId, nLn, nLpos, nXoffs, nRow)

      // Bookmark in current buffer and in visible area
      if (nId == nCid) and (nLn >= nTop) and (nLn < nBottom)
        nY = nWindowY1 + nLn - nTop

        // bookmark indicator on left border
        if g_fBookmarks and fDrawOnBorder
          PutOemStrXY(nWindowX1 - 1, nY, Chr(nI), nAttr, iif(Query(SpecialEffects) & _USE_3D_CHARS_, _USE3D_, 0))
        endif

#if NEW_BOOKMARKED_LINE_BEHAVIOR
        // highlite the line when bookmark is not at cursorline
        if g_fHiliteBookmarks and nLn <> nCursorLine
#else
        // highlite the line
        if g_fHiliteBookmarks
#endif
          nY = nWindowY1 + nLn - nTop

          // there may be more than one bookmark in one block to display
          nTempColStart = nColStart
          nTempColEnd   = nColEnd

          // bookmark within block
          fInBlock = iif(nLn >= nLineStart and nLn <= nLineEnd, TRUE, FALSE)

          // this block requiers special handling
          if fInBlock and nBlock == _INCLUSIVE_ and nLineStart <> nLineEnd
            if nLineStart == nLn            // first line, mark block partial to end of line
              nTempColEnd = nWindowCols + 2
            elseif nLineEnd == nLn          // last line, mark block partial from start
              nTempColStart = 0
            else                            // lines between, mark full line
              nTempColStart = 0
              nTempColEnd   = nWindowCols + 2
            endif
          endif

          // do the coloring
          for nX = 0 to nWindowCols - 1
            nWinXpos    = nWindowX1 + nX
            nHiliteAttr = g_attrHiliteBookmarksAttr

            // bookmark within block, use different color
            if fInBlock
              if nX >= nTempColStart and nX < nTempColEnd
                nHiliteAttr = g_attrInBlockBookmark
              endif
            endif

            GetStrAttrXY(nWinXpos, nY, s, a, 3)

            // special display mode
            if (DisplayMode() == _DISPLAY_HEX_)
              // check for window section divider
              // divider consists of space+separator+space, for example " | "
              if (s[1] == " ") and (s[3] == " ")
                // character ->                    |    
                and (Asc(s[2]) in 179, 186, 219, 222, 124, 221)
                and (Asc(a[2]) in nCurrWinBorderAttr, nOtherWinBorderAttr)

                // the hex-side will only get the background colored
                PutAttrXY(nWinXpos + 3, nY,
                          ((Query(TextAttr) & 0x0F) | (g_attrHiliteBookmarksAttr & 0xF0)),
                          nWindowCols - nX - 3)
                break                       // were done here
              endif
            endif

            // text transparency on bookmark background
            if g_fHiliteBookmarksTransparent
              PutAttrXY(nWinXpos, nY, ((Asc(a[1]) & 0x0F) | (nHiliteAttr & 0xF0)), 1)
            else
              PutAttrXY(nWinXpos, nY, nHiliteAttr, 1)
            endif
          endfor
        endif
      endif
    endif
  endfor
end


proc DrawLeftBorder()
  string s[1]   = ""
  string a[1]   = ""
  integer fDrawMark = FALSE
  integer fIsDisplayBoxed  = Query(DisplayBoxed) == 2 or (Query(DisplayBoxed) == 1 and Query(MouseEnabled))
  integer fShowLineNumbers = Set(ShowLineNumbers, FALSE)
  integer nX    = Query(WindowX1) - 1
  integer nY    = Query(WindowY1)
  integer nRows = Query(WindowRows)
  integer nYmax = nRows
  integer nIs3D = iif(Query(SpecialEffects) & _USE_3D_CHARS_, _USE3D_, 0)
  integer nI, nJ

  // create my working area
  Window(1, 1, Query(ScreenCols), Query(ScreenRows))

  // look for border type in boxed mode only
  if fIsDisplayBoxed
    // verify this is left border
    GetStrXY(nX, nY, s, 1)
    GetStrXY(nX, nY + nRows - 1, a, 1)

    // character ->/**/ up arrow      /**/ down arrow
    if Asc(s[1]) <> 30 and Asc(a[1]) <> 31
      // clear the left border
      for nI = 0 to nRows - 1
        GetStrAttrXY(nX, nY + nI, s, a, 1)
        // character ->                   |    
        if (Asc(s[1]) in 179, 186, 219, 222, 124, 221)
          // found border type and attr, draw stuff on the border
          // redraw the left border, to erase old sidechar/bookmarks
          for nJ = nY to nY + nRows - 1
            PutOemStrXY(nX, nJ, s[1], Asc(a[1]), nIs3D)
          endfor
          // found border
          fDrawMark = TRUE
          break
        endif
      endfor
    endif
  endif

  // switch linenumbers back to old state after cleaning is done
  Set(ShowLineNumbers, fShowLineNumbers)
  // may have changed while switching linenumbers
  nX = Query(WindowX1) - 1

  // look for border type and attr when linenumbers are active,
  // because they are drawn at a different position
  if   (fShowLineNumbers and fIsDisplayBoxed)
    or (fShowLineNumbers and fIsDisplayBoxed == FALSE and DisplayMode() == _DISPLAY_TEXT_)

    // limit to end of file mark
    if (CurrLine() - CurrRow() + nYMax + 1) > NumLines()
      nYMax = NumLines() - CurrLine() + CurrRow()
    endif

    // clear the marks in range only
    for nI = 0 to nYmax
      GetStrAttrXY(nX, nY + nI, s, a, 1)
      // space is used when showing line numbers
      if s[1] == " "
        // found border type and attr, draw stuff on the border
        // redraw the left border, to erase old sidechar/bookmarks
        for nJ = nY to nYMax
          PutOemStrXY(nX, nJ, s[1], Asc(a[1]), nIs3D)
        endfor
        // found border
        fDrawMark = TRUE
        break
      endif
    endfor
  endif

  // draw left border -> cursor mark
  if fDrawMark and g_fSidechar
    PutOemStrXY(nX, nY + CurrRow() - 1, Chr(g_chSidechar),
      Transparent(g_fSidecharTransparent,
        iif(BrowseMode(), g_attrSidecharBrowse, g_attrSidechar), Asc(a[1])), nIs3D)
  endif

  // draw bookmarks
  DrawBookmarks(fDrawMark, Asc(a[1]))

  // close my working area
  FullWindow()
end


// DrawWhitespace()
// this is more complex than it would seem.
proc DrawWhitespace(integer fDraw)
  string s[255] = ""
  string a[255] = ""
  string st[32]
  integer nXoffs = CurrXoffset()
  integer nTopLn, nRow, nCid, nI, nCh, nAttr

  if not g_fWhitespace
    or CurrLine() > NumLines()
    or DisplayMode() <> _DISPLAY_TEXT_
    or BinaryMode() <> 0
    return()
  endif

  // get string from screen
  GetStrAttrXY(Query(WindowX1), Query(WindowY1)+CurrRow() - BOXED_Y_OFFSET, s, a, Query(WindowCols))
  PushPosition()

  // fix trailing whitespace
  EndLine()
  if CurrCol() - nXoffs <= Query(WindowCols)
    s = SubStr(s, 1, CurrCol() - nXoffs - 1)
    s = Format(s:-Query(WindowCols):" ")
  endif

  // draw whitespace characters
  GotoColumn(nXoffs + 1)
  repeat
    nI = CurrCol() - nXoffs
    case CurrChar()
    when 32
      nCh = iif(fDraw, g_chSpace, Asc(" "))
      nAttr = Asc(a[nI])

      if g_fHiliteWhitespace and CurrPos() > PosLastNonWhite()
        nAttr = Transparent(g_fWhitespaceAttrTransparent, g_attrWhitespaceAttr, nAttr)
      endif

      PutOemStrXY(Query(WindowX1) + nI - 1, Query(WindowY1) + CurrRow() - BOXED_Y_OFFSET, Chr(nCh), nAttr, FALSE)

    when 9
      nI = (((CurrCol() - 1) / Query(TabWidth)) + 1) * Query(TabWidth) - nXoffs
      nCh = 0
      nAttr = Asc(a[nI])

      if nI <= Query(WindowCols)
        nCh = iif(fDraw, g_chTab, Asc(" "))
      endif

      if g_fHiliteWhitespace and CurrPos() > PosLastNonWhite()
        nAttr = Transparent(g_fWhitespaceAttrTransparent, g_attrWhitespaceAttr, nAttr)
      endif

      PutOemStrXY(Query(WindowX1) + nI - 1, Query(WindowY1) + CurrRow() - BOXED_Y_OFFSET, Chr(nCh), nAttr, FALSE)

    when _AT_EOL_
      if g_fEOL
        nCh = iif(fDraw, g_chEOL, Asc(" "))
        nAttr = Asc(a[nI])

        if g_fHiliteWhitespace
          nAttr = Transparent(g_fWhitespaceAttrTransparent, g_attrWhitespaceAttr, nAttr)
        endif

        PutOemStrXY(Query(WindowX1) + nI - 1, Query(WindowY1) + CurrRow() - BOXED_Y_OFFSET, Chr(nCh), nAttr, FALSE)
      endif
      break

    when _BEYOND_EOL_
      break
    endcase
    Right()
  until nI >= Query(WindowCols)

  // put string to screen
  PopPosition()

  // update previous line if CursorAttr == TextAttr
  if fDraw and Query(CursorAttr) == Query(TextAttr)
    // get current info
    nRow   = CurrRow()
    nTopLn = CurrLine() - CurrRow() + 1
    nCid   = GetBufferId()

    // goto/create window info buffer
    if g_idWindows
      GotoBufferId(g_idWindows)
    else
      g_idWindows = CreateTempBuffer()

      do 32 times      // actual max is 9, allow for growth
        AddLine()
      enddo
    endif

    // check if need to force redraw
    GotoLine(WindowId())
    st = GetText(1, CurrLineLen())

    if nRow <> Val(GetToken(st, " ", 1))
       and nTopLn == Val(GetToken(st, " ", 2))
       and nXoffs == Val(GetToken(st, " ", 3))
       and nCid   == Val(GetToken(st, " ", 4))
      GotoBufferId(nCid)
      PushPosition()
      GotoRow(Val(GetToken(st, " ", 1)))
      DrawWhitespace(FALSE)
      PopPosition()
    endif

    // record current window info
    GotoBufferId(g_idWindows)
    BegLine()
    KillToEol()
    InsertText(Format(nRow; nTopLn; nXoffs; nCid))
    GotoBufferId(nCid)
  endif
end


integer proc FMarginFile()
  return ((Length(CurrExt()) and Pos(CurrExt() + ".", g_stMarginExts) or g_stMarginExts == '')
           and DisplayMode() == _DISPLAY_TEXT_)
end


proc DrawMargin(integer col, integer DoNotCheckMarginChr)
  string s[1]   = ""
  string a[1]   = ""
  string Ch[1]  = Chr(g_ChMargin)
  integer nX    = Query(WindowX1) + col - CurrXoffset()
  integer nY
  integer nYmax = Query(WindowRows)
  integer nTop  = CurrLine() - CurrRow() + nYmax + 1
  integer nWindowY1   = Query(WindowY1) - BOXED_Y_OFFSET
  integer nMarginAttr = g_attrMargin & 0x0F   // margin is always displayed in transparent
  integer nMarginChr  =  iif(DoNotCheckMarginChr, 0, g_ChMargin)

  // limit to end of file mark
  if ntop > NumLines()
    nYMax = NumLines() - CurrLine() + CurrRow()
  endif

  for nY = nWindowY1 + 1 to nWindowY1 + nYmax
    GetStrAttrXY(nX, nY, s, a, 1)

    if (Asc(s[1]) in 32, nMarginChr)
      PutOemStrXY(nX, nY, Ch, (Asc(a[1]) & 0xF0) | nMarginAttr, FALSE)
    endif
  endfor
end


proc Color_EOL()
  integer nX, nR, nWest, nEast, nWidth, nCc, nLn, nPl

  if not g_fColorEOL
     or DisplayMode() <> _DISPLAY_TEXT_
     or BinaryMode()  <> 0
    return ()
  endif

  // Save the state
  PushPosition()
  nX     = CurrCol()
  nWidth = Query(WindowCols)
  nWest  = CurrXOffset()
  nEast  = nWest + nWidth
  nR     = 1
  nPl    = 0

  while (nR <= Query(WindowRows))
    GotoRow(nR)
    GotoColumn(nEast + 2)
    nCc = CurrChar()
    nLn = CurrLine()
    GotoColumn(nX)
    GotoXoffset(nWest)

    if (nLn <= nPl)
      break
    endif

    nPl = nLn

    if (nCc <> _BEYOND_EOL_)
      PutAttrXY(Query(WindowX1) + nWidth - 1, nR + Query(WindowY1) - 1, g_ColorEOLmark, 1)
    endif

    nR = nR + 1
  endwhile

  // Restore the state
  PopPosition()
end


proc Idle()
  if Query(RightMargin) <> g_colMargin
    g_colMargin = Query(RightMargin)
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  endif
end


proc AfterUpdateDisplay()
  integer fMargin
  integer nXoff   = CurrXoffset()
  integer nArea   = nXoff + Query(WindowCols)
  integer nMargin = Query(RightMargin)

  if BufferType() == _NORMAL_
    // don't update the screen until all is drawn
    BufferVideo()

    // draw stuff on left border
    DrawLeftBorder()

    // whitespace
    DrawWhitespace(TRUE)

    // vertical margin lines
    fMargin = FMarginFile()

#if NEW_MARGIN_DRAWING_BEHAVIOR
    // only draw in visible area, always draw margin
    if g_fMargin and fMargin and ((nArea > nMargin) and (nXoff <= nMargin))
      DrawMargin(nMargin, g_fHiliteBookmarksTransparent)
    endif
#else
    // only draw in visible area, only draw margin when not 80
    if g_fMargin and fMargin and ((nArea > nMargin) and (nXoff <= nMargin))
      and nMargin <> 80
      DrawMargin(nMargin, g_fHiliteBookmarksTransparent)
    endif
#endif

    Color_EOL()

    // all is drawn, update the screen now
    UnbufferVideo()
  endif
end


proc GetSettings()
  g_fWhitespace = GetProfileInt(c_stSection, c_stWhitespace, TRUE)
  g_chSpace = GetProfileInt(c_stSection, c_stSpaceChr, 250)
  g_chTab = GetProfileInt(c_stSection, c_stTabChr, 170)
  g_chEOL = GetProfileInt(c_stSection, c_stEOLChr, 20)
  g_fEOL = GetProfileInt(c_stSection, c_stEOL, FALSE)
  g_fHiliteWhitespace = GetProfileInt(c_stSection, c_stHiliteWhitespace, FALSE)
  g_attrWhitespaceAttr = GetProfileInt(c_stSection, c_stWhitespaceAttr, Color(Bright Red on Black))
  g_fWhitespaceAttrTransparent = GetProfileInt(c_stSection, c_stWhitespaceAttrTransparent, TRUE)

  g_fBookmarks = GetProfileInt(c_stSection, c_stBookmarks, TRUE)
  g_attrBookmark = GetProfileInt(c_stSection, c_stBookmarkAttr, Color(blue on white))
  g_fBookmarkTransparent = GetProfileInt(c_stSection, c_stBookmarkAttrTransparent, FALSE)

  g_fHiliteBookmarks = GetProfileInt(c_stSection, c_stHiliteBookmarks, TRUE)
  g_attrHiliteBookmarksAttr = GetProfileInt(c_stSection, c_stHiliteBookmarksAttr, Color(Blue on White))
  g_fHiliteBookmarksTransparent = GetProfileInt(c_stSection, c_stHiliteBookmarksAttrTransparent, TRUE)
  g_attrInBlockBookmark = GetProfileInt(c_stSection, c_stHiliteInBlockAttr, Query(CursorInBlockAttr))

  g_fSidechar = GetProfileInt(c_stSection, c_stSidechar, TRUE)
  g_chSidechar = GetProfileInt(c_stSection, c_stSidecharChr, 16)
  g_attrSidechar = GetProfileInt(c_stSection, c_stSidecharAttr, Color(Bright Red on Black))
  g_attrSidecharBrowse = GetProfileInt(c_stSection, c_stSidecharAttrBrowse, Color(Bright Green on Black))
  g_fSidecharTransparent = GetProfileInt(c_stSection, c_stSidecharAttrTransparent, TRUE)

  g_fColorEOL = GetProfileInt(c_stSection, c_stColorEOL, FALSE)
  g_ColorEOLmark = GetProfileInt(c_stSection, c_stColorEOLmark, 78)

  g_fMargin = GetProfileInt(c_stSection, c_stMargin, TRUE)
  g_ChMargin = GetProfileInt(c_stSection, c_stMarginChar, 179)
  g_attrMargin = GetProfileInt(c_stSection, c_stMarginAttr, Query(TextAttr))
  g_stMarginExts = GetProfileStr(c_stSection, c_stMarginExts, "")

  // This is here to automatically save only the new missing default values
  // when the version changes. The old data remains unchanged
  if GetProfileInt(c_stSection, c_stVersion, -1) <> WHTSPC_VERSION
    WriteSettings()
  endif
end


proc WriteSettings()
  WriteProfileInt(c_stSection, c_stWhitespace, g_fWhitespace)
  WriteProfileInt(c_stSection, c_stSpaceChr, g_chSpace)
  WriteProfileInt(c_stSection, c_stTabChr, g_chTab)
  WriteProfileInt(c_stSection, c_stEOLChr, g_chEOL)
  WriteProfileInt(c_stSection, c_stEOL, g_fEOL)
  WriteProfileInt(c_stSection, c_stHiliteWhitespace, g_fHiliteWhitespace)
  WriteProfileInt(c_stSection, c_stWhitespaceAttr, g_attrWhitespaceAttr)
  WriteProfileInt(c_stSection, c_stWhitespaceAttrTransparent, g_fWhitespaceAttrTransparent)

  WriteProfileInt(c_stSection, c_stBookmarks, g_fBookmarks)
  WriteProfileInt(c_stSection, c_stBookmarkAttr, g_attrBookmark)
  WriteProfileInt(c_stSection, c_stBookmarkAttrTransparent, g_fBookmarkTransparent)

  WriteProfileInt(c_stSection, c_stHiliteBookmarks, g_fHiliteBookmarks)
  WriteProfileInt(c_stSection, c_stHiliteBookmarksAttr, g_attrHiliteBookmarksAttr)
  WriteProfileInt(c_stSection, c_stHiliteBookmarksAttrTransparent, g_fHiliteBookmarksTransparent)
  WriteProfileInt(c_stSection, c_stHiliteInBlockAttr, g_attrInBlockBookmark)

  WriteProfileInt(c_stSection, c_stSidechar, g_fSidechar)
  WriteProfileInt(c_stSection, c_stSidecharChr, g_chSidechar)
  WriteProfileInt(c_stSection, c_stSidecharAttr, g_attrSidechar)
  WriteProfileInt(c_stSection, c_stSidecharAttrBrowse, g_attrSidecharBrowse)
  WriteProfileInt(c_stSection, c_stSidecharAttrTransparent, g_fSidecharTransparent)

  WriteProfileInt(c_stSection, c_stColorEOL, g_fColorEOL)
  WriteProfileInt(c_stSection, c_stColorEOLmark, g_ColorEOLmark)

  WriteProfileInt(c_stSection, c_stMargin, g_fMargin)
  WriteProfileInt(c_stSection, c_stMarginChar, g_ChMargin)
  WriteProfileInt(c_stSection, c_stMarginAttr, g_attrMargin)
  WriteProfileStr(c_stSection, c_stMarginExts, g_stMarginExts)

  WriteProfileInt(c_stSection, c_stVersion, WHTSPC_VERSION)
end


string proc OnOffStr(integer f)
  return(iif(f, "On", "Off"))
end


proc ToggleVar(var integer f)
  f = not f
  g_fChanged = TRUE
end


proc GetString(var string s)
  string ss[255] = s

#if SHOW_SELECTED_COLORS_IN_MENU
  StopColorMenuValues()
#endif
  if Read(ss)
    s = ss
    g_fChanged = TRUE
  endif

#if SHOW_SELECTED_COLORS_IN_MENU
  StartColorMenuValues()
#endif
end


proc GetColor(var integer nN)
#if SHOW_SELECTED_COLORS_IN_MENU
  StopColorMenuValues()
#endif
  if PickColor(nN)
    g_fChanged = TRUE
  endif

#if SHOW_SELECTED_COLORS_IN_MENU
  StartColorMenuValues()
#endif
end


proc drawAscii()
  PutOemStrXY(VWhereX(), VWhereY(), GetText(1,255), Query(Attr), FALSE)
  ClrEol()
end


// mAsciiChart
// display ASCII chart and returns selected character; <Esc> to abort.
proc GetAscii(var integer nCh)
  integer nI, nCid, nChartId

#if SHOW_SELECTED_COLORS_IN_MENU
  StopColorMenuValues()
#endif
  nCid     = GetBufferId()
  nChartId = CreateTempBuffer()

  if nChartId
    nI = 0

    while AddLine(Format(nI:5, Str(nI, 16):4, Chr(nI):4)) and nI < 255
      nI = nI + 1
    endwhile

    GotoLine(nCh + 1)
    ScrollToCenter()
    HookDisplay(drawAscii,,,)

    if List("DEC HEX Char", 20)
      nCh = CurrLine() - 1
      g_fChanged = TRUE
    endif

    UnHookDisplay()
  endif

  GotoBufferId(nCid)
  AbandonFile(nChartId)

#if SHOW_SELECTED_COLORS_IN_MENU
  StartColorMenuValues()
#endif
end


menu Options()
  title = "WHTSPC [V 1.11] Options"
  history

  "&A - Whitespace" [OnOffStr(g_fWhitespace):3]
  , ToggleVar(g_fWhitespace)
  , _MF_CLOSE_ALL_AFTER_
  , "Show whitespace on cursor line."

  "&B - Space character..." [Chr(g_chSpace):1]
  , GetAscii(g_chSpace)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&C - Tab character..." [Chr(g_chTab):1]
  , GetAscii(g_chTab)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&D - EOL character..." [Chr(g_chEOL):1]
  , GetAscii(g_chEOL)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&E - Show EOL" [OnOffStr(g_fEOL):3]
  , ToggleVar(g_fEOL)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)
  , "Show EOL marker on cursor line."

  "&F - Highlite Trailing Whitespace" [OnOffStr(g_fHiliteWhitespace):3]
  , ToggleVar(g_fHiliteWhitespace)
  ,_MF_CLOSE_ALL_AFTER_ | iif(g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)
  , "Highlite trailing whitespace to make it more visible."

  "&G - Highlite attribute..." [format(g_attrWhitespaceAttr:2:"0":16):2]
  , GetColor(g_attrWhitespaceAttr)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fHiliteWhitespace and g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&H - Transparent attribute" [OnOffStr(g_fWhitespaceAttrTransparent):3]
  , ToggleVar(g_fWhitespaceAttrTransparent)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fHiliteWhitespace and g_fWhitespace, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "",,_MF_DIVIDE_

  "&I - Bookmarks" [OnOffStr(g_fBookmarks):3]
  , ToggleVar(g_fBookmarks)
  , _MF_CLOSE_ALL_AFTER_
  , "Display a marker for bookmarked lines, on the left window border."

  "&J - Marker attribute..." [format(g_attrBookmark:2:"0":16):2]
  , GetColor(g_attrBookmark)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fBookmarks, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&K - Transparent attribute" [OnOffStr(g_fBookmarkTransparent):3]
  , ToggleVar(g_fBookmarkTransparent)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fBookmarks, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "",,_MF_DIVIDE_

  "&L - Highlite Bookmarked Lines" [OnOffStr(g_fHiliteBookmarks):3]
  , ToggleVar(g_fHiliteBookmarks)
  , _MF_CLOSE_ALL_AFTER_
  , "Highlite lines that are bookmarked."

  "&M - Highlite attribute..." [format(g_attrHiliteBookmarksAttr:2:"0":16):2]
  , GetColor(g_attrHiliteBookmarksAttr)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fHiliteBookmarks, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&N - Highlite in block attribute..." [format(g_attrInBlockBookmark:2:"0":16):2]
  , GetColor(g_attrInBlockBookmark)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fHiliteBookmarks, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&O - Text transparency" [OnOffStr(g_fHiliteBookmarksTransparent):3]
  , ToggleVar(g_fHiliteBookmarksTransparent)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fHiliteBookmarks, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "",,_MF_DIVIDE_

  "&P - Cursor Line Marker" [OnOffStr(g_fSidechar):3]
  , ToggleVar(g_fSidechar)
  , _MF_CLOSE_ALL_AFTER_
  , "Display a marker for the current line, on the left window border."

  "&Q - Marker character" [Chr(g_chSidechar):1]
  , GetAscii(g_chSidechar)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fSidechar, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&R - Marker attribute edit..." [format(g_attrSidechar:2:"0":16):2]
  , GetColor(g_attrSidechar)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fSidechar, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&S - Marker attribute browse..." [format(g_attrSidecharBrowse:2:"0":16):2]
  , GetColor(g_attrSidecharBrowse)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fSidechar, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&T - Transparent attribute" [OnOffStr(g_fSidecharTransparent):3]
  , ToggleVar(g_fSidecharTransparent)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fSidechar, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "",,_MF_DIVIDE_

  "&U - Color EOL" [OnOffStr(g_fColorEOL):3]
  , ToggleVar(g_fColorEOL)
  , _MF_CLOSE_ALL_AFTER_
  , "Mark Lines longer than visible area."

  "&V - EOL attribute..." [format(g_ColorEOLmark:2:"0":16):2]
  , GetColor(g_ColorEOLmark)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fColorEOL, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "",,_MF_DIVIDE_

  "&W - Margin" [OnOffStr(g_fMargin):3]
  , ToggleVar(g_fMargin)
  , _MF_CLOSE_ALL_AFTER_
  , "Display a vertical line at the right margin."

  "&X - Margin character..." [Chr(g_ChMargin):1]
  , GetAscii(g_ChMargin)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fMargin, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&Y - Margin transparent attribute..." [format(g_attrMargin:2:"0":16):2]
  , GetColor(g_attrMargin)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fMargin, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)

  "&Z - Margin file types" [g_stMarginExts:15]
  , GetString(g_stMarginExts)
  , _MF_CLOSE_ALL_AFTER_ | iif(g_fMargin, _MF_ENABLED_, _MF_GRAYED_ | _MF_SKIP_)
  , "File types that show right margin."
end


proc DoOptions()
  g_fChanged = FALSE

#if SHOW_SELECTED_COLORS_IN_MENU
  StopColorMenuValues()
  Hook(_NONEDIT_IDLE_, NonEditIdle)
  StartColorMenuValues()
#endif
  BufferVideo()

  // looping allows us to refresh the display after each change
  while Options()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  endwhile

  UnBufferVideo()
#if SHOW_SELECTED_COLORS_IN_MENU
  StopColorMenuValues()
  UnHook(NonEditIdle)
  UpdateDisplay(_STATUS_LINE_REFRESH_)
#endif
  // if anything changed, write the settings to the ini file
  if g_fChanged
    WriteSettings()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  endif
end


proc WhenLoaded()
  GetSettings()
  Hook(_AFTER_UPDATE_DISPLAY_, AfterUpdateDisplay)
  Hook(_IDLE_, Idle)
end


proc WhenPurged()
  if g_idWindows
    AbandonFile(g_idWindows)
  endif
end

#if SHOW_SELECTED_COLORS_IN_MENU
proc StartColorMenuValues()
  g_PutColorToMenuValues = TRUE
end


proc StopColorMenuValues()
  g_PutColorToMenuValues = FALSE
end


proc NonEditIdle()
  if g_PutColorToMenuValues
    PutAttrXY(XPOS_MENU_COLOR,  7, g_attrWhitespaceAttr, 2)       // "G"
    PutAttrXY(XPOS_MENU_COLOR, 11, g_attrBookmark, 2)             // "J"
    PutAttrXY(XPOS_MENU_COLOR, 15, g_attrHiliteBookmarksAttr, 2)  // "M"
    PutAttrXY(XPOS_MENU_COLOR, 16, g_attrInBlockBookmark, 2)      // "N"
    PutAttrXY(XPOS_MENU_COLOR, 21, g_attrSidechar, 2)             // "R"
    PutAttrXY(XPOS_MENU_COLOR, 22, g_attrSidecharBrowse, 2)       // "S"
    PutAttrXY(XPOS_MENU_COLOR, 26, g_ColorEOLmark, 2)             // "V"
    PutAttrXY(XPOS_MENU_COLOR, 30, g_attrMargin, 2)               // "Y"
    Message("")                                                   // this makes it work
  endif
end
#endif

proc Main()
  DoOptions()
end


public proc WHTSPC_DrawWhitespace()
  DrawWhitespace(TRUE)
end


