/*************************************************************************
  Where       A "whereis" command for TSE

  Author:     Ian Campbell

  Date:       Aug 22, 1994  Initial version
              Aug 28, 1994  First revision
              Apr 11, 1995  Second revision for TSE 2.02e beta
              May 28, 1996  Converted to TSE32 (no sort by extension)
              Oct 11, 1996  Merging changes by Chris Antos & SemWare
              Oct 15, 1996  Change "Sort Order" quick key from 'r' to 'o'.
              Feb  7, 1997  SEM - change labels to not conflict with new continue keyword
              Jan 30, 1998  SEM - can't load files with spaces in the name: fixed
              Feb 18, 1998  SEM - in win32, convert filelist to ascii
              Aug 21, 1998  SEM - use EditFile(QuotePath(fn)) in mViewFile()
              Aug 24, 1998  SEM - use _HIDDEN_ with CreateBuffer to avoid conflicts
                            with other macros.
              Jun 18, 1999  SEM - bug fix - where.dat stored in loaddir, but
                            loaded from mac dir - fix loading code.
              Aug  2, 1999  Also remove dir_buff in whenpurged - thanks to Chris Antos
              Sep 15, 2000  SEM - fix long-standing speed-search bug
              Feb  9, 2002  JHB - Made helpdef text non-OEM font compliant
              Feb 13, 2002  JHB - LFN fix for mSmartEditFile().
                                  LFN fix for mAcceptFiles().
                                  Cosmetic centering of Titles on Ask() dialogs.
                                  TAG_CHAR changed to "-" for font compatibility.
                                  TAG_DONE_CHAR changed  to "+"
                                  Standardised colors of border/text in mPopUpHelp()
              Feb 18, 2002  JHB - Win32 version now stores Where.dat data in TSE.INI
              Dec 06, 2010  SEM - Use builtin StartPgm instead of ShellExecute DLL
                                - Add copy filename to Windows/Editor clipboard
              Nov 04, 2011  SEM - Changed SORT_MAXIMUM to 100,000 (faster machines)
              Jun 30, 2015  SEM - use same viewer as f.s
              May 14, 2016  SEM - show correct filesize for large files when creating
                                  filelist.tse
              Mar 27, 2020  SEM - remove all the old DOS logic
              Nov 02, 2021  SEM - add <alt z> - zip tagged files
              Jan 03, 2023  SEM - try to fix broken drive processing

  Version     0.97

  Overview:

  This command will search a drive or multiple drives for the
  specified file(s), and present the user with a list of all such
  found files. From this list, you can either edit or delete the file.

  Keys:
        <CtrlAlt W>         Where()

  Usage notes:

  Normally, Where searches the drives that have been specified via
  the configuration menu.  However, you can override the default by
  specifying one or more drives along with the filename, as in:
  "cde:*.doc", which would search drives "c", "d", and "e" for all
  "*.doc" files.

  Searches may also include a path, for instance:

  C:\TSE\*.MAC will find all ".MAC" files in all of the TSE
  subdirectories. (eg) C:.\*.EXE will find all ".EXE" files
  beginning with the current directory.

*************************************************************************/

/*
                                WHERE

Version 0.95    Original release.
*/
/*
Version 0.96    Allowed multiple definition of drive letters in the
                search file spec (eg) "CDE:*.DOC" will search drives "C"
                through "E" for all files matching the ".DOC" extension.
                Requested by David Marcus.

                Allowed entry of partial paths.  (eg) C:\TSE\*.MAC will
                find all ".MAC" files in all of the TSE subdirectories.
                (eg) C:.\*.EXE will find all ".EXE" files beginning with
                the current directory.  Requested by David Marcus.

                Changed the sort mechanism so that the same cursor
                position no longer sticks to the filename.  Also,
                _DEFAULT_ is now used for all sort options.

                Bug fix.  Entries without extensions could not be
                matched.  Fixed.

                Misc. mouse cleanup for some of the prompts.

                A few minor cosmetic cleanups.
*/
/*
Version 0.97    WHERE can now search for multiple files concurrently.
                Simply enter the names of the files in the prompt box,
                each separated by a space character, and then begin the
                search.  WHERE will find all of them, and present
                them in the "found files" picklist.

                Added file tagging to the "found files" picklist.  Now
                multiple matching filenames may be tagged and
                conveniently loaded into the editor.  This tagging works
                the same as the tagging found in the macro EDITFILE.MAC.

                Cleaned up the Sort Menu function to make it a little
                easier to use and more like the one in EDITFILE.MAC.
                Sort fields include Date, Extension, Name, Pathname,
                Size, and Attribute.

                In addition to searching for files with the normal
                attribute set, WHERE now automatically searches for
                files with the SYSTEM, HIDDEN, AND READ-ONLY attributes
                set.  These attributes are displayed in the found files
                list to the right of the time.  Also, the new sort menu
                has been enhanced to allow sorting on these attributes.

                Added auto-sizing logic to select the vertical size of
                the list display as a proportion of the vertical size of
                the screen.  Also, horizontally extended the width of
                the list buffer from 78 to 80 characters.

                The "Searching Drive" section on the display will now
                flash while filenames are being searched.  This makes it
                a bit easier to tell when a search has completed.

                Special checks have been added to reduce the time to
                respond to the escape key when searching a series of non
                existent network drives.  Also, the drive letter is now
                displayed on the title line while the search is being
                attempted so that you always know what is happening.

                Various cosmetic cleanups / speedups.

*/

/*
Constants.

*/

constant FN_COMP = 0x4
constant    SORT_MAXIMUM    = 100000,   // Complain if higher than 100,000 entries
            TAG_COL         = 1,        // The tag character column
            TAG_CHAR        = 45,       // Character to use for tagging
            TAG_DONE_CHAR   = 43        // Character to flag a tag completion

constant    YES_NO          = 2         // for MsgBox, hardcoded so source code is portable to TSE 2.5

constant
            ffALLNV         = _DIRECTORY_|_ARCHIVE_|_HIDDEN_|_SYSTEM_|_READONLY_,
            ffALL           = ffALLNV|_VOLUME_,
            MONTH_COL       = 2,        // The month column
            MONTH_WIDTH     = 2,        // The month width
            DAY_COL         = 5,        // The day column
            DAY_WIDTH       = 2,        // The day width
            YEAR_COL        = 8,        // The year column
            YEAR_WIDTH      = 2,        // The year width
            TIME_COL        = 11,       // The time column
            TIME_WIDTH      = 5,        // The time width
            SIZE_COL        = 16,       // The size column
            SIZE_WIDTH      = 12,       // The size width
            ATTR_COL        = 29,       // The attribute column
            ATTR_WIDTH      = 4,        // The attribute width
            NAME_COL        = 34,       // The name column
            NAME_WIDTH      = 32,       // The width of the name
            //$32 not sure good way to do this in TSE/32
            //EXT_COL         = 10,       // The extension column
            //EXT_WIDTH       = 4,        // The extension width
            DIR_COL         = 67,       // The directory column
            DIR_WIDTH       = 80        // The directory width

constant MAXPATH = _MAX_PATH_
constant MAXNAME = MAXPATH
constant MAXEXT = 12

#define SW_HIDE             0
#define SW_SHOWNORMAL       1
#define SW_NORMAL           1
#define SW_SHOWMINIMIZED    2
#define SW_SHOWMAXIMIZED    3
#define SW_MAXIMIZE         3
#define SW_SHOWNOACTIVATE   4
#define SW_SHOW             5
#define SW_MINIMIZE         6
#define SW_SHOWMINNOACTIVE  7
#define SW_SHOWNA           8
#define SW_RESTORE          9
#define SW_SHOWDEFAULT      10
#define SW_MAX              10

proc change_dir(string path)
    if path[2] == ':'
        LogDrive(path[1])
    endif
    ChDir(path)
end

proc start(string fn)
    string save_dir[_MAX_PATH_]

    save_dir = CurrDir()
    change_dir(SplitPath(fn, _DRIVE_|_PATH_))

    if Lower(SplitPath(fn, _EXT_)) == ".exe"
        lDos(fn, "", _RUN_DETACHED_|_DONT_WAIT_|_DONT_PROMPT_|_DONT_CLEAR_)
    else
        StartPgm(fn)
    endif

    change_dir(save_dir)
end

/*
Global Strings.

*/
string GlobalSearchString[MAXPATH]  // The complete file(s) search string

string DriveLtrs[30] = "cdefghijklmnopqrstuvwxyz"
string DriveLetters[30] = ""        // The working drive letters
string DriveLetter[1] = ""          // The current drive letter
string PathStart[MAXPATH] = ""      // If a path is specified, use it as start dir
string SearchName1[MAXPATH]         // 1st string (_NAME_) to match (if it exists)
string SearchName2[MAXPATH] = ""    // 2nd string (_EXT_) to match (if it exists)
string SearchName3[MAXPATH] = ""    // Alternate to 1st string (an OR)
string ListPath[MAXNAME] = ""       // The filename when the file picklist activated
string mpath[MAXPATH] = ""          // Complete pathname for the search

string where_title[] = " Where "
string where_title2[] = "Where"

/*
Global Integers.

*/
integer file_history            // History number
integer drive_history           // History number
integer DriveLtr = 1            // Indexes to the appropriate drive letter
integer dir_buff                // Scratch buffer used to mark directory names
integer pick_buff               // The filename list buffer ID
integer DirCount = 0            // Count the number of directories found
integer ListVertical            // The vertical size of the list buffer
integer Accepted = FALSE        // Set if a file was edited from the list buffer
integer TagCount = 0            // A dynamic count of the number of current tags
integer PermissionGranted = 0   // Only grant > 1000 sort item permission once
integer longest_line            // longest line in the where buffer - for listing

forward proc mViewZip()

/*
This help information is displayed when the user selects "Help" from
the "Search for file:" prompt.

*/
helpdef PopupHelp1
    TITLE = "WHERE Help"

    " <Alt L>    View File List"
    ""
    "   Show the file list from a previous search.  This selection is only"
    "   available if a previous search has been done, matching files were found,"
    "   and the list buffer has not been deleted."
    "______________________________________________________________________________"
    ""
    " <Alt C>   Configure"
    ""
    " - Edit Drive Letters "
    ""
    "   Enter a list of drive letters between A and Z in the order that you"
    "   want the directories to be searched.  Drives that do not exist will be"
    "   ignored.  By default, drives C to Z will be checked in alphabetical"
    "   order."
    ""
    " - AutoLoad WHERE"
    ""
    "   Determines whether or not WHERE will automatically load when TSE "
    "   starts.  "
    ""
    " All changes will be preserved across editing sessions."
    "______________________________________________________________________________"
    "<Alt Z> Zip tagged files                                                      "
    "______________________________________________________________________________"
    ""
    ' <Escape>  Exit'
    ''
    '   Exit the prompt box without taking any action.'
    "______________________________________________________________________________"
    ''
    ' <Alt X>  Exit and release buffer'
    ''
    '   Exit the prompt box.  Release any memory used to store the last file'
    '   list information (if one exists)'
    "______________________________________________________________________________"
    ''
    ' General:'
    ''
    ' This macro may be popped up either by executing it, or via <CtrlAlt W>'
    ' if it has already been loaded.  This key may be changed or'
    ' eliminated -- it is located at the end of the WHERE.S source file.'
    ''
    ' Enter the filename to search for -- you may optionally include DRIVE'
    ' LETTERS, a partial PATH, a NAME, and an EXTENSION.  (eg) "\TSE\*.MAC will'
    ' extract all ".MAC" files in C:\TSE, and all of its subdirectories.  If no'
    ' drive letter is specified, then all drives will be searched.  Finally, press'
    ' <Enter> to begin the search.  Multiple filenames per search are now accepted.'
    ' Note that drive letters are only accepted at the beginning of the prompt,'
    ' and they apply to multiple filenames if more than one is given.'
    ''
    ' The "*" may be used as a wildcard -- the "?", if found, will be treated'
    ' as an "*".  In addition to its standard usage, the "*" may also be entered'
    ' at the beginning of the name.  In this case, all files containing the'
    ' search characters anywhere within the name will be matched.'
    ''
    ' While a search is progressing, you may press the <Escape> key to'
    ' immediately terminate the remainder of the search and present a PickList'
    ' of any found items, or you may press the <Enter> key to immediately edit'
    ' a found file.'
    "______________________________________________________________________________"
    ''
    ' EXAMPLES:'
    ''
    ' abc               Exact match to "ABC", ALL extensions'
    ' abc.*             Exact match to "ABC", ALL extensions'
    ' c:abc*.exe        File begins with "ABC" on drive C: and ".EXE" extension'
    ' abc*.             File begins with "ABC", NO extension'
    ' abc.              Exact match to "ABC", NO extension'
    ' abc*.e*           File begins with "ABC", extension begins with "E"'
    ''
    ' *abc.exe          Matches "ABC" anywhere in filename, extension is ".EXE"'
    ' *abc.*            Matches "ABC" anywhere in filename, ALL extensions match'
    ' *abc              Matches "ABC" anywhere in filename, ALL extensions match'
    ''
    ' c:*.*             Matches ALL files on drive C:'
    ' cde:*.*           Matches ALL files on drives C:, D: and E:'
    ' *.*               Matches ALL files on ALL drives'
    ''
    ' .\abc.exe         Exact match to "ABC.EXE", beginning in the current'
    '                   directory and including the rest of its subdirectories'
    ' .                 List ALL files beginning with the current directory and'
    '                   including the rest of its subdirectories'
    ' ..                List ALL files beginning with the parent directory and'
    '                   including the rest of its subdirectories'
    ' c:\tse\*.mac      List ALL ".MAC" files beginning with the TSE main directory'
    '                   and including all of its subdirectories'
    ''
    ' cde:*.tse *.s *.mac'
    '                   This will find all ".TSE", ".S", and ".MAC" files across '
    '                   three disk drives.'
    ''
end PopupHelp1

/*
This help information is displayed when the user selects "Help" from
the "File List" display.

*/
helpdef PopupHelp2
    TITLE = "WHERE Help"
    ""
    " SORTING"
    ""
    " HotKeys used to invoke the sort:"
    ""
    "   <Ctrl S>"
    "     or"
    "   <Alt S>"
    ""
    "   Files may be sorted by Date, Extension, Name, PathName, Size,  or"
    "   Attribute."
    ""
    "   Some supported Norton Commander shortcut sort keys include:"
    ""
    "   <Ctrl F3>   Sort by Name"
    "   <Ctrl F4>   Sort by Extension"
    "   <Ctrl F5>   Reverse sort by Date"
    "   <Ctrl F6>   Reverse sort by Size"
    "________________________________________________________________"
    ""
    " TAGGING"
    ""
    " HotKeys used for tagging include:"
    ""
    "   <Spacebar>      toggle a tag On or Off"
    "   <LeftBtn>       Click the left mouse button to toggle a tag"
    "   <Grey*>         ALL tags On"
    "   <Grey/>         ALL tags Off"
    "   <Grey+>         Current tag On"
    "   <Insert>        Current tag On"
    "   <Grey->         Current tag Off"
    "   <Ctrl F>        Go to First tag"
    "   <Ctrl T>        Go to lasT tag"
    "   <Ctrl N>        Go to Next tag"
    "   <Ctrl P>        Go to Previous tag"
    "________________________________________________________________"
    ""
    ' LOADING'
    ''
    ' HotKey <Enter> or <RightBtn>'
    ''
    "   Press <Enter> or click the mouse <RightBtn> to load all "
    "   tagged filenames into TSE's ring of files.  Control will be"
    '   returned to the file list when this operation has completed.'
    ''
    '   If no files are tagged, then the selected file will be'
    '   loaded and switched to in the normal way.'
    "________________________________________________________________"
    ''
    ' VIEWING'
    ''
    ' HotKey <Alt V>'
    ''
    '   Press <Alt V> to conveniently view a found file without'
    '   permanently loading it.'
    "________________________________________________________________"
    ''
    ' FILE DELETION'
    ''
    ' HotKey <Del>'
    ''
    "   Press <Del> to erase ALL marked files, or the selected"
    '   file if no files are marked.  A confirmation prompt will'
    '   be issued before any action is taken.'
    ''
    "   This provides a mechanism to eliminated unwanted files.  For example,"
    '   suppose you wanted to delete all backup files across all disks.  You'
    '   could search for "*.bak", and then tag and delete them.'
    "______________________________________________________________________________"
    ''
    " VIEW RELATED DIRECTORY"
    ""
    " HotKey <Alt L>"
    ""
    "   Provides a PickList of files located in the same directory as the"
    "   highlighted file.  Any file in this directory may be highlighted and"
    "   edited by pressing <Enter>.  If a file is selected in this manner,"
    "   the current directory will be changed to match this directory."
    "______________________________________________________________________________"
    ""
    ' Exit File List (Keep List)'
    ''
    ' HotKey <Esc>'
    ''
    '   Return to the "Search for file" prompt.  Retain the list buffer for '
    '   possible later use.'
    "______________________________________________________________________________"
    ""
    " EXIT FILE LIST (Delete List)"
    ""
    " HotKey <Alt X>"
    ""
    '   Return to the "Search for file" prompt.  Delete the list buffer and '
    '   return the used memory back to the system.'
    "______________________________________________________________________________"
    ''
    " SAVE LIST BUFFER TO DISK"
    ""
    " HotKey <Alt F>"
    ""
    '  Save a permanent copy of your filenames to disk.  You will be'
    '  prompted for a name ("filelist.tse" is the default).'
    ''
    '  For example, to extract all filenames on all disks, simply search'
    '  for "*.*".  To extract all "doc" files, search for "*.doc", etc.'
    "______________________________________________________________________________"
    "<Alt Z> Zip tagged files                                                      "
    "______________________________________________________________________________"
end PopupHelp2

/*

Display one of two QuickHelp type of screens, depending on the value in
helptype.

Called by:  keydef ListKeys, mWhereIs().

*/
proc mPopupHelp(integer helptype)

    case helptype
        when 1
            QuickHelp(PopUpHelp1)
        when 2
            QuickHelp(PopUpHelp2)
    endcase

end mPopupHelp

/*
Writes a header and footer message to the filelist detailing how the
search is progressing, and providing some keystroke help.  This
information is updated periodically while the drives are being searched.

*/
proc FileMessage(string DriveLetter, integer pick_buff)
    integer ID = GetBufferID()
    string TitleMessage1[30]
    string TitleMessage2[30] = " Press {<Esc>} to stop "
    string FooterMsg[80]
    string UprDriveLetter[1] = Upper(DriveLetter)
    string Count[10]

    TitleMessage1 = " Searching Drive {"
    TitleMessage1 = TitleMessage1 + UprDriveLetter
    TitleMessage1 = TitleMessage1 + ":} "
    GotoBufferID(pick_buff)
    if DriveLetter <> "\"
        VGotoXY(Query(WindowX1) + 4, Query(WindowY1) - 1)
//        Set(MenuTextAttr, Query(MenuTextAttr) | 0x80)
        PutHelpLine(TitleMessage1)
//        Set(MenuTextAttr, Query(MenuTextAttr) & 0x7f)
        VGotoXY(Query(WindowX1) + Query(WindowCols) - Length(TitleMessage2),
            Query(WindowY1) - 1)
        PutHelpLine(TitleMessage2)
    endif
    FooterMsg = " Found {"
    Count = Str(NumLines())
    FooterMsg = FooterMsg + Count
    FooterMsg = FooterMsg + "} files in {"
    Count = Str(DirCount)
    FooterMsg = FooterMsg + Count
    FooterMsg = FooterMsg + "} directories "
    WindowFooter(FooterMsg)
    GotoBufferID(ID)
end FileMessage

string proc pbname_ex()
    return (ExpandTilde(PBName()))
end

integer proc mAddFileToBuffer(integer pick_buff, integer n)
    integer OldID, displayed, len
    string path[_MAX_PATH_]

    displayed = FALSE
    OldID = GotoBufferID(pick_buff)
    GotoLine(NumLines())
    path = SplitPath(mpath, _DRIVE_ | _PATH_)
    len = length(path) + length(FFName()) + 47  // account for size, date, etc
    longest_line = max(longest_line, len)
    AddFFInfoToBuffer(pick_buff, SqueezePath(Path, _USE_HOME_PATH_))
    if n > 100 or n == 0
        UpdateDisplay()
        displayed = TRUE
    endif
    GotoBufferID(OldID)
    return (displayed)
end mAddFileToBuffer

/*
Compare the filename in the ffblk to the search filename.  If they match,
then return TRUE, otherwise return FALSE

Called by:  mMatchAllFileNames()
*/
integer proc mCmpFileName(string fn)
    string s1[MAXPATH]
    string s2[MAXPATH]
    string s3[MAXPATH]

    // first, check for a leading wildcard
    if Length(SearchName1) and SearchName1[1] == "*"
        s1 = fn
        // s2 = the search filename without leading "*"
        s2 = SearchName1[2 : Length(SearchName1) - 1]
        // s3 = the alternate search filename without the leading "*"
        s3 = SearchName3[2 : Length(SearchName3) - 1]
        // now, do the compare -- any kind of match is acceptable
        if Pos(s2, s1) or Pos(s3, s1)
            return(TRUE)
        endif
    // Next, check for a search filename with a trailing "+".  A "+" is
    // used to indicate that the search filename ended with a "*." combination.
    elseif Length(SearchName1) and SearchName1[Length(SearchName1)] == "+"
        // filter out the "+", and place the search filename into s2
        s2 = SearchName1[1 : Length(SearchName1) - 1]
        // extract the filename from the ffblk into s1
        s1 = fn[1:Length(s2)]              // GDB: CA added this, not quite certain why
        s1 = fn                            // GDB: Especially when this overwrites it right away....
        // names match?
        if s2 == s1
            // extract the name again from ffblk
            s1 = fn
            // check for a "." -- not allowed with the "*." syntax
            if not Pos(".", s1)
                return(TRUE)
            endif
        endif
    // try for an exact name match now
    elseif SearchName1 == fn[1 : Length(SearchName1)]
        // is there a second search filename (an extension)?  If so, then
        // it must match too.
        if Length(SearchName2)
            s1 = SplitPath(fn, _EXT_) + Chr(0)                             // GDB: Change to make sure extensions make exact match
            // Does the extension match?
            if Pos(SearchName2, s1)
                return(TRUE)
            endif
        else
            return(TRUE)
        endif
    // is there an alternate name to try to match?
    elseif Length(SearchName3)
        if SearchName3 == fn[1 : Length(SearchName3)]
            return(TRUE)
        endif
    endif
    return(FALSE)
end mCmpFileName

/*
Search an entire subdirectory, looking for files or subdirectory entries.
As files are found, compare them to the search filename, and if they match,
then add them to list buffer pick_buff.  As subdirectories are found, add them to
scratch buffer dir_buff for later searching.  Pass the first letter of both the
name and the extension to the binary so that the binary can act as a filter
for faster searching.

There are three SearchName global strings.
If SearchName1 (_NAME_) exists, then try to match it to the found filename.
If SearchName2 (_EXT_) exists, then it must match also (an AND condition).
If SearchName3 exists and if it matches the filename, then s1 does not matter
   (an OR condition).

Called by:  mHookViewFiles()

*/
proc mMatchAllFileNames(integer TokenIndex)
    string s1[MAXPATH]
    string s2[MAXNAME]
    integer OldID
    integer count = 0
    integer h
    integer total = 0, last_display = 0

    OldID = GotoBufferID(dir_buff)               // dir buffer

    // Search an entire subdirectory, and add all sudirectories to scratch
    // buffer dir_buff.
    // (DOS:  The binary will return any filenames with a matching first
    // letter in either the _NAME_ component or the _EXT_ component.)
    // Compare these names with the search filename, and if a match occurs,
    // add the file components to list buffer pick_buff.
    h = FindFirstFile(mpath, -1)
    if h <> -1
        repeat                              // loop until all names found
            count = count + 1
            total = total + 1
            if FFAttribute() & _DIRECTORY_ // is this a directory?
                // place the directory name in string s
                s1 = (SplitPath(mpath, _DRIVE_ | _PATH_))
                s1 = Lower(s1)
                s2 = FFName()
                s1 = s1 + s2
                if s2 <> "." and s2 <> ".."
                    if TokenIndex == 1
                        DirCount = DirCount + 1
                    endif
                    AddLine(s1, dir_buff)        // add the pathname directory entry
                endif
            else
                // compare the ffblk filename to the search filename
                if mCmpFileName(Lower(FFName()))
                    // matched -- add the file info to the pick_buff list buffer
                    FileMessage(mpath[1], pick_buff)
                    if mAddFileToBuffer(pick_buff, total - last_display)
                        last_display = total
                        count = 0
                    endif
                elseif count >= 99
                    // Update the prompt box header/footer messages
                    FileMessage(mpath[1], pick_buff)
                    GotoBufferid(pick_buff)
                    UpdateDisplay()
                    GotoBufferId(dir_buff)
                    count = 0                   // reduce the updating load
                endif
            endif
        // repeat until all files are searched
        until not FindNextFile(h, -1)
        FindFileClose(h)
        // Done, update the prompt box header/footer messages
        FileMessage(mpath[1], pick_buff)
    endif
    GotoBufferID(OldID)         // restore original buffer ID
end mMatchAllFileNames

/*
This routine saves the drive letter configuration information to the
TSE.INI file.  It returns TRUE on success, otherwise FALSE.

*/
integer proc UpdateDiskDat()
    return( WriteProfileStr(SplitPath(CurrMacroFileName(),_NAME_),"Drives",DriveLtrs))
end UpdateDiskDat

/*
Edit the passed filename if it exists and flag the file as "Accepted".  If the
file is one of several known binary extensions, then edit it in binary mode.

Called by:  mDisplayListBuffer()

*/
integer proc mSmartEditFile(string s)
    string s1[MAXPATH] = QuotePath(s)

    case Lower(SplitPath(s1, _EXT_))
        when    ".exe", ".bin", ".com", ".dll", ".obj", ".pif",
                ".bmp", ".xls", ".xlm", ".zip", ".ico", ".ovl",
                ".qwk", ".slc"
            // JHB: LFN BugFix - it was using: s1 = "-b " + s
            s1 = "-b " + s1      // set to a binary load for above extensions
    endcase
    if FileExists(s)
        if EditFile(s1)
            Accepted = TRUE     // user picked a file
            UpdateDisplay()
            return(TRUE)
        endif
    endif
    return(FALSE)
end mSmartEditFile

/*
Provides a sort menu while viewing the list buffer.  Filenames may be sorted
in a number of ways.

Called by:  mSortListFiles()

*/

/****************************************************************************
   Helper routines for Sort on the Options menu.
 ***************************************************************************/
integer sort_flags      // used for Sorting

string proc ShowSortFlag()
    return (iif(sort_flags & 1, "Descending", "Ascending"))
end

proc ToggleSortFlag(integer which)
    if sort_flags & which
        sort_flags = sort_flags & ~ which
    else
        sort_flags = sort_flags | which
    endif
end

/***********************************************************************
    Sort the pick list.  This routine uses the SortMenu() to determine
    what keys to sort on:

    Option      Primary key             Secondary key           Third key
    ------      -----------             -------------           ---------
    NAME        Name                    Extension
    EXTENSION   Extension               Name
    SIZE        Size                    Name
    DATE        Date                    Time                    Name
    ATTRIBUTE   Attribute               Name
    Also uses the sort_flags variable to determine order.
***********************************************************************/
constant
    SORT_NAME_EXT = 1,
    SORT_EXT_NAME = 2,
    SORT_SIZE     = 3,
    SORT_DATE     = 4,
    SORT_ATTR     = 5,
    SORT_PATH     = 6

proc SortList(integer sortby)
    string flags[3], key[MAXPATH]
    integer y = CurrRow()

    key = PBName()
    flags = ""
    case sortby
        when 0  return()
        when 1  flags = "NE"   // Name
        when 2  flags = "EN"   // Ext
        when 3  flags = "SN"   // Size
        when 4  flags = "DTN"  // Date
        when 5  flags = "AN"   // Attribute
        when 6  flags = "PNE"  // Path
    endcase
    if NOT sort_flags          // Decending order?
        flags = lower(flags)   // if so, lower case our flags
    endif
    Set(PickFileSortOrder, flags)  // set the sort order
    Sort(_PICK_SORT_)
    lFind(key, "g")
    ScrollToRow(y)
end

proc mSortByDateDescending()
    integer flags = sort_flags

    sort_flags = 1
    SortList(SORT_DATE)
    sort_flags = flags
end

proc mSortBySizeDescending()
    integer flags = sort_flags

    sort_flags = 1
    SortList(SORT_SIZE)
    sort_flags = flags
end

menu SortMenu()
    Title = "WHERE Sort Menu"
    History
    Command = SortList(MenuOption())
    "&Name"
    "&Extension"
    "&Size"
    "&Date"
    "&Attribute"
    "&Path"
    "", , Divide
    "Sort &Order"   [ShowSortFlag() : 10], ToggleSortFlag(1), DontClose
end SortMenu

/*
This routine checks the number of filenames that must be sorted, and if it
exceeds SORT_MAXIMUM, it issues a warning to the user, since the sort may be
quite slow.

Called by:  Assigned to <Ctrl S> in ListKeys.

*/
proc mSortListFiles()
    integer OldY1 = Set(Y1, ((Query(ScreenRows) - 6) / 2) - 4)
    integer OldX1 = Set(X1, (Query(ScreenCols) - 20) / 2)

    if (NumLines() > SORT_MAXIMUM) and not PermissionGranted
        if MsgBox(where_title, "There are more than "
                    + format(SORT_MAXIMUM) + " items," + Chr(13)
                    + "so sorting may be quite slow." + Chr(13) + Chr(13) + "Continue?", YES_NO) == 1
            PermissionGranted = TRUE
            SortMenu()
        endif
    else
        SortMenu()
    endif
    Set(X1, OldX1)
    Set(Y1, OldY1)
end mSortListFiles

/*
Writes data to the top and bottom border sections of the window.  The
mouse is hidden to try and prevent any possible slowdown with some slow
mouse drivers.

Called by:  mPutListMessages()

*/
proc mFastHeaderFooterMsg(integer X, integer Y, string msg)
    HideMouse()
    VGotoXY(X, Y)
    PutHelpLine(msg)
    ShowMouse()
end mFastHeaderFooterMsg

/*
Keeps the WindowHeader and WindowFooter messages current (in case the user
scrolls horizontally), and the mouse walks on it.

Called by:  Attached to the cursor keys in ListKeys and called from
            mListStartup().

*/
string PutTitleMessage1[] = " {<Ctrl S>} Sort Files "
string PutTitleMessage2[] = " {<SpaceBar>} Tag Files "
proc mPutListMessages(integer TitleRefresh)
    if TitleRefresh
        // JHB: Center Title
        mFastHeaderFooterMsg((Query(ScreenCols)-Length(where_title))/2, Query(WindowY1) - 1, where_title)
        mFastHeaderFooterMsg(5, Query(WindowY1) - 1, PutTitleMessage1)
        mFastHeaderFooterMsg(53, Query(WindowY1) - 1, PutTitleMessage2)
    endif
    mFastHeaderFooterMsg(4, Query(WindowY1) + Query(PopWinRows),
                        " {<F1>} {H}elp           {" + Format(NumLines())
                        + "} files in {"
                        + Format(DirCount)
                        + "} directories             {Tags}: "
                        + Format(TagCount:-4))
end mPutListMessages

/*
Add "filelist.tse" to the edit history, and then envoke "SaveAs()".

Called by:  Assigned to <Alt F> in listkeys.

*/
string attr_pos[] = Chr(_READONLY_) + Chr(_HIDDEN_) + Chr(_SYSTEM_) + Chr(_VOLUME_) + Chr(_DIRECTORY_) + Chr(_ARCHIVE_)
string attr_ltr[] = "rhsvda"
string proc mPBAttribute()
    integer a, i
    string astr[6] = "      "

    a = PBAttribute()
    for i = 1 to Length(attr_pos)
        if a & Asc(attr_pos[i])
            astr[i] = attr_ltr[i]
        endif
    endfor

    return (astr)
end

string proc mPBSize()
    return (iif(PBAttribute() & _DIRECTORY_,
            "   <Dir>", PBSizeStr()))
end

/**************************************************************************
  List buffer needs to be hidden to prevent conflicts with other macros.
  However, we don't want undo info originally, so create it as _SYSTEM_.
 **************************************************************************/
integer proc PBToAscii(integer pb_id)
    integer list_id
    string path[MAXPATH]

    Message("Creating list file...")
    PushPosition()

    list_id = CreateTempBuffer()

    GotoBufferId(pb_id)
    BegFile()
    repeat
        path = PBName()
        AddLine(Format(mPBSize():12; PBDateStr(); PBTimeStr(); mPBAttribute()), list_id)
        GotoBufferId(list_id)
        GotoPos(CurrLineLen() + 2)
        InsertText(path)
        GotoBufferId(pb_id)
    until not Down()

    GotoBufferId(pb_id)
    PopPosition()

    GotoBufferId(list_id)
    BufferType(_HIDDEN_)
    BegFile()
    return (list_id)
end

proc mSaveListFile()
    integer id

    PushPosition()
    id = PBToAscii(GetBufferId())
    AddHistoryStr("filelist.tse", _EDIT_HISTORY_)
    SaveAs()
    PopPosition()
    AbandonFile(id)
end mSaveListFile

/*
Place/remove the appropriate tag to the left of the selected filename in
the filelist.  Track the tag count on the bottom of the file list window.

Called by:  mLeftBtn(), ListKeys

*/
proc mToggleTags(integer function, integer GoDown)
    integer CurrentPosition = CurrPos()
    string tag[1] = GetText(TAG_COL, 1)

    tag = iif(tag == Chr(TAG_DONE_CHAR), ' ', tag)
    BegLine()
    case function
        when 0              // tag off
            InsertText(" ", _OVERWRITE_)
        when 1              // tag on
            InsertText(Chr(TAG_CHAR), _OVERWRITE_)
        when 2              // toggle tag
            InsertText(iif(tag == Chr(TAG_CHAR), " ", Chr(TAG_CHAR)),
                _OVERWRITE_)
    endcase
    if tag <> GetText(TAG_COL, 1)
        TagCount = iif(tag <> Chr(TAG_CHAR), TagCount + 1, TagCount - 1)
    endif
    GotoPos(CurrentPosition)
    if GoDown
        Down()
    endif
    UpdateDisplay()
    mPutListMessages(FALSE)
end mToggleTags

/*
Load all tagged files into TSE's ring of files, and update the tag count
at the bottom of the window.  If no files are tagged, then pass the name
of the selected file and exit the file list.

Called by:  mRightBtn(), ListKeys

*/
proc mAcceptFiles()
    integer id = GetBufferID()
    string fn[MAXPATH]
//    integer row = CurrRow()
    integer bmode

    if TagCount
        PrevFile(_DONT_LOAD_)       // get the current file's filename
        GotoBufferID(id)
        PushPosition()
        BegFile()
        lFind(Chr(TAG_CHAR), "^")
        repeat
//            ScrollToRow(row)
//            UpdateDisplay()
            if not TagCount
                break
            endif
            // file tagged?
            if GetText(TAG_COL, 1) == Chr(TAG_CHAR)
                BegLine()
                TagCount = TagCount - 1
//                mPutListMessages(FALSE)
                fn = PBName()
                bmode = 0
                case Lower(SplitPath(fn, _EXT_))
                    when    ".exe", ".bin", ".com", ".dll", ".obj", ".pif",
                            ".bmp", ".xls", ".xlm", ".zip", ".ico", ".ovl",
                            ".qwk", ".slc"
                        bmode = 64 // set to a binary load for above extensions
                endcase
                // JHB: LFN fix...
                AddFileToRing(QuotePath(fn), bmode)
                GotoBufferID(id)            // back to the control buffer
                BegLine()
                InsertText(Chr(TAG_DONE_CHAR), _OVERWRITE_)
            endif
            if KeyPressed()
                if GetKey() == <Escape>
                    break
                endif
            endif
        until not lFind(Chr(TAG_CHAR), "^")
        PopPosition()
    else
        // accept the single selection and edit that file
        EndProcess(1)
    endif
    mPutListMessages(FALSE)
end mAcceptFiles

/*
Delete the selected filename in the file list.

Called by:  mDeleteFiles()

Returns TRUE if the file is deleted, otherwise FALSE.

*/
integer proc mDeleteSelectedFile(string file)
    mPutListMessages(FALSE)
    if not EraseDiskFile(file)
        warn('Error, Unable to delete "' + file +'"')
        return(FALSE)
    endif
    return(TRUE)
end mDeleteSelectedFile

/*
Delete all tagged files, or the highlighted file or directory if no
files are tagged.

Called by:
*/
proc mDeleteFiles()
    string s[MAXPATH], tag[1]
//    integer row = CurrRow()

    if TagCount
            if MsgBox(where_title2, "Delete ALL marked files?", YES_NO) == 1
                PushPosition()
                BegFile()
                lFind(Chr(TAG_CHAR), "^")
                repeat
                continue_label:
                    if KeyPressed()
                        if GetKey() == <Escape>
                            break
                        endif
                    endif
//                    ScrollToRow(row)
//                    UpdateDisplay()
                    if not TagCount
                        break
                    endif
                    // file tagged?
                    if GetText(TAG_COL, 1) == Chr(TAG_CHAR)
                        if mDeleteSelectedFile(PBName())
                            TagCount = TagCount - 1
                            KillLine()
                            goto continue_label
                        endif
                    endif
                until not Down()
                PopPosition()
                mPutListMessages(FALSE)
            endif
    else                            // one single file or directory
        s = PBName()
        if MsgBox(where_title2, 'Delete "' + s + '" ?', YES_NO) == 1
            // get the current tag state
            tag = GetText(TAG_COL, 1)
            if mDeleteSelectedFile(s)
                // file was tagged?
                if tag == Chr(TAG_CHAR)
                    TagCount = TagCount - 1
                endif
                KillLine()
                mPutListMessages(FALSE)
            endif
        endif
    endif
    mPutListMessages(FALSE)
    if Numlines() == 0  // !!! GDB: Mod to fix Deletion Bug
        EndProcess()    // !!! GDB:
    endif               // !!! GDB:
end mDeleteFiles

/*------------------------------------------------------------------------
  Query for the zip name
  Create a temp file, with the files to zip
  Then run:
  dos zip -@tempfile zipname
  delete the temp file
 ------------------------------------------------------------------------*/
proc mZipFiles()
    string zipfn[MAXPATH] = "", tempname[MAXPATH]
    integer temp_id

    if TagCount == 0
        Warn("No files tagged - zip not created")
        return ()
    endif

    loop
        if not AskFilename("Target zipfile name:", zipfn) or trim(zipfn) == ""
            return ()
        endif
        if splitpath(zipfn, _EXT_) == ""
            zipfn = zipfn + ".zip"
        endif
        if not FileExists(zipfn)
            break
        elseif MsgBox("File Exists", "Update existing zipfile?", _YES_NO_CANCEL_) == 1
            break
        endif
    endloop

    tempname = MakeTempName(CurrDir())
    PushLocation()
    temp_id = EditFile(tempname)
    PopLocation()

    PushLocation()
    BegFile()
    lFind(Chr(TAG_CHAR), "^")
    repeat
        // file tagged?
        if GetText(TAG_COL, 1) == Chr(TAG_CHAR)
            AddLine(QuotePath(pbname_ex()), temp_id)

            BegLine()
            InsertText(" ", _OVERWRITE_)
            TagCount = TagCount - 1
        endif
        if KeyPressed()
            if GetKey() == <Escape>
                break
            endif
        endif
    until TagCount == 0 or not lFind(Chr(TAG_CHAR), "^")
    GotoBufferId(temp_id)
    //Can't use SaveAndQuitFile, as it can quit the editor
    SaveFile()
    AbandonFile()
    PopLocation()

    if not Dos("zip -@ " + QuotePath(zipfn) + " <" + tempname, _DONT_PROMPT_|_START_HIDDEN_)
        MsgBox("Zip failed", "Zip failed")
    else
        EraseDiskFile(tempname)
    endif

    mPutListMessages(FALSE)
end

/*
Tag all filenames in the file list

Called by:  ListKeys

*/
proc mTagAllFiles(integer TagChar)
    PushBlock()
    UnmarkBlock()                       // clear any possible existing block
    PushPosition()
    MarkColumn(1, TAG_COL, NumLines(), TAG_COL)
    TagCount = iif(TagChar == TAG_CHAR, NumLines(), 0)
    // fill with either TAG_CHAR or space
    FillBlock(Chr(TagChar))
    PopPosition()
    PopBlock()
    mPutListMessages(FALSE)
end mTagAllFiles

/*
Count any files with a TAG_DONE_CHAR in the tag column, and update the
footer message when done.

Called by:  mReTag()
*/
proc mCountTags()
    PushPosition()
    TagCount = 0
    BegFile()
    if lFind(Chr(TAG_CHAR), "^")
        repeat
            TagCount = TagCount + 1
        until not lFind(Chr(TAG_CHAR), "^+")
    endif
    PopPosition()
    mPutListMessages(FALSE)
end mCountTags

/*
Retag any files with an "*" in the tag column.

Called by:  Assigned to a key in ListKeys
*/
proc mReTag()
    PushPosition()
    lReplace(Chr(TAG_DONE_CHAR), Chr(TAG_CHAR), "^gn")
    mCountTags()
    PopPosition()
end mReTag

/*
This handles the left mouse click while in the file list.

Called by:  ListKeys
*/
proc mLeftBtn()
    if ProcessHotSpot()
        if Query(MouseY) == 0 or Query(MouseX) == 0
            EndProcess()
        elseif Query(MouseY) <= Query(PopWinRows)
                and Query(MouseX) <= Query(PopWinCols)
            mToggleTags(2, FALSE)
        endif
    else
        EndProcess()
    endif
    mPutListMessages(FALSE)
end mLeftBtn

/*
This handles the right mouse click while in the file list.

Called by:  ListKeys
*/
proc mRightBtn()
    if ProcessHotSpot()
        if Query(MouseY) == 0 or Query(MouseX) == 0
            EndProcess()
        elseif Query(MouseY) <= Query(PopWinRows)
                and Query(MouseX) <= Query(PopWinCols)
            mAcceptFiles()
        endif
    else
        EndProcess()
    endif
    mPutListMessages(FALSE)
end mRightBtn

/*
Find one of the following:
    1.  the first tag
    2.  the last tag
    3.  the next tag
    4.  the previous tag

Called by:  ListKeys

*/
proc mFindTag(integer key)
    integer row

    if TagCount
        row = CurrRow()
        case key
            when <Ctrl F>
                lFind(Chr(TAG_CHAR), '^g')
            when <Ctrl T>
                lFind(Chr(TAG_CHAR), '^bg')
            when <Ctrl N>
                lFind(Chr(TAG_CHAR), '^+')
            when <Ctrl P>
                lFind(Chr(TAG_CHAR), '^b')
        endcase
        ScrollToRow(row)
        UpdateDisplay()
    endif
end mFindTag

proc mShell()
    string
        olddir[MAXPATH] = CurrDir(),
        newdir[MAXPATH] = SplitPath(pbname_ex(), _DRIVE_|_PATH_)

    ChDir(newdir)
    Shell()
    ChDir(olddir)
end

integer copy_buf_id
proc copy_string_clipboard(string s)
    PushBlock()
    PushPosition()

    if copy_buf_id == 0
        copy_buf_id = CreateTempBuffer()
    endif
    GotoBufferId(copy_buf_id)
    EmptyBuffer()
    InsertText(s)
    MarkAll()
    Copy()

    PopPosition()
    PopBlock()
end

menu WhereMenu()
    history
    "S&tart Current File",          Start(QuotePath(pbname_ex()))
    "&Sort Menu",                   mSortListFiles()
    "Set as &Current Directory",    ChDir(SplitPath(pbname_ex(), _DRIVE_|_PATH_))
    "&Dos Shell",                   mShell()
    "&File Manager",                 ExecMacro("f " + QuotePath(pbname_ex()))
    "Copy Filename to &Windows Clipboard", CopyToWinClip(pbname_ex())
    "Copy Filename to &Editor Clipboard",  copy_string_clipboard(pbname_ex())
end

/*
These keys are enabled in mListStartup().  They are used in conjunction with
the list buffer pick_buff.

*/
keydef ListKeys
    <Alt L>      EndProcess(-1)
    <SpaceBar>   mToggleTags(2, TRUE)
    <Grey+>      mToggleTags(1, TRUE)
    <Ins>        mToggleTags(1, TRUE)
    <GreyIns>    mToggleTags(1, TRUE)
    <Grey->      mToggleTags(0, TRUE)
    <Alt X>      EndProcess(-2)
    <Alt F>      mSaveListFile()
    <Alt S>      mSortListFiles()
    <Ctrl S>     mSortListFiles()
    <Ctrl Grey*> mReTag()
    <Del>        mDeleteFiles()
    <Ctrl F>     mFindTag(<Ctrl F>)
    <Ctrl T>     mFindTag(<Ctrl T>)
    <Ctrl N>     mFindTag(<Ctrl N>)
    <Ctrl P>     mFindTag(<Ctrl P>)
    <Alt V>      mViewZip()
    <Alt Z>      mZipFiles()
    <Alt H>      mPopupHelp(2)
    <F1>         mPopupHelp(2)
    <Enter>      mAcceptFiles()
    <GreyEnter>  mAcceptFiles()
    <Grey*>      mTagAllFiles(TAG_CHAR)
    <Grey/>      mTagAllFiles(Asc(" "))
    <LeftBtn>    mLeftBtn()
    <RightBtn>   mRightBtn()
    <ctrl v>     mViewZip()
    <f10>        WhereMenu()

    // sorting
    <Ctrl F3> SortList(SORT_NAME_EXT)
    <Ctrl F4> SortList(SORT_EXT_NAME)
    <Ctrl F5> mSortByDateDescending()
    <Ctrl F6> mSortBySizeDescending()

    <CursorUp> Up()
    <CursorDown> Down()

    <Ctrl Home> BegWindow()
    <Ctrl End>  EndWindow()

    <Home> if Query(LastKey) == <Home> BegFile() endif BegLine()
    <End>  if Query(LastKey) == <End>  EndFile() BegLine() else EndLine() endif

    <CursorRight> RollRight() mPutListMessages(FALSE)
    <CursorLeft> RollLeft() mPutListMessages(FALSE)

end ListKeys

/*
This routine allows one to quickly view a file without permanently
loading that file.

Called by:  <Alt V> in ListKeys

*/
proc mViewZip()
    integer id

    id = GetBufferId()
    Disable(ListKeys)
//    ExecMacro("zipview " + QuotePath(PBName()))  'f' works better than zipview
    ExecMacro("f -z " + pbname_ex())

    GotoBufferId(id)
    Enable(ListKeys)
end

/*
This enables the ListKeys, and places the header/footer messages on the list
buffer display border.

Hooked to _LIST_STARTUP_ in mDisplayListBuffer()

*/
proc mListStartup()
    Enable(ListKeys)
    UnHook(mListStartup)
    UpdateDisplay()
    mPutListMessages(TRUE)
    MarkColumn(1, 14, NumLines(), 15 + 255)
end mListStartup

/*
Find and highlight the filename in the list buffer in the edit pickfile
buffer, and then try to center it vertically.

Hooked to _PICKFILE_STARTUP_ in mDisplayListBuffer()

*/
proc mPickListUp()
    lFind(ListPath, "gi")
    ScrollToCenter()
end mPickListUp

/*
Display the list buffer, and wait for a return key.  Possible return keys
include:

    <Escape>        Exit the buffer list, keep the buffer list
    <Alt X>         Exit the buffer list, and delete the buffer list
    <Alt L>         Request a file picklist display
    <Enter>         Edit the selected filename

Called by:  mScanAllDirs(), mWhereIs()

*/
integer proc mDisplayListBuffer()
    integer OldX1 = Query(X1)
    integer OldY1 = Query(Y1)
    integer OldID = GetBufferID()
    integer ReturnCode = FALSE
    integer ListReturnCode
    string fn[MAXPATH]

    GotoBufferID(pick_buff)
    if not NumLines()
        GotoBufferID(OldID)
        return(FALSE)
    endif
    Accepted = FALSE
ListSpin:
    Set(X1, 1)
    Set(Y1, (Query(ScreenRows) - ListVertical
        + iif(Query(StatusLineAtTop), 1, -1)) / 2)
    Hook(_LIST_STARTUP_, mListStartup)
    PushBlock()
    // This is the one that shows the files in the final list
    ListReturnCode = lList(where_title2, longest_line, ListVertical, _ENABLE_SEARCH_ | _ENABLE_HSCROLL_ | _BLOCK_SEARCH_)
    PopBlock()
    UnHook(mListStartup)
    fn = PBName()
    case ListReturnCode
        when 0                                  // escape
            GotobufferID(OldID)                 // back to original buffer
            ReturnCode = FALSE
        when -1                                // Alt L
            ReturnCode = FALSE

            Hook(_PICKFILE_STARTUP_, mPickListUp)
            ListPath = fn
            fn = PickFile(SplitPath(fn, _DRIVE_|_PATH_))
            UnHook(mPickListUp)
            if Length(fn)
                if mSmartEditFile(fn)
                    ChDir(SplitPath(fn, _DRIVE_ | _PATH_))
                    ReturnCode = TRUE
                endif
            endif
            if not ReturnCode
                goto ListSpin
            endif
        when -2                                 // Alt X
            if pick_buff
                EmptyBuffer(pick_buff)        // empty the file list buffer
            endif
            GotobufferID(OldID)                 // back to original buffer
            ReturnCode = FALSE
        otherwise
            Down()
            if mSmartEditFile(fn)
                ReturnCode = TRUE
            else
                GotobufferID(OldID)             // back to original buffer
                ReturnCode = FALSE
            endif
        endcase
    Set(X1, OldX1)
    Set(Y1, OldY1)
    return(ReturnCode)
end mDisplayListBuffer

/*
Interpret and massage any wildcard requirements, and any specified drive
requirements.

Called by:  mScanAllDirs()

*/
integer proc mHandleWildCards(string s1)
    integer i

    SearchName1 = Lower(s1)
    SearchName2 = ""
    SearchName3 = ""

    // Replace any "?" with an "*" in SearchName1
    for i = 1 to Length(SearchName1)
        if SearchName1[i] == "?"
            SearchName1[i] = "*"
        endif
    endfor

    if Length(SearchName1) > 1
        if SearchName1[Length(SearchName1) - 1 : 2] == ".."
            SearchName1 = SearchName1 + "\*.*"
        elseif SearchName1[Length(SearchName1) - 1 : 2] == "\."
            SearchName1 = SearchName1 + "\*.*"
        endif
    endif
    if SearchName1 == "."
        SearchName1 = SearchName1 + "\*.*"
    endif
    if SearchName1[Length(SearchName1)] == "\"
        SearchName1 = SearchName1 + "*.*"
    endif
    if Pos("\", SearchName1)
        PathStart = SplitPath(SearchName1, _PATH_)
        PathStart = ExpandPath(PathStart)
        PathStart = SplitPath(PathStart, _PATH_)
        SearchName1 = Lower(SplitPath(SearchName1, _NAME_ | _EXT_))
    else
        PathStart = ""
    endif
    if not Length(SearchName1)
        return(FALSE)
    endif
    if SearchName1 == ".*"
        SearchName1 = ""
    elseif Pos("*.*", SearchName1)
        SearchName1 = SearchName1[1 : Pos("*.*", SearchName1) - 1]
    elseif Pos(".*", SearchName1)
        // terminate with a null
        SearchName3 = SearchName1[1 : Pos(".*", SearchName1) - 1]
        // terminate with a dot
        SearchName1 = SearchName1[1 : Pos(".*", SearchName1)]
    elseif Pos("*.", SearchName1)
        if SearchName1[Length(SearchName1)] == "."
            SearchName1 = SearchName1[1 : Pos("*.", SearchName1) - 1] + "+"
        else
            SearchName2 = SearchName1[Pos("*.", SearchName1) + 1
                : Length(SearchName1)]
            if Pos("*", SearchName2)
                SearchName2 = SearchName2[1 : Pos("*", SearchName2) - 1]
            else
                if Length(SearchName2) and SearchName2 <> "."
                    SearchName2 = SearchName2 + Chr(0)  // GDB: Force Strict (ie, non-wildcard) extension matching
                endif
            endif
            SearchName1 = SearchName1[1 : Pos("*.", SearchName1) - 1]
        endif
    elseif SearchName1[Length(SearchName1)] == "."
        SearchName1 = SearchName1[1 : Pos(".", SearchName1) - 1]
    elseif SearchName1 == "*"
        SearchName1 = ""
    elseif SearchName1[Length(SearchName1)] == "*"
        SearchName1 = SearchName1[1 : Length(SearchName1) - 1]
    else
        if not Pos("*", SearchName1)
            if not Pos(".", SearchName1)        // no dot?
                SearchName3 = SearchName1 + "." // then add a dot to SearchName3
            endif
            SearchName1 = SearchName1
        endif
    endif
    return(TRUE)
end mHandleWildCards

/*
Remove duplicate drive letters, remove drive colons, and ensure that
only alphabetics are allowed.  Return only lower case drive letters.

Called by:  SplitDriveLetters(), mEditDriveLtrs(), WhenLoaded()

*/
string proc FilterDriveLtrs(string s)
    integer i
    string s0[30] = s
    string s1[30] = ""

    s0 = Lower(s0)
    for i = 1 to Length(s0)
        if not Pos(s0[i], s0[i + 1 : Length(s0)])
            case s0[i]
                when "a" .. "z"
                    s1 = s1 + s0[i]
            endcase
        endif
    endfor
    return(s1)
end FilterDriveLtrs

/*
If drive letters exist, split them from the string, and store them in
global string DriveLetters.  If they do not exist, then go with the
defaults found in DriveLtrs[].  Modify string s1 so that it no longer
contains drive letters, if they exist.  In any case, the drive letter
colon is discarded.

Called by:  mScanAllDirs()

*/
integer proc SplitDriveLetters(var string s1)
    integer index

    DriveLetters = ""
    // Split the drive letters from the searchname
    index = Pos(":", s1)            // is there a drive letter(s)?
    if index
        DriveLetters = s1[1 : index - 1]
        s1 = s1[index + 1 : Length(s1)]
    endif
    // Found a second colon? -- illegal!
    if Pos(":", s1)
        return(-1)
    endif
    // Found a space in the drive letters? -- illegal!
    if Pos(" ", DriveLetters)
        return(-2)
    endif
    // Make sure no duplicates, and all generally ok
    DriveLetters = FilterDriveLtrs(DriveLetters)
    return(1)
end SplitDriveLetters

/*
When the list buffer is displayed, start scanning all drives and all
subdirectories.  As matches are found, add them to the list buffer.

Called by:  Hooked to _LIST_STARTUP_ in mScanAllDirs()

*/
proc mHookViewFiles()
    string s[MAXPATH]
    string s1[MAXPATH]
    integer ID, MyKey, TokenCount, TokenIndex, unc_name

    UnHook(mHookViewFiles)
    // JHB: Center List Title
    VGotoXY((Query(ScreenCols)-Length(where_title))/2, Query(WindowY1) - 1)
    PutHelpLine(where_title)
    GotoBufferID(dir_buff)                       // go to the scratch buffer
    EmptyBuffer()                           // initialize the scratch buffer
    Set(Attr, Query(MenuTextAttr))
    ClrScr()                                // clear the list buffer display
    TokenCount = NumFileTokens(GlobalSearchString)
    TokenIndex = 1
    DriveLtr = 1                            // new drive letters -- start fresh

    longest_line = 0
    loop                                    // search all drive letters
        s1 = GetFileToken(GlobalSearchString, TokenIndex)
        DriveLetter = DriveLetters[DriveLtr]

        /*
        Handle the wildcard situation by creating up to three global strings
        in order to handle them.

        */
        mHandleWildCards(s1)
        unc_name = PathStart[1:2] == "\\"
        FileMessage(iif(unc_name, "\", DriveLetter), pick_buff)
        if unc_name
            s = ExpandPath(PathStart)
            if FileExists(s)
                if TokenIndex == 1
                    DirCount = DirCount + 1     // count the root directory
                endif
                mpath = s
                mMatchAllFileNames(TokenIndex)  // search the root directory
            endif
        elseif Length(PathStart) and PathStart <> '\'
            s = DriveLetters[DriveLtr] + ":"
            if s == ":"
                s = ""
            endif
            s = s + PathStart
            if FileExists(s) & _DIRECTORY_
                if TokenIndex == 1
                    DirCount = DirCount + 1     // count the directory
                endif
                mpath = ExpandPath(s + "*.*")
                mMatchAllFileNames(TokenIndex)  // search the directory
            endif
        else
            if DriveLetters == ""
                s = PathStart + "*.*"
            else
                s = DriveLetters[DriveLtr] + ":*.*"
            endif
            if FileExists(s)
                if TokenIndex == 1
                    DirCount = DirCount + 1     // count the root directory
                endif
                mpath = ExpandPath(s)
                mMatchAllFileNames(TokenIndex)  // search the root directory
            endif
        endif
        BegFile()
        repeat                              // read all subdirs on the drive
            s = GetText(1, CurrLineLen())
            if not Length(s)                // illegal state
                break
            endif
            mpath = s + "\*.*"              // the directory pathname to search
            mMatchAllFileNames(TokenIndex)  // search an entire subdir
            if KeyPressed()                 // key pressed?
                MyKey = GetKey()
                case MyKey
                    when <Escape>, <Alt X>, <RightBtn>  // terminate the search
                        EndProcess(0)
                        return()
                    when <Enter>, <CursorUp>, <CursorDown>
                        ID = GetBufferID()
                        GotoBufferID(pick_buff)
                        if NumLines()
                            PushKey(MyKey)   // reprocess the keystroke
                            EndProcess(0)
                            return()
                        endif
                        GotoBufferID(ID)
                endcase
            endif
            BegFile()
            KillLine()                  // kill each subdirectory as it's used
        until not NumLines()            // loop until all subdirectories killed
        if KeyPressed()
             case GetKey()
                when <Escape>, <Alt X>, <RightBtn>  // terminate the search
                    break
             endcase
        endif
        if TokenIndex < TokenCount
            TokenIndex = TokenIndex + 1
        else
            if unc_name or DriveLtr >= Length(DriveLetters) // searched all drive letters?
                break                       // all done, exit the loop
            endif
            DriveLtr = DriveLtr + 1     // advance to the next disk drive letter
            TokenIndex = 1
        endif
    endloop
    GotoBufferID(pick_buff)
    EndProcess(0)
end mHookViewFiles

/*
This routine adds "LinesToAdd" blank lines to the current buffer.

Called by:  mScanAllDirs()

*/
proc mAddBlankLines(integer LinesToAdd)
    while LinesToAdd
        AddLine()
        LinesToAdd = LinesToAdd - 1
    endwhile
end mAddBlankLines

/*
Create all of the needed buffers (dir_buff and pick_buff).  Interpret and massage any
wildcard requirements, and any specified drive requirements.

Called by:  mWhereIs()

*/
integer proc mScanAllDirs(var string GlobalSearchString)
    integer ID0 = GetBufferID()
    integer OldX1, OldY1
    integer OldCursor

    DriveLetters = DriveLtrs                // get the drive letter active list
    case SplitDriveLetters(GlobalSearchString)   // split out the drive letters
        when -1, -2
            warn("Only be ONE set of drive letters allowed - and at the BEGINNING!")
            return(FALSE)
    endcase
    if not Length(GlobalSearchString)
        return(FALSE)
    endif

    if not pick_buff
        pick_buff = CreateTempBuffer()
        DisplayMode(_DISPLAY_PICKFILE_)
    endif
    if not dir_buff
        dir_buff = CreateTempBuffer()
    endif
    if (not dir_buff) or (not pick_buff)
        warn("Out of Memory -- Cannot create buffer -- ABORTING!")
        GotoBufferID(ID0)
        PurgeMacro(CurrMacroFilename())
        return(TRUE)
    endif
    EmptyBuffer(pick_buff)
    GotoBufferID(dir_buff)
    EmptyBuffer(dir_buff)
    mAddBlankLines(ListVertical)
    DirCount = 0
    DriveLtr = 1
    HideMouse()
    OldCursor = Set(Cursor,Off)        // turn off the cursor
    // center the list buffer
    OldX1 = Set(X1, 1)
    OldY1 = Set(Y1, (Query(ScreenRows) - ListVertical
        + iif(Query(StatusLineAtTop), 1, -1)) / 2)
    Hook(_LIST_STARTUP_, mHookViewFiles)
    Message("")                         // wipe the message line
    // this is the incremental listing...
    lList(where_title2, Query(ScreenCols), ListVertical, _ENABLE_HSCROLL_)
    UnHook(mHookViewFiles)
    Set(X1, OldX1)
    Set(Y1, OldY1)
    Message("")                         // wipe the message line
    Set(Cursor, OldCursor)              // restore the cursor
    ShowMouse()
    GotoBufferID(pick_buff)           // go to the new buffer just created
    DriveLtr = 1
    if NumLines()
        if not mDisplayListBuffer()     // re-display the list buffer
            GotoBufferID(ID0)
            return(FALSE)
        endif
    else
        GotoBufferID(ID0)               // back to original buffer
        return(FALSE)
    endif
    return(TRUE)
end mScanAllDirs

/*
End the process if the mouse is clicked in the prompt window.

Called by:  keydef RestrictLtrs()

*/
proc mLeftMouseBtn()
    if Query(MouseX) >= Query(WindowX1)
            and Query(MouseX) < Query(WindowX1) + Query(WindowCols)
            and Query(MouseY) >= Query(WindowY1)
            and Query(MouseY) < Query(WindowY1) + Query(WindowRows)
        EndProcess(1)
    endif
end mLeftMouseBtn

/*
Restrict the prompt keydef to the following alphabetic keys:

Called by:  mRestrictLtrs()

*/
keydef RestrictLtrs
    <a> SelfInsert()
    <Shift a> SelfInsert()
    <b> SelfInsert()
    <Shift b> SelfInsert()
    <c> SelfInsert()
    <Shift c> SelfInsert()
    <d> SelfInsert()
    <Shift d> SelfInsert()
    <e> SelfInsert()
    <Shift e> SelfInsert()
    <f> SelfInsert()
    <Shift f> SelfInsert()
    <g> SelfInsert()
    <Shift g> SelfInsert()
    <h> SelfInsert()
    <Shift h> SelfInsert()
    <i> SelfInsert()
    <Shift i> SelfInsert()
    <j> SelfInsert()
    <Shift j> SelfInsert()
    <k> SelfInsert()
    <Shift k> SelfInsert()
    <l> SelfInsert()
    <Shift l> SelfInsert()
    <m> SelfInsert()
    <Shift m> SelfInsert()
    <n> SelfInsert()
    <Shift n> SelfInsert()
    <o> SelfInsert()
    <Shift o> SelfInsert()
    <p> SelfInsert()
    <Shift p> SelfInsert()
    <q> SelfInsert()
    <Shift q> SelfInsert()
    <r> SelfInsert()
    <Shift r> SelfInsert()
    <s> SelfInsert()
    <Shift s> SelfInsert()
    <t> SelfInsert()
    <Shift t> SelfInsert()
    <u> SelfInsert()
    <Shift u> SelfInsert()
    <v> SelfInsert()
    <Shift v> SelfInsert()
    <w> SelfInsert()
    <Shift w> SelfInsert()
    <x> SelfInsert()
    <Shift x> SelfInsert()
    <y> SelfInsert()
    <Shift y> SelfInsert()
    <z> SelfInsert()
    <Shift z> SelfInsert()
    <CursorRight> Right()
    <CursorLeft> Left()
    <CursorUp> Up()
    <CursorDown> Down()
    <Home> BegLine()
    <End> EndLine()
    <Escape> EndProcess(0)
    <Alt X> EndProcess(0)
    <BackSpace> BackSpace()
    <Del> DelChar()
    <Enter> EndProcess(1)
    <RightBtn> EndProcess(0)
    <LeftBtn> mLeftMouseBtn()
end RestrictLtrs

/*
Restrict the prompt keydef to the following alphabetic keys:

Called by:  Hooked to _PROMPT_STARTUP_ in routine mEditDriveLtrs().

*/
proc mRestrictLtrs()
    enable(RestrictLtrs, _EXCLUSIVE_)
end mRestrictLtrs

/*
Center a prompt box when it is presented.  Hook a restriction to
the allowed prompt letters,  a - z only.  If the drive letters are
changed via this prompt, then remove any impossible non alphabetic
characters, and any drive duplicates.  If no drive letters, then
add the default drive letters "c - z".

Save the new drive letters on disk.  Advise the user of the new drive
letters on the statusline.

Called by:  ConfigMenu()

*/
proc mEditDriveLtrs()
    string s0[27] = DriveLtrs
    integer AskReturn, OldY1, OldX1, Cursor

    OldY1 = Set(Y1, (Query(ScreenRows)) / 2)
    OldX1 = Set(X1, (Query(ScreenCols) - 27) / 2)
    Hook (_PROMPT_STARTUP_, mRestrictLtrs)
    AskReturn = Ask("Drive letters:", s0, DRIVE_HISTORY)
    UnHook(mRestrictLtrs)
    // remove any duplicate drive letters and make lower case
    s0 = FilterDriveLtrs(s0)
    if s0 <> DriveLtrs              // any change?
        if AskReturn
            if Length(s0)
                DriveLtrs = s0
            else
                // no letters -- go with the defaults
                DriveLtrs = "cdefghijklmnopqrstuvwxyz"
            endif
            // turn off the cursor
            cursor = Set(Cursor,Off)
            // save the directory database with new drive letters on disk
            if UpdateDiskDat()
                Message('Drive letters changed to "' + DriveLtrs + '"')
            endif
            // wait for a keypress and then blank the statusline
            while not KeyPressed()
            endwhile
            Message("")
            // restore the cursor
            Set(Cursor, Cursor)
        endif
    endif
    Set(Y1, OldY1)
    Set(X1, OldX1)
end mEditDriveLtrs

/*
   Procedure:   ToggleAutoLoad()

   Notes:       Toggles Autoloading ON/OFF

*/
proc ToggleAutoLoad()
    if isAutoLoaded(CurrMacroFileName())
        if not DelAutoLoadMacro(SplitPath(CurrMacroFileName(),_NAME_))
            MsgBox("Error!","WHERE could NOT be removed from the AutoLoad list")
        endif
    else
        if not AddAutoLoadMacro(SplitPath(CurrMacroFileName(),_NAME_))
            MsgBox("Error!","WHERE could NOT be added to the AutoLoad list")
        endif
    endif
end

/****************************************************************************
   Procedure:   OnOffStr()

   Notes:       Menu helper function.

****************************************************************************/
string proc OnOffStr(integer flag)
    return (iif(flag, "On","Off"))
end

/*
Provides a menu for configuring WHERE.  Drive letters and AutoLoad
functionality are handled here.

Called by:  mWhereIs()

*/
Menu ConfigMenu()
    Title = "WHERE Setup Menu"
    "&Edit Drive Letters",          mEditDriveLtrs()
        , DontClose, "Select disk drives to search (if they exist) when searching for filenames"
    "",,Divide
    "&Autoload WHERE" [OnOffStr(isAutoLoaded(CurrMacroFileName())):3]
            ,ToggleAutoLoad()
            ,_MF_DONT_CLOSE_
            ,"Automatically load WHERE when TSE is started?"
end ConfigMenu

/*
Adds the following keys to the prompt:

    <F1>        Quick help display
    <Alt H>     Quick help display
    <SpaceBar>  Display list buffer (if it exists)
    <Alt X>     Exit and delete list buffer (if it exists)
    <Alt C>     Display the configuration menu

Enabled in: mPromptStartup()

*/
keydef PromptKeys
    <F1> EndProcess(-1)         // Quick help display
    <Alt H> EndProcess(-1)      // Quick help display
    <Alt L> EndProcess(-2)      // Display list buffer (if it exists)
    <Alt X> EndProcess(-3)      // Exit and delete list buffer (if it exists)
    <Alt C> EndProcess(-4)      // Display the configuration menu
end PromptKeys

/*
Add a header/footer message to the "Search for file:" prompt and enable
the PromptKeys keydef.

Hooked to:  _PROMPT_STARTUP_.

*/
proc mPromptStartup()
    string TitleMessage1[26]  = " {<Alt L>} View File List "
    string FooterMessage0[15] = " {<F1>} {H}elp "
    string FooterMessage1[23] = " {<Alt C>} {C}onfigure "
    integer ID = GetBufferID()

    if pick_buff
        GotoBufferID(pick_buff)
    endif
    Enable(PromptKeys)
    // JHB: Center Title
    VGotoXY((Query(ScreenCols)-Length(where_title))/2, Query(WindowY1) - 2)
    PutHelpLine(where_title)
    if pick_buff and NumLines()
        VGotoXY(50, Query(WindowY1) - 2)
        PutHelpLine(TitleMessage1)
    endif
    VGotoXY(5, Query(WindowY1) + 1)
    PutHelpLine(FooterMessage0)
    VGotoXY(55, Query(WindowY1) + 1)
    PutHelpLine(FooterMessage1)
    GotoBufferID(ID)
end mPromptStartup

/*
This is the routine where it all starts.  If list buffer pick_buff exists, and the
"Accepted" flag is set, meaning that a file was loaded from a previous search,
the list buffer is immediately displayed again.

Otherwise, display a "Search for file:" prompt routine, and hook
mPromptStartup to _PROMPT_STARTUP_.  Possible key returns from the prompt
include:

    <Enter>             Accept prompt and begin the search via mScanAllDirs()
    <Alt H>             Quick help display
    <SpaceBar>          Display list buffer (if it exists)
    <Escape>            Exit the macro
    <Alt X>             Exit and delete list buffer (if it exists)
    <Alt C>             Display the configuration menu

Called by:  Main() and the <CtrlAlt W> key combination.

*/
proc mWhereIs2()
    string SaveString[1] = ""
    integer ID0 = GetBufferID()
    integer AskReturn, save_readflags
    integer OldX1 = Query(X1)
    integer OldY1 = Query(Y1)
    integer OldCursor

    ListVertical = (Query(ScreenRows) * 2) / 3
RepeatWhereStart:
    if pick_buff and Accepted
        Message("")
        if mDisplayListBuffer()
            return()
        endif
    endif
RepeatWhere:
    UpdateDisplayFlags(_STATUSLINE_REFRESH_)
    Set(Y1, (Query(ScreenRows) - 4) / 2)
    Hook(_PROMPT_STARTUP_, mPromptStartup)
    save_readflags = Set(ReadFlags, Query(ReadFlags) | FN_COMP)
    AskReturn = Ask("Search for file: [" + SqueezePath(CurrDir(), _USE_HOME_PATH_) + "]", GlobalSearchString, FILE_HISTORY)
    Set(ReadFlags, save_readflags)
    UnHook(mPromptStartup)
    case AskReturn
        when 0                  // escape pressed
            GotoBufferID(ID0)   // back to original buffer
        when -1                 // help request
            mPopupHelp(1)
            GlobalSearchString = SaveString
            goto RepeatWhere    // spin back and start again
        when -2                 // spacebar
            GlobalSearchString = SaveString
            Accepted = TRUE
            goto RepeatWhereStart   // spin back and start again
        when -3                 // escape pressed
            GotobufferID(ID0)   // back to original buffer
            if pick_buff
                EmptyBuffer(pick_buff)
            endif
        when -4                 // Alt C
            // center the configuration menu
            Set(Y1, (Query(ScreenRows) - 6) / 2)
            Set(X1, (Query(ScreenCols) - 38) / 2)
            ConfigMenu()        // access the configuration menu
            GlobalSearchString = SaveString     // restore the default prompt history
            goto RepeatWhere    // spin back and start again
        otherwise               // Enter
            if Pos("+", GlobalSearchString)
                OldCursor = Set(Cursor,Off)
                HideMouse()
                warn('The "+" character is invalid!')
                ShowMouse()
                Set(Cursor, OldCursor)
                GlobalSearchString = SaveString
                goto RepeatWhere            // spin back and start again
            else
                //GlobalSearchString = Upper(GlobalSearchString)
                TagCount = 0
                PermissionGranted = FALSE
                if not mScanAllDirs(GlobalSearchString)     // begin the search now
                    GlobalSearchString = SaveString
                    goto RepeatWhere
                else
                    UpdateDisplayFlags(_STATUS_LINE_REFRESH_)
                endif
            endif
    endcase
    Set(X1, OldX1)
    Set(Y1, OldY1)
end mWhereIs2

proc mWhereIs()
    integer save_eqh

    save_eqh = Set(EquateEnhancedKBD, On)
    mWhereIs2()
    Set(EquateEnhancedKBD, save_eqh)
end

/*
Retrieve the drive letters from the TSE.INI file

Called by:  TSE when it loads

*/
proc WhenLoaded()
    file_history = GetFreeHistory("Where:file")
    drive_history= GetFreeHistory("Where:drive")

    // Read the drive letters from the TSE.INI file
    DriveLtrs = GetProfileStr(SplitPath(CurrMacroFileName(),_NAME_),"Drives",DriveLtrs)

    DriveLtrs = FilterDriveLtrs(DriveLtrs)
end WhenLoaded

/*
Call the main mWhereIs() macro, and get things started.  This is an
alternative to the <CtrlAlt W> hotkey.

Called by:  TSE when WHERE is executed

*/
proc Main()
    if Length(Query(MacroCmdLine))
        PushKey(<Enter>)
        PushKeyStr(Query(MacroCmdLine))
    endif
    mWhereIs()
end Main

/*
Purge any list buffer if one exists when WHERE is purged.

Called by:  TSE when it purges the WHERE macro

*/
proc WhenPurged()
    AbandonFile(pick_buff)
    AbandonFile(dir_buff)
end WhenPurged

/*
The hotkey that gets things going.  This is an alternative to executing the
macro.

*/
<CtrlAlt W> mWhereIs()

