
/*** TIMELOG -- SUPPORT MACRO FOR PROJECTS ******************************
 *                                                                      *
 * SOFTWARE:    PROJECT                                                 *
 * VERSION:     1.00                                                    *
 * DATE:        December 31, 1993                                       *
 * REV. DATE:   July 15th, 1994                                         *
 * AUTHOR:      Ian Campbell                                            *
 * TYPE:        External Macro                                          *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 * This file contains all of the functions needed to save all of the    *
 * times associated with a project file into a projectname.log file     *
 * in the ".PRJ" subdirectory.  It will be automatically started by     *
 * PROJECTS when it starts.                                             *
 *                                                                      *
 * If this macro is executed, a log will automatically display.         *
 *                                                                      *
 ************************************************************************/

//Global integers
integer ProjectStartTime = -1
integer CurrentID = 0

//Global strings
string CurrentFn[80]

/*** mGetDurationTimeInSeconds ******************************************
 *                                                                      *
 * Returns the time in seconds between two time periods.                *
 *                                                                      *
 * Called by:   mUpdateLogEntry(), mMergeAndSaveTimeLogs()              *
 *                                                                      *
 * Enter With:  the current time and the older time in seconds          *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mGetDurationTimeInSeconds(integer CurrentTimeInSeconds,
    integer OldTimeInSeconds)
        constant FullDaySeconds = 24 * 60 * 60

        if CurrentTimeInSeconds >= OldTimeInSeconds  // before midnight
            return(CurrentTimeInSeconds - OldTimeInSeconds)
        endif
        return(FullDaySeconds - OldTimeInSeconds + CurrentTimeInSeconds)
end mGetDurationTimeInSeconds

/*** mUpdateLogEntry ****************************************************
 *                                                                      *
 * This routine updates a system buffer created at load time each time  *
 * the current file is changed.  Entry time in seconds, and cumulative  *
 * time in seconds are recorded each time a file is entered/left.       *
 * Entries are recorded for each filename.                              *
 *                                                                      *
 * Called by:   mCreateLogBuffer(), mMergeAndSaveTimeLogs(),            *
 *              mTrackFileChanges()                                     *
 *                                                                      *
 * Enter With:  A flag indicating whether entering or leaving the       *
 *              buffer, and a string with the name of the buffer.       *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mUpdateLogEntry(integer EnteringNewBuffer, string fn)
    integer OldTimeInSeconds, CurrentTimeInSeconds, DurationTimeInSeconds = 0
    integer SaveBufferID = GetBufferID(), LogBufferID
    integer hours, minutes, seconds, hundreds

    LogBufferID = GetBufferID("+++PRJLOG+++")
    if LogBufferID
        GotoBufferID(LogBufferID)
        GetTime(hours, minutes, seconds, hundreds)
        CurrentTimeInSeconds = (hours * 60 * 60) + (minutes * 60) + seconds
        if ProjectStartTime == -1
            ProjectStartTime = CurrentTimeInSeconds // initialize things
        endif
        EndFile()
        if (lFind(fn, "b$"))
            OldTimeInSeconds = Val(GetText(1, 6))
            DurationTimeInSeconds = Val(GetText(8, 6))
            if not EnteringNewBuffer
                DurationTimeInSeconds = DurationTimeInSeconds
                    + mGetDurationTimeInSeconds(CurrentTimeInSeconds,
                        OldTimeInSeconds)
            endif
            BegLine()
            KillLine()       // retrieved both values, delete the line
        endif
        AddLine(Format(Str(CurrentTimeInSeconds):6:"0", " ",
            Str(DurationTimeInSeconds):6:"0", " ", fn), LogBufferID)
    endif
    GotoBufferID(SaveBufferID)
end mUpdateLogEntry

/*** mCreateLogBuffer ***************************************************
 *                                                                      *
 * Entered each time a file is first edited.  A check is made to ensure *
 * that system buffer "+++PRJLOG+++ exists If it does not, it is        *
 * created, some values are inited, and a call is made to               *
 * mUpdateLogEntry() to record the start time.                          *
 *                                                                      *
 * Called by:  an "_ON_FIRST_EDIT_" hook set up a WhenLoaded() time.    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mCreateLogBuffer()
    integer SaveBufferID = GetBufferID()

    // if the log buffer does not already exist, this will create it
    if CreateBuffer("+++PRJLOG+++", _SYSTEM_)
        GotoBufferID(SaveBufferID)
        CurrentID = GetBufferID()
        CurrentFn = CurrFilename()
        mUpdateLogEntry(TRUE, CurrFilename())
    endif
end mCreateLogBuffer

/*** GetFormattedText ***************************************************
 *                                                                      *
 * Returns the current time in a formatted string for hours and         *
 * minutes.                                                             *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  the time in seconds.                                    *
 *                                                                      *
 * Returns:     the current time in hrs and minutes.                    *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc GetFormattedText(integer TimeInSeconds)
    string s[80]
    integer hours = TimeInSeconds/(60*60),
            minutes = (TimeInSeconds mod (60 * 60))/60

    if (TimeInSeconds mod (60 * 60)) mod 60 >= 30
        minutes = minutes + 1       // round up with 30 seconds
        hours = iif(minutes == 60, hours + 1, hours)
    endif
    if hours
        s = Format(hours:5:" ", " hr, ", minutes:2:" ", " min")
    else
        s = Format("          ", minutes:2:" ", " min")
    endif
    return(s)
end GetFormattedText

/*** mGetTimeString *****************************************************
 *                                                                      *
 * Returns the time in hours:minutes format.                            *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  the time in seconds.                                    *
 *                                                                      *
 * Returns:     the time in hours:minutes format (eg) "10:29am".        *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc mGetTimeString(integer TimeInSeconds)
    string indicator[4] = "am"
    integer hours = TimeInSeconds/(60*60),
            minutes = (TimeInSeconds mod (60 * 60))/60,
            seconds = (TimeInSeconds mod (60 * 60)) mod 60
    case hours
        when 0
            hours = 12
        when 12
            indicator = "pm"
        when 13..23
            hours = hours - 12
            indicator = "pm"
    endcase
    return(Format(Str(hours):2:" ", ":", Str(minutes):2:"0", ":",
        Str(seconds):2:"0", indicator))
end mGetTimeString

/*** mDirExists *********************************************************
 *                                                                      *
 * Check for the existence of a directory and return non zero if it     *
 * does exist.                                                          *
 *                                                                      *
 * Called by:   mGetLogDir()                                            *
 *                                                                      *
 * Enter With:  the name of the directory                               *
 *                                                                      *
 * Returns:     non zero if the directory exists                        *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mDirExists(string s)
    return(FileExists(s) & _DIRECTORY_)
end mDirExists

/*** mGetLogDir ********************************************************
 *                                                                      *
 * Check for the existence of a PRJ directory just after LoadDir.  If   *
 * it does not exist, then try for a PRJ directory based on the global  *
 * string stored by PROJECTS.                                           *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     environment LOADDIR entry, otherwise TSE startup        *
 *              directory.                                              *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc mGetLogDir()
    string pn[80] = LoadDir()

    if not mDirExists(pn + "prj\")
        pn = SplitPath(GetGlobalStr("ProjectName"), _DRIVE_ | _PATH_)
    endif

    if Length(pn)
        pn = pn + "PRJ\"
    endif
    return(Upper(pn))
end mGetLogDir

/*** mGetDayOfWeek ******************************************************
 *                                                                      *
 * Converts integer dow to an actual day string.                        *
 *                                                                      *
 * Called by:   mGetDayDateStr()                                        *
 *                                                                      *
 * Enter With:  dow                                                     *
 *                                                                      *
 * Returns:     string with the actual day in it.                       *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc mGetDayOfWeek(integer dow)
    string DayOfWeek[3] = ""

    case dow
        when 1
            DayOfWeek = "Sun"
        when 2
            DayOfWeek = "Mon"
        when 3
            DayOfWeek = "Tue"
        when 4
            DayOfWeek = "Wed"
        when 5
            DayOfWeek = "Thu"
        when 6
            DayOfWeek = "Fri"
        when 7
            DayOfWeek = "Sat"
    endcase
    return(DayOfWeek)
end mGetDayOfWeek

/*** mGetMonthOfYear ****************************************************
 *                                                                      *
 * Converts integer mon to an actual month string.                      *
 *                                                                      *
 * Called by:   mGetDayDateStr()                                        *
 *                                                                      *
 * Enter With:  dow                                                     *
 *                                                                      *
 * Returns:     string with the actual month in it.                     *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc mGetMonthOfYear(integer mon)
    string MonthOfYear[3] = ""

    case mon
        when 1
            MonthOfYear = "Jan"
        when 2
            MonthOfYear = "Feb"
        when 3
            MonthOfYear = "Mar"
        when 4
            MonthOfYear = "Apr"
        when 5
            MonthOfYear = "May"
        when 6
            MonthOfYear = "Jun"
        when 7
            MonthOfYear = "Jul"
        when 8
            MonthOfYear = "Aug"
        when 9
            MonthOfYear = "Sep"
        when 10
            MonthOfYear = "Oct"
        when 11
            MonthOfYear = "Nov"
        when 12
            MonthOfYear = "Dec"
    endcase
    return(MonthOfYear)
end mGetMonthOfYear

/*** mGetDayDateStr *****************************************************
 *                                                                      *
 * Returns the day and the date as in the following example:            *
 *                                                                      *
 *      Fri, Dec-31-93                                                  *
 *                                                                      *
 * Called by:   mUpdateFileCumulativeTime(), mUpdateProjectHistory()    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     the day and date string.                                *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string proc mGetDayDateStr()
    integer mon, day, year, dow

    GetDate(mon, day, year, dow)
    return(Format(mGetDayOfWeek(dow), ", ", mGetMonthOfYear(mon), "-",
        day:2:"0", "-", SubStr(Str(year), 3, 2)))
end mGetDayDateStr

/*** mUpdateFileCumulativeTime ******************************************
 *                                                                      *
 * Updates the cumulative time for all the filenames in the log file.   *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  Buffer ID for "+++PRJLOG+++", and buffer ID for the     *
 *              temporary buffer holding the projectname.log file.      *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mUpdateFileCumulativeTime(integer LogBufferID, integer TempBufferID)
    string fn[80]
    integer DurationTimeInSeconds, OldFileTime

    GotoBufferID(LogBufferID)
    BegFile()
    loop
        DurationTimeInSeconds = Val(GetText(8, 6))
        if DurationTimeInSeconds > 30
            fn = GetText(15, 90)
            if GotoBufferID(TempBufferID)
                BegFile()
                if lFind("File Cumulative Time", "")
                    Down()
                    PushBlock()
                    UnMarkBlock()
                    BegLine()
                    MarkColumn()
                    if lFind("Project Cumulative Time", "")
                        Up()
                        GotoColumn(100)
                        MarkColumn()
                        // finally, search for the matching filename
                        OldFileTime = 0
                        if lFind(fn, "$lb")
                            OldFileTime = (Val(GetText(1, 5)) * 60 * 60)
                                + (Val(GetText(11, 2)) * 60)
                            KillLine()
                            Up()
                        else
                            Up()
                        endif
                        UnMarkBlock()
                        AddLine(GetFormattedText(OldFileTime
                            + DurationTimeInSeconds)
                            + ", "
                            + mGetDayDateStr()
                            + ", "
                            + fn)
                    endif
                    PopBlock()
                endif
            endif
        endif
        GotoBufferID(LogBufferID)
        if not Down()
            break
        endif
    endloop
    GotoBufferID(TempBufferID)
end mUpdateFileCumulativeTime

/*** mUpdateProjectHistory **********************************************
 *                                                                      *
 * Add an entry to the project history section.                         *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  the project start time and the current time in seconds. *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mUpdateProjectHistory(integer ProjectStartTime,
    integer CurrentTimeInSeconds)

    BegFile()
    if lFind("File Cumulative Time", "")
        BegLine()
        Up(2)
        if Abs(CurrentTimeInSeconds - ProjectStartTime) > 30
            AddLine(GetFormattedText(mGetDurationTimeInSeconds(CurrentTimeInSeconds,
                ProjectStartTime))
                + ", "
                + mGetDayDateStr()
                + ", "
                + mGetTimeString(ProjectStartTime)
                + " - "
                + mGetTimeString(CurrentTimeInSeconds))
        endif
    endif
end mUpdateProjectHistory

/*** mUpdateProjectCumulativeTime ***************************************
 *                                                                      *
 * Update the project cumulative time section.                          *
 *                                                                      *
 * Called by:   mMergeAndSaveTimeLogs()                                 *
 *                                                                      *
 * Enter With:  the project start time and the current time in seconds. *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mUpdateProjectCumulativeTime(integer ProjectStartTime,
    integer CurrentTimeInSeconds)
    integer OldFileTime

    if lFind("Project Cumulative Time", "")
        BegLine()
        Down()
        OldFileTime = (Val(GetText(1, 5)) * 60 * 60)
            + (Val(GetText(11, 2)) * 60)
        KillLine()
        Up()
        if Abs(CurrentTimeInSeconds - ProjectStartTime)
            + OldFileTime > 30
                AddLine(GetFormattedText(OldFileTime +
                    mGetDurationTimeInSeconds(CurrentTimeInSeconds,
                        ProjectStartTime)))
        else
            AddLine("")
        endif
    endif
end mUpdateProjectCumulativeTime

/*** mMergeAndSaveTimeLogs **********************************************
 *                                                                      *
 * This routine is called when a project session is ending.  Its        *
 * purpose is to merge +++PRJLOG+++ into the projectname.log stored in  *
 * the "PRJ" subdirectory.                                              *
 *                                                                      *
 * Called by:   hooked to _ON_ABANDON_EDITOR_, and called by            *
 *              mFileQuit (which is hooked to _ON_FILE_QUIT_)           *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mMergeAndSaveTimeLogs()
    string ProjectName[80] = GetGlobalStr("ProjectName")
    string LogName[80] = mGetLogDir()
        + SplitPath(ProjectName, _NAME_) + ".LOG"
    integer TempBufferID,
            OriginalBufferID = GetBufferID(),
            LogBufferID,
            hours,
            minutes,
            seconds,
            hundreds,
            CurrentTimeInSeconds

    // buffer flushed, nothing to do, so exit
    if GetBufferID("+++PRJLOG+++") == 0
        return()
    endif
    // no project, just exit
    if not Length(ProjectName)
        return()
    endif

    GetTime(hours, minutes, seconds, hundreds)
    CurrentTimeInSeconds = (hours * 60 * 60)
        + (minutes * 60) + seconds
    Set(MsgLevel, _NONE_)
    mUpdateLogEntry(FALSE, CurrFilename())
    TempBufferID = CreateTempBuffer()
    if TempBufferID
        PushBlock()
        if not InsertFile(LogName)          // retrieve any existing log
            AddLine("-------------- Project History -------------------")
            AddLine("")
            AddLine("-------------- File Cumulative Time --------------")
            AddLine("")
            AddLine("-------------- Project Cumulative Time -----------")
            AddLine("")
            AddLine("--------------------------------------------------")
        endif
        PopBlock()
        LogBufferID = GetBufferID("+++PRJLOG+++")
        if LogBufferID
            mUpdateFileCumulativeTime(LogBufferID, TempBufferID)
            mUpdateProjectHistory(ProjectStartTime, CurrentTimeInSeconds)
            mUpdateProjectCumulativeTime(ProjectStartTime, CurrentTimeInSeconds)
            // cleanup -- erase any possible !!LAST!!.LOG filename
            EraseDiskFile(SplitPath(LogName, _Drive_ | _Path_)
                + "!!LAST!!" + ".log")
            SaveAs(LogName, _OVERWRITE_)
        endif
    endif
    Set(MsgLevel, _ALL_MESSAGES_)
    GotoBufferID(OriginalBufferID)
    AbandonFile(GetBufferID("+++PRJLOG+++"))
    AbandonFile(TempBufferID)
end mMergeAndSaveTimeLogs

/*** mTrackFileChanges **************************************************
 *                                                                      *
 * Updates log entries whenever a file changes.  Tracks the CurrentID   *
 * and the CurrentFn.                                                   *
 *                                                                      *
 * Called by:   hooked to _ON_CHANGING_FILES_ at WhenLoaded() time.     *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mTrackFileChanges()
    if GetBufferID() <> CurrentID
        CurrentID = GetBufferID()
        mUpdateLogEntry(FALSE, CurrentFn)
        mUpdateLogEntry(TRUE, CurrFilename())
        CurrentFn = CurrFilename()
    endif
end mTrackFileChanges

/*** mFileQuit **********************************************************
 *                                                                      *
 * When no more files exist within TSE, the project times are written   *
 * to the disk project file (used when changing projects without        *
 * exiting the editor).                                                 *
 *                                                                      *
 * Called by:   hooked to _ON_FILE_QUIT_ at WhenLoaded() time.          *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mFileQuit()
    if NumFiles() == 1          // quitting last file?
        mMergeAndSaveTimeLogs()
    endif
end mFileQuit

/*** WhenLoaded *********************************************************
 *                                                                      *
 * This macro is executed when the macro file is loaded.  It sets up    *
 * a few hooks, in preparation for tracking times.                      *
 *                                                                      *
 * Called by:   TSE at load time.                                       *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc WhenLoaded()
    Hook(_ON_FIRST_EDIT_, mCreateLogBuffer)
    Hook(_ON_ABANDON_EDITOR_, mMergeAndSaveTimeLogs)
    Hook(_ON_CHANGING_FILES_, mTrackFileChanges)
    Hook(_ON_FILE_QUIT_, mFileQuit)
end WhenLoaded

proc terminate()
     BreakHookChain()
end terminate

/*** FlashMessage *******************************************************
 *                                                                      *
 * Display a flashing message on the top line.  The message will flash  *
 * for the FlashDelay time period, and then stop flashing.              *
 *                                                                      *
 * Called by:                                                           *
 *                                                                      *
 * Enter With:  Message to display, whether or not to update display    *
 *              on exit, the time to flash the message in seconds.      *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc FlashMessage(string Msg, integer Update, integer FlashDelay)
    integer OldMsgAttr, BlinkDelay

    OldMsgAttr = Set(MsgAttr, Query(MsgAttr) | 0x80)
    Message(Msg)
    UpdateDisplay()
    Set(MsgAttr, OldMsgAttr)
    BlinkDelay = GetClockTicks()
    repeat
        ExecHook(_IDLE_)
    until ((GetClockTicks() > BlinkDelay + (FlashDelay * 18))
            or (GetClockTicks() < BlinkDelay)
            or (KeyPressed()))
    Message(Msg)
    if Update
        UpdateDisplay()
    endif
end FlashMessage

/*** mViewTimelog() *****************************************************
 *                                                                      *
 * Views the ".LOG" file for this project if one exists.                *
 *                                                                      *
 * Called by:   bound to a key in ProjectKeys.                          *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mViewTimelog()
    string LogName[80]
    string ProjectName[80] = GetGlobalStr("ProjectName")

    if not Length(ProjectName)
        or Upper(SplitPath(ProjectName, _NAME_)) == "!!LAST!!"
            FlashMessage("This project does NOT have a name!", FALSE, 2)
    else
        LogName = mGetLogDir() + SplitPath(ProjectName, _NAME_)
            + ".LOG"
        if FileExists(LogName)
            Hook(_ON_CHANGING_FILES_, terminate)    // disable event
            if EditFile(LogName)
                lFind("File Cumulative Time", "ig")
                Up(2)
                ScrollToRow(Query(ScreenRows))
                lList("Time Log", 66, Query(ScreenRows) - 4,
                    _ENABLE_SEARCH_ | _ENABLE_HSCROLL_)
                QuitFile()
            endif
            unhook(terminate)           // re-enable events
        else
            FlashMessage(Logname + " does NOT exist!", FALSE, 2)
        endif
    endif
end mViewTimelog

/*** WhenLoaded *********************************************************
 *                                                                      *
 * This macro is executed when the macro file is executed.  It          *
 * displays the timelog for viewing purposes.                           *
 *                                                                      *
 * Called by:   TSE when the macro is executed.                         *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc Main()
    Set(MsgLevel, _NONE_)
    mViewTimelog()
    Set(MsgLevel, _ALL_MESSAGES_)
end Main

