/*************************************************************************
  Template    Expands user-defined templates within the current file.

  Author:     SemWare

  Date:       Jul  8, 1993 (Steve Watkins) Original version
              Jun 21, 1997 (Chris Antos)
                                - temporarily disables RemoveTrailingWhite to
                                  work correctly with hard tabs.
              Nov 11, 1997 SEM
                    Allow short-names to end with non-wordset characters.

                    Make the current extension the default when prompted for
                    the extension.
              Feb  3, 1999 SEM
                    Allow short-names to be of the following form:

                    :W+(:w+){$|[ \t]}                    as originally implemented
                    :W+(:w+:W+){$|[ \t]}                 added Nov 11, 1997
                    {^|[ \t]}(:W+){$|[ \t]}              added Feb  3, 1999
                    {^|[ \t]|:w+}:W+(:W+:w+:W+){$|[ \t]} added Oct  6, 1999

                    Where:
                    :w represents wordset characters
                    :W represents non-wordset characters

              Oct  6, 1999 SEM
                    Equate .html to .htm, and .java to .jav.
                    This is necessary since Template only supports
                    1..3 char extensions.

                    To equate other extensions to their 3 char counterparts,
                    see routine ExtensionType().

              Oct 26, 2001 Peter Birch. Add \f.

                    usage example:

                    class \f\c
                    {
                    public:
                                        \f();
                        virtual      ~\f();
                    private:

                    protected:

                    };

              May 5, 2003 SEM: add user options to turn off \c
                    and \f actions.

              October 2003: Updated to support C#, by Greg Macdonald.

              June 26, 2007: SEM: Add histories for prompts.  Thanks to
                Knud van Eeden for the suggestion.

              Jan 10 2023: SEM: Change templates to allow up to 12
                character file extensions (including the .).
                a converion routine had to be written to convert
                old template files.
                Thanks to Knud van Eeden for the suggestion.
              Jan 12 2023: SEM: templates were being expanded even
                when not found for the current file extension.
                Fixed.
                Added template expansion on/off to the menu.
                Thanks to Knud van Eeden for the report and suggestion.

  Overview:

  This macro will expand a user-defined template into the current
  file. Templates may be extension specific or global.  Several
  templates are defined for SAL, C, and Pascal source files.  But
  the user may add/delete/expand their own templates.

  Keys:
              <Tab>         Expand word to left of cursor
              <Ctrl Tab>    Pull up Template Menu

  Usage notes:

  Simply press <Tab> after the word you want expanded.  The word
  will be looked up for the current extension and expanded into the
  current file.

  For example, while editing a TSE source file such as "test.s" type
  the word 'if' (without quotes) and press <Tab> to expand 'if' to

        if ()
        endif

  To define your own template, simply mark the text with a block,
  Press <Ctrl Tab> and Select "Add Marked Text as Template".  The
  user is then prompted for a short name and extension for this
  new template.

  NOTE: When a template is expanded the cursor is placed at the end
        of the block. If '\c' is specified within the template, the
        cursor is placed at the '\c' and the '\c' is removed.  If
        '\C' (upper-case) is specified and insert mode is OFF, the
        cursor is placed at the '\C' and insert mode is turned on
        until the cursor is moved off the line.

  The user may also pull up the Template Menu with <Ctrl Tab> and
  List the current templates.  The templates may be added, deleted,
  or expanded from the list.
*************************************************************************/

// user-definable settings

constant MESSAGE_IF_NOT_FOUND = FALSE
constant TEMPLATE_MENU      =   <Ctrl Tab>

// If HONOR_INSERT is TRUE and the insert is off, the entire
// word upon which the cursor sits will be expanded.
constant HONOR_INSERT       =   FALSE

//if TEMPLATE_EXPAND is not set to <Tab> see "key-definitions" below
constant TEMPLATE_EXPAND    =   <Tab>

constant HONOR_CURSOR_POSITION = TRUE
constant HONOR_FILENAME_EXPANSION = TRUE

// end user-definable settings

//  Note: Global Integers affected: Template->Editing
//                                  Template->Expanded

constant EXT_LEN        =   13      // extension length including the period
constant MAX_WORDLEN    =   16

constant TEMPLATE_pDELTA  = 1,
         TEMPLATE_lDELTA  = 4,
         TEMPLATE_pEXT    = TEMPLATE_pDELTA + TEMPLATE_lDELTA,  // 1 + 4 = 5
         TEMPLATE_lEXT    = EXT_LEN,                            // 13
         TEMPLATE_PWORD   = TEMPLATE_pEXT + TEMPLATE_lEXT,      // 5 + 13 = 18
         TEMPLATE_LWORD   = MAX_WORDLEN                         // 16

constant CTRL_pWORD     =   1,
         CTRL_lWORD     =   MAX_WORDLEN + 4,                    // 16 + 4 = 20
         CTRL_pEXT      =   CTRL_pWORD + CTRL_lWORD,            // 1 + 16 = 17
         CTRL_lEXT      =   EXT_LEN,                            // 13
         CTRL_pLINE     =   CTRL_lWORD + CTRL_lEXT + 1,         // 20 + 13 + 1 = 34
         CTRL_lLINE     =   8                                   //  9

constant LIST_WIDTH     = CTRL_lWORD + CTRL_lEXT

constant TAB = 9, SPACE = 32

string EyeCatcher[] = "SemWare Template file... Do not edit"
string NewEyeCatcher[] = "SemWare Template file13... Do not edit"

integer template_id, ctrl_id, list_id, keyword_history, ext_history
string  template_fn[128]

// return TRUE if the character is 0-9 or empty
integer proc isNumeric(string ch)
    if ch == ''
        return (TRUE)
    endif
    return (ASC(ch) >= ASC('0') and ASC(ch) <= ASC('9'))
end

/**************************************************************************
  return TRUE if the current character is space, tab, or beyond the text on
  the line.
 **************************************************************************/
integer proc isWhiteSpace()
    return (CurrChar() in TAB, SPACE, _AT_EOL_, _BEYOND_EOL_)
end

string proc ExtensionType()
    string ext[32] = CurrExt()

    case ext
        when ".s", ".cfg", ".ui", ".si"
            return ('.s')
        when ".java"
            return ('.jav')
        when ".html", ".shtml"
            return ('.htm')
        when ".pas", ".dpr", ".pp"
            return ('.pas')
        when ".rb", ".rbw"
            return ('.rb')
    endcase

    case ext[2]
        when 'c'
            // check for alternatives to the 'c' extension
            if (ext in '.cs', '.c', '.cc', '.cpp', '.java') or (isNumeric(ext[3]) and isNumeric(ext[4]))
                return ('.c')
            endif
        when 's'
            // check for alternatives to the 's' extension
            if isNumeric(ext[3]) and isNumeric(ext[4])
                return ('.s')
            endif
    endcase
    return (ext)
end

integer proc SaveTemplateFile()
    integer n, eof_type, level

    eof_type = Set(EOFType, _NONE_)
    level = Set(MsgLevel, _NONE_)
    n = SaveAs(template_fn, _OVERWRITE_)
    Set(MsgLevel, level)
    Set(EOFType, eof_type)
    return (n)
end

integer proc BlockLines()
    return (val(GetText(TEMPLATE_pDELTA,TEMPLATE_lDELTA)))
end

/*------------------------------------------------------------------------

 ------------------------------------------------------------------------*/
integer proc CtrlLine()
    return (val(GetText(CTRL_pLINE, CTRL_lLINE)))
end

/*
Template format:
-4- ------13-----
1234123456789012312
2   .c           do
1234567890123456789
1   5            18

Ctrl file format:
-------20---------- -----13----- ----8---
12345678901234567890123456789012312345678
sproc               .s                349
str                 .s                352
while               .s                354
1234567890123456789012345678901234567890
1                   21           34
*/
proc GenerateControlFile()
    integer id, n
    string ext[TEMPLATE_lEXT], wordst[TEMPLATE_lWORD]

    EmptyBuffer(ctrl_id)

    id = GotoBufferId(template_id)
    if NumLines() > 2
        GotoLine(2)
        loop
            n    = Val(GetText(TEMPLATE_pDELTA  ,TEMPLATE_lDELTA))
            ext  =     GetText(TEMPLATE_pEXT    ,TEMPLATE_lEXT  )
            wordst =   GetText(TEMPLATE_pWORD   ,TEMPLATE_lWORD )
            if n == 0
                break
            endif
            AddLine(format(wordst:-CTRL_lWORD, ext:-CTRL_lEXT, CurrLine():CTRL_lLINE), ctrl_id)
            GotoLine(CurrLine() + n + 1)
            if CurrLine() >= NumLines()
                break
            endif
        endloop
    endif
    GotoBufferId(id)
end

/*------------------------------------------------------------------------
  Convert an old template.dat file to the new format.
  old format:
123412341234567890123456 --ruler
2   .c  do
do {
} while (\c);

  new format:
123412345678901231234567890123456 --ruler
2   .c           do
do {
} while (\c);
 ------------------------------------------------------------------------*/
integer proc ConvertOldToNew()
    constant xTEMPLATE_lEXT    = 4,
             xTEMPLATE_PWORD   = TEMPLATE_pEXT + xTEMPLATE_lEXT
    integer n
    string ext[EXT_LEN], wordst[MAX_WORDLEN]

    BegFile()
    KillToEOL()
    InsertText(NewEyeCatcher)
    if NumLines() > 2
        GotoLine(2)
        loop
            BegLine()
            n    = Val(GetText(TEMPLATE_pDELTA, TEMPLATE_lDELTA))
            ext  =     GetText(TEMPLATE_pEXT,   xTEMPLATE_lEXT  )
            wordst =   GetText(xTEMPLATE_pWORD, TEMPLATE_lWORD )
            if n == 0
                break
            endif
            KillToEOL()
            InsertText(format(n:-TEMPLATE_lDELTA, ext:-EXT_LEN, wordst))
            GotoLine(CurrLine() + n + 1)
            if CurrLine() >= NumLines()
                break
            endif
        endloop
    endif
    return (TRUE)
end

integer proc LoadTemplateFile()
    string just_fn[12]
    integer success = FALSE, eof_type

    PushPosition()

    ctrl_id     = CreateTempBuffer()
    list_id     = CreateTempBuffer()
    template_id = CreateTempBuffer()

    if template_id
        just_fn = SplitPath(CurrMacroFileName(), _NAME_) + ".dat"
        template_fn = SearchPath(just_fn, Query(TSEPath) ,"mac")

        if template_fn == ''
            if YesNo("Create setup file '"+just_fn+"'") == 1
                BegFile()
                InsertLine(NewEyeCatcher)
                template_fn = just_fn
                success = SaveTemplateFile()
            endif
        else
            PushBlock()
            eof_type = Set(EOFType, _NONE_)
            success = InsertFile(template_fn, _DONT_PROMPT_)
            Set(EOFType, eof_type)
            if not success
                Warn("Error loading '",template_fn,"'")
            else
                BegFile()
                if GetText(1,sizeof(NewEyeCatcher)) <> NewEyeCatcher
                    if GetText(1,sizeof(EyeCatcher)) <> EyeCatcher
                        success = Warn("Invalid template file '",template_fn,"'")
                    else
                        success = ConvertOldToNew()
                        if success
                            SaveAs(template_fn, _OVERWRITE_|_DONT_PROMPT_)
                        endif
                    endif
                endif
            endif
            PopBlock()
        endif
    else
        Warn("Out of memory")
    endif

    if success
        GenerateControlFile()
    endif

    PopPosition()

    return (success)
end

integer cursor_line

proc RestoreInsert()
    if (CurrLine() <> cursor_line)
        Set(Insert,OFF)
        UnHook(RestoreInsert)
        SetGlobalInt("Template->Editing",FALSE)
    endif
end

proc IndentBlock()
    integer goal, delta

    if isBlockMarked() <> _COLUMN_
        GotoBlockBegin()

        // instead of commented line below...
        goal = CurrCol()
        PushPosition()
        GotoPos(PosFirstNonWhite())
        if CurrCol() < goal
            goal = CurrCol()
        endif
        PopPosition()
        //goal = iif(PosFirstNonWhite() < CurrCol(), PosFirstNonWhite(), CurrCol())

        PushPosition()

        goto kick_start

        while isCursorInBlock()
            GotoPos(PosFirstNonWhite())
            delta = goal + CurrCol() - 2
            if delta > 0
                BegLine()
                while isWhite()
                    DelChar()
                endwhile

                if Query(TabType) == _HARD_
                    while delta >= Query(TabWidth)
                        InsertText(chr(9),_INSERT_)
                        delta = delta - Query(TabWidth)
                    endwhile
                endif
                InsertText(format('':delta:' '),_INSERT_)
            endif

            kick_start:
            BegLine()
            if not Down()
                break
            endif
        endwhile
        PopPosition()
    endif
end

proc InsertBlock(integer line)
    integer delta, id

    id = GotoBufferId(template_id)

    GotoLine(line)
    delta = BlockLines()

    Down()
    BegLine()

    UnMarkBlock()
    MarkChar()

    GotoLine(CurrLine() + delta - 1)
    GotoPos(PosLastNonWhite() + 1)

    GotoBufferId(id)
    CopyBlock()
end

proc PositionCursor()
    integer delta

    GotoBlockEnd()      // bring end of block onto screen

    GotoBlockBegin()    // bring top of block onto screen
    delta = CurrLine() - CurrRow()

    // see if user has specified cursor location
    if HONOR_CURSOR_POSITION and lFind('\c','gli')
        ScrollToRow(CurrLine() - delta)
        DelChar()
        if (CurrChar() == ASC('C')) // special handling required!
            if Set(Insert, ON) == OFF
                cursor_line = CurrLine()
                SetGlobalInt("Template->Editing",TRUE)
                Hook(_AFTER_UPDATE_DISPLAY_, RestoreInsert)
            endif
        endif
        DelChar()
    else
        GotoBlockEnd()
        if CurrChar() >= 0 and not isWhite()
            InsertText(" ",_INSERT_)
        else
            Right()
        endif
    endif
end

proc ExpandSpecialChars()
    if HONOR_FILENAME_EXPANSION
        // Change any \f into the filename
        lreplace("\f", SplitPath(CurrFilename(), _NAME_), "ln")
    endif
end

proc Expand(integer line)
    // assume we're in source
    InsertBlock(line)
    IndentBlock()
    ExpandSpecialChars()
    PositionCursor()
end

/*
12345678901234567890123456789012312345678
while               .s                354
*/
integer proc FindTemplate(string wordst, string ext)
    integer id, line

    id = GotoBufferId(ctrl_id)
    line = iif(lFind(format(wordst:-CTRL_lWORD, ext:-CTRL_lEXT), "^giw"), CtrlLine(), 0)
    GotoBufferId(id)

    return (line)
end

public integer proc ExpandTemplate()
    integer line, found, pass, try
    string wordst[MAX_WORDLEN], ext[EXT_LEN]
    integer rtw

    line = 0
    pass = iif (HONOR_INSERT, not Query(Insert), 0)

    PushBlock()
    UnMarkBlock()

    repeat

        // see if cursor is immediately to the right of a word
        // get the word/template
        PushPosition()
        if pass == 0
            found = Left() and isWord()
            if found
                // get the word to the left of the cursor
                UnMarkBlock()
                MarkColumn()
                BegWord()
                MarkColumn()
            endif
        else
            // get the entire word before (and possibly including the cursor)
            found = Left() and MarkWord()
        endif
        PopPosition()

        if not found and (pass > 0 or not HONOR_INSERT)
            // no 'short-name' found.  Try to find a short-name that doesn't end in
            // the currently defined word-set.
            if isWhiteSpace()
                PushPosition()
                if Left() and not isWhiteSpace()
                    // look for a word-char
                    while Left()
                        if isWhiteSpace()
                            Right()
                            break
                        endif
                        if isWord()
                            // look for a non-word-char
                            while Left()
                                if not isWord()
                                    Right()
                                    break
                                endif
                            endwhile

                            break

                        endif
                    endwhile

                    MarkChar()
                    PopPosition()
                    MarkChar()
                    found = TRUE
                else
                    PopPosition()
                endif
            endif
        endif

        if found
            wordst = GetMarkedText()
            ext = ExtensionType()

            for try = 1 to 3
                // search for template in the control list
                line = FindTemplate(wordst, ext)      // look for current ext
                if line == 0
                    line = FindTemplate(wordst, '')   // look for any extension
                endif

                if line
                    rtw = Set(RemoveTrailingWhite, OFF)
                    KillBlock()
                    Expand(line)
                    Set(RemoveTrailingWhite, rtw)
                    break
                elseif try == 1
                    PushPosition()
                    GotoBlockBegin()
                    if not Left() or isWhiteSpace() or isWord()
                        PopPosition()
                        break
                    endif
                    MarkColumn()
                    wordst = GetMarkedText()
                    PopPosition()
                elseif try == 2
                    PushPosition()
                    GotoBlockBegin()
                    if not Left() or isWhiteSpace() or isWord()
                        PopPosition()
                        break
                    endif
                    loop
                        if not Left()
                            break
                        endif
                        if isWhiteSpace() or isWord()
                            Right()
                            break
                        endif
                    endloop
                    MarkColumn()
                    wordst = GetMarkedText()
                    PopPosition()
                endif
            endfor
        endif
        pass = pass + 1
        if not HONOR_INSERT
            break
        endif
    until pass >= 2

    if line == 0 and MESSAGE_IF_NOT_FOUND
        Message("No template found for '",wordst,"'")
    endif

    PopBlock()

    SetGlobalInt("Template->Expanded", line)

    return (line)
end

proc WhenLoaded()
    integer id = GetBufferId()

    if not LoadTemplateFile()
        PurgeMacro(CurrMacroFileName())
    endif

    GotoBufferId(id)
    keyword_history = GetFreeHistory("Template:keyword")
    ext_history     = GetFreeHistory("Template:ext")
end

proc WhenPurged()
    AbandonFile(ctrl_id)
    AbandonFile(template_id)
    AbandonFile(list_id)
    Set(TemplateExpansion, OFF)
end

integer proc lDelTemplate(integer line)
    integer id

    id = GotoBufferId(template_id)

    GotoLine(line)
    KillLine(BlockLines() + 1)
    SaveTemplateFile()

    GotoBufferId(id)
    GenerateControlFile()

    return (TRUE)
end

integer proc DelTemplate()
    if YesNo("Delete "+trim(GetText(CTRL_pWORD,CTRL_lWORD))) == 1
        return (lDelTemplate(CtrlLine()))
    endif
    return (FALSE)
end

menu TemplateExistsMenu()
    history = 3

    "&Add Duplicate Template"
    "&Replace Template"
    "&Cancel"
end

proc AddLines(integer n)
    BegLine()
    PushPosition()
    while n
        AddLine()
        n = n - 1
    endwhile
    PopPosition()
    Down()
end

string ext_added[EXT_LEN]

#if 0
integer in_wordset

proc CheckWordset()
    PushBlock()
    BegLine()
    MarkWord()
    in_wordset = Query(BlockEndCol) == CurrLineLen() and Query(BlockBegCol) == 1
    PopBlock()
end

integer proc GetKeyword(var string word)
    integer n

    Hook(_PROMPT_CLEANUP_, CheckWordset)
    loop
        n = Ask("Keyword:",word)
        if n and not in_wordset
            Warn("Keyword include characters not in wordset")
        else
            break
        endif
    endloop
    UnHook(CheckWordset)
    return (n)
end
#endif

integer proc AddTemplate()
    integer id, insert_line_blocks_above, line, last_line
    string wordst[MAX_WORDLEN]

    if isBlockMarked()
        wordst = ''

//        if GetKeyword(wordst)
        if Ask("Keyword:", wordst, keyword_history)
            ext_added = CurrExt()
            if Ask("Extension: (clear for all extensions)", ext_added, ext_history)
                ext_added = trim(ext_added)
                if ext_added <> '' and ext_added[1] <> '.'
                    ext_added = '.' + ext_added
                endif

                line = FindTemplate(wordst, ext_added)
                if line
                    case TemplateExistsMenu("'" + wordst + "' exists")
                        when 0,3
                            // abort
                            return (FALSE)
                        when 1
                            // do nothing
                        when 2
                            // delete current extension
                            lDelTemplate(line)
                    endcase
                endif

                id = GotoBufferId(template_id)
                BegFile()

                AddLine()

                PushPosition()

                case isBlockMarked()
                    when _LINE_
                        // do nothing
                    when _COLUMN_
                        // make space for template
                        AddLines(Query(BlockEndLine) - Query(BlockBegLine) + 1)
                    otherwise
                        AddLines(1)
                endcase

                PushBlock()

                insert_line_blocks_above = Set(InsertLineBlocksAbove, OFF)
                CopyBlock()
                Set(InsertLineBlocksAbove, insert_line_blocks_above)

                GotoBlockEnd()
                if Down() and PosFirstNonWhite() == 0
                    DelLine()
                endif
                last_line = CurrLine()

                PopPosition()

                BegLine()
                InsertText(format(last_line - CurrLine() - 1:-TEMPLATE_lDELTA,
                               lower(ext_added):-EXT_LEN,wordst))

                PopBlock()

                SaveTemplateFile()
                GotoBufferId(id)

                GenerateControlFile()
                return (TRUE)
            endif
        endif
    else
        Warn("No block marked")
    endif
    return (FALSE)
end

constant listESCAPE     =   0,
         listENTER      =   1,
         listDEL        =   2,
         listINS        =   3,
         listTOGGLE     =   4

HelpDef ListKeysHelp
    title = "Help on keys"
    x = 27
    y = 5

    "<Enter>        Expand entry  "
    ""
    "<Ins>          Add new entry "
    ""
    "<Del>          Delete entry  "
    ""
    "<Escape>       Close list    "
    ""
    "<Ctrl Enter>   Toggle full/short list   "
end

proc HelpHook()
    BreakHookChain()
end

proc HelpOnListKeys()
    if Hook(_LIST_STARTUP_, HelpHook)
        QuickHelp(ListKeysHelp)
        UnHook(HelpHook)
    endif
end

keydef ListKeys
    <Escape>        EndProcess(listESCAPE)
    <Enter>         EndProcess(listENTER)
    <Del>           EndProcess(listDEL)
    <GreyDel>       EndProcess(listDEL)
    <Ins>           EndProcess(listINS)
    <GreyIns>       EndProcess(listINS)
    <Ctrl Enter>    EndProcess(listTOGGLE)
    <F1>            HelpOnListKeys()
end

proc ListStartup()
    if Enable(ListKeys)
        WindowFooter(" {F1}-Help ")
        BreakHookChain()
    endif
end

integer proc ListTemplates()
    integer id, line, list_all, msg_level, n
    string curr_ext[EXT_LEN]

    line = 0
    curr_ext = ExtensionType()

    id = GotoBufferId(ctrl_id)

    Hook(_LIST_STARTUP_, ListStartup)
    list_all = FALSE

    loop
        EmptyBuffer(list_id)
        GotoBufferId(ctrl_id)
        // generate file
        if not list_all and NumLines()
            BegFile()
            repeat
                case trim(GetText(CTRL_pEXT, CTRL_lEXT))
                    when '', curr_ext
                        AddLine(GetText(1, CurrLineLen()), list_id)
                endcase
            until not Down()
            GotoBufferId(list_id)
        endif

        if NumLines()

            PushBlock()
            MarkColumn(1,1,NumLines(), LIST_WIDTH)
            msg_level = Set(MsgLevel, _NONE_)
            Sort(_IGNORE_CASE_)
            Set(MsgLevel, msg_level)
            BegFile()
            n = lList("Templates", LIST_WIDTH, NumLines(), _BLOCK_SEARCH_ | _ENABLE_SEARCH_ | _ANCHOR_SEARCH_)
            PopBlock()

            case n
                when listENTER
                    line = CtrlLine()
                    break
                when listDEL
                    DelTemplate()
                when listINS
                    AddTemplate()
                when listTOGGLE
                    list_all = not list_all
                when listESCAPE
                    break
            endcase
        else
            if not list_all
                list_all = TRUE
            else
                Warn("No templates exist")
                break
            endif
        endif
    endloop

    UnHook(ListStartup)

    EmptyBuffer(list_id)
    GotoBufferId(id)

    if line
        PushBlock()
        Expand(line)
        PopBlock()
    endif
    return (line)
end

string proc OnOffStr(integer i)
    return (iif(i, "On", "Off"))
end

integer proc mQueryTemplate()
    if Query(TemplateExpansion)
        return (TRUE)
    endif
    return (FALSE)
end

proc mToggleTemplateExpand()
    Toggle(TemplateExpansion)
    mQueryTemplate()
end

menu _TemplateMenu()
    NoKeys
    title = "Template Menu"

    "E&xpand Word Before Cursor    " + KeyName(TEMPLATE_EXPAND) ,   ExpandTemplate()
    "&Add Marked Text as Template"                              ,   AddTemplate()
    "&List Templates (Add/Delete/Expand)"                       ,   ListTemplates()
    ""                          ,                   ,   Divide
    "Template Expa&nsion"   [OnOffStr(mQueryTemplate()):3], mToggleTemplateExpand(), DontClose
end

public integer proc TemplateMenu()
    return (_TemplateMenu())
end

proc main()
    ExpandTemplate()
end

// key-definitions
<TEMPLATE_MENU>     TemplateMenu()

// assuming ExpandTemplate _is_ assigned to <Tab>
<TEMPLATE_EXPAND>   if not Query(TemplateExpansion) or isCursorInBlock() or not ExpandTemplate()
                        ChainCmd()
                    endif

