/*************************************************************************
  SpellChk    Spell checks a word, block, or file.

  Author:     SemWare (Sammy Mitchell)
  =======

  Date:       Jun  4  1993 - Initial version
  =====

    May  27 1994 - Miscellaneous cleanup
    Sep  22 1994 - On Novell drives, FileExists returns TRUE
                  if an empty string is passed!  This causes the
                  routine LoadsWordFile() to call InsertFile()
                  passing an empty string, which in turn pulls up
                  the directory picker.  The fix is to check to
                  see if an empty (or blanks only) string is
                  passed in.
    Dec  9 1994 -  Handle words of the form ''abc'' (tex/latex)
    Feb  5 1995 - add "flag these words as misspelt" handling
    Apr  4 1997 - Add a command line interface.
                  SpellChk can now be called from other macros.
                  To spellchk a file, block, or word:

                  ExecMacro("spellchk file")

                  Replace file as appropriate.

                  If the file was checked (without the user
                  pressing escape) MacroCmdLine is set to "true".
                  Otherwise, it is set to "false".

                  As an example application, a macro that
                  spellchks all the files in the ring:

      proc main()
          PushPosition()
          do NumFiles() + (BufferType() <> _NORMAL_) times    // for each file
              ExecMacro("spellchk file")                      // spell check
              if Query(MacroCmdLine) <> "true"
                  KillPosition()
                  return ()
              endif
              NextFile()
          enddo
          PopPosition()
      end

    Apr 25 1997 - Minor mods to "always misspelt" handling.  Don't
                  Show the misspelt word as one of the suggestions.

    Oct 15 1998 - SEM
                  For .html files, only check what is not between
                  html tags and comments.  Also, skip &word; style
                  things.

                  For .c style source files, only check what is in
                  comments and strings, ignoring the rest of the
                  source code.

                  For other files, if the line starts with
                  quote_string and SKIP_QUOTED_LINES is true, then
                  do so.

    Jul  5, 2000 - Peter Birch
                   Changed to not count \" as a quotation mark in C/C++ programs.

    Mar 13 2003 - Added simple Tex support.  Ignore words that
                  Start with \ when spell checking a .tex file.

    October 2003 - Greg Macdonald - added C# support

    June 1 2005 - Make comment/string checking optional,
                suggested by Shane Brinkman-Davis.

    July 9, 2005 - Bug fix: Words after end-of-line hyphennated words
                are not handled properly.  Fix is to reset the
                word-length.  See @@@1.  Thanks to Reed Truitt for the
                report.

  Overview:
  =========

  This macro allows you to check the spelling of the word at the cursor,
  a marked block, from the cursor to the end of file, or the entire
  file.

  Keys:       <F12> - also available on the TSE Util menu.
  =====

  Usage notes:
  ============

  Uses 4 word lists:

    semware.lex (supplied by SemWare)

    user.lex (user supplied)
      SpellCheck will add to and update (on disk) this file after
      every invocation of SpellCheck.

    personal.lex (user supplied)
      SpellCheck will load this file if found, but will never
      update it.

    wrong.lex (user supplied)
      Any words appearing in this file will always be considered
      misspelled, regardless of what other .lex file they might
      appear in.

  Word lists are loaded by using SearchPath(name, TSEPath, "spell")

  Copyright 1992-1994 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/

/************************************************************************
  If the dll is compiled with Borland (4.5, 5), use "foo"
  If the dll is compiled with Microsoft C (VC 4), use "_foo@??" to get ?? look in the map!!!
  If the dll is compiled with Watcom C (10, 10.5), use "_Foo"

  Currently compiled with Watcom c 10.5.
 ************************************************************************/

constant PATHLEN = 255
dll "spell.dll"
    integer proc OpenSpell(string fn) : "_OpenSpell"
    integer proc CloseSpell() : "_CloseSpell"
    integer proc SpellCheckWord(string st) : "_SpellCheckWord"
    integer proc SuggestWord(string st) : "_SuggestWord"
            proc GetSuggestion(var string st, integer n) : "_GetSuggestion"
            proc RemoveQuotes(var string st) : "_RemoveQuotes"
end

constant
    MINIMUM_WORD_LENGTH = 2,
    IGNORE_NON_ALPHA_WORDS = TRUE,
    SKIP_QUOTED_LINES = TRUE

constant
    SINGLE_QUOTE = 0x27

constant
    SKIPONCE = 12, IGNORE, EDIT, ADD, TAG, QUIT,
    spESCAPE = 0, spFILE, spREST_OF_FILE, spBLOCK, spWORD,
    MAXWORD = 32,
    TEX_KEYWORD_CHAR = Asc("\")

integer count, ignore_id, addl_id, added_to_addl, wrong_id,
    suggestion, number_suggestion, change_word_history

string main_disk_fn[PATHLEN] = "semware.lex",
    addl_buffer_name[] = "+++addl@words+++",
    addl_disk_fn[PATHLEN] = "user.lex",
    ignore_buffer_name[] = "+++ignore@words+++",
    ignore_disk_fn[PATHLEN] = "personal.lex",
    wrong_buffer_name[] = "+++wrong@words+++",
    wrong_disk_fn[PATHLEN] = "wrong.lex",
    quote_string[] = ">"

string section_name[] = "Spellcheck"
string heading_name[] = "ignore-code"

string wrong_word[MAXWORD]

// variables for parsing the file
string
    token[MAXWORD]                  // currently found word

integer ignore_code, ignore_code_grayed, tag_all

constant
    UNKNOWN_EXT = 0, HTML_EXT, C_EXT, TEX_EXT

string proc OnOffStr(integer i)
    return (iif(i, "On", "Off"))
end

proc toggle_ignore_code()
    ignore_code = not ignore_code
end

integer proc ignore_code_flags()
    if ignore_code_grayed
        return (_MF_GRAYED_)
    endif
    return (_MF_ENABLED_)
end

menu SpellMenu()
    Title = "Spell Check"
    History

    "&File"
    "&Rest of File"
    "&Block"
    "&Word"
    "",,Divide
    "Ignore &Code" [OnOffStr(ignore_code):3], toggle_ignore_code(), ignore_code_flags()
end

string proc GetNextSuggestion()
    string wordst[MAXWORD]

    loop
        wordst = ""

        if number_suggestion == 0
            break
        endif

        number_suggestion = number_suggestion - 1
        suggestion = suggestion + 1

        GetSuggestion(wordst, suggestion)
        if wordst <> wrong_word

            if Length(wordst) < sizeof(wordst)
                wordst = Format(' ', wordst:-MAXWORD)
            endif
            break
        endif
    endloop
    return (wordst)
end

menu WhatToDoMenu()
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    ""[GetNextSuggestion():MAXWORD]
    "",,DIVIDE
    "&Skip Once"
    "&Ignore for Rest of Session"
    "&Edit"
    "&Add to Word List"
    "&Tag"
    "&Quit"
end

integer proc FindWordIn(string wordst, integer id)
    integer ok, curr_id

    curr_id = GotoBufferId(id)
    ok = lFind(wordst, "gw")
    GotoBufferId(curr_id)
    return (ok)
end

integer last_ticks, reset_ticks

proc tag_it(string raw_word)
    token = Chr(174) + raw_word + Chr(175)
    lreplace(raw_word, token, "n1")
    Right()
end

integer proc SpellCheck()
    integer i, ticks
    string wordst[MAXWORD], raw_word[MAXWORD], msg[20],
        tword[MAXWORD] = ""

    ticks = GetClockTicks()
    if reset_ticks
        last_ticks = ticks
        reset_ticks = FALSE
    elseif ticks - last_ticks > 12 or ticks < last_ticks
        last_ticks = ticks
        UpdateDisplay(_CLINE_REFRESH_)
        if Query(MsgLevel) == _ALL_MESSAGES_
            Message((CurrLine() * 100) / NumLines(), "%")
        endif
        if tag_all and KeyPressed()
            GetKey()
            if MsgBox("Spell check", "Quit Tagging?", _YES_NO_CANCEL_) == 1
                tag_all = false
            endif
        endif
    endif

    loop

        msg = " not found"
        wrong_word = ""

        raw_word = token
        wordst = Lower(raw_word)
        RemoveQuotes(wordst)

        if Length(wordst) < MINIMUM_WORD_LENGTH or SpellCheckWord(raw_word)
            if not FindWordIn(wordst, wrong_id)
                return (TRUE)
            endif

            msg = " found in wrong.lex"
            wrong_word = raw_word
        endif

        //https://mail.google.com/mail/u/0/#sent/FMfcgzQVxbfwpmNlCRsjmPDgVcllhCJB
        if wordst in "https", "http"
            right(length(wordst))
            if CurrChar() == Asc(":")
                while CurrChar() >= 0 and not isWhite()
                    Right()
                endwhile
            endif
            token = ""
            return (TRUE)
        endif

        if Length(wrong_word) == 0

            // try to handle hyphenated words at the eol
            // unfortunately, we can't (yet) correct them

            PushPosition()
            GotoPos(CurrPos() + Length(raw_word))
            if CurrPos() == PosLastNonWhite() and CurrChar() == Asc('-')
                EndLine()
                if WordRight() and isWord() and SpellCheckWord(raw_word + GetWord())
                    token = GetWord()   // @@@1: reset word length
                    KillPosition()
                    return (TRUE)
                endif
            endif
            PopPosition()
            // end of eol hyphenation code...

            if FindWordIn(wordst, addl_id) or FindWordIn(wordst, ignore_id)
                return (TRUE)
            endif

            if IGNORE_NON_ALPHA_WORDS
                for i = 1 to Length(wordst)
                    if not (wordst[i] in 'a'..'z', "'")
                        return (TRUE)
                    endif
                endfor
            endif
        endif

        ScrollToRow(3 - (Query(DisplayBoxed) <> 0))

        if raw_word[1] == "'" and raw_word[Length(raw_word)] == "'"
            raw_word = raw_word[2:Length(raw_word) - 2]
        endif

        if tag_all
            tag_it(raw_word)
            return (true)
        endif

        reset_ticks = TRUE

        lFind(raw_word, "")
        HiLiteFoundText()

        Set(Y1, 14)
        Set(X1, 60)

        suggestion = 0
        number_suggestion = SuggestWord(raw_word)

        case WhatToDoMenu(chr(34) + raw_word + chr(34) + msg)
            when 1 .. 10
                if MenuOption() <= suggestion
                    GetSuggestion(tword, MenuOption())
                    lreplace(raw_word, tword, "n1")
                    token = tword
                else
                    break
                endif

            when SKIPONCE
                return (TRUE)

            when IGNORE
                AddLine(wordst, ignore_id)
                return (TRUE)

            when EDIT
                tword = raw_word
                if ask("Change word:", tword, change_word_history)
                    lreplace(raw_word, tword, "n1")
                    token = tword
                endif

            when ADD
                AddLine(wordst, addl_id)
                added_to_addl = added_to_addl + 1
                return (TRUE)

            when TAG
                tag_it(raw_word)
                return (TRUE)

            when QUIT, 0
                return (FALSE)

            otherwise
                break

        endcase
    endloop
    return (TRUE)
end

integer proc LoadWordsFile(integer id, string b_name, string disk_fn, integer orig_id)
    if id == 0
        id = GetBufferId(b_name)
        if id == 0
            PushBlock()

            id = CreateBuffer(b_name, _HIDDEN_)
            SetUndoOff()
            if Length(Trim(disk_fn)) and FileExists(disk_fn)
                InsertFile(disk_fn, _DONT_PROMPT_)
            endif

            PopBlock()
            GotoBufferId(orig_id)
        endif
    endif
    return (id)
end

proc SaveAddlWordList()
    integer curr_id

    if added_to_addl
        curr_id = GotoBufferId(addl_id)
        SaveAs(addl_disk_fn, _OVERWRITE_)
        GotoBufferId(curr_id)
    endif
end

/**************************************************************************
  GetTok must always return something - mainloop depends on it!
 **************************************************************************/
integer proc GetTok(var integer token_is_word, integer ext_type)
    token_is_word = FALSE
    repeat
        while CurrPos() > CurrLineLen()
            if not NextChar()
                return (FALSE)
            endif
        endwhile

        while isWhite()
            if not NextChar()
                return (FALSE)
            endif
        endwhile

        if ext_type == TEX_EXT and CurrChar() == TEX_KEYWORD_CHAR
            repeat
            until not Right() or CurrPos() > CurrLineLen() or ((not isWord()) and CurrChar() <> TEX_KEYWORD_CHAR)
        endif

    until CurrPos() <= CurrLineLen()

    PushPosition()
    if not isWord()
        token = Chr(CurrChar())
    else
        token = ""
        token_is_word = TRUE
        repeat
            token = token + Chr(CurrChar())
        until Length(token) == sizeof(token) or not Right() or CurrPos() > CurrLineLen() or not isWord()
    endif
    PopPosition()
    return (TRUE)
end

proc Skip(integer n)
    Right(n)
end

proc SkipOver(string s)
    if lFind(s, "")
        Skip(Length(s))
    else
        EndFile()
    endif
end

/**************************************************************************
  Determine whether token is a start of comment/quote in the specified language
 **************************************************************************/
integer proc isCommentFound(integer ext_type, var string end_cmt, var integer cmt_line)
    end_cmt = ""
    cmt_line = 0

    case ext_type
        when HTML_EXT
            case token[1]
                when '<'
                    Skip(1)
                    // see if comment <!--
                    if GetText(CurrPos(), 3) == "!--"
                        Skip(3)
                        end_cmt = "-->"
                    else
                        end_cmt = ">"
                    endif
                    return (TRUE)
                when '&'
                    Skip(1)
                    end_cmt = ";"
                    return (TRUE)
            endcase

        when C_EXT
            case token[1]
                when '/'
                    Skip(1)
                    if GetText(CurrPos(), 1) == "*"
                        Skip(1)
                        end_cmt = "*/"
                        return (TRUE)
                    elseif GetText(CurrPos(), 1) == "/"
                        Skip(1)
                        end_cmt = ""
                        cmt_line = CurrLine()
                        return (TRUE)
                    endif
                when '"'
                    Skip(1)
                    end_cmt = '"'
                    return (TRUE)
                when "'"
                    Skip(1)
                    end_cmt = "'"
                    return (TRUE)
            endcase
    endcase
    return (FALSE)
end

/**************************************************************************
  Determine whether token is a end of comment/quote in the specified language
 **************************************************************************/
integer proc isEndCommentFound(integer ext_type, string end_cmt, integer cmt_line, string string_escape_char)
    if end_cmt == "" and cmt_line == 0
        return (FALSE)
    endif

    case ext_type
        when HTML_EXT
            if cmt_line
                Warn("Logic error 1")
            elseif end_cmt <> ""
                SkipOver(end_cmt)
                return (TRUE)
            else
                Warn("Logic error 2")
            endif
        when C_EXT
            if cmt_line and end_cmt <> ""
                Warn("Logic error 3")
            elseif cmt_line
                if CurrLine() > cmt_line
                    return (TRUE)
                endif
            elseif end_cmt == GetText(CurrPos(), Length(end_cmt))
                if (end_cmt in '"', "'") and CurrPos() > 1 and
                        (GetText(CurrPos() - 1, 1) == string_escape_char and
                        GetText(CurrPos() - 2, 1) <> string_escape_char)
                else
                    Skip(Length(end_cmt))
                    return (TRUE)
                endif
            endif
    endcase
    return (FALSE)
end

integer proc PastEndOfBlock()
    integer endcol

    if CurrLine() < Query(BlockEndLine)
        return (FALSE)
    endif

    if CurrLine() > Query(BlockEndLine)
        return (TRUE)
    endif

    // must be equal

    if CurrPos() >= CurrLineLen()
        return (TRUE)
    endif

    endcol = Query(BlockEndCol)
    case isBlockInCurrFile()
        when _LINE_
        when _INCLUSIVE_
        when _NONINCLUSIVE_
            if CurrPos() >= endcol
                return (TRUE)
            endif
        when _COLUMN_
            if CurrCol() >= endcol
                return (TRUE)
            endif
    endcase
    return (FALSE)
end

integer proc GotoInitialPosition(integer range_type, integer block_type)
    case range_type
        when spFILE
            BegFile()

        when spREST_OF_FILE

        when spBLOCK
            GotoBlockBegin()
            if block_type == _LINE_
                BegLine()
            endif

        when spWORD
            //do magic stuff
            PushPosition()
            repeat
            until BegWord() or not Left()
            repeat
            until BegWord() or not Right() or CurrPos() > CurrLineLen()
            if not isWord()
                PopPosition()
                Message("Cursor not on a word...")
                return (FALSE)
            endif
            KillPosition()
    endcase
    return (TRUE)
end

string cmdline[255]
proc Main()
    integer
        range_type, block_type, ext_type, errors, in_cmt, spell_in_cmt,
        done, skip_token, token_is_word, tmp,
        cmt_line                        // if eol comment, line number of comment
    string
        old_wordset[32],
        end_cmt[32],                    // end comment string for current comment, or ""
        string_escape_char[1]           // for C-like languages, the "\"" style character

Set(Break, on)

    count = 0
    added_to_addl = 0
    errors = FALSE

    cmt_line = 0
    end_cmt = ""
    string_escape_char = ""

    case CurrExt()
        when ".html", ".htm", ".shtml", ".shtm"
            ext_type = HTML_EXT
            spell_in_cmt = FALSE
        when ".cs", ".c", ".h", ".cpp", ".hpp", ".cc", ".cxx", ".hxx", ".java", ".js"
            string_escape_char = "\"
            ext_type = C_EXT
            spell_in_cmt = TRUE
        when ".s", ".si", ".ui"
            ext_type = C_EXT
            spell_in_cmt = TRUE
        when ".tex"
            ext_type = TEX_EXT
            spell_in_cmt = FALSE
        otherwise
            ext_type = UNKNOWN_EXT
            spell_in_cmt = FALSE
    endcase

    ignore_code_grayed = ext_type == UNKNOWN_EXT
    cmdline = Trim(Query(MacroCmdLine))
    if cmdline == ""
        range_type = SpellMenu()
    else
        case Lower(cmdline)
            when "-file"
                range_type = spFILE
            when "-rest"
                range_type = spREST_OF_FILE
            when "-block"
                range_type = spBLOCK
            when "-word"
                range_type = spWORD
            when "-tag"
                tag_all = true
            otherwise
                Warn("Unrecognized cmdline: ", cmdline)
                range_type = spESCAPE
        endcase
    endif

    //get ignore_code from tse.ini
    //if different from ignore_code, update it

    tmp = EquiStr(GetProfileStr(section_name, heading_name, ""), "yes")
    if tmp <> ignore_code
        WriteProfileStr(section_name, heading_name, iif(ignore_code, "yes", "no"))
    endif

    if not ignore_code
        ext_type = UNKNOWN_EXT
        spell_in_cmt = FALSE
    endif

    if range_type == spESCAPE
        return ()
    endif

    block_type = isBlockInCurrFile()
    if block_type == 0 and range_type == spBLOCK
        Warn("No block")
        return ()
    endif

    if not OpenSpell(main_disk_fn)
        warn("Can't load word list: ", main_disk_fn)
        return ()
    endif

    // at this point, main lexicon is loaded

    old_wordset = Set(WordSet, ChrSet("A-Za-z'0-9\d128-\d165"))

    PushBlock()
    PushPosition()
    Set(Marking, off)

    if not GotoInitialPosition(range_type, block_type)
        errors = TRUE
    elseif range_type == spWORD
        GetTok(token_is_word, ext_type)
        if not SpellCheck()
            errors = TRUE
        endif
    else
        in_cmt = FALSE
        done = FALSE
        while not done and GetTok(token_is_word, ext_type)
            skip_token = FALSE

            if ext_type == UNKNOWN_EXT
                if SKIP_QUOTED_LINES and token == quote_string and CurrPos() == PosFirstNonWhite()
                    EndLine()
                    skip_token = TRUE
                endif
            else
                if in_cmt
                    if isEndCommentFound(ext_type, end_cmt, cmt_line, string_escape_char)
                        in_cmt = FALSE
                        skip_token = TRUE
                    elseif end_cmt == "'" and token[Length(token)] == "'"
                        token = DelStr(token, Length(token), 1)
                    endif
                else // not in_cmt
                    if isCommentFound(ext_type, end_cmt, cmt_line)
                        in_cmt = TRUE
                        skip_token = TRUE
                    endif
                endif
            endif

            if range_type == spBLOCK
                if not isCursorInBlock()
                    if PastEndOfBlock()
                        done = TRUE
                    elseif not skip_token
                        skip_token = TRUE
                        Skip(Length(token))
                    endif
                endif
            endif

            if not done and not skip_token
                if token_is_word
                    if ext_type == UNKNOWN_EXT or in_cmt == spell_in_cmt
                        if SpellCheck()
                            count = count + 1
                        else
                            errors = TRUE
                            done = TRUE
                        endif
                    endif
                endif
                Skip(Length(token))
            endif

        endwhile
    endif

    SaveAddlWordList()
    Set(WordSet, old_wordset)
    CloseSpell()
    if errors
        KillPosition()
    else
        PopPosition()
        if Query(MsgLevel) == _ALL_MESSAGES_ and Length(cmdline) == 0
            Message("Finished...")
        endif
    endif
    PopBlock()

    Set(MacroCmdLine, iif(errors, "false", "true"))
end

string proc FindFile(string fn)
    return (SearchPath(fn, Query(TSEPATH), "SPELL\"))
end

proc WhenLoaded()
    string fn[PATHLEN]
    integer orig_id

    orig_id = GetBufferId()

    fn = FindFile(main_disk_fn)
    if Length(fn)
        main_disk_fn = fn
    endif

    fn = FindFile(addl_disk_fn)
    if Length(fn) == 0
        addl_disk_fn = SplitPath(main_disk_fn, _DRIVE_|_PATH_) + addl_disk_fn
    else
        addl_disk_fn = fn
    endif

    ignore_disk_fn = FindFile(ignore_disk_fn)
    wrong_disk_fn = FindFile(wrong_disk_fn)

    addl_id = LoadWordsFile(addl_id, addl_buffer_name, addl_disk_fn, orig_id)
    ignore_id = LoadWordsFile(ignore_id, ignore_buffer_name, ignore_disk_fn, orig_id)
    wrong_id = LoadWordsFile(wrong_id, wrong_buffer_name, wrong_disk_fn, orig_id)
    change_word_history = GetFreeHistory("SpellChk:ChangeWord")

//get ignore_code from tse.ini
    ignore_code = TRUE
    if EquiStr(GetProfileStr(section_name, heading_name, ""), "no")
        ignore_code = FALSE
    endif
end

