/*****************************************************************************

  Filename      : SpeedKey.s

  Created       : Originally by Ian Campbell (QKeys.s)

  Modified by   : Ross Boyd

  Date          : Feb 2002 (beta release)
                  Apr  1, 2002 Fixed EquateEnhancedKbd handling.
                               Corrected "SpeedKey" text to uniform case in
                               all dialogs.
                               Some extra comments and hotkeys in ConfigMenu()
                : Nov 2003 (SEM) proc isAutoLoaded did not work correctly if
                        the autoload file (tseload.dat) did not exist.

  Version       : 3.01

  Compatibility : TSEPro 2.6 versions and higher

 Description:
 This macro accelerates the rate of cursor movement and hence scroll speed.
 Essentially, the macro detects when the Arrow keys are being pressed and
 repeats the commands associated with them eg. Up(),Down(),Right(),Left()

 Use the configuration menu via <CtrlAlt S> to experiment with the
 horizontal and vertical repeat rates and the initial repeat delay.
 Once satisfied with your configuration, use the Save Settings option to
 make your settings permanent.

 For best results add SpeedKey to your AutoLoad list.

 Caveats:
 The slowest acceleration is actually quite fast.

*****************************************************************************/

/*****************************************************************************
 API Calls and Wrappers
*****************************************************************************/
dll "<user32.dll>"
    integer proc GetKBSettings(integer uiAction,integer uiParam
            ,var integer pvParam,integer fWiniIni): "SystemParametersInfoA"
    integer proc SetKBSettings(integer uiAction,integer uiParam
            ,integer pvParam,integer fWiniIni): "SystemParametersInfoA"
end

// From the Windows header files
#define SPI_GETKEYBOARDSPEED    10
#define SPI_SETKEYBOARDSPEED    11
#define SPI_GETKEYBOARDDELAY    22
#define SPI_SETKEYBOARDDELAY    23


// Wrapper for API Call to retrieve Window's Repeat Rate
integer proc GetWinKbSpeed()
    integer  i
    GetKBSettings(SPI_GETKEYBOARDSPEED,0,i,0)
    return(i)
end

// Wrapper for API Call to retrieve Window's Repeat Delay
integer proc GetWinKbDelay()
    integer  i
    GetKBSettings(SPI_GETKEYBOARDDELAY,0,i,0)
    return(i)
end

/*****************************************************************************
 Forward Declarations
*****************************************************************************/
forward menu ConfigMenu()

/*****************************************************************************
 Global Constants
*****************************************************************************/

// Change the hotkey here. The ConfigMenu() will display the correct hotkey
// automatically.
constant CONFIG_MENU_KEY = <CtrlAlt s>

constant SK_VERTICAL = 1               // SAL supports enumerated constants!
        ,SK_HORIZONTAL                 // eg. SK_HORIZONTAL will now = 2
        ,SK_DELAY
        ,SK_DEFAULTS
        ,SK_PURGE
        ,SK_AUTOLOAD
        ,SK_WIN_KEYRATE
        ,SK_WIN_KEYDELAY
        ,SK_SAVESETTINGS
        ,SK_VERSION = 301

constant INITIAL_DEFAULT = 2

// Parallel arrays to store default settings
constant MAX_DEFAULTS = 6

string   aDefaultV[MAX_DEFAULTS] = "012346",
         aDefaultH[MAX_DEFAULTS] = "012346",
         aDefaultD[MAX_DEFAULTS] = "011111"


string   SECTION[]          = "SpeedKey"
        ,HDG_VERTI_KEYS[]   = "VerticalSpeedUp"
        ,HDG_HORIZ_KEYS[]   = "HorizontalSpeedUp"
        ,HDG_DELAY_KEYS[]   = "KeyDelayBeforeSpeedUp"
        ,HDG_VERSION[]      = "Version"

/*****************************************************************************
 Global Variables
*****************************************************************************/

// the number of additional Up()/Down() commands that are
// processed by each <CursorUp>, <CursorDown> keystroke.
integer  VerticalRepeats

// the number of additional Left() or Right() commands that are
// processed by each <CursorLeft>, <CursorRight> keystroke.
integer  HorizontalRepeats

// the number of consecutive cursor keystrokes before speedup occurs
integer  DelayBeforeSpeedup

integer  CursorKeyCount = 0
integer  StoredTicks    = 0
integer  OldCursorKey   = 0
integer  isConfigured   = FALSE

/****************************************************************************
   Procedure:   Configure()

   Notes:       Provides Public configuration menu via Execute Macro dialog.

****************************************************************************/
public proc Configure()

    // Its in a loop so we can refresh the menu appearance 'on the fly'...
    repeat
    until ConfigMenu() == 0 or isConfigured
end

/*****************************************************************************
   Procedure:   SpeedKey()

   Notes:       This is the main macro.  It adds the extra cursor key
                functions, and sorts out the smoothing mechanism for
                performing these functions at the appropriate time.

   Called by:   Assigned to the cursor keys by a direct key assignment
                at the bottom of this macro.

*****************************************************************************/
proc SpeedKey(integer CursorKey)

    integer  StreamingState = False

    // NOTE: These two constants should be left alone.
    constant STREAM_TIMEOUT = 2
    constant STREAM_HOLDOFF = 1

    if CursorKey <> OldCursorKey
        CursorKeyCount = 0
        OldCursorKey   = CursorKey
    else
        StreamingState = TRUE
    endif

    case CursorKey
    when <CursorRight>
        Right()
        if HorizontalRepeats == 0
            return()
        endif
    when <CursorLeft>
        Left()
        if HorizontalRepeats == 0
            return()
        endif
    when <CursorUp>
        Up()
        if VerticalRepeats == 0
            return()
        endif
    when <CursorDown>
        Down()
        if VerticalRepeats == 0
            return()
        endif
    endcase

    // Was the cursor key released or are we recording/running a kbd macro?
    if GetClockTicks() > StoredTicks + STREAM_TIMEOUT
        or Query(KbdMacroRecording) or Query(KbdMacroRunning)
        StreamingState = FALSE         // Yep, so switch off streaming
        CursorKeyCount = 0             // and reset keycount
    endif

    if StreamingState
        and CursorKeyCount >= DelayBeforeSpeedup
        and GetClockTicks() <= StoredTicks + STREAM_HOLDOFF

        case CursorKey
        when <CursorRight>
            Right(HorizontalRepeats)
        when <CursorLeft>
            Left(HorizontalRepeats)
        when <CursorUp>
            Up(VerticalRepeats)
        when <CursorDown>
            Down(VerticalRepeats)
        endcase
    endif

    StoredTicks    = GetClockTicks()
    CursorKeyCount = CursorKeyCount + 1

end

/****************************************************************************
   Procedure:   intSaneReadNum()

   Notes:       Read command that returns the numeric value of the user input.
                Forces the values to be within a specified range
****************************************************************************/
integer proc intSaneReadNum(integer n,integer width, integer minimum, integer maximum)
    string   s[3] = Str(n)
    if lReadNumeric(s,width)
        return (Min(Max(Val(s),minimum),maximum))
    endif
    return (n)
end

/****************************************************************************
   Procedure:   SaveIniSettings()

   Notes:       Save Acceleration Settings to TSE.INI file.
****************************************************************************/
proc SaveIniSettings()
    if not (WriteProfileInt(SECTION, HDG_VERTI_KEYS,  VerticalRepeats)
        and WriteProfileInt(SECTION, HDG_HORIZ_KEYS,  HorizontalRepeats)
        and WriteProfileInt(SECTION, HDG_DELAY_KEYS,  DelayBeforeSpeedup)
        and WriteProfileInt(SECTION, HDG_VERSION,     SK_VERSION))
        Warn("Error writing tse.ini settings...")
    endif

end

/****************************************************************************
   Procedure:   LoadIniSettings()

   Notes:       Load and Set Acceleration Settings from TSE.INI file.
****************************************************************************/
proc LoadIniSettings()

    integer  ConfigRequired = False

    VerticalRepeats = GetProfileInt(SECTION,HDG_VERTI_KEYS,-1)
    if VerticalRepeats == -1
        VerticalRepeats = Val(aDefaultV[INITIAL_DEFAULT])
        ConfigRequired  = True
    endif

    HorizontalRepeats = GetProfileInt(SECTION,HDG_HORIZ_KEYS,-1)
    if HorizontalRepeats == -1
        HorizontalRepeats = Val(aDefaultH[INITIAL_DEFAULT])
        ConfigRequired    = True
    endif

    DelayBeforeSpeedup = GetProfileInt(SECTION,HDG_DELAY_KEYS,-1)
    if DelayBeforeSpeedup == -1
        DelayBeforeSpeedup = Val(aDefaultD[INITIAL_DEFAULT])
        ConfigRequired     = True
    endif

    // This is here to force a reconfig for existing users of the 2.00 version.
    if GetProfileInt(SECTION,HDG_VERSION,-1) == -1
        ConfigRequired = True
    endif

    // Handle maiden voyage by setting up defaults if necessary
    if ConfigRequired
        SaveIniSettings()
        Configure()
    endif
end

menu DefaultMenu()
    history
    noescape
    "&No Acceleration"     ,,_MF_CLOSE_AFTER_
    "&Faster"              ,,_MF_CLOSE_AFTER_
    "&2x Faster"           ,,_MF_CLOSE_AFTER_
    "&3x Faster"           ,,_MF_CLOSE_AFTER_
    "&4x Faster"           ,,_MF_CLOSE_AFTER_
    "&6x Faster"           ,,_MF_CLOSE_AFTER_
    "E&xit"                ,,_MF_CLOSE_AFTER_
end

/****************************************************************************
   Procedure:   ConfigHelper()

   Notes:       Support function for ConfigMenu() menu
****************************************************************************/
proc ConfigHelper(integer option)

    integer  i

    isConfigured = False

    case option
    when SK_VERTICAL
        VerticalRepeats = intSaneReadNum(VerticalRepeats,1,0,9)
    when SK_HORIZONTAL
        HorizontalRepeats = intSaneReadNum(HorizontalRepeats,1,0,9)
    when SK_DELAY
        DelayBeforeSpeedup = intSaneReadNum(DelayBeforeSpeedup,2,0,99)
    when SK_WIN_KEYRATE
        SetKbSettings(SPI_SETKEYBOARDSPEED,intSaneReadNum(GetWinKbSpeed(),2,0,31),0,0)
    when SK_WIN_KEYDELAY
        SetKbSettings(SPI_SETKEYBOARDDELAY,intSaneReadNum(GetWinKbDelay(),1,0,3),0,0)
    when SK_DEFAULTS
        i = DefaultMenu()
        if i in 1..MAX_DEFAULTS
            VerticalRepeats    = Val(aDefaultV[i])
            HorizontalRepeats  = Val(aDefaultH[i])
            DelayBeforeSpeedup = Val(aDefaultD[i])

            if GetWinKbSpeed() < 31 or GetWinKbDelay() > 0
                if 1 == MsgBox("Speedkey","Optimise the MS Windows system settings also? (Recommended)",_YES_NO_)
                    SetKbSettings(SPI_SETKEYBOARDSPEED,31,0,0)
                    SetKbSettings(SPI_SETKEYBOARDDELAY,00,0,0)
                endif
            endif
        endif
        Set(Key,-1)
    when SK_PURGE
        PurgeMacro(CurrMacroFileName())
        isConfigured = TRUE
    when SK_AUTOLOAD
        if isAutoLoaded(CurrMacroFileName())
            if not DelAutoLoadMacro(SplitPath(CurrMacroFileName(),_NAME_))
                MsgBox("Error!","SpeedKey could NOT be removed from the AutoLoad list")
            endif
        else
            if not AddAutoLoadMacro(SplitPath(CurrMacroFileName(),_NAME_))
                MsgBox("Error!","SpeedKey could NOT be added to the AutoLoad list")
            endif
        endif
    when SK_SAVESETTINGS
        SaveIniSettings()
        isConfigured = TRUE
    endcase
end

/****************************************************************************
   Procedure:   OnOffStr()

   Notes:       Menu helper function.

****************************************************************************/
string proc OnOffStr(integer flag)
    return (iif(flag, "On","Off"))
end

/****************************************************************************
   Menu:        ConfigMenu()

   Notes:       Allows user to configure acceleration settings.
                Provides options to Save Settings, Load Defaults or Purge
                this macro, and Add/Remove from AutoLoad List.
****************************************************************************/
menu ConfigMenu()

    Title = "SpeedKey"
    History

    "&Vertical Speedup Factor"[VerticalRepeats:1]
            ,ConfigHelper(SK_VERTICAL)
            ,_MF_DONT_CLOSE_
            ,"Vertical speed multiplier (0-9)"
    "&Horizontal Speedup Factor"[HorizontalRepeats:1]
            ,ConfigHelper(SK_HORIZONTAL)
            ,_MF_DONT_CLOSE_
            ,"Horizontal speed multiplier (0-9)"
    "&Delay"[DelayBeforeSpeedup:2]
            ,ConfigHelper(SK_DELAY)
            ,_MF_DONT_CLOSE_
            ,"Number of keystrokes to delay acceleration (0-99)"
    "MS Window's Repeat &Rate" [GetWinKbSpeed():2]
            ,ConfigHelper(SK_WIN_KEYRATE)
            ,_MF_DONT_CLOSE_
            ,"Set MS Window's Keyboard Repeat Rate (0-31) Recommended=31"
    "MS &Window's Repeat Delay" [GetWinKbDelay():1]
            ,ConfigHelper(SK_WIN_KEYDELAY)
            ,_MF_DONT_CLOSE_
            ,"Set MS Window's Keyboard Repeat Delay (0-3) Recommended=0"
    "",,_MF_DIVIDE_
    "&Choose a default..."
            ,ConfigHelper(SK_DEFAULTS)
            ,_MF_CLOSE_AFTER_
            ,"Quick and Easy Default Settings"
    "&Purge SpeedKey now"
            ,ConfigHelper(SK_PURGE)
            ,_MF_CLOSE_AFTER_
            ,"Purge SpeedKey from memory"
    "",,_MF_DIVIDE_
    "&Autoload SpeedKey" [OnOffStr(isAutoLoaded(CurrMacroFileName())):3]
            ,ConfigHelper(SK_AUTOLOAD)
            ,_MF_DONT_CLOSE_
            ,"Automatically load SpeedKey at TSE startup"
    "&Save Settings"
            ,ConfigHelper(SK_SAVESETTINGS)
            ,_MF_CLOSE_AFTER_
            ,"Save the current settings to disk"
    // Here is a little trick to get the menu's hotkey to display without
    // hard coding it.
    "Invoke this menu with"
            ,Configure()
            ,_MF_GRAYED_|_MF_SKIP_
end

/****************************************************************************
   Procedure:   WhenLoaded()
****************************************************************************/
proc WhenLoaded()
    LoadIniSettings()
end

/****************************************************************************
  Cursor Key Definitions/Remapping
****************************************************************************/
<CursorRight>       SpeedKey(<CursorRight>)
<CursorLeft>        SpeedKey(<CursorLeft>)
<CursorUp>          SpeedKey(<CursorUp>)
<CursorDown>        SpeedKey(<CursorDown>)

// These 4 handle EquateEnhancedKbd being set to OFF
<GreyCursorRight>   SpeedKey(<CursorRight>)
<GreyCursorLeft>    SpeedKey(<CursorLeft>)
<GreyCursorUp>      SpeedKey(<CursorUp>)
<GreyCursorDown>    SpeedKey(<CursorDown>)

/****************************************************************************
  Configuration Menu Hotkey Assignment
****************************************************************************/
<CONFIG_MENU_KEY> Configure()

// EOF SpeedKey.s

