/****************************************************************************
   TSE Jr style scratch buffer package

  If called from pulldown menus, and command successful, try to clear the
  pulldown menus by pushing escape on the stack, twice.

  Format:

  nameclip cmdline

  if cmdline starts with '-' assumed a direct entry into the command.

  Assumptions:

    If MacroCmdLine is empty, we're called from the pulldown menus.

    When called from the pulldown menus, there are two menus on the screen.

  --- Updates ---

  August 10, 1999 "Delete" menu item changed to "Empty" at the request
        of the documentation person - don't blame me, I'm only the
        programmer!
 ****************************************************************************/

integer named_clip_hist,
    ok                      // flag: was operation successful?

constant    _DELSCRATCH = 0, _PASTEOVER = 1, _PASTE = 2,     // code depends on this order
            _COPY = 3, _COPYAPPEND = 4, _CUT = 5, _CUTAPPEND = 6

proc NameClip(integer operation)
    integer cid, id, SaveClipBoardId
    string BufferName[40], msg[30]

    if operation > _PASTE and (NOT (isBlockInCurrFile() or Query(UseCurrLineIfNoBlock)))
        return ()
    endif
    BufferName = ""
    SaveClipBoardId = Query(ClipBoardId)          // save id
    case operation
        when _COPY        msg = "Copy to"
        when _COPYAPPEND  msg = "Copy Append to"
        when _PASTE       msg = "Paste from"
        when _PASTEOVER   msg = "Paste Over from"
        when _CUT         msg = "Cut to"
        when _CUTAPPEND   msg = "Cut Append to"
        when _DELSCRATCH  msg = "Delete which"
    endcase
    if ask(msg + " ClipBoard:", BufferName, named_clip_hist) and Length(BufferName)   // get scratch name
        BufferName = "+++" + BufferName         // Fudge for scratch
        id = GetBufferId(BufferName)             // See if already there
        if operation <> _PASTE and id == 0
            cid = GetBufferId()
            id = CreateBuffer(BufferName, _SYSTEM_)    // create a buffer
            GotoBufferId(cid)
        endif
        if id <> 0                              // if it worked
            if operation == _DELSCRATCH
                AbandonFile(id)
            else
                Set(ClipBoardId, id)                  // new ClipBoard
                case operation
                    when _COPY       Copy()
                    when _COPYAPPEND Copy(_APPEND_)
                    when _PASTE      Paste()
                    when _PASTEOVER  Paste(_OVERWRITE_)
                    when _CUT        Cut()
                    when _CUTAPPEND  Cut(_APPEND_)
                endcase
                Set(ClipBoardId, SaveClipBoardId)     // restore ClipBoard
                ok = TRUE
            endif
        else
            warn("Could not create/find buffer")
        endif
    endif
    return ()
end

integer proc BrowseModeMenuFlags()
    return (iif(BrowseMode(), _MF_GRAYED_, _MF_CLOSE_BEFORE_))
end

Menu NamedClipBoardMenu()
    history
    width = 18

    "Cu&t..."           ,   nameclip(_CUT),             BrowseModeMenuFlags()
    "C&ut Append..."    ,   nameclip(_CUTAPPEND),       BrowseModeMenuFlags()
    "&Copy..."          ,   nameclip(_COPY),            _MF_CLOSE_BEFORE_
    "Cop&y Append..."   ,   nameclip(_COPYAPPEND),      _MF_CLOSE_BEFORE_
    ""                  ,   ,                           Divide
    "&Paste..."         ,   nameclip(_PASTE),           BrowseModeMenuFlags()
    "&Paste &Over..."   ,   nameclip(_PASTEOVER),       BrowseModeMenuFlags()
    ""                  ,   ,                           Divide
    "&Empty..."         ,   nameclip(_DELSCRATCH),      _MF_CLOSE_BEFORE_
end

/****************************************************************************
  Main routine.

  If called from pulldown menus, and command successful, try to clear the
  pulldown menus by pushing escape on the stack, twice.

  Assumptions:

    If MacroCmdLine is empty, we're called from the pulldown menus.

    When called from the pulldown menus, there are two menus on the screen.
 ****************************************************************************/
proc main()
    integer close_menus
    string cmdline[128]

    cmdline = Query(MacroCmdLine)
    if cmdline[1] == '-'
        case Lower(cmdline[2:sizeof(cmdline) - 1])
            when "cut"          nameclip(_CUT)
            when "cutappend"    nameclip(_CUTAPPEND)
            when "copy"         nameclip(_COPY)
            when "copyappend"   nameclip(_COPYAPPEND)
            when "paste"        nameclip(_PASTE)
            when "pasteover"    nameclip(_PASTEOVER)
            when "delete"       nameclip(_DELSCRATCH)
        endcase
    else
        close_menus = cmdline == ""
        ok = FALSE

        NamedClipBoardMenu(cmdline)
        if close_menus and ok
            PushKey(<Escape>)
            PushKey(<Escape>)
        endif
    endif
end

string here_before[] = "NAMECLIP:HereBefore"

proc WhenLoaded()
    if named_clip_hist == 0
        named_clip_hist = GetFreeHistory("NAMECLIP:NamedClipboards")
    endif
    if NOT GetGlobalInt(here_before)
        SetGlobalInt(here_before, 1)
        DelHistory(named_clip_hist)
    endif
end

