/*------------------------------------------------------------------------
  *ini*:ini-filename

  Need to save/restore current buffer

  In general, if a function does not work, return an empty MacroCmdLine

  25 Jul 2019
  TODO:
  we need to update the changed profiles.
  Windows appears to do it pretty quickly.
  Idea:

  store all the changed profiles in a buffer
  hook on idle, when purged, and exit.
  If any of those files are changed, save then, and remove them from the buffer.

  Or maybe somehow use global variables?

  Or, do we add "property lists" to TSE?

  Bugs:
  1 in set_profile_name(), if name was quoted, caused problems.
    fix - remove the quotes in set_profile_name()

  2 If section names are pre-quoted, e.g.: [foo], then causes problem.
    Fix - in init(), don't add the [] if already there.

  3 26 Mar 2020  6:25 am SEM get_next_profile_section_name() should not include
    the "[]" Fixed - in get_next_profile_section_name(), strip them.  Reported
    by Carlo Hogeveen.

  4 12/30/2021  8:36 am SEM flush_profile() does not restore the current buffer.
    Reported by Carlo Hogeveen.

 ------------------------------------------------------------------------*/
string profile[MAXSTRINGLEN]
string arg1[MAXSTRINGLEN]
string arg2[MAXSTRINGLEN]
string arg3[MAXSTRINGLEN]
string glob_ok[] = "ok"
//debug: string save_cmdline[MAXSTRINGLEN]

integer cur_bid, // current editing buffer
        prof_id, // id of the profile
        pi_bid,  // id of the item names
        ps_bid,  // id of the section names
        prof_buf // id of list of changed profile buffers

string  prefix[] = "<*ini*>:"

string proc UnQuote(string s0, string c1, string c2)
    string s[MAXSTRINGLEN] = s0

    if s[1] == c1
        s = DelStr(s, 1, 1)
    endif
    if s[Length(s)] == c2
        s = DelStr(s, Length(s), 1)
    endif
    return (s)
end

string proc Quote(string s0, string c1, string c2)
    string s[MAXSTRINGLEN] = s0

    if s[1] <> c1
        s = c1 + s
    endif

    if s[Length(s)] <> c2
        s = s + c2
    endif
    return (s)
end

proc create_section(string section)
    EndFile()
    AddLine(section)
end

proc save_profile()
    string fn[_MAX_PATH_]
    integer id

    if NumLines() > 0
        BegFile()
        fn = GetText(1, MAXSTRINGLEN)
        KillLine()
        id = GetBufferId(prefix + fn)

        PushLocation()
        if id == 0
            Warn("save_profile, profile not found:"; fn)
        else
            GotoBufferId(id)
            SaveAs(fn, _OVERWRITE_|_DONT_PROMPT_)
        endif
        PopLocation()

    endif
end

proc save_profiles()
    PushLocation()
    GotoBufferId(prof_buf)
    while NumLines() > 0
        save_profile()
    endwhile
    PopLocation()
end

proc save_one_profile()
    PushLocation()
    GotoBufferId(prof_buf)
    if NumLines() > 0
        save_profile()
    else
        UnHook(save_one_profile)
    endif
    PopLocation()
end

proc queue_update()
    PushLocation()
    GotoBufferId(prof_buf)
    if not lFind(profile, "gi^$")
        AddLine(profile)
    endif
    PopLocation()
    Hook(_IDLE_, save_one_profile)
end

/*------------------------------------------------------------------------
   Cursor is at the beginning of a section name, e.g.
   [foo]
   Marks the section, including the section name.
   Leaves a block on the stack; callers should pop it.
 ------------------------------------------------------------------------*/
integer proc mark_section()
    if not lFind(arg1, "^giw")
        return (FALSE)
    endif
    PushBlock()
    PushLocation()
    MarkLine()
    EndLine()
    // look for next section
    if lFind("[", "^")
        Up()                // don't include it in the block
    else
        EndFile()
    endif
    MarkLine()
    PopLocation()
    return (TRUE)
end

/*------------------------------------------------------------------------
  Load the passed args; load the .ini file
 ------------------------------------------------------------------------*/
integer proc init(integer args, integer sections)
    string s1[MAXSTRINGLEN]
    string s2[MAXSTRINGLEN]
    string s3[MAXSTRINGLEN]

    //debug: save_cmdline = Query(MacroCmdLine)

    s1 = GetFileToken(Query(MacroCmdLine), 1)
    s2 = GetFileToken(Query(MacroCmdLine), 2)
    s3 = GetFileToken(Query(MacroCmdLine), 3)

    if args > 0 and sections > 0
        arg1 = Quote(s1, "[", "]")
        sections = sections - 1
        args = args - 1
    elseif args > 0
        arg1 = s1
        args = args - 1
    endif

    if args > 0 and sections > 0
        arg2 = Quote(s2, "[", "]")
        sections = sections - 1
        args = args - 1
    elseif args > 0
        arg2 = s2
        args = args - 1
    endif

    if args > 0 and sections > 0
        arg3 = Quote(s3, "[", "]")
        sections = sections - 1
        args = args - 1
    elseif args > 0
        arg3 = s3
        args = args - 1
    endif

    Set(MacroCmdLine, "")

    cur_bid = GetBufferId()
    prof_id = GetBufferId(prefix + profile)
    if prof_id <> 0
        GotoBufferId(prof_id)
    else
        prof_id = CreateBuffer(prefix + profile, _HIDDEN_)
        if not LoadBuffer(profile)
            //debug: warn("init: could not load:"; profile)
        endif

        FileChanged(FALSE)
    endif
    return (prof_id <> 0)
end

proc cleanup()
    GotoBufferId(cur_bid)
end

proc common_multi_stuff(var integer bid)
    integer nlines

    nlines = NumLines()

    cleanup()

    if nlines == 0
        AbandonFile(bid)
        bid = 0
    endif

    // If nothing there, if a bid, must be "find first" caller, return ok
    if Query(MacroCmdLine) == "" and bid <> 0
        Set(MacroCmdLine, glob_ok)
    endif
end

/*------------------------------------------------------------------------
  Get the next line from buf bid
  Handle errors
 ------------------------------------------------------------------------*/
proc get_next_common(var integer bid)
    Set(MacroCmdLine, "")

    if not GotoBufferId(bid)
        return ()
    endif

    if NumLines() > 0
        BegFile()
        Set(MacroCmdLine, GetText(1, MAXSTRINGLEN))
        KillLine()
    endif

    common_multi_stuff(bid)
end

/*************************************************************************
  Externally callable routines from here on
 *************************************************************************/

/*------------------------------------------------------------------------
  profile filename passed on command line
 ------------------------------------------------------------------------*/
protected proc set_profile_name()
    profile = UnQuote(Query(MacroCmdLine), '"', '"')
    Set(MacroCmdLine, glob_ok)
end

/*------------------------------------------------------------------------
  arg1 = section name
  arg2 = item name
 ------------------------------------------------------------------------*/
protected proc get_profile_str()
    integer ok

    init(2, 1)

    if not mark_section()
        UpdateDisplay()
        //debug: warn("get_profile_str: could not find section"; arg1; "in"; profile; "cmdl"; save_cmdline)
        cleanup()
        return ()
    endif

    ok = lFind(arg2, "^lgiw")
    PopBlock()

    if not ok
        //debug: warn("get_profile_str: could not find item"; arg2)
        cleanup()
        return ()
    endif

    if not lFind("=", "c")
        cleanup()
        return ()
    endif

    Right()
    // skip spaces
    while Currchar() >= 0 and CurrChar() == Asc(" ")
        Right()
    endwhile

    Set(MacroCmdLine, GetText(CurrPos(), MAXSTRINGLEN))
    cleanup()
end

/*------------------------------------------------------------------------
  arg1 = section name
  arg2 = item name
  arg3 = value

  [section]  x  - add it

  [section]
             x  - insert it

  ***changes profile
 ------------------------------------------------------------------------*/
protected proc write_profile_str()
    integer ok, found, line

    init(3, 1)
    found = FALSE

    line = 0
    if not mark_section()
        create_section(arg1)
    else
        ok = lFind(arg2, "^lgiw")
        line = Query(BlockEndLine)
        PopBlock()
        if ok
            if not lFind("=", "c")
                KillLine()
                // so we can do addline below - see case 1
                Up()
            else
                Right()
                KillToEol()
                InsertText(arg3)
                found = TRUE
            endif
        endif
    endif

    if not found
        if line > 0
            GotoLine(line)
        endif
        AddLine(arg2 + "=" + arg3)
    endif
    queue_update()
    Set(MacroCmdLine, glob_ok)
    cleanup()
end

/*------------------------------------------------------------------------
  arg1 = section name

  ***changes profile
 ------------------------------------------------------------------------*/
protected proc remove_profile_section()
    init(1, 1)

    if not mark_section()
        //debug: warn("remove_profile_section: could not find section"; arg1)
        cleanup()
        return ()
    endif

    KillBlock()
    PopBlock()

    queue_update()
    Set(MacroCmdLine, glob_ok)
    cleanup()
end

/*------------------------------------------------------------------------
  arg1 = section name
  arg2 = item name

  ***changes profile
 ------------------------------------------------------------------------*/
protected proc remove_profile_item()
    integer ok

    init(2, 1)

    if not mark_section()
        //debug: warn("remove_profile_item: could not find section"; arg1)
        cleanup()
        return ()
    endif

    ok = lFind(arg2, "^lgiw")
    PopBlock()

    if not ok
        //debug: warn("remove_profile_item: could not find item"; arg2)
        cleanup()
        return ()
    endif

    KillLine()

    queue_update()
    Set(MacroCmdLine, glob_ok)
    cleanup()
end

/*------------------------------------------------------------------------
  arg1 = old section name
  arg2 = new section name

  ***changes profile
 ------------------------------------------------------------------------*/
protected proc rename_profile_section()
    init(2, 2)

    if not lFind(arg1, "^giw")
        //debug: warn("rename_profile_section: could not find section"; arg1)
        cleanup()
        return ()
    endif

    if not lReplace(arg1, arg2, "cgn")
        cleanup()
        return ()
    endif

    queue_update()
    Set(MacroCmdLine, glob_ok)
    cleanup()
end

/*------------------------------------------------------------------------
  Load the profile_item - pi_bid - buf is item names.
  arg1 = section name
 ------------------------------------------------------------------------*/
protected proc load_profile_section()
    init(1, 1)

    if not lFind(arg1, "^giw")
        //debug: warn("load_profile_section: could not find section"; arg1)
        cleanup()
        return ()
    endif

    mark_section()
    pi_bid = GetWorkBuffer()
    CopyBlock()
    PopBlock()

    // first line should be [section name] - remove it
    BegFile()
    KillLine()

    common_multi_stuff(pi_bid)
end

/*------------------------------------------------------------------------
  Return the next item name from the profile_item buf.
  0 args
 ------------------------------------------------------------------------*/
protected proc get_next_profile_item()
    get_next_common(pi_bid)
end

/*------------------------------------------------------------------------
  Load the profile_section - ps_bid - buf with section names.
  0 args
 ------------------------------------------------------------------------*/
protected proc load_profile_section_names()
    init(0, 0)

    ps_bid = GetWorkBuffer()
    GotoBufferId(prof_id)
    if lFind("[", "g^")
        repeat
            AddLine(GetText(1, MAXSTRINGLEN), ps_bid)
        until not lRepeatFind()
    endif

    GotoBufferId(ps_bid)

    common_multi_stuff(ps_bid)
end

/*------------------------------------------------------------------------
  Return the next section name from the profile_section buf.
  0 args
 ------------------------------------------------------------------------*/
protected proc get_next_profile_section_name()
    string s[255]
    get_next_common(ps_bid)
    s = Query(MacroCmdLine)
    if s[1] == '['
        s = DelStr(s, 1, 1)
    endif
    if s[length(s)] == ']'
        s = DelStr(s, Length(s), 1)
    endif
    Set(MacroCmdLine, s)
end

/*------------------------------------------------------------------------
  0 args
 ------------------------------------------------------------------------*/
protected proc flush_profile()
    init(0, 0)
    SaveAs(profile, _OVERWRITE_|_DONT_PROMPT_)
    cleanup()
end

proc whenloaded()
    PushLocation()
    prof_buf = CreateTempBuffer()
    PopLocation()
    Hook(_ON_ABANDON_EDITOR_, save_profiles)
end

/*************************************************************************
  Unit tests
  SEM - 26 Mar 2020  6:35 am How to run these?  I should have made better
    notes :(
 *************************************************************************/

#if 0
string t_pname[_MAX_PATH_]

proc t_init()
    t_pname = ExpandPath("c:.\test.ini")
end

proc t_remove_from_memory()
    integer id

    id = GetBufferId(prefix + t_pname)
    if id > 0
        AbandonFile(id)
    endif
end

proc t_remove_from_disk()
    EraseDiskFile(t_pname)
end

proc t_create_item(integer s_n, integer i_n, integer v_n)
    Set(MacroCmdLine, Format("section", s_n; "item", i_n; "value", v_n))
    write_profile_str()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_create_item: write_profile_str failed 1"; s_n; i_n; v_n)
        Halt
    endif

    Set(MacroCmdLine, Format("section", s_n; "item", i_n))
    get_profile_str()
    if Query(MacroCmdLine) <> ("value" + str(v_n))
        Warn("t_create_item write_profile_str failed 1.1")
        Halt
    endif
end

proc t_5_sections_3_items(var integer s, var integer i, var integer v)
    // create a profile with 5 sections, each with 3 items
    v = 0
    for s = 1 to 5
        for i = 1 to 3
            v = v + 1
            t_create_item(s, i, v)
        endfor
    endfor
end

/*------------------------------------------------------------------------
tested
[x] set_profile_name()
[x] get_profile_str()
[x] write_profile_str()
[x] remove_profile_section()
[x] remove_profile_item()
[x] rename_profile_section()
[x] load_profile_section()
[x] get_next_profile_item()
[x] load_profile_section_names()
[x] get_next_profile_section_name()
[x] flush_profile()
[x] saving changed profiles
 ------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
  arg1 = section name
  arg2 = item name
  arg3 = value
 ------------------------------------------------------------------------*/
public proc t_write_profile_str()
    string section[7] = "section"

    string item[4]  = "item"
    string item2[5] = "item2"

    string value[6]  = "value"
    string value2[7] = "value2"
    string value3[7] = "value3"

    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // 1 create a new section
    Set(MacroCmdLine, Format(section; item; value))
    write_profile_str()
    if Query(MacroCmdLine) <> glob_ok
        Warn("write_profile_str failed 1")
        Halt
    endif

    Set(MacroCmdLine, Format(section; item))
    get_profile_str()
    if Query(MacroCmdLine) <> value
        Warn("write_profile_str failed 1.1")
        Halt
    endif

    // 2 add to an existing section
    Set(MacroCmdLine, Format(section; item2; value2))
    write_profile_str()
    if Query(MacroCmdLine) <> glob_ok
        Warn("write_profile_str failed 2")
        Halt
    endif

    Set(MacroCmdLine, Format(section; item2))
    get_profile_str()
    if Query(MacroCmdLine) <> value2
        Warn("write_profile_str failed 2.1")
        Halt
    endif

    // 3 update an existing item
    Set(MacroCmdLine, Format(section; item; value3))
    write_profile_str()
    if Query(MacroCmdLine) <> glob_ok
        Warn("write_profile_str failed 3")
        Halt
    endif

    Set(MacroCmdLine, Format(section; item))
    get_profile_str()
    if Query(MacroCmdLine) <> value3
        Warn("write_profile_str failed 3.1")
        Halt
    endif
end

/*------------------------------------------------------------------------
  arg1 = section

  1 - remove middle section
  2 - remove 1st section
  3 - remove last section
 ------------------------------------------------------------------------*/
public proc t_remove_profile_section()
    integer s, i, v
    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // create a profile with 5 sections, each with 3 items
    t_5_sections_3_items(s, i, v)

    // remove middle
    Set(MacroCmdLine, "section" + str(3))
    remove_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_remove_profile_section: 1")
        Halt
    endif

    // remove last
    Set(MacroCmdLine, "section" + str(5))
    remove_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_remove_profile_section: 1.1")
        Halt
    endif

    // remove first
    Set(MacroCmdLine, "section" + str(1))
    remove_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_remove_profile_section: 1.1")
        Halt
    endif

end

/*------------------------------------------------------------------------
  arg1 = section
  first in section
  last in section
  last line of file
 ------------------------------------------------------------------------*/
public proc t_remove_profile_item()
    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // create a profile with 1 section, 5 items
    t_create_item(1, 1, 1)
    t_create_item(1, 2, 2)
    t_create_item(1, 3, 3)
    t_create_item(1, 4, 4)
    t_create_item(1, 5, 5)

    Set(MacroCmdLine, Format("section1"; "item1"))
    remove_profile_item()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_remove_profile_item: 1")
        Halt
    endif

    Set(MacroCmdLine, Format("section1"; "item5"))
    remove_profile_item()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_remove_profile_item: 1.1")
        Halt
    endif
end


/*------------------------------------------------------------------------
  arg1 = old section
  arg2 = new section

  1 - rename middle section
  2 - rename 1st section
  3 - rename last section
 ------------------------------------------------------------------------*/
public proc t_rename_profile_section()
    integer s, i, v
    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // create a profile with 5 sections, each with 3 items
    t_5_sections_3_items(s, i, v)

    // rename middle
    Set(MacroCmdLine, Format("section" + str(3); "newmiddle"))
    rename_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_rename_profile_section: 1")
        Halt
    endif

    // rename last
    Set(MacroCmdLine, Format("section" + str(5); "newlast"))
    rename_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_rename_profile_section: 1.1")
        Halt
    endif

    // rename first
    Set(MacroCmdLine, Format("section" + str(1); "newfirst"))
    rename_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_rename_profile_section: 1.1")
        Halt
    endif
end

public proc t_load_profile_section()
    integer s, i, v
    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // create a profile with 5 sections, each with 3 items
    t_5_sections_3_items(s, i, v)

    Set(MacroCmdLine, "section2")
    load_profile_section()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_load_profile_section: 1")
        Halt
    endif
end

public proc t_get_next_profile_item()
    get_next_profile_item()
    if Query(MacroCmdLine) <> "item1=value4"
        Warn("t_get_next_profile_item: 1:"; Query(MacroCmdLine))
        Halt
    endif

    get_next_profile_item()
    if Query(MacroCmdLine) <> "item2=value5"
        Warn("t_get_next_profile_item: 1.1"; Query(MacroCmdLine))
        Halt
    endif
    get_next_profile_item()
    if Query(MacroCmdLine) <> "item3=value6"
        Warn("t_get_next_profile_item: 1.2"; Query(MacroCmdLine))
        Halt
    endif
end

public proc t_load_profile_section_names()
    integer s, i, v
    t_init()

    t_remove_from_memory()
    t_remove_from_disk()

    Set(MacroCmdLine, t_pname)
    set_profile_name()

    // create a profile with 5 sections, each with 3 items
    t_5_sections_3_items(s, i, v)

    load_profile_section_names()
    if Query(MacroCmdLine) <> glob_ok
        Warn("t_load_profile_section_names: 1")
        Halt
    endif
end

// SEM 26 Mar 2020  6:14 am - Carlo Hogeveen - should not have the []
public proc t_get_next_profile_section_name()
    t_load_profile_section_names()

    get_next_profile_section_name()
    if Query(MacroCmdLine) <> "section1"
        Warn("t_get_next_profile_section_name: 1:"; Query(MacroCmdLine))
        Halt
    endif

    get_next_profile_section_name()
    if Query(MacroCmdLine) <> "section2"
        Warn("t_get_next_profile_section_name: 1.1"; Query(MacroCmdLine))
        Halt
    endif
    get_next_profile_section_name()
    if Query(MacroCmdLine) <> "section3"
        Warn("t_get_next_profile_section_name: 1.2"; Query(MacroCmdLine))
        Halt
    endif
end
#endif
