/***************************************************************
  04 Nov 2009 v.01  Initial release

  A simple function lister for SQL.

  Not intended for other languages (including SAL).

  Usage notes:

  [ctrl g]      displays a function list
  [alt pgdn]    goes to the next function
  [alt pgup]    goes to the previous function

  Once loaded, it automatically enables and disables itself based
  on the current file extension.

  'function' id ['(' {parms} ')'] 'return' id 'is'|'as'
  'procedure' id ['(' {parms} ')'] 'is'|'as'

  tokens:
  '(', ')', id

  find function|procedure
  skip
  get_id
  if accept_paren
    accept_parms
    accept_paren
  end
  if is_function
    expect_return
    expect_id
  end
  expect_is

 14 Mar 2019 SEM: Also accept "deterministic", as in: return varchar2 deterministic is

 ***************************************************************/
constant sym_eof, sym_ident, sym_lparen, sym_num,
    sym_punct, sym_rparen, sym_semi, sym_string

constant lang_none, lang_c, lang_cpp, lang_cs, lang_java

constant context_lines = 4

// string constants
string eol[] = Chr(10)
string b_fn[] = "filename"
string b_changes[] = "changes"
string b_numlines[] = "numlines"
string b_undocount[] = "undocount"

string fun_or_proc[] = "{function}|{procedure}"

// global variables
string ch[1]
string token_str[MAXSTRINGLEN]
string ident[MAXSTRINGLEN]
integer end_of_file
integer sym

integer copy_mode
integer fun_buf
integer fun_line

integer last_msg

forward proc getsym()

integer proc buffer_numlines(integer buf)
    integer nlines

    PushPosition()
    nlines = iif(GotoBufferId(buf), NumLines(), 0)
    PopPosition()
    return (nlines)
end

integer proc get_line_no()
    return (Val(GetText(1, 6)))
end

proc append_printing_char(var string s, string c)
    if c >= ' '
        s = s + c
    endif
end

proc append_one_space(var string s)
    if Length(s) > 0 and s[Length(s)] <> ' '
        s = s + ' '
    endif
end

proc append_char(var string s, string c)
    if c > ' '
        append_printing_char(s, c)
    else
        append_one_space(s)
    endif
end

/***************************************************************
  Scanner
 ***************************************************************/

proc getch()
    if CurrChar() < 0
        ch = eol
    else
        ch = Chr(CurrChar())
    endif
    if not NextChar()
        end_of_file = True
    endif
    if copy_mode
        append_char(token_str, ch)
    endif
end

string proc remove_last_token(string s)
    integer i

    i = Length(s)
    while i > 0 and s[i] == ' '
        i = i - 1
    endwhile

    while i > 0 and s[i] <> ' '
        i = i - 1
    endwhile

    while i > 0 and s[i] == ' '
        i = i - 1
    endwhile

    return (DelStr(s, i + 1, Length(s) - i + 1))
end

// Note: We treat foo.bar as one identifier
integer proc is_ident(string ch)
    if end_of_file
        return (false)
    endif
    return (not (ch in '"', "'", '/', '-', '(', ')', ';', ' ', chr(9), eol))
end

proc read_ident()
    sym = sym_ident
    ident = ""
    repeat
        ident = ident + ch
        getch()
    until end_of_file or not is_ident(ch)
end

proc read_string()
    sym = sym_string
    getch()
    while not end_of_file and ch <> '"' and ch <> eol
        if ch == '\'
            getch()
        endif
        getch()
    endwhile
    getch()
end

proc read_char_lit()
    sym = sym_punct
    getch()             // skip the opening '
    while not end_of_file and ch <> "'" and ch <> eol
        if ch == '\'
            getch()
        endif
        getch()         // skip literal
    endwhile
    getch()             // and skip the closing '
end

proc read_comment()
    integer save_copy_mode

    save_copy_mode = copy_mode
    // remove extraneous comment character
    if copy_mode and token_str <> "" and
            RightStr(token_str, 1) == "/"
        token_str = Trim(DelStr(token_str,
                Length(token_str), 1))
    endif
    copy_mode = False

    getch()
    if ch == '/'                // eol comment
        repeat
            getch()
        until end_of_file or ch == eol
        getch()
        getsym()
    elseif ch == '*'            // multi-line comment
        getch()
        repeat
            while not end_of_file and ch <> '*'
                getch()
            endwhile
            getch()
        until end_of_file or ch == '/'
        getch()
        getsym()
    else
        sym = sym_punct
        getch()
    endif

    copy_mode = save_copy_mode
end

proc read_eol_comment()
    integer save_copy_mode

    save_copy_mode = copy_mode
    // remove extraneous comment character
    if copy_mode and token_str <> "" and
            RightStr(token_str, 1) == "/"
        token_str = Trim(DelStr(token_str,
                Length(token_str), 1))
    endif
    copy_mode = False

    getch()
    if ch == '-'                // eol comment
        repeat
            getch()
        until end_of_file or ch == eol
        getch()
        getsym()
    else
        sym = sym_punct
        getch()
    endif

    copy_mode = save_copy_mode
end

// main scanner routine
proc getsym()
    while not end_of_file and ch <= ' '
        getch()
    endwhile
    if end_of_file
        sym = sym_eof
        return ()
    endif

    case ch
        when '"'                        read_string()
        when "'"                        read_char_lit()
        when '/'                        read_comment()
        when '-'                        read_eol_comment()

        when '(' sym = sym_lparen       getch()
        when ')' sym = sym_rparen       getch()
        when ';' sym = sym_semi         getch()
        otherwise                       read_ident()
    endcase
end

proc clear_fun()
    token_str = ""
    fun_line = 0
end

proc must_be_function(string s0)
    string s[MAXSTRINGLEN]

    s = Trim(s0)
    if RightStr(s, 1) in '{', '/', ':'
        s = LeftStr(s, Length(s) - 1)
    endif

    AddLine(Format(fun_line:6, ': ', s), fun_buf)
    if CurrLine() - last_msg >= 1000
        Message(CurrLine(), ':', NumLines())
        last_msg = CurrLine()
    endif
    clear_fun()
end

integer proc find_matching(integer a, integer b)
    integer count = 1
    while sym <> sym_eof and count > 0
        if sym == a
            count = count + 1
        elseif sym == b
            count = count - 1
        endif
        getsym()
    endwhile
    return (count == 0)
end

proc init()
    if fun_buf == 0
        PushPosition()
        fun_buf = CreateTempBuffer()
        PopPosition()
    endif

    // init
    copy_mode = False
    ch = ' '
    end_of_file = False
    last_msg = 0
    clear_fun()
end

integer proc accept(integer sym0)
    if sym == sym0
        getsym()
        return (True)
    endif
    return (False)
end

integer proc accept_id(string id)
    if sym == sym_ident and EquiStr(ident, id)
        getsym()
        return (True)
    endif
    return (False)
end

proc build_function_list_guts()
    integer is_fun

    loop
        clear_fun()
        if not lFind(fun_or_proc, "ixw")
            break
        endif

        if lFind('--', "cb")
            EndLine()
            goto loop_continue
        endif

        copy_mode = True
        getsym()                // get function|procedure keyword
        is_fun = EquiStr(ident, "function")
        fun_line = CurrLine()
        getsym()                // get id
        if not accept(sym_ident)
            goto loop_continue
        endif
        // ['(' {parms} ')']
        if accept(sym_lparen)
            if not find_matching(sym_lparen, sym_rparen)
                goto loop_continue
            endif
        endif
        // 'return' id
        if is_fun
            if not accept_id("return")
                goto loop_continue
            endif
            if not accept(sym_ident)
                goto loop_continue
            endif
            accept_id("deterministic")
        endif
        copy_mode = False
        // is
        if not accept_id("is") and not accept_id("as")
            goto loop_continue
        endif
        token_str = remove_last_token(token_str)

        must_be_function(token_str)

        loop_continue:
    endloop
end

integer proc build_function_list()
    init()

    if buffer_numlines(fun_buf) > 0 and
        GetBufferStr(b_fn, fun_buf) == CurrFilename() and
        GetBufferInt(b_changes, fun_buf) == FileChanged() and
        GetBufferInt(b_numlines, fun_buf) == NumLines() and
        GetBufferInt(b_undocount, fun_buf) == UndoCount()
        return (fun_buf)
    endif

    SetBufferStr(b_fn, CurrFilename(), fun_buf)
    SetBufferInt(b_changes, FileChanged(), fun_buf)
    SetBufferInt(b_numlines, NumLines(), fun_buf)
    SetBufferInt(b_undocount, UndoCount(), fun_buf)
    EmptyBuffer(fun_buf)

    PushPosition()
    BegFile()

    build_function_list_guts()

    PopPosition()
    if last_msg > 0
        UpdateDisplay(_STATUSLINE_REFRESH_)
    endif
    return (fun_buf)
end

integer proc is_valid_extension()
    return (CurrExt() == ".sql")
end

integer proc get_function_line(integer direction)
    integer buf, start, line

    start = CurrLine()
    buf = build_function_list()
    PushPosition()
    GotoBufferId(buf)
    if direction > 0
        BegFile()
        repeat until get_line_no() > start or not Down()
    elseif direction < 0
        EndFile()
        repeat until get_line_no() < start or not Up()
    endif
    line = get_line_no()
    PopPosition()
    return (line)
end

proc show_function_list()
    integer start_line, cv_id, buf
    string fn[_MAX_PATH_] = CurrFilename()

    start_line = CurrLine()

    // create a "view finds" buffer if not there already
    PushPosition()
    cv_id = Query(ViewFindsId)
    if not GotoBufferId(cv_id)
        cv_id = CreateTempBuffer()
    endif
    EmptyBuffer()
    PopPosition()

    // get the function list buffer
    buf = build_function_list()
    if buffer_numlines(buf) == 0
        Warn("No functions found")
        return ()
    endif

    // copy it to compressed view buffer
    PushPosition()
    GotoBufferId(buf)
    PushBlock()
    MarkLine(1, NumLines())
    GotoBufferId(cv_id)
    CopyBlock()
    UnmarkBlock()
    PopBlock()
    PopPosition()

    // put in the header line
    PushPosition()
    GotoBufferId(cv_id)
    BegFile()
    InsertLine()
    InsertText(Format("File: ", QuotePath(fn)))
    InsertText(Format("  ", NumLines() - 1, " occurrences found"))
    // find the function we're in
    BegFile()
    repeat
        if get_line_no() == start_line
            break
        endif
        if get_line_no() > start_line
            Up()
            break
        endif
    until not Down()
    // and show it
    Set(ViewFindsId, cv_id)
    if ViewFinds()
        KillPosition()
    else
        PopPosition()
    endif
end

public proc next_para()
    integer next, row = CurrRow(),
            room = Query(WindowRows) - CurrRow() - context_lines

    next = get_function_line(1)
    if next <= CurrLine()
        repeat until not Down()
    else
        if room > 0 and next - CurrLine() < room
            Down(next - CurrLine())
        else
            GotoLine(next)
            ScrollToRow(row)
        endif
    endif
end

public proc prev_para()
    integer prev, row = CurrRow(),
            room = CurrRow() - 1 - context_lines

    prev = get_function_line(-1)
    if prev >= CurrLine()
        repeat until not Up()
    else
        if room > 0 and CurrLine() - prev < room
            Up(CurrLine() - prev)
        else
            GotoLine(prev)
            ScrollToRow(row)
        endif
    endif
end

public proc foo()
    getsym()
    warn(ident)
end

proc main()
    if is_valid_extension()
        case Lower(Query(MacroCmdLine))
            when "-nextpara"    next_para()
            when "-prevpara"    prev_para()
            when "-list"        show_function_list()
            otherwise           show_function_list()
        endcase
    endif
end

proc WhenLoaded()
//    Hook(_ON_FILE_SAVE_, on_file_save)
//    Hook(_ON_FILE_QUIT_, on_file_quit)
end

