/***************************************************************
  29 Jan 2007 v.01  Initial release

  30 Jan 2007 v.02  Recognize 'catch' keyword:
                    catch() { --not a function
                    '[]' can occur in function definitions:
                    ident[] ident() {}

  01 Feb 2007 v.03  Handle character literals
                    Recognize '['ident']' in function
                    definitions:
                    [ident] ident() {}

  02 Feb 2007 v.04  Handle destructor names: ~ident() {}
                    Handle leading underscores.

  02 Feb 2007 v.05  Allow 'const'|'override' in C++ functions:
                    ident() const {}
                    Recognize '__except' keyword:
                    __except() { --not a function

  03 Dec 2007 v.06  Allow multi-character char literals
                    e.g., instead of just: 'a', allow 'abc'
                    Thanks to Chris Antos for the fix.

  07 May 2008 v.07  Allow old-style C function declarations.

  20 Jun 2016 v.08  Added "each" to keywords, thanks to Joe Souza.

  16 Oct 2020 v.09  Allow 'noexcept'|'try' in C++ functions:
                    ident() noexcept {}

  A simple function lister for C/C++/C#/Java.

  Not intended for other languages (including SAL).

  Usage notes:

  [ctrl g]      displays a function list
  [alt pgdn]    goes to the next function
  [alt pgup]    goes to the previous function

  Once loaded, it automatically enables and disables itself based
  on the current file extension.
 ***************************************************************/
constant sym_amp, sym_backslash, sym_colon, sym_eof, sym_eol,
    sym_equal, sym_gtr, sym_hash, sym_ident, sym_lbrace,
    sym_lbrak, sym_lparen, sym_lss, sym_num, sym_punct,
    sym_rbrace, sym_rbrak, sym_rparen, sym_semi, sym_star,
    sym_string, sym_tilde, sym_comma

constant lang_none, lang_c, lang_cpp, lang_cs, lang_java

constant context_lines = 4

// string constants
string eol[] = Chr(10)
string b_fn[] = "filename"
string b_changes[] = "changes"
string b_numlines[] = "numlines"
string b_undocount[] = "undocount"

// global variables
string ch[1], last_ch[1], next_to_last_ch[1]
string token_str[MAXSTRINGLEN]
string ident[MAXSTRINGLEN]
string sym_str[MAXSTRINGLEN]
integer end_of_file
integer sym

integer copy_mode
integer fun_buf
integer fun_line
integer language

integer last_msg

forward proc getsym()

integer proc buffer_numlines(integer buf)
    integer nlines

    PushPosition()
    nlines = iif(GotoBufferId(buf), NumLines(), 0)
    PopPosition()
    return (nlines)
end

integer proc get_line_no()
    return (Val(GetText(1, 6)))
end

proc append_printing_char(var string s, string c)
    if c >= ' '
        s = s + c
    endif
end

proc append_one_space(var string s)
    if Length(s) > 0 and s[Length(s)] <> ' '
        s = s + ' '
    endif
end

proc append_char(var string s, string c)
    if c > ' '
        append_printing_char(s, c)
    else
        append_one_space(s)
    endif
end

/***************************************************************
  Scanner
 ***************************************************************/

proc set_sym(integer s)
    sym = s
    case sym
        when sym_amp        sym_str = "amp"
        when sym_backslash  sym_str = "backslash"
        when sym_colon      sym_str = "colon"
        when sym_eof        sym_str = "eof"
        when sym_eol        sym_str = "eol"
        when sym_equal      sym_str = "equal"
        when sym_gtr        sym_str = "gtr"
        when sym_hash       sym_str = "hash"
        when sym_ident      sym_str = "ident"
        when sym_lbrace     sym_str = "lbrace"
        when sym_lbrak      sym_str = "lbrak"
        when sym_lparen     sym_str = "lparen"
        when sym_lss        sym_str = "lss"
        when sym_num        sym_str = "num"
        when sym_punct      sym_str = "punct"
        when sym_rbrace     sym_str = "rbrace"
        when sym_rbrak      sym_str = "rbrak"
        when sym_rparen     sym_str = "rparen"
        when sym_semi       sym_str = "semi"
        when sym_star       sym_str = "star"
        when sym_string     sym_str = "string"
        when sym_tilde      sym_str = "tilde"
        when sym_comma      sym_str = "comma"
    endcase
end

proc getch()
    next_to_last_ch = last_ch
    last_ch = ch

    if CurrChar() < 0
        ch = eol
    else
        ch = Chr(CurrChar())
    endif
    if not NextChar()
        end_of_file = True
    endif
    if copy_mode
        append_char(token_str, ch)
    endif
end

string proc remove_token(string s)
    integer i, n

    n = Length(s)
    i = 1
    while i <= n and s[i] == ' '
        i = i + 1
    endwhile

    while i <= n and s[i] <> ' '
        i = i + 1
    endwhile

    return (DelStr(s, 1, i - 1))
end

proc append_token_str(string tok)
    if token_str == "" or fun_line == 0
        fun_line = CurrLine()
    endif
    if Length(tok) == MAXSTRINGLEN
        token_str = tok
        return ()
    endif

    while Length(token_str) + Length(tok) + 1 > sizeof(token_str)
        token_str = remove_token(token_str)
    endwhile

    token_str = token_str + tok
    if tok <> '~'
        token_str = token_str + " "
    endif
end

/***************************************************************
  Handle floats and trailing modifiers
 ***************************************************************/
integer proc is_numeric(string ch)
    return (isDigit(ch) or (Lower(ch) in '.', 'e','d','l','u'))
end

// Note: We treat foo.bar as one identifier
integer proc is_ident(string ch)
    return (isAlphaNum(ch) or (ch in '_', '.'))
end

proc read_number()
    set_sym(sym_num)
    if ch == '0'
        getch()
        if Lower(ch) == 'x'     // hex
            repeat
                getch()
            until end_of_file or not isHexDigit(ch)
            return ()
        endif
    endif

    while not end_of_file and is_numeric(ch)
        getch()
    endwhile
end

proc read_ident()
    set_sym(sym_ident)
    ident = ""
    repeat
        ident = ident + ch
        getch()
    until end_of_file or not is_ident(ch)
end

proc read_string()
    set_sym(sym_string)
    getch()
    while not end_of_file and ch <> '"'
        if ch == '\'
            getch()
        endif
        getch()
    endwhile
    getch()
end

proc read_char_lit()
    set_sym(sym_punct)
    getch()             // skip the opening '
    while not end_of_file and ch <> "'"
        if ch == '\'
            getch()
        endif
        getch()         // skip literal
    endwhile
    getch()             // and skip the closing '
end

proc read_lss()
    set_sym(sym_lss)
    getch()
    if ch == '='
        set_sym(sym_punct)
        getch()
    endif
end

proc read_gtr()
    set_sym(sym_gtr)
    getch()
    if ch == '='
        set_sym(sym_punct)
        getch()
    endif
end

proc read_comment()
    integer save_copy_mode

    save_copy_mode = copy_mode
    // remove extraneous comment character
    if copy_mode and token_str <> "" and
            RightStr(token_str, 1) == "/"
        token_str = Trim(DelStr(token_str,
                Length(token_str), 1))
    endif
    copy_mode = False

    getch()
    if ch == '/'                // eol comment
        repeat
            getch()
        until end_of_file or ch == eol
        getch()
        set_sym(sym_eol)
    elseif ch == '*'            // multi-line comment
        getch()
        repeat
            while not end_of_file and ch <> '*'
                getch()
            endwhile
            getch()
        until end_of_file or ch == '/'
        getch()
        getsym()
    else
        set_sym(sym_punct)
        getch()
    endif

    copy_mode = save_copy_mode
end

// main scanner routine
proc getsym()
    while not end_of_file and ch <= ' '
        if ch == eol
            set_sym(sym_eol)
            getch()
            return ()
        endif
        getch()
    endwhile
    if end_of_file
        set_sym(sym_eof)
        return ()
    endif

    case ch
        when 'a'..'z', 'A'..'Z', '_'    read_ident()
        when '0'..'9'                   read_number()
        when '<'                        read_lss()
        when '>'                        read_gtr()
        when '"'                        read_string()
        when "'"                        read_char_lit()
        when '/'                        read_comment()

        when '#' set_sym(sym_hash)      getch()
        when '&' set_sym(sym_amp)       getch()
        when '(' set_sym(sym_lparen)    getch()
        when ')' set_sym(sym_rparen)    getch()
        when '*' set_sym(sym_star)      getch()
        when ':' set_sym(sym_colon)     getch()
        when ';' set_sym(sym_semi)      getch()
        when '=' set_sym(sym_equal)     getch()
        when '[' set_sym(sym_lbrak)     getch()
        when '\' set_sym(sym_backslash) getch()
        when ']' set_sym(sym_rbrak)     getch()
        when '{' set_sym(sym_lbrace)    getch()
        when '}' set_sym(sym_rbrace)    getch()
        when '~' set_sym(sym_tilde)     getch()
        when ',' set_sym(sym_comma)     getch()
        otherwise set_sym(sym_punct)    getch()
    endcase
end

proc clear_fun()
    token_str = ""
    fun_line = 0
end

proc must_be_function(string s0)
    string s[MAXSTRINGLEN]

    s = Trim(s0)
    if RightStr(s, 1) in '{', '/', ':'
        s = LeftStr(s, Length(s) - 1)
    endif

    AddLine(Format(fun_line:6, ': ', s), fun_buf)
    if CurrLine() - last_msg >= 1000
        Message(CurrLine(), ':', NumLines())
        last_msg = CurrLine()
    endif
    clear_fun()
end

integer proc is_keyword()
    if ident in
            "break", "case", "continue", "default", "do", "else",
            "enum", "extern", "for", "goto", "if", "return",
            "sizeof", "struct", "switch", "typedef", "union",
            "while"
        return (True)
    endif
    if language in lang_cpp, lang_cs, lang_java
        if ident in "catch", "class", "each", "finally", "foreach",
                "namespace", "new", "throw", "try", "using"
            return (True)
        endif
    endif
    if language in lang_cpp
        if ident in "__except"
            return (True)
        endif
    endif
    return (False)
end

proc handle_macro()
    integer previous_sym

    repeat
        previous_sym = sym
        getsym()
    until sym == sym_eol or sym == sym_eof
    if sym == sym_eol and previous_sym == sym_backslash
        handle_macro()
    endif
    clear_fun()
end

proc handle_ident()
    if is_keyword()
        clear_fun()
    else
        append_token_str(ident)
    endif
    getsym()
end

integer proc find_matching(integer a, integer b)
    integer count = 1
    while sym <> sym_eof and count > 0
        if sym == sym_hash
            handle_macro()
        else
            if sym == a
                count = count + 1
            elseif sym == b
                count = count - 1
            endif
            getsym()
        endif
    endwhile
    return (count == 0)
end

proc skip_eol()
    while sym == sym_eol
        getsym()
    endwhile
end

// handle old style function definitions:
// fun(a, b) int a; int b; {
// For C: ident+ '(' idlist+ ')' { (idlist ';')+ } '{' ...
proc maybe_function(string s)
    integer count, n_idents

    // ??? review this - is it right ???
    if language == lang_cpp
        skip_eol()
        if sym == sym_ident and ident in "const", "override", "noexcept", "try"
            getsym()
        endif
        skip_eol()
        if not (sym in sym_colon, sym_lbrace)
            return ()
        endif
    endif
    /*******************************************************
      Search for opening '{'. If we find '}' or ';' before
      the '{', it can't be a function.  Also handle old style
      C function declarations.
     *******************************************************/
    count = 0

    n_idents = 0
    loop
        case sym
            when sym_lbrak
                count = count + 1
            when sym_rbrak
                count = count - 1
                if count < 0
                    break
                endif
            when sym_lbrace
                must_be_function(s)
                break
            when sym_eof, sym_rbrace
                break
            when sym_semi
                if language <> lang_c
                    break
                endif
                if n_idents < 2
                    break
                endif
                n_idents = 0
            when sym_ident
                if is_keyword()
                    if language <> lang_c
                        break
                    endif
                    if not (ident in "struct", "union")
                        break
                    endif
                endif
                n_idents = n_idents + 1
            when sym_equal
                break
        endcase
        getsym()
    endloop
end

proc handle_lparen()
    integer match_found

    if token_str == ""
        getsym()
        find_matching(sym_lparen, sym_rparen)
    else
        token_str = Trim(token_str) + "("
        append_printing_char(token_str, ch)
        copy_mode = True
        getsym()
        // foo(), or foo(void), is not a function
        match_found = find_matching(sym_lparen, sym_rparen) and sym <> sym_comma
        copy_mode = False
        if language == lang_c and sym == sym_lparen
            getsym()
            find_matching(sym_lparen, sym_rparen)
        elseif match_found and sym <> sym_semi
            maybe_function(token_str)
        endif
    endif
    if sym <> sym_ident
        getsym()            // skip '{', '}' or ';'
    endif
    clear_fun()
end

proc handle_special()
    if token_str <> "" or (language == lang_cs and last_ch == '[')
        token_str = Trim(token_str)
        if next_to_last_ch in ' ', eol
            append_one_space(token_str)
        endif
        token_str = token_str + last_ch
        if ch in ' ', eol
            append_one_space(token_str)
        endif
    endif
    getsym()
end

proc handle_resynch()
    clear_fun()
    getsym()
end

proc handle_colon()
    if language <> lang_c and
            (ident in "public", "private", "protected")
        handle_resynch()
    else
        handle_special()
    endif
end

proc handle_tilde()
    if language == lang_c
        handle_resynch()
    else
        append_token_str(last_ch)
    endif
    getsym()
end

proc init()
    if fun_buf == 0
        PushPosition()
        fun_buf = CreateTempBuffer()
        PopPosition()
    endif

    // init
    copy_mode = False
    ch = ' '
    last_ch = ' '
    end_of_file = False
    last_msg = 0
    clear_fun()
end

integer proc build_function_list()
    init()

    if buffer_numlines(fun_buf) > 0 and
        GetBufferStr(b_fn, fun_buf) == CurrFilename() and
        GetBufferInt(b_changes, fun_buf) == FileChanged() and
        GetBufferInt(b_numlines, fun_buf) == NumLines() and
        GetBufferInt(b_undocount, fun_buf) == UndoCount()
        return (fun_buf)
    endif

    SetBufferStr(b_fn, CurrFilename(), fun_buf)
    SetBufferInt(b_changes, FileChanged(), fun_buf)
    SetBufferInt(b_numlines, NumLines(), fun_buf)
    SetBufferInt(b_undocount, UndoCount(), fun_buf)
    EmptyBuffer(fun_buf)

    PushPosition()
    BegFile()
    getsym()
    while sym <> sym_eof
        case sym
            when sym_ident  handle_ident()
            when sym_lparen handle_lparen()
            when sym_hash   handle_macro()
            when sym_colon  handle_colon()
            when sym_tilde  handle_tilde()

            when sym_amp, sym_gtr, sym_lbrak, sym_lss,
                    sym_rbrak, sym_star
                handle_special()

            when sym_backslash, sym_eol, sym_punct, sym_comma
                getsym()

            when sym_equal, sym_lbrace, sym_num, sym_rbrace,
                    sym_rparen, sym_semi, sym_string
                handle_resynch()

            otherwise
                Warn("Unexpected sym"; sym; "(",
                        CurrLine(); CurrCol(), ")")
                Warn("Please report this problem")
                break
        endcase
    endwhile
    PopPosition()
    if last_msg > 0
        UpdateDisplay(_STATUSLINE_REFRESH_)
    endif
    return (fun_buf)
end

integer proc is_valid_extension()
    case CurrExt()
        when ".c"
            language = lang_c
        when ".cc", ".cpp", ".cxx"
            language = lang_cpp
        when ".cs"
            language = lang_cs
        when ".java"
            language = lang_java
        otherwise
            language = lang_none
    endcase
    return (language <> lang_none)
end

integer proc get_function_line(integer direction)
    integer buf, start, line

    start = CurrLine()
    buf = build_function_list()
    PushPosition()
    GotoBufferId(buf)
    if direction > 0
        BegFile()
        repeat until get_line_no() > start or not Down()
    elseif direction < 0
        EndFile()
        repeat until get_line_no() < start or not Up()
    endif
    line = get_line_no()
    PopPosition()
    return (line)
end

proc show_function_list()
    integer start_line, cv_id, buf
    string fn[_MAX_PATH_] = CurrFilename()

    start_line = CurrLine()

    // create a "view finds" buffer if not there already
    PushPosition()
    cv_id = Query(ViewFindsId)
    if not GotoBufferId(cv_id)
        cv_id = CreateTempBuffer()
    endif
    EmptyBuffer()
    PopPosition()

    // get the function list buffer
    buf = build_function_list()
    if buffer_numlines(buf) == 0
        Warn("No functions found")
        return ()
    endif

    // copy it to compressed view buffer
    PushPosition()
    GotoBufferId(buf)
    PushBlock()
    MarkLine(1, NumLines())
    GotoBufferId(cv_id)
    CopyBlock()
    UnmarkBlock()
    PopBlock()
    PopPosition()

    // put in the header line
    PushPosition()
    GotoBufferId(cv_id)
    BegFile()
    InsertLine()
    InsertText(Format("File: ", QuotePath(fn)))
    InsertText(Format("  ", NumLines() - 1, " occurrences found"))
    // find the function we're in
    BegFile()
    repeat
        if get_line_no() == start_line
            break
        endif
        if get_line_no() > start_line
            Up()
            break
        endif
    until not Down()
    // and show it
    Set(ViewFindsId, cv_id)
    if ViewFinds()
        KillPosition()
    else
        PopPosition()
    endif
end

public proc next_para()
    integer next, row = CurrRow(),
            room = Query(WindowRows) - CurrRow() - context_lines

    next = get_function_line(1)
    if next <= CurrLine()
        repeat until not Down()
    else
        if room > 0 and next - CurrLine() < room
            Down(next - CurrLine())
        else
            GotoLine(next)
            ScrollToRow(row)
        endif
    endif
end

public proc prev_para()
    integer prev, row = CurrRow(),
            room = CurrRow() - 1 - context_lines

    prev = get_function_line(-1)
    if prev >= CurrLine()
        repeat until not Up()
    else
        if room > 0 and CurrLine() - prev < room
            Up(CurrLine() - prev)
        else
            GotoLine(prev)
            ScrollToRow(row)
        endif
    endif
end

proc main()
    is_valid_extension()
    case Lower(Query(MacroCmdLine))
        when "-nextpara"    next_para()
        when "-prevpara"    prev_para()
        when "-list"        show_function_list()
        otherwise           show_function_list()
    endcase
    if sym_str == ""
    endif
end

proc WhenLoaded()
//    Hook(_ON_FILE_SAVE_, on_file_save)
//    Hook(_ON_FILE_QUIT_, on_file_quit)
end

