/*************************************************************************
  Expr        Calculates the value of a specified mathematical expression

  Author:     SemWare

  Date:       Mar 23, 1994
              Apr 18, 1996  Peter Birch: Added Exponentiation - **
              Mar  7, 1997  Peter Birch: Added paste-result ability
              Mar 2004  SEM: force grey-operator keys to
                self-insert - suggested by Ron Perrella.
              Jun 2005 SEM: Use commas with big numbers.
              Mar 2023 Eckhard Hillmann:
                -Reworked the code
                -Added >>, shr, <<, shl, mod, div, not, pow
                -Added copy to Winclip
                -Added Hex input using prefix $ ($ff) Motorola style
                -Changed error-handling, don't terminate and close
                 the macro on error, show message
                -Updated help
                -Added keywords comp, xor, and, or
                -Insert Formatted Decimal result
                -Updated help
              Jul 2023 Eckhard Hillmann
                -Operator % mod: added check for division by 0

  Overview:

  This macro operates as a simple calculator, and handles decimal or
  hexadecimal numbers.  For decimal numbers, only whole numbers are
  accepted.

  Keys:       (none)

  Usage notes:

  This macro does not have any key assignments.  To use, simply select
  it from the Potpourri menu.

  The following operations are available:

            >      SHR      (bitwise shift right)
            <      SHL      (bitwise shift left)
            ~      COMP     (2's complement)
            %      MOD      (modulo division)
            &      AND      (bitwise AND)
            |      OR       (bitwise OR)
            ^      XOR      (bitwise eXclusive-OR)
            +      Addition
            -      Subtraction
            *      Multiplication
            **     Exponentiation--binary (Power)
            /      Division

  Precedence is as in SAL, but can be forced by using parenthesis.

  Hex number may be entered using C format, or ending with an 'h' or
  '$'.  Regardless, the first character _must_ be numeric.
*************************************************************************/

HelpDef ExprHelp
   "The following operators are available within an Expression:"
   ""
   "Operator        Meaning                     Precedence"
   ""
   "    ()     Sub-expression                        1"
   ""
   " ~ comp    Bitwise NOT (2's Complement)          2"
   " ! not     Logic NOT                             2"
   ""
   "    *      Multiplication                        3"
   "** pow     Exponentiation                        3"
   " / div     Division                              3"
   " % mod     Modulo Division                       3"
   " ^ xor     Bitwise eXclusive-OR                  3"
   " > >> shr  Bitwise Shift Right                   3"
   " < << shl  Bitwise Shift Left                    3"
   " & and     Bitwise AND                           3"
   ""
   "    +      Addition                              4"
   "    -      Subtraction                           4"
   " | or      Bitwise OR                            4"
   ""
   "    Alt-P  Option to insert result at cursor or"
   "           to copy it to Windows clipboard."
   ""
   "Hex numbers may be entered by using prefix '$' or '0x'"
   "or using postfix 'h' or '$'."
   ""
   "Binary numbers may be entered using postfix '#'."
   ""
   "         Integerarithmetic (32 Bit) only!"
   ""
end


constant ins_DECIMAL        =   0x01,   // insert the sum in decimal
         ins_HEX            =   0x02,   // insert the sum in hex
         ins_FORMATTED      =   0x04    // insert the formatted decimal result

constant ERROR_SYNTAX = 1,
         ERROR_PARENTHESIS,
         ERROR_DIVISION,
         ERROR_SYMBOL

/*
   These are "magic" values that GetNextToken will store in the global variable
   symbol, to indicate what is the next item coming.
*/
constant
  symNumber           = 0,
  symAddition         = 1,
  symSubtraction      = 2,
  symMultiplication   = 3,
  symDivision         = 4,
  symLeftParen        = 5,
  symRightParen       = 6,
  symAND              = 7,
  symOR               = 8,
  symXOR              = 9,
  symCOMP             = 10,
  symMOD              = 11,
  symSHR              = 12,
  symSHL              = 13,
  symPower            = 14,
  symNot              = 15,
  symEndOfString      = -1,
  symError            = -2

/*
   Work variables
*/

string  InputLine[80]
integer Col               // current column in InputLine
string  Token[40]         // currently parsed token from InputLine
integer symbol            // numeric "code" representing what token is
string  Ch[1]             // next Char from InputLine

integer finalResult
integer bufferId

string  ErrorMessage[40]  // contains the message for display
integer ExprErrorFlag     // TRUE is error

integer hex               // is current symbol hex? used only by Number/isHex

forward integer proc SimpleExpression()
forward string proc InsertCommas(string s)

proc GetNextChar()
  Ch  = Upper(SubStr(InputLine, col, 1))
  col = col + 1
end

integer proc isNumeric()
  return (Ch >= '0' and Ch <= '9')
end

integer proc isHexChar()
  if Ch >= 'A' and Ch <= 'F'
    hex = TRUE
    return (TRUE)
  endif

  return (FALSE)
end

// **************************************************************************
// Peter Birch  04/18/96
// my own cheesy pow() function
// overflows are easy with any exponent larger than 30
//
integer proc pow(integer base, integer exponent)
  integer iRetval, save_exponent, check

  save_exponent = exponent
  iRetval = base

  while exponent > 1
    iRetval = iRetval * base
    exponent = exponent - 1
  endwhile

  check = iRetval
  exponent = save_exponent
  while exponent > 1
    check = check / base
    exponent = exponent - 1
  endwhile

  if check <> base
    Warn("Overflow! ", base, " ** ", save_exponent, " : Result is too large")
  endif

  return (iRetval)
end

proc SetError(integer ErrorCode)
  if ExprErrorFlag                    // set error only once
    return()
  endif

  case ErrorCode
  when ERROR_SYNTAX
    ErrorMessage = "Expression syntax error"

  when ERROR_PARENTHESIS
    ErrorMessage = "Missing right parenthesis"

  when ERROR_DIVISION
    ErrorMessage = "Division by 0"

  when ERROR_SYMBOL
    ErrorMessage = "Wrong or missing symbol"

  otherwise
    ErrorMessage = "Unspecified error"

  endcase

  warn(ErrorMessage)
  ExprErrorFlag = TRUE                // oops, error, red alert...
end

// Number _must_ set hex to FALSE on startup and exit.
proc Number()
  Token = Ch
  hex   = FALSE

  if not isNumeric() and not isHexChar() and Ch <> "$"
    symbol = symError
  else
    if Ch == "$"                      // Motorola style hex, $ is prefix ($ff)
      hex   = TRUE                    // number is Hex
      Token = ""                      // reset Token
    endif

    symbol = symNumber                // Tag this as an integer value
    GetNextChar()

    if (Ch == "X")                    // Allow c-style hex
      hex = TRUE
      GetNextChar()
    endif

    while isNumeric() or isHexChar()
      Token = Token + Ch
      GetNextChar()
    endwhile

    if Ch == 'H' or Ch == '$'         // Allow assembly/Turbo Pascal hex strings
      hex = TRUE
      GetNextChar()
    elseif Ch == '#'
      GetNextChar()
      Token = Token + 'B'
    endif

    if hex
      Token = Token + 'H'
    endif
  endif

  hex = FALSE
end

// GetNextToken
proc GetNextToken()

  while Ch == " "                     //  Skip over any leading blanks
    GetNextChar()
  endwhile

  case Ch
  when ">"                            // Shift Right
    symbol = symSHR
    if equistr(">>", substr(InputLine, col - 1, 2))
      col = col + 1
    endif

  when "<"                            // Shift Left
    symbol = symSHL
    if equistr("<<", substr(InputLine, col - 1, 2))
      col = col + 1
    endif

  when "S"                            // Shift Right or Shift Left
    if equistr("SHR", substr(InputLine, col - 1, 3))
      symbol = symSHR
      col    = col + 2
    elseif equistr("SHL", substr(InputLine, col - 1, 3))
      symbol = symSHL
      col    = col + 2
    else
      symbol = symError
    endif

  when "~"                            // Bitwise NOT (2's Complement)
    symbol = symCOMP

  when "C"                            // Bitwise NOT (2's Complement)
    if equistr("COMP", substr(InputLine, col - 1, 4))
      symbol = symCOMP
      col    = col + 3
    else                              // might be Hex without prefix
      Number()
      return()
    endif

  when "%"                            // Modulo
    symbol = symMOD

  when "M"                            // Modulo
    if equistr("MOD", substr(InputLine, col - 1, 3))
      symbol = symMOD
      col    = col + 2
    else
      symbol = symError
    endif

  when "&"                            // And
    symbol = symAND

  when "A"                            // And
    if equistr("AND", substr(InputLine, col - 1, 3))
      symbol = symAND
      col    = col + 2
    else                              // might be Hex without prefix
      Number()
      return()
    endif

  when "|"                            // Or
    symbol = symOR

  when "O"                            // Or
    if equistr("OR", substr(InputLine, col - 1, 2))
      symbol = symOR
      col    = col + 1
    else
      symbol = symError
    endif

  when "^"                            // Xor
    symbol = symXOR

  when "X"                            // Xor
    if equistr("XOR", substr(InputLine, col - 1, 3))
      symbol = symXOR
      col    = col + 2
    else
      symbol = symError
    endif

  when "+"                            // Addition
    symbol = symAddition

  when "-"                            // Subtraction
    symbol = symSubtraction

  when "*"                            // Power or Multiplication
    if equistr("**", substr(InputLine, col - 1, 2))
      symbol = symPower
      col    = col + 1
    else
      symbol = symMultiplication
    endif

  when "P"                            // Power
    if equistr("POW", substr(InputLine, col - 1, 3))
      symbol = symPower
      col    = col + 2
    else
      symbol = symError
    endif

  when "/"                            // Division
    symbol = symDivision

  when "D"                            // Division
    if equistr("DIV", substr(InputLine, col - 1, 3))
      symbol = symDivision
      col    = col + 2
    else                              // might be Hex without prefix
      Number()
      return()
    endif

  when "("
    symbol = symLeftParen

  when ")"
    symbol = symRightParen

  when "!"                            // Logic NOT
    symbol = symNot

  when "N"                            // Logic NOT
    if equistr("NOT", substr(InputLine, col - 1, 3))
      symbol = symNot
      col    = col + 2
    else
      symbol = symError
    endif

  when ""                             // The End
    symbol = symEndOfString

  otherwise                           // if token is numeric, collect rest of digits
    Number()
    return()

  endcase

  GetNextChar()
end

integer proc Factor()
  integer result, base = 10

  case symbol
  when symNumber                      // found a number
    if Token[Length(Token)] == 'H'
      Token = substr(Token, 1, Length(Token) - 1)
      base  = 16
    elseif Token[Length(Token)] == 'B'
      Token = substr(Token, 1, Length(Token) - 1)
      base  = 2
    endif

    result = Val(Token, base)         // get its value
    GetNextToken()                    // and skip to next

  when symLeftParen                   // found a left paren
    GetNextToken()                    // skip over "("
    result = SimpleExpression()

    if symbol <> symRightParen        // make sure the right paren exists
      SetError(ERROR_PARENTHESIS)
    endif

    GetNextToken()                    // skip over ")"

  when symCOMP                        // Bitwise NOT (2's Complement)
    GetNextToken()
    result = ~ Factor()

  when symNot                         // Logic NOT
    GetNextToken()
    result = not Factor()

  when symSubtraction                 // Subtraction
    GetNextToken()
    result = - Factor()

  otherwise
    SetError(ERROR_SYNTAX)            // not number or paren--error

  endcase

  return (result)
end

integer proc Term()
  integer result, sym, temp

  result = Factor()

  while (symbol == symMultiplication) or  // "*"  sign, or
        (symbol == symDivision)       or  // "/"  sign...
        (symbol == symPower)          or  // "**" sign...
        (symbol == symMOD)            or  // "%"  sign...
        (symbol == symXOR)            or  // "^"  sign...
        (symbol == symSHR)            or  // ">"  sign...
        (symbol == symSHL)            or  // "<"  sign...
        (symbol == symAND)                // "&"  sign...
    sym = symbol
    GetNextToken()                    // skip over math op

    case sym
    when symMultiplication
      result = result * Factor()      // multiplication

    when symPower                     // Power
      result = pow(result, Factor())

    when symDivision                  // Division
      temp = Factor()
      if temp <> 0
        result = result / temp
      else
        SetError(ERROR_DIVISION)
      endif

    when symAND                       // And
      result = result & Factor()

    when symXOR                       // Xor
      result = result ^ Factor()

    when symMOD                       // Modulo
      temp = Factor()
      if temp <> 0
        result = result mod temp
      else
        SetError(ERROR_DIVISION)
      endif

    when symSHR                       // Shift Right
      result = result shr Factor()

    when symSHL                       // Shift Left
      result = result shl Factor()

    endcase
  endwhile

  if symbol == symError
    SetError(ERROR_SYMBOL)
  endif

  return (result)
end

integer proc SimpleExpression()
  integer result, sym

  result = Term()

  while ((symbol == symNumber     ) or  // another integer coming, or
         (symbol == symOR         ) or  // "|" sign, or
         (symbol == symAddition   ) or  // "+" sign, or
         (symbol == symSubtraction)     // "-" sign....
        ) and (not ExprErrorFlag  )     // as long as no error came up

    sym = symbol
    if sym == symNumber               // if another integer came immediately
      sym = symAddition               // then default to adding
    else
      GetNextToken()                  // else skip over the math op
    endif

    case sym
    when symAddition                  // Addition
      result = result + Term()

    when symSubtraction               // Substraction
      result = result - Term()

    when symOR                        // Or
      result = result | Term()

    endcase
  endwhile

  if symbol == symError
    SetError(ERROR_SYMBOL)
  endif

  return (result)
end

// Added by PLB 03/07/97
proc InsertExpr(integer flags)
  integer iPromptId

  iPromptId = getbufferId()
  gotoBufferId(bufferId)
  if flags & ins_FORMATTED
    InsertText(InsertCommas(Str(finalResult)), _INSERT_)
  else
    InsertText(iif(flags & ins_DECIMAL, str(finalResult, 10), str(finalResult, 16) + "h"), _INSERT_)
  endif
  gotoBufferId(iPromptId)
end

proc CopyExpr(integer flags)
  if flags & ins_FORMATTED
    CopyToWinClip(InsertCommas(Str(finalResult)))
  else
    CopyToWinClip(iif(flags & ins_DECIMAL, str(finalResult, 10), str(finalResult, 16) + "h"))
  endif
end

Menu InsertExprMenu()
  title = "Insert Result at Cursor"
  history

  "&Decimal Result"           , InsertExpr(ins_DECIMAL)
  "&Formatted Decimal Result" , InsertExpr(ins_FORMATTED)
  "&Hex Result"               , InsertExpr(ins_HEX)
  "Copy to Windows clipboard",, _MF_DIVIDE_
  "D&ecimal Result"           , CopyExpr(ins_DECIMAL)
  "F&ormatted Decimal Result" , CopyExpr(ins_FORMATTED)
  "He&x Result"               , CopyExpr(ins_HEX)
end

keydef ExprHelpKey
  <F1>            QuickHelp(ExprHelp)
  <Alt p>         InsertExprMenu()
  <grey/>         SelfInsert()
  <grey*>         SelfInsert()
  <grey->         SelfInsert()
  <grey+>         SelfInsert()
end

proc PromptStartup()
  Enable(ExprHelpKey)
  UnHook(PromptStartup)
end

proc PromptCleanup()
  Disable(ExprHelpKey)
  UnHook(PromptCleanup)
end

// Assumes input string has no leading zeroes...
// Returns string with formatting commas inserted
string proc InsertCommas(string s)
  string  ret[26] = s
  integer sign = iif(s[1] == "-", 1, 0)
  integer idx = ((length(s) - sign) mod 3) + 1

  if length(s) - sign > 3
    while idx < length(ret)
      if idx > 1
        ret = InsStr(",", ret, idx + sign)
        idx = idx + 4
      else
        idx = idx + 3
      endif
    endwhile
  endif

  return (ret)
end

//
//  Set up globals so SimpleExpression starts off on the right foot
//
proc Main()
  integer from_cmdline, hist
  bufferId = getBufferId()

  hist = GetFreeHistory("expr:main")

  InputLine    = Trim(Query(MacroCmdLine))
  from_cmdline = Length(InputLine) > 0

  loop
    if not from_cmdline
      Hook(_PROMPT_STARTUP_, PromptStartup)
      Hook(_PROMPT_CLEANUP_, PromptCleanup)

      if not Ask("Enter Expression: (<F1> for help)", InputLine, hist) or Length(InputLine) == 0
        break
      endif
    endif

    ExprErrorFlag = FALSE             // reset error
    ErrorMessage  = ""                // no message

    // Prime the pump...
    Col   = 1
    Token = ''
    GetNextChar()
    GetNextToken()

    finalResult = SimpleExpression()

    if ExprErrorFlag
      Message(ErrorMessage)
      finalResult = 0
    else
      Message("The result is: ", InsertCommas(Str(finalResult)), " (", str(finalResult, 16), "h)", " (", str(finalResult, 2), "b)")
    endif

    if from_cmdline
      break
    endif
  endloop
end

