/*************************************************************************
  CUAMark     Enables <Shift>+cursor-keys to emulate CUA-style block marking

  Author:     SemWare
              Additional help from Gary Weinfurther and Bradley Small.

  Date:       Apr  5, 1994 - Initial version
              Jun 21, 1994 - Enhancements and improvements
              Jan 30, 1997 - Rewrite to be more CUA compliant
                             Make blocks non-persistent
                             Use the Windows Clipboard
                             Make Cursor movement sensitive to block
                             Toggle Cut/Copy/Paste/Delete keys
                             Block mark found text

              Mar 11, 1997 - Rewrite using a different method, which allows us
                             to more closely emulate Windows shift marking
                             behavior.

              Mar 12, 1997 - Fix bug reported by Chris Antos - if no block
                             marked, EquateEnhancedKbd off, <grey del>
                             pressed, get macro stack overflow.

              Jul 12, 1997 - Fix bug reported by Bill Stewart - if a twokey
                             command is executed while a block is being
                             marked, the block is deleted.  Fix was to
                             recognize that we're in a twokey in the
                             _AFTER_GETKEY_ hook.

              Jul 18, 1997 - Fix bug reported by Bill Stewart - if marking
                             right to left, the typing replaces block feature
                             doesn't work correctly - needs to backspace
                             first.  Thanks to Bill for the suggested fix.

              Jul 23, 1997 - Fix problem with Find.  Using repeat find would
                             leave Find block marked, causing the cursor to
                             jump when pressing a cursor key after doing
                             repeat find. Wrote wrapper for RepeatFind that
                             cleans up the previous mark, and marks new text
                             (if found). Also, fixed problem where trying to
                             continue marking after a find would lose current
                             block.

              Aug  5, 1997 - did not co-exist with LineDrawing.  Fix is to
                             ignore cursor movement keys when LineDrawing is
                             on.

              Jan  8, 1998 - RepeatFind backwards did not work.  This is
                             because of placing the cursor at the end of the
                             found text.  Fix is to leave the cursor at the
                             beginning of the found text.

                           - On initial load, see whether user wants win95
                             style cuamark keys enabled.  Save in tse.ini.

              Jan 21, 1998 - Did not co-exist with other macros that, while a
                             block is marked, do a GetKey() and want to check
                             the non-shifted cursor keys.  This can now be
                             handled by the macro in question setting Marking
                             off, and then restoring it once the GetKey() is
                             complete.

                             However, there is currently no solution for the
                             shifted cursor keys, other than purging cuamark.

              Mar  2, 1998 - Some people don't like some of the CUA
                             behavior, such as the new CUA cut/copy/past
                             keys (ctrl x, ctrl c, ctrl v) and the behavior
                             of find marking the selected text.  On initial
                             load, this macro will configure itself (using
                             tse.ini).

              Aug  6, 1998 - SEM:  Rework AfterGetKey logic slightly, to work
                             a little better with other macros that use
                             GetKey().  If we notice that we are not 'shift
                             marking', for regular cursor keys, just bail
                             out, allowing the higher up's to handle it.

                             Also, fix/cleanup what happens when a
                             non-shifted cursor key is pressed after a block
                             has been marked.

              Aug 19, 1999 - SEM: Correct misspellings <sigh>

              Jan 25, 2001 - SEM: Fix problem with hard tabs, relating to
                             BlockBegCol vs. CurrCol() vs. CurrPos().  Thanks
                             to Bruce Riggins for finding the problem.

              Jul  3, 2001 - SEM: Add command line option: -setup
                             Forces cuamark to configure itself.

              Jun 20, 2002 - SEM: on setup, force 2nd option default to
                             no.  Also, if already configured, set yes
                             no appropriately.

              Dec 4, 2002 -  SEM: respect the editors
                             UseCurrLineIfNoBlock variable.

              Oct    2003 -  SEM: respect the editors UnmarkAfterPaste
                             variable.

                             Add support for column blocks, using <alt>
                             in additional to shift cursor keys.

                             Add support for cua-style marking using the
                             mouse left-button, and the mouse
                             shift-left-button keys.

              Jan    2004 -  SEM: more work on the <del>/<greydel> problem.

              2006 May 05 -  [WAS] First stab at enabling prompt box marking.
                             Changed <CtrlShift Home> and <CtrlShift End> to
                             do BegFile/EndFile.

              2006 May 08 -  [WAS] Second try. This one replaces the
                             editing_key variable with editing_state and
                             eliminates the AfterNoneditCommand function:
                             QueryEditState() returns 0x3000 in a prompt box.

              2006 May 10 -  [WAS] Third try for 4.40.16. Uses the
                             _STATE_PROMPTED_ constant instead.

              2006 May 11 -  [WAS] Updated for 4.40.17. Changed
                             <CtrlShift PgUp> and <CtrlShift PgDn> to do
                             BegWindow/EndWindow.

              2006 May 17 -  [WAS] Added support for Ctrl-A.

              2006 May 18 -  [WAS] Fixed paste problem in prompt box:
                             don't push <Home> followed by <Shift End>
                             if CUA Cut/Copy/Paste is not enabled.
                             Thanks to Joe Souza for the report.

              2006 May 23 -  [WAS] Bug in Ctrl-A processing. If Ctrl-A
                             was assigned, the keystroke was ignored.
                             Created separate Ctrl-A keydef.

              2019 Apr 22 - Carlo Hogeveen - When configuring the CUAmark macro, two of
                            the options also default to "No" when they were previously
                            configured to "Yes", ignoring the user's existing
                            configuration, and possibly confusing the user about what
                            their configuration was. Methinks the intention was and is to
                            only default these two options to "No" when they are not
                            configured yet.

                            I made this change in the CUAmark macro to make its
                            configuration work as I expect it to:

                            // if EquiStr(option, "no") or (HEADING == HEADING_FIND) or (HEADING == HEADING_CTRL_A)

                            if EquiStr(option, "no") or (option == "" and (HEADING in HEADING_FIND, HEADING_CTRL_A))

              2019 May 01 - Carlo Hogeveen -   Well, it has been a long-time bug for me,
                            and I finally took the time to solve it.

                            I have all TSE's standard CUAmark macro's features enabled, so
                            when I do a Find with <Ctrl F> then CUAmark executes it.
                            CUAmark uses block marking instead of the built-in Find's
                            highlighting. Useful, as we can immediately copy the found
                            text.

                            The problem occurs when I successfully Find something, and
                            immediately afterwards do a Replace without the "n" option:
                            Text gets marked from the Find position to the places that
                            Replace stops at to ask its (Yes/No/...) question. Grr!

                            Looking in CUAmark's source code, continued marking seems very
                            deliberate, but testing CUAmark it seems to have no function,
                            because for instance I cannot continue shift-marking the found
                            text, because shift-marking continues after the found text.

                            I made the following change in the CommonFind() proc in the
                            CUAmark macro:

                                // Set(Marking, TRUE)

                                Set(Marking, FALSE)

                            This solves my problem, and now if I continue shift-marking it
                            starts at the found text. I found no other changes in
                            behaviour.

                            A word of warning though: The CUAmark macro is big and
                            complex, and I could not determine why the marking was
                            deliberately set on, so maybe I missed something that
                            reviewing and testing the change so far have not revealed.

              2023 Jun 04 - Carlo Hogeveen
                            If cumark is used for a new search with the "v"
                            option and a found line is selected from the
                            ViewFinds panel, then the text cursor now also goes
                            to the column of the found string.


  Overview:

  This macro enables CUA-style (Common User Access) key assignments in
  the editor for marking and manipulating character and column blocks.
  When you execute this command, the shifted cursor-movement keys can
  be used for marking blocks, and additional block operations can be
  accomplished using select shifted keys.

  CUA-style blocks are also called non-persistent blocks.  This means
  that in general, if you press a non Block Mark/extend key, the block
  is unmarked.

  When using CUA-style blocks, typing and pasting (from the Windows
  Clipboard) will replace the block with the typed character or the
  contents of the Windows Clipboard.

  Keys:
      <Shift CursorLeft>      Mark/extend Block, move left one column
      <Shift CursorRight>     Mark/extend Block, move right one column
      <Shift CursorUp>        Mark/extend Block, move up one line
      <Shift CursorDown>      Mark/extend Block, move down one line
      <Shift Home>            Mark/extend Block, go to beginning of line
      <Shift End>             Mark/extend Block, go to end of line
      <Shift PgUp>            Mark/extend Block, move up one page
      <Shift PgDn>            Mark/extend Block, move down one page
      <CtrlShift CursorRight> Mark/extend Block, move right one word
      <CtrlShift CursorLeft>  Mark/extend Block, move left one word
      <CtrlShift Home>        Mark/extend Block, go to beginning of file
      <CtrlShift End>         Mark/extend Block, go to end of file
      <CtrlShift PgUp>        Mark/extend Block, go to beginning of window
      <CtrlShift PgDn>        Mark/extend Block, go to end of window

      Adding the <alt> key to any of the above combinations will instead
      Mark/extend a Column Block.

      <Shift Ins>             PasteReplace (from the Windows ClipBoard)
      <Shift Del>             Cut (to the Windows ClipBoard)
      <Ctrl Ins>              Copy (to the Windows ClipBoard)
      <Ctrl Del>              DelBlock (delete marked Block)

      <Del>                   delete marked Block or delete character
      <GreyDel>               delete marked Block or delete character
      <BackSpace>             delete marked Block or backspace

      Optionally, (by selecting CUAMark options) the following keys will also be
      assigned:

      <LeftBtn>               Mark/extend Block via the mouse by dragging
      <Shift LeftBtn>         Mark/extend Block via the mouse clicking

      <Ctrl v>                PasteReplace (from the Windows ClipBoard)
      <Ctrl c>                Copy (to the Windows ClipBoard)
      <Ctrl x>                Cut (to the Windows ClipBoard)

      <Ctrl F>                Find.  If text is found, it is marked as
                              a CUA-style block.

      <Ctrl L>                Repeat Find commands.  If text is found,
      <F3>                    it is marked as a CUA-style block.

      <CtrlShift L>           Repeat Find Reverse commands.  If text is
      <Shift F3>              found, it is marked as a CUA-style block.

  Usage notes:

  If you wish to make this macro a permanent part of your
  configuration, then simply add "CUAMark" to the AutoLoad list on
  the Macro menu.

  The first time the CUAMark macro is loaded, it will self-configure
  by asking you several questions.  To reconfigure CUA Marking at
  another time, press <escape>, <enter>.

  Copyright 1992-2003 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/
constant
    REVERSE = TRUE,
    NORMAL = FALSE

// [WAS] renamed editing_key to editing_state
integer shift_marking, keep_marking, editing_state
// [WAS] Variables for prompt box processing
integer promptbox, markstate, shift_markstate

// [WAS] Moved these variables to the top of the file
// (the PromptStartup function uses the first one)
integer enabled_use_cua_cut_copy_keys
integer enabled_use_cua_find_keys
integer enabled_use_cua_mouse_keys
integer enabled_use_ctrl_a  // [WAS] Added

forward proc CleanUp()
forward keydef DelCutCopyKeys

integer level

/**************************************************************************
  Delete the block marked with the shifted cursor keys.

  Bug!  If EquateEnhancedKbd is off, the keyboard manager gets stuck in
  a loop calling this routine.  Internally, <greydel> is relaxed to become
  <del> if nothing is assigned to <greydel>.

  Since <del> and <greydel> are both in this local keymap, this command is
  found, and it is called.  Then chaincmd gets called, but the chained key
  is still set to <greydel>, so the process repeats itself, recursively.

  The fix here is if we are called recursively and EquateEnhancedKbd is off,
  then we disable our local keydef, so we will not be found.  After the
  chaincmd, we re-enable our local keydef.
 **************************************************************************/
proc mDelete()
    integer equate_enhanced_kbd
    level = level + 1

    if shift_marking and isBlockInCurrFile()
        // DelBlock()
        // [WAS] Use KillBlock if we're in a prompt box
        iif(promptbox, KillBlock(), DelBlock())
    else
        if level == 1
            Disable(DelCutCopyKeys)
            PressKey(Query(Key))
            Enable(DelCutCopyKeys)
        else
            equate_enhanced_kbd = Set(EquateEnhancedKbd, On)
            Disable(DelCutCopyKeys)
            PressKey(Query(Key))
            Enable(DelCutCopyKeys)
            Set(EquateEnhancedKbd, equate_enhanced_kbd)
        endif
    endif
    CleanUp()

    level = level - 1
end

proc mPaste()
    integer unmark_after_paste

    if shift_marking and isBlockInCurrFile()
        // DelBlock()
        // [WAS] Use KillBlock() if we're in a prompt box
        iif(promptbox, KillBlock(), DelBlock())
    endif
    unmark_after_paste = Set(UnmarkAfterPaste, Off)
    PasteFromWinClip()
    Set(UnmarkAfterPaste, unmark_after_paste)
    GotoBlockEnd()
    if unmark_after_paste
        UnmarkBlock()
    endif
    PushBlock()
    CleanUp()
    PopBlock()
end

proc mCopy()
    if isBlockInCurrFile()
        PushBlock()
        CopyToWinClip()
        PopBlock()
        keep_marking = TRUE
    elseif not isBlockMarked() and Query(UseCurrLineIfNoBlock)
        CopyToWinClip()
    endif
end

proc mCut()
    if isBlockInCurrFile()
        CutToWinClip()
        Cleanup()
    elseif not isBlockMarked() and Query(UseCurrLineIfNoBlock)
        CutToWinClip()
    endif
end

proc UnAssignedKey()
    if shift_marking
        keep_marking = TRUE
    else
        Cleanup()
    endif
end

integer proc isBlockBegCol()
    case isCursorInBlock()
        when _COLUMN_
            return (Query(BlockBegCol) == CurrCol())
        when _LINE_, _INCLUSIVE_, _NONINCLUSIVE_
            return (Query(BlockBegCol) == CurrPos())
    endcase
    return (FALSE)
end

proc AfterCommand()
    integer Key

    if shift_marking and keep_marking
        keep_marking = FALSE
    else
        Key = Query(Key)
        if shift_marking and isBlockInCurrFile() and Query(Marking)
            // [WAS] QueryEditState() returns _STATE_PROMPTED_ in a prompt box
            if isTypeableKey(Key) and ((editing_state == 0) or (editing_state & _STATE_PROMPTED_))
                if isBlockBegCol()                  // Marking right to left?
                    BackSpace()                     // then backspace to get to proper position
                endif
                // [WAS] Use KillBlock if we're in a prompt box
                // DelBlock()
                iif(promptbox, KillBlock(), DelBlock())
                InsertText(Chr(LoByte(Key)))
            endif
        endif
        Cleanup()
    endif
end

proc Cleanup()
    if shift_marking
        UnmarkBlock()
    endif
    shift_marking = FALSE
    Unhook(AfterCommand)
    Unhook(UnAssignedKey)
end

proc CommonInit()
    shift_marking = TRUE
    Hook(_AFTER_COMMAND_, AfterCommand)
    Hook(_ON_UNASSIGNED_KEY_, UnAssignedKey)
end

proc init(integer blocktype)
    if not shift_marking or not isBlockInCurrFile() or not Query(Marking)
        UnmarkBlock()
        Mark(blocktype)
        CommonInit()
    endif
    keep_marking = TRUE
end

// do *not* position at end of block - causes find backwards to fail
proc CommonFind()
    UnMarkBlock()
    MarkFoundText()
    Set(Marking, False) /* was TRUE - see comments in macro header */
    keep_marking = TRUE

    CommonInit()
end

proc mLeftBtn()
    Set(Marking, Off)
    if ProcessHotSpot()
        if isCursorInBlock()
            if isCursorInBlock() == _LINE_
                BegLine()
                MarkChar()
                EndLine()
            else
                GotoBlockEnd()
            endif
            Set(Marking, On)
            CommonInit()
            keep_marking = TRUE
        else
            UnMarkBlock()
            Cleanup()
        endif
    else
        ChainCmd()
    endif
end

proc mTrackMouseCursor()
    if GotoMouseCursor()
        TrackMouseCursor()
    endif
end

proc mShiftLeftBtn()
    integer blocktype = isCursorInBlock()
    if blocktype == 0
        blocktype = _NONINCLUSIVE_
        UnMarkBlock()
        MarkChar()
    endif
    mTrackMouseCursor()
    Mark(blocktype)

    Set(Marking, On)
    CommonInit()
    keep_marking = TRUE
end

proc mFind()
    string options [11] = ''
    if Find()
        options = Lower(GetHistoryStr(_FINDOPTIONS_HISTORY_, 1))
        if Pos('v', options)
          options = options[1: Pos('g', options) - 1] +
                    options[Pos('g', options) + 1: MAXSTRINGLEN]
          options = options[1: Pos('v', options) - 1] +
                    options[Pos('v', options) + 1: MAXSTRINGLEN]
          options = 'cg' + options
          lFind(GetHistoryStr(_FIND_HISTORY_, 1), options)
        endif
        CommonFind()
    endif
end

proc mRepeatFind(integer reverse)
    if shift_marking
        Cleanup()
    endif
    if reverse
        if RepeatFind(_REVERSE_)
            CommonFind()
        endif
    else
        if RepeatFind()
            CommonFind()
        endif
    endif
end

integer proc isMarkingRightToLeft()
    if Query(BlockBegLine) == CurrLine()
        case isCursorInBlock()
            when _LINE_
                return (TRUE)
            when _COLUMN_
                return (Query(BlockBegCol) == CurrCol())
            otherwise
                return (Query(BlockBegCol) == CurrPos())
        endcase
    endif
    return (FALSE)
end

proc AfterGetkey()
    editing_state = QueryEditState()
    if (editing_state <> 0) or Query(LineDrawing)
        return()
    endif

//  if QueryEditState() <> 0 or Query(LineDrawing)
//      editing_key = FALSE
//      return ()
//  endif

//  editing_key = TRUE
    case Query(key)
        when <CtrlShift PgUp>, <CtrlShift GreyPgUp>                 init(_noninclusive_) BegWindow()
// [WAS] Ctrl-Home/Ctrl-End should do BegFile/EndFile (Windows standard)
// [WAS] Ctrl-PgUp/Ctrl-PgDn do BegWindow/EndWindow
        when <Shift Home>, <Shift GreyHome>                         init(_noninclusive_) BegLine()
//      when <CtrlShift Home>, <CtrlShift GreyHome>                 init(_noninclusive_) BegWindow()
        when <CtrlShift Home>, <CtrlShift GreyHome>                 init(_noninclusive_) BegFile()
        when <Shift GreyCursorDown>, <Shift CursorDown>             init(_noninclusive_) Down()
        when <CtrlShift PgDn>, <CtrlShift GreyPgDn>                 init(_noninclusive_) EndWindow()
        when <Shift End>, <Shift GreyEnd>                           init(_noninclusive_) EndLine()
//      when <CtrlShift End>, <CtrlShift GreyEnd>                   init(_noninclusive_) EndWindow()
        when <CtrlShift End>, <CtrlShift GreyEnd>                   init(_noninclusive_) EndFile()
        when <Shift GreyCursorLeft>, <Shift CursorLeft>             init(_noninclusive_) Left()
        when <Shift PgDn>, <Shift GreyPgDn>                         init(_noninclusive_) PageDown()
        when <Shift PgUp>, <Shift GreyPgUp>                         init(_noninclusive_) PageUp()
        when <Shift GreyCursorRight>, <Shift CursorRight>           init(_noninclusive_) Right()
        when <CtrlShift GreyCursorDown>, <CtrlShift CursorDown>     init(_noninclusive_) RollDown()
        when <CtrlShift GreyCursorUp>, <CtrlShift CursorUp>         init(_noninclusive_) RollUp()
        when <Shift GreyCursorUp>, <Shift CursorUp>                 init(_noninclusive_) Up()
        when <CtrlShift GreyCursorLeft>, <CtrlShift CursorLeft>     init(_noninclusive_) WordLeft()
        when <CtrlShift GreyCursorRight>, <CtrlShift CursorRight>   init(_noninclusive_) WordRight()

        when <CtrlAltShift PgUp>, <CtrlAltShift GreyPgUp>                 init(_column_) BegFile()
        when <AltShift Home>, <AltShift GreyHome>                         init(_column_) BegLine()
//      when <CtrlAltShift Home>, <CtrlAltShift GreyHome>                 init(_column_) BegWindow()
        when <CtrlAltShift Home>, <CtrlAltShift GreyHome>                 init(_column_) BegFile()
        when <AltShift GreyCursorDown>, <AltShift CursorDown>             init(_column_) Down()
        when <CtrlAltShift PgDn>, <CtrlAltShift GreyPgDn>                 init(_column_) EndFile()
        when <AltShift End>, <AltShift GreyEnd>                           init(_column_) EndLine()
//      when <CtrlAltShift End>, <CtrlAltShift GreyEnd>                   init(_column_) EndWindow()
        when <CtrlAltShift End>, <CtrlAltShift GreyEnd>                   init(_column_) EndFile()
        when <AltShift GreyCursorLeft>, <AltShift CursorLeft>             init(_column_) Left()
        when <AltShift PgDn>, <AltShift GreyPgDn>                         init(_column_) PageDown()
        when <AltShift PgUp>, <AltShift GreyPgUp>                         init(_column_) PageUp()
        when <AltShift GreyCursorRight>, <AltShift CursorRight>           init(_column_) Right()
        when <CtrlAltShift GreyCursorDown>, <CtrlAltShift CursorDown>     init(_column_) RollDown()
        when <CtrlAltShift GreyCursorUp>, <CtrlAltShift CursorUp>         init(_column_) RollUp()
        when <AltShift GreyCursorUp>, <AltShift CursorUp>                 init(_column_) Up()
        when <CtrlAltShift GreyCursorLeft>, <CtrlAltShift CursorLeft>     init(_column_) WordLeft()
        when <CtrlAltShift GreyCursorRight>, <CtrlAltShift CursorRight>   init(_column_) WordRight()

        otherwise
            if (Query(key) in <Tab>, <Shift Tab>) and
                    DisplayMode() <> _DISPLAY_HEX_ and
                    Query(TabShiftsBlock) and isCursorInBlock() and Query(Insert)
                keep_marking = TRUE
                return ()
            elseif Query(Marking) == FALSE or shift_marking == FALSE
                return ()
            else
                case Query(key)
                    when <CursorDown>, <GreyCursorDown>
                        if isMarkingRightToLeft()
                            GotoBlockEnd()
                            Down()
                        else
                            Down()
                        endif
                        Cleanup()

                    when <CursorRight>, <GreyCursorRight>
                        if isMarkingRightToLeft()
                            GotoBlockEnd()
                        else
                            Right()
                        endif
                        Cleanup()

                    when <CursorLeft>, <GreyCursorLeft>
                        if isMarkingRightToLeft()
                            Left()
                        else
                            GotoBlockBegin()
                        endif
                        Cleanup()

                    when <CursorUp>, <GreyCursorUp>
                        if isMarkingRightToLeft()
                            Up()
                        else
                            GotoBlockBegin()
                            Up()
                        endif
                        Cleanup()

                    otherwise
                        return ()
                endcase
            endif
    endcase
    Set(key, -1)    // ignore this keystroke, by setting the key to -1
                    // this causes the 'do-nothing' (NoOp) command to be issued
end

keydef CUAMouseKeys
    <LeftBtn>               mLeftBtn()
    <Shift LeftBtn>         mShiftLeftBtn()
end

keydef DelCutCopyKeys
    <Del>                           mDelete()
    <BackSpace>                     mDelete()

    <Shift Ins>                     mPaste()
    <Ctrl Ins>                      mCopy()
    <Shift Del>                     mCut()
    <Ctrl Del>                      mDelete()
end

keydef CUACutCopyKeys
    <Ctrl v>                        mPaste()
    <Ctrl c>                        mCopy()
    <Ctrl x>                        mCut()
end

Keydef CUAFindKeys
    <Ctrl F>        mFind()

    <Ctrl L>        mRepeatFind(NORMAL)
    <F3>            mRepeatFind(NORMAL)

    <CtrlShift L>   mRepeatFind(REVERSE)
    <Shift F3>      mRepeatFind(REVERSE)
end

// [WAS] Handle Ctrl-A with a keydef
keydef CtrlAKey
    <Ctrl A>        init(_noninclusive_) MarkAll()
end

// [WAS] CUA marking initialization for prompt boxes
proc PromptInit()
    if not shift_marking or not isBlockInCurrFile() or not Query(Marking)
        UnmarkBlock()
        MarkChar()
        shift_marking = TRUE
        Hook(_AFTER_NONEDIT_COMMAND_, AfterCommand)
        Hook(_ON_UNASSIGNED_KEY_, UnAssignedKey)
    endif
    keep_marking = TRUE
end

// [WAS] keydef for use inside prompt boxes
keydef PromptKeys
    <Shift CursorLeft>              PromptInit() Left()
    <Shift GreyCursorLeft>          PromptInit() Left()
    <Shift CursorRight>             PromptInit() Right()
    <Shift GreyCursorRight>         PromptInit() Right()
    <CtrlShift CursorLeft>          PromptInit() WordLeft()
    <CtrlShift GreyCursorLeft>      PromptInit() WordLeft()
    <CtrlShift CursorRight>         PromptInit() WordRight()
    <CtrlShift GreyCursorRight>     PromptInit() WordRight()
    <Shift Home>                    PromptInit() BegLine()
    <Shift GreyHome>                PromptInit() BegLine()
    <Shift End>                     PromptInit() EndLine()
    <Shift GreyEnd>                 PromptInit() EndLine()
end

// [WAS] Hook: Execute this when a prompt box appears
proc PromptStartup()
    PushBlock()                      // Save any current block
    markstate = Set(Marking, Off)    // Turn off marking
    shift_markstate = shift_marking  // Save shift-marking state

    Enable(PromptKeys)
    Enable(DelCutCopyKeys)

    if enabled_use_cua_cut_copy_keys
        Enable(CUACutCopyKeys)
        // Select the initial text in the prompt box

//        BegLine()
//        PromptInit()
//        EndLine()

        PushKey(<Shift end>)
        PushKey(<Home>)
    endif

    promptbox = TRUE

end

// [WAS] Hook: Execute when a prompt box disappears
proc PromptCleanup()
    Set(Marking, markstate)
    shift_marking = shift_markstate
    PopBlock()
    promptbox = FALSE
end

string SECTION[] = "CUAMark",
        HEADING_CUT_COPY[] = "UseCUACutCopyPasteKeys",
        HEADING_FIND[] =     "UseCUAFindKeys",
        HEADING_MOUSEMARKING[] = "UseCUAMouseKeys",
        HEADING_CTRL_A[] = "UseCtrlA"  // [WAS] Added

string proc GetOptions(string HEADING, string msg, integer force)
    string option[3]

    option = GetProfileStr(SECTION, HEADING, "")

    if force or option == ""
        // [WAS] Ctrl-A defaults to "no", same as find
        if EquiStr(option, "no") or (option == "" and (HEADING in HEADING_FIND, HEADING_CTRL_A))
            PushKey(<cursorright>)
        endif
        if MsgBox("", msg, _YES_NO_) == 1
            option = "yes"
        else
            option = "no"
        endif
        WriteProfileStr(SECTION, HEADING, option)
    endif

    return (option)
end

proc configure(integer force)
    string use_cua_cut_copy_keys[3],
        use_cua_find_keys[3],
        use_cua_mouse_keys[3],
        use_ctrl_a[3]  // [WAS] Added

    use_cua_cut_copy_keys = GetOptions(HEADING_CUT_COPY,
            "Enable CUA Cut/Copy/Paste (Ctrl x, Ctrl c, Ctrl v, respectively) keys in addition to the normal CUAMark assignments?",
            force)

    use_cua_find_keys     = GetOptions(HEADING_FIND,
            "Enable CUA Find (Ctrl f, F3, Shift F3, respectively) keys in addition to the normal CUAMark assignments?" + Chr(13) +
            "Warning - in this mode, found text is marked as a block, and is replaced/deleted in the same way as other shift-marked blocks.",
            force)

    use_cua_mouse_keys     = GetOptions(HEADING_MOUSEMARKING,
            // [WAS] corrected spelling of the word "persistent"
            "Enable CUA Mouse Left-Button marking (non-persistent block marking)?",
            force)

    // [WAS] Added
    use_ctrl_a = GetOptions(HEADING_CTRL_A,
            "Enable <Ctrl A> to mark entire file?",
            force)

    if use_cua_cut_copy_keys == "yes"
        if not enabled_use_cua_cut_copy_keys
            Enable(CUACutCopyKeys)
            enabled_use_cua_cut_copy_keys = TRUE
        endif
    else
        Disable(CUACutCopyKeys)
        enabled_use_cua_cut_copy_keys = FALSE
    endif

    if use_cua_find_keys == "yes"
        if not enabled_use_cua_find_keys
            Enable(CUAFindKeys)
            enabled_use_cua_find_keys = TRUE
        endif
    else
        Disable(CUAFindKeys)
        enabled_use_cua_find_keys = FALSE
    endif

    if use_cua_mouse_keys == "yes"
        if not enabled_use_cua_mouse_keys
            Enable(CUAMouseKeys)
            enabled_use_cua_mouse_keys = TRUE
        endif
    else
        Disable(CUAMouseKeys)
        enabled_use_cua_mouse_keys = FALSE
    endif

    // [WAS]
    enabled_use_ctrl_a = use_ctrl_a == "yes"
    if enabled_use_ctrl_a
        Enable(CtrlAKey)
    else
        Disable(CtrlAKey)
    endif
end

public proc setup()
    configure(TRUE)
end

proc main()
    configure(TRUE)
end

proc WhenLoaded()
    Hook(_AFTER_GETKEY_, AfterGetkey)
    // [WAS] Hook _PROMPT_STARTUP_ and _PROMPT_CLEANUP_
    Hook(_PROMPT_STARTUP_, PromptStartup)
    Hook(_PROMPT_CLEANUP_, PromptCleanup)
    Enable(DelCutCopyKeys)
    configure(FALSE)
end

