/*************************************************************************
  Compile     Compiles and marks errors for several compilers.

  Author:     SemWare

  Date:       Jul 21, 1993 (Steve Watkins)
              Jul 28, 1995 (Steve Watkins)
                Added support for macro commands via '@'
                Added support for merging previous compile.dat files
                Added support for macro macro_debugging
              Oct 26, 1995 (Sammy Mitchell)
                Some compilers do not like file locking.
                Added conditional code to unlock the file before compiling,
                 and then lock it back after the compile.  To implement,
                 compile with "sc -DLOCKFIX compile"
              Oct 8, 1997 SEM
                Fix truncation bug with long extensions
              Nov 20, 1997 SEM
                Change for i = constant to constant loop to a
                do constant times loop
              Feb 13, 1998 SEM
                Did not work if TSE installed in a path with spaces in it.
                Fixed by liberal sprinkling of QuotePath()
              Apr 4, 1998 SEM
                For win32 compilers, call them directly instead of going
                through command/cmd/4dos/4nt/whatever.  This is somewhat
                faster, and, in Win95/98, it keeps us from having to resize
                the window to a "safe" size (e.g., 80x25).  sc32, bcc32,
                msc, wcl386 and others benefit from this.  Also, we can
                capture both stdout and stderr by doing it this way.
              May 21, 1998 SEM
                Increase the cmd string from 128 to 180 to allow for larger
                batch file creation.
              Jul 16, 1998 SEM
                The method used to capture stdout/stderr in the Apr 4 1998
                fix did not work on Windows NT.  So, I bit the bullet and
                added back in "tee" support.  Works under both Win95/98 and
                WinNT.  However, both win32 implementations apparently
                buffer redirected output, so that we don't see anything
                until > 512 characters has been written.  Bummer.
                Changed the error output filename destination to the TEMP
                dir.
                Increased the maximum command line to 255 chars (up from
                180).
                Use the short-name of the file being compiled if the program
                to be run is a 16-bit app.
              Sep 24, 1998 SEM
                Need to get the exact case of the file we are compiling, as
                some compilers (such as Javac) require this.
              Feb 2, 1999 SEM
                Use new 3.00 features, including WhichOS() and is32BitApp().
                Rework pgm finding logic to only search the PATH if the pgm
                in question does not include a '\' in the specification.
              Feb 18, 1999 SEM
                BUG: Barfs on PATH's > 255 characters.  Need to increase the
                maximum string length somehow.
                For now, work-around is to use the Win32 SearchPath API to
                search the path.
              May 17, 1999 SEM
                CreateProcess, under W98/95 will take a string such as
                c:\Program Files\tse32\sc32.exe and execute it.  Under
                NT 4, in order for this to work, you must quote the path. So,
                We now quote the path we send to CreateProcess, if needed.
              September 24, 1999 KSW
                (1) - Adjusted call to fCreate to remove remove leading and
                trailing quotes (if present) from the passed filename.
                i.e., code segment:
                h = iif(FileExists(inpath), -1, fCreate(inpath))
                changed to:
                h = iif(FileExists(inpath), -1, fCreate(mRemoveQuotesFrom(inpath)))
                (2) - Adjusted code that determines if a 16-bit application is
                being called from the macro. i.e., code segment:
                elseif (not win32_app)
                changed to:

                elseif not win32_app and
                    (Lower(SplitPath(pgm, _EXT_)) <> ".bat" or
                    Pos("16", SplitPath(pgm, _NAME_)) <> 0)

                *** If the application being called is a batch file, it's treated
                as a 32 bit application unless "16" appears anywhere within the
                application filename (less path and extension), in which case it
                is treated as a 16 bit application.  If the application is
                determined to be a 16 bit application, then a passed
                "long filename" is converted to its "short filename" equivalent
                before passing it to the 16 bit application.
              October 6, 1999 SEM
                Force saving the source file if it does not exist on disk.
              November 11, 1999 SEM
                Need to add QuotePath to the load/execute macro calls, so
                that we can properly load macros with spaces in the filenames.
                Thanks to Ross Boyd for pointing this one out.
              November 8, 2000 SEM
                Added -p and -b command line switches.
                Added user-settable variables, save_changed_files, query_save_changed_files,
                to modify normal operation.
              January 31 2001 KSW
                Fix to handle error output that includes tabs in the error strings.
              March 2, 2001 SEM
                Allow the errors window to scroll horizontally when error messages wider
                than the screen are on-screen.
              March 13, 2001 KSW
                Pretty up the keynames when they show in the help.
              March 22, 2001 SEM
                Fix problem with isOutputRedirected and long filenames.
              December 7, 2001 SEM (reported by KSW)
                Warn if compile.dat not found
              September 27, 2002 SEM
                Add -2 tag feature for finding Python error
                column
              December 5, 2002 SEM
                In mDos(), try to remove the current directory from paths, in order to
                make the paths a little shorter.
              February 26, 2003 SEM
                Pressing <escape> will close the error window
              March 2004, SEM
                Made the 'command' and 'output' prompts a little
                larger.
              April 2005, SEM
                Add the ability to parse a file of errors, without
                compiling.
              January  24, 2007 SEM
                Add ability to work with hard to parse error message
                (e.g., the IBM Jikes Java compiler) by utilizing a
                custom macro to parse the error messages.
              March 2010 SEM
                Changes for Linux.
                - Use correct path separator based on WhichOS.
                - If compiler is in current dir, need to have "./" prepended, if no path
                - To capture both stdout and stderr, append: 2>&1 to the compile string
                - Drop ".exe" from sc32.
              December 2010 SEM
                - Don't use DLL (GetShortPathName) under Linux.
              April 2019
                - Some compilers - qb64, for instance - don't handle relative paths.
                  Remove the code that tries to shorten paths.
              August 4, 2019 CHO
                - Substitute <Ctrl E> for <Ctrl Enter> for Linux.
                - Retain a macro's and ui's case when compiling it.
              April 2019
                - Add new process option: no prompt, no clear, run hidden
              Jan 08 2022
                - If file-to-compile contains "&" and ";", force quotting it.
                  Thanks to Fred Holmes for the note.

  Overview:

  Compiles source code, tags the errors, and allows the user to step
  through errors for compilers whose output files are in a regular
  pattern.

  Keys:
              <Shift F9>    CompileMenu()
              <Ctrl F9>     CompileCurrentFile()
              <Alt F8>      NextErrKey()
              <Alt F7>      PrevErrKey()

  Usage notes:

  After loading this macro, simply press <Ctrl F9> to compile the
  file being edited.  The current file will be saved (if necessary),
  compiled, and the errors will be marked.  The windows will be set
  up with the source in the top and errors in the bottom window.
  Press <Alt F7> and <Alt F8> to go to the previous/next errors,
  respectively.  Pressing <F1> within the Source window provides
  help on the keys.

  If a TSE macro source files in compiled, the user is asked to
  execute the macro if the compile is successful.

  Invoke <Shift F9> to invoke the Compile Menu() for additional
  selections like selecting alternate compilers and setting up new
  or modifying existing compiler setups.

  Copyright 1992-2001 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

  Switches:

  -c  Causes the current file to be compiled, same as "compile fn"

  -m  Used to bring up the compile main menu

  -p[error_fn [fn]] allows one to override the default error filename

  -x.ext  Whenever you want to invoke the compile macro on a filename whose
      extension is not a pre-defined extension OR in a case when you want to
      invoke a compiler (based on a specified extension) on a file which
      does not have that extension. For example, if you want to check the
      parsability of a ".h" file you could do an exec macro on "compile
      -x.c" .  Or in my most immediate case, I want to invoke the ".s"
      compiler for a ".ui" file.

      -x.ext must occur before any other switches or parameters.

  -b [fn] allows the compile to work in the background.  -b and -d are
      mutually exclusive.

  -d [fn] Used by the macro debugger. -d and -b are mutually exclusive.
*************************************************************************/

integer
    save_changed_files = FALSE,
    query_save_changed_files = FALSE

string separator[] = "\"

constant MAXPATH = _MAX_PATH_
string tee32[MAXPATH], tee16[MAXPATH]       // paths to tee32.exe and tee16.exe

dll "<kernel32.dll>"
    integer proc WinSearchPath(
            integer lpPath,             // address of search path - set this to 0
            string lpFileName:cstrval,  // address of filename
            integer lpExtension,        // address of extension - set this to 0, make sure lpPath includes extension
            integer nBufferLength,      // size, in characters, of buffer
            var string lpBuffer:strptr, // address of buffer for found filename
            var integer lpFilePart      // address of pointer to file component
   ) : "SearchPathA"
end

/**************************************************************************
  Search the PATH environment variable for a file using the Win32 API
 **************************************************************************/
string proc SearchPathForFile(string fn)
    string path[MAXPATH] = Format("":MAXPATH)
    integer len, ptr_to_fn = 0
    len = WinSearchPath(0, fn, 0, SizeOf(path), path, ptr_to_fn)
    if (len > MAXPATH)
        Warn("Path of ", fn, " is larger than can currently be handled.")
        return ("")
    endif
    return (path[1:len])
end

constant BACKGROUND_FLAG = 0x01, PROCESS_ERROR_FILE_FLAG = 0x02, INVOKE_LIST_FLAG = 0x04

string compile_error_fn[] = "compile->error_fn"
string compile_source_fn[] = "compile->source_fn"

/************************************************************************
    To customize the compile keys to your liking modify these
    keys and recompile this macro.
 ************************************************************************/

// Default compile keys

constant NEXT_ERROR_KEY     = <Alt F8>
constant PREV_ERROR_KEY     = <Alt F7>
constant COMPILE_FILE_KEY   = <Ctrl F9>
constant COMPILE_MENU_KEY   = <Shift F9>
constant NOT_DEFINED        = 0xffff // not defined
constant SWITCH_WINDOW_KEY  = NOT_DEFINED

// if a path is not specified for the the batch_fn, it will be placed in the
// same directory as the compile macro.

string batch_fn[]   = "$$tmp$$.bat"
string DATA_FN[]    = "compile.dat"         // list of compilers and how to handle each one
string compile_fn[MAXPATH]                  // name of the file being compiled
string dbg_ext[4] = ".dbg"

constant NOISE      =  TRUE

// END USER-DEFINABLE VARIABLES

string full_output_fn[MAXPATH]              // errors written here...
integer dbg_enabled

/**************************************************************************
  Return the first filename-like argument from the passed string
 **************************************************************************/
string proc GetFirstArg(string argstr)
    return (GetFileToken(argstr, 1))
end

/**************************************************************************
  Search for pgm + ext in path, return full path in pgm and TRUE if found
 **************************************************************************/
integer proc TryFind(var string pgm, string ext)
    string p[MAXPATH]

    if Pos(separator, pgm)
        if FileExists(pgm + ext)
            pgm = pgm + ext
            return (TRUE)
        else
            return (FALSE)
        endif
    endif

    p = SearchPath(pgm + ext, GetEnvStr("PATH"), ".")
    if p == ""
        p = SearchPathForFile(pgm + ext)
    endif
    if p <> ""
        pgm = p
        return (TRUE)
    endif
    return (FALSE)
end

/**************************************************************************
  See if we can find a pgm, using the standard search methods.  If no
  extension, try .bat, .com, and finally, .exe.
 **************************************************************************/
string proc FindProgram(string cmdstr)
    string pgm[MAXPATH]

    if cmdstr == ""
        return ("")
    endif

    // find the pgm - we really need to use a better search here
    pgm = GetFirstArg(cmdstr)
    if SplitPath(pgm, _EXT_) <> ""
        return (iif(TryFind(pgm, ""), pgm, ""))
    else
        if TryFind(pgm, ".bat")
            return (pgm)
        elseif TryFind(pgm, ".com")
            return (pgm)
        elseif TryFind(pgm, ".exe")
            return (pgm)
        endif
    endif
    return ("")
end

/**************************************************************************
  is the passed token a built-in OS command?
 **************************************************************************/
integer proc isOSCommand(string cmdline)
    string pgm[255]

    pgm = Trim(Lower(GetFirstArg(cmdline)))
    return (pgm in "copy", "move")
end

integer proc StartRedirect(string fn, var integer _fd1, var integer _fd2)
    integer fd1, fd2, fd_out, fd_err

    fd_out = fCreate(fn)
    fd1 = fDup(_STDOUT_)
    fDup2(fd_out, _STDOUT_)

    fd_err = fd_out
    fd2 = fDup(_STDERR_)
    fDup2(fd_err, _STDERR_)

    fClose(fd_out)
    _fd1 = fd1
    _fd2 = fd2
    return (TRUE)
end

integer proc EndRedirect(integer fd1, integer fd2)
    fDup2(fd1, _STDOUT_)
    fClose(fd1)

    fDup2(fd2, _STDERR_)
    fClose(fd2)

    return (TRUE)
end

/**************************************************************************
  Remove all quotes from path.
  A better routine would be RemoveCharsFrom(string del_set, string s)
 **************************************************************************/
string proc mRemoveQuotesFrom(string path)
    string s[255]
    integer i

    s = ""
    for i = 1 to Length(path)
        if path[i] <> '"'
            s = s + path[i]
        endif
    endfor
    return (s)
end

/**************************************************************************
  Force quoting a path if it contains & or ; or ( ) characters
 **************************************************************************/
string proc mQuotePath(string path)
    string s[255]

    s = QuotePath(path)
    if s[1] <> '"'
        if Pos("&", path) > 0 or Pos(";", path) > 0 or Pos("(", path) > 0 or Pos(")", path) > 0
            s = '"' + s + '"'
        endif
    endif
    return (s)
end

/**************************************************************************
  See if the specified program exists in the LoadDir() (might be revectored
  with -i) or LoadDir(1) (the actual editor start dir)
 **************************************************************************/
proc find_editor_program(string fn, var string path)
    path = LoadDir() + fn
    if not FileExists(path)
        path = SplitPath(LoadDir(1), _DRIVE_|_PATH_) + fn
        if not FileExists(path)
            path = SearchPath(fn, Query(TSEPath))
            if not FileExists(path)
                path = LoadDir() + "mac" + separator + fn
                if not FileExists(path)
                    path = ""
                endif
            endif
        endif
    endif
    path = QuotePath(path)
end

/**************************************************************************
  Convert a short path to a long path.
 **************************************************************************/
string proc mGetLongPath(string ashort_path)
    string s[255], short_path[255]
    integer n, i, len

    short_path = ExpandPath(ashort_path, TRUE)
    n = NumTokens(short_path, separator)
    s = SplitPath(short_path, _DRIVE_)
    for i = 2 to n
        len = Length(s)
        s = s + separator + GetToken(short_path, separator, i)
        if FindThisFile(s, -1)
            s = s[1:len] + separator + FFName()
        endif
    endfor
    return (s)
end

/**************************************************************************
  Convert and replace compile_fn to/with a short name, if found in cmdstr
 **************************************************************************/
proc Handle16BitApps(var string cmdstr, string compile_fn)
    integer p
    string short_name[MAXPATH]

    p = Pos(compile_fn, cmdstr)
    if p
        short_name = GetShortPath(compile_fn)
        if short_name <> ""
            cmdstr = DelStr(cmdstr, p, Length(compile_fn))
            cmdstr = InsStr(short_name, cmdstr, p)
        endif
    endif
end

/**************************************************************************
  See if '>' preceeds output_fn, optionally separated by spaces
  Return the position of the '>', or 0 if not found.

  Need to handle the following problematic strings:
  output_fn = err file

    grep "foo>bar"*.c>"err file"
    grep "foo>bar" *.c>"err file"
    grep "foo>bar" *.c > "err file"
    grep "foo>bar"*.c "err file"

  output_fn may or may not be quoted.  But, if it contains spaces, it
  will be quoted in cmdstr.
 **************************************************************************/
integer proc isOutputRedirected(string cmdstr, string output_fn)
    integer p

    // get position of output_fn in cmdstr, less 1
    p = Pos(output_fn, cmdstr) - 1
    if p > 0
        if cmdstr[p] == '"'     // ignore a possible leading quote
            p = p - 1
        endif
        // ignore 0 or more spaces or tabs
        while p > 0 and cmdstr[p] in " ", Chr(9)
            p = p - 1
        endwhile
        // if cmdstr[p] is '>', return the position, else 0
        return (iif(p > 0 and cmdstr[p] == ">", p, 0))
    endif
    return (FALSE)
end

//proc stripCurrDirFrom(var string cmdstr)
//    string dir[_MAXPATH_] = CurrDir()
//
//    if Pos(dir, cmdstr) == 1
//        cmdstr = StrReplace(dir, cmdstr, "./", "1")
//    endif
//
//    cmdstr = StrReplace(dir, cmdstr, "")
//end

/**************************************************************************
  Format is: Dos(cmdstring, flags)

  Problem:  tee32 really needs to know if he is calling a 16-bit app on NT,
  so that he can take care to run it via cmd.exe, who somehow magically
  loads an NTVDM if needbe.

  So, for now, if that is the case, have tee32 try to call command.com, or
  cmd.exe.

  Make sure tee??.exe exists in the editors actual load directory.
 **************************************************************************/
integer proc mDos(string acmdstr, integer flags, integer background)
    integer result, win32_app, manual_redir, h1, h2, output_redirected
    string pgm[128], cmd[128], cmdstr[255], efn[255]

    cmdstr = acmdstr
//    stripCurrDirFrom(cmdstr)  -- some compilers - qb64, for instance -- cannot handle relative paths.
    if WhichOS() == _LINUX_
        if Pos(">", cmdstr) > 0
            cmdstr = cmdstr + " 2>&1"
        endif
        return (Dos(cmdstr, flags))
    endif

    result = FALSE
    pgm = FindProgram(cmdstr)
    win32_app = is32BitApp(pgm) > 0
    if pgm == ""
        if not isOSCommand(cmdstr)
            return (Warn(GetFirstArg(cmdstr), " not found!"))
        endif
    elseif not win32_app and (Lower(SplitPath(pgm, _EXT_)) <> ".bat" or Pos("16", SplitPath(pgm, _NAME_)) <> 0)
        Handle16BitApps(cmdstr, compile_fn)
    endif
    if not isGUI() and not ((Query(ScreenCols) in 80) and (Query(ScreenRows) in 25,43,50))
        flags = flags | _CREATE_NEW_CONSOLE_
    endif

    if background
        return (Dos(cmdstr, flags | _RUN_DETACHED_|_DONT_WAIT_|_START_HIDDEN_))
    endif

    output_redirected = isOutputRedirected(cmdstr, QuotePath(full_output_fn))

    if pgm == "" or (flags & _TEE_OUTPUT_) == 0 or (not output_redirected)
        result = Dos(cmdstr, flags)
    else
    /**************************************************************************
      The LoadDir() value can actually be revectored, if the -i editor cmdline
      option is set, or the TSELOADDIR environment variable is set.  So, use
      LoadDir(1), which can not be revectored.
    **************************************************************************/
        if win32_app or (WhichOS() == _WINDOWS_NT_)
            flags = flags | _DONT_CHANGE_VIDEO_|_PRESERVE_SCREEN_
            if not FileExists(tee32)
                result = Warn("tee32.exe", " not found.")
        //*** 32-bit app, any platform ***
            elseif win32_app

                if dbg_enabled
                    Warn(tee32; cmdstr; flags)
                endif
                result = lDos(tee32, cmdstr, flags)
            else
        /**************************************************************************
          16-bit app, on WinNT:
          NT seems to run 16-bit apps better when run from command.com, so,
          try to load the app via command.com, or cmd.exe, or GetEnv("COMSPEC").
         **************************************************************************/
                cmd = "command.com"
                if SearchPath(cmd, GetEnvStr("PATH")) == ""
                    cmd = "cmd.exe"
                    if SearchPath(cmd, GetEnvStr("PATH")) == ""
                        cmd = GetEnvStr("COMSPEC")
                    endif
                endif
                result = lDos(tee32, cmd + " /c " + cmdstr, flags)
            endif
        /**************************************************************************
          16-bit app, Win95/98.  Pipes cause use trouble.  So use a 16-bit DOS
          program to tee the output, which seems to work just fine.
         **************************************************************************/
        else
            if not FileExists(tee16)
                result = Warn("tee16.exe", " not found.")
            endif

            if (flags & _CREATE_NEW_CONSOLE_) and output_redirected
                cmdstr = DelStr(cmdstr, output_redirected, Length(cmdstr) - output_redirected + 1)
                efn = full_output_fn
                manual_redir = TRUE
                StartRedirect(efn, h1, h2)
            else
                manual_redir = FALSE
            endif

            if Upper(SplitPath(pgm, _EXT_)) == '.BAT'
                result = Dos(cmdstr, flags)
            else
                result = Dos(tee16 + " " + cmdstr, flags)
            endif

            if manual_redir
                EndRedirect(h1, h2)
            endif

        endif
    endif

    return (result)
end

constant EXT_LEN = 12
string SCEXE[] = "sc32.exe" // set appropriately in WhenLoaded()

/**************************************************************************
  Borland C's startup processing interprets \" special.  To get a literal
  \", we must use \\"".

  Microsoft C is different, we would use \\"
 **************************************************************************/

// Helper routines

proc SetXY()
    Set(x1,WhereXAbs())
    Set(y1,WhereYAbs())
end

proc mAlarm()
    if noise
        Alarm()
    endif
end

string press_a_key[] = ". Press a key"

proc Msg(string s)
    integer n

    if dbg_enabled
        n = Length(s)
        if n - SizeOf(press_a_key) > Query(ScreenCols)
            n = Query(ScreenCols) - SizeOf(press_a_key)
        endif
        Message(s:n,press_a_key)
        GetKey()
    else
        Message(s)
    endif
end

proc DbgMsg(string s)
    if dbg_enabled
        Msg(s)
    endif
end

/**************************************************************************
  We depend on CurrExt() returning a lower-cased extension.
 **************************************************************************/
string proc mCurrExt()
    string ext[EXT_LEN] = CurrExt()
    if ext[1] <> '.'
        ext = '.' + ext
    endif
    #ifdef INHOUSE
    if ext == ".si" and SubStr(SplitPath(CurrFilename(),_NAME_),1,4) == "bind"
        ext = ".bnd"
    endif
    #endif
    return (ext)
end

integer batch_id, data_id, work_id, err_id
integer err_no, err_row, err_col, num_errors
integer errors_in_source
string  err_msg[80]

string full_data_fn[MAXPATH]
integer new_file

integer output_no, output_type

string output_types[] = Chr(_DEFAULT_) +
                        Chr(_DONT_PROMPT_) +
                        Chr(_DONT_CLEAR_ | _DONT_PROMPT_) +
                        Chr(_TEE_OUTPUT_ | _DONT_PROMPT_) +
                        Chr(_TEE_OUTPUT_ | _DONT_PROMPT_) +
                        Chr(_DONT_CLEAR_ | _DONT_PROMPT_)

proc SetOutputType(integer n)
    output_no = n
    output_type = Asc(output_types[n])
// Kludge!  _START_HIDDEN_ has a value > 255, so it can't fit in a char
// Have to manually OR it in if it is selected.
    if n in 5..6
        output_type = output_type | _START_HIDDEN_
    endif

end

// data for compiler records

string EyeCatcher[] = "Semware compile macro data file"

constant FIND_WIDTH = 80

integer errs_in_src
string  ext[EXT_LEN]
string  desc[48]
string  flags[8]
string  cmd[255]
string  err_ss[FIND_WIDTH], err_opts[8], err_tag[3]
string  fn_ss [FIND_WIDTH], fn_opts [8], fn_tag [3]
string  ln_ss [FIND_WIDTH], ln_opts [8], ln_tag [3]
string  col_ss[FIND_WIDTH], col_opts[8], col_tag[3]
string  msg_ss[FIND_WIDTH], msg_opts[8], msg_tag[3]
string  rule_mac[_MAX_PATH_]

string InstallSuccessfulMsg[80]
string LoadableUI[MAXPATH]
string LoadableCfg[MAXPATH]

constant LIST_WIDTH = SizeOf(desc) + 8

integer compiler_start, compiler_end

proc ReadLn(var string s)
    s = GetText(1,CurrLineLen())
    Down()
end

proc ReadSetupInfo()
    string s[EXT_LEN + 1] = ''

    ReadLn(s)
    ReadLn(desc)
    ReadLn(flags)
    errs_in_src = Asc(flags[1])
    SetOutputType(Asc(flags[2]))

    ext = SubStr(s,2,SizeOf(s))
end

proc ReadRecord()
    integer id

    id = GotoBufferId(data_id)

    compiler_start = CurrLine()

    ReadSetupInfo()

    ReadLn(cmd)

    ReadLn(err_ss)
    ReadLn(err_opts)
    ReadLn(err_tag)

    ReadLn(fn_ss)
    ReadLn(fn_opts)
    ReadLn(fn_tag)

    ReadLn(ln_ss)
    ReadLn(ln_opts)
    ReadLn(ln_tag)

    ReadLn(col_ss)
    ReadLn(col_opts)
    ReadLn(col_tag)

    ReadLn(msg_ss)
    ReadLn(msg_opts)
    ReadLn(msg_tag)

    rule_mac = ""
    if GetText(1, 1) <> Chr(31)
        ReadLn(rule_mac)
    endif

    compiler_end = CurrLine()
    GotoLine(compiler_start)

    GotoBufferId(id)
end

proc WriteLn(string s)
    // only write if line changed so FileChanged() is accurate

    if GetText(1,CurrLineLen()) <> s
        BegLine()
        KillToEol()
        InsertText(s)
    endif
    if not Down()
        AddLine()
    endif
    BegLine()
end

proc SaveCompileData()
    integer id, level, eoltype

    level = Set(MsgLevel, _NONE_)

    eoltype = Set(EOLType, 2)
    id = GotoBufferId(data_id)
    if FileChanged()
        if new_file
            SetXY()
            if YesNo("Create setup file: '"+full_data_fn+"'") == 1
                BegFile()
                InsertLine(EyeCatcher)
                InsertLine(Format('DBG=',dbg_ext))
                if SaveAs(full_data_fn, _OVERWRITE_)
                    new_file = FALSE
                endif
            endif
        else
            SaveAs(full_data_fn, _OVERWRITE_)
        endif
    endif
    Set(EOLType, eoltype)

    Set(MsgLevel, level)
    GotoBufferId(id)
end

proc WriteRecord()
    integer id

    id = GotoBufferId(data_id)

    flags = Chr(errs_in_src) + Chr(output_no)

    WriteLn(''+ext)
    WriteLn(desc)
    WriteLn(flags)
    WriteLn(cmd)

    WriteLn(err_ss)
    WriteLn(err_opts)
    WriteLn(err_tag)

    WriteLn(fn_ss)
    WriteLn(fn_opts)
    WriteLn(fn_tag)

    WriteLn(ln_ss)
    WriteLn(ln_opts)
    WriteLn(ln_tag)

    WriteLn(col_ss)
    WriteLn(col_opts)
    WriteLn(col_tag)

    WriteLn(msg_ss)
    WriteLn(msg_opts)
    WriteLn(msg_tag)

    WriteLn(rule_mac)

    WriteLn('')

    GotoBufferId(id)
end

integer num_compilers

proc GenerateCompilerList(string extension)
    integer line, level

    EmptyBuffer(work_id)

    GotoBufferId(data_id)
    PushPosition()

    num_compilers = 0

    if not lFind(''+extension,'^i$g') and not lFind('','^g')
        AddLine("        No compilers found", work_id)
    else
        repeat
            line = CurrLine()
            ReadSetupInfo()
            AddLine(Format(ext:-8,desc:-SizeOf(desc),line:4), work_id)
            num_compilers = num_compilers + 1
        until not lRepeatFind()
    endif

    PopPosition()

    GotoBufferId(work_id)
    MarkColumn(1,1,NumLines(),5)
    level = Set(MsgLevel, _NONE_)
    Sort(_IGNORE_CASE_)
    Set(MsgLevel, level)
    MarkColumn(1,1,NumLines(),LIST_WIDTH)
end

menu OutputMenu()
    history = output_no
    command = SetOutputType(MenuOption())

    "Clear Screen and Prompt After Shell"                   ,   ,   CloseBefore
    "Clear Screen but Don't Prompt After Shell"             ,   ,   CloseBefore
    "Don't Clear Screen or Prompt After Shell"              ,   ,   CloseBefore
    "Tee Output and Don't Prompt After Shell"               ,   ,   CloseBefore
    "Tee Output, Run Hidden and Don't Prompt After Shell"   ,   ,   CloseBefore
    "Run Hidden, Don't Clear Screen or Prompt After Shell"  ,   ,   CloseBefore

end

menu RulesMenu()
    "&Error   " [Format(err_ss:-60):-60]  ,Read(err_ss)     , DontClose
    " Options"  [Format(err_opts:-8):-8]  ,Read(err_opts)   , DontClose
    "&Filename" [Format(fn_ss:-60):-60]   ,Read(fn_ss)      , DontClose
    " Options"  [Format(fn_opts:-8):-8]   ,Read(fn_opts)    , DontClose
    " Tag    "  [fn_tag:-3]               ,Read(fn_tag)     , DontClose
    "&Line    " [Format(ln_ss:-60):-60]   ,Read(ln_ss)      , DontClose
    " Options"  [Format(ln_opts:-8):-8]   ,Read(ln_opts)    , DontClose
    " Tag    "  [ln_tag:-3]               ,Read(ln_tag)     , DontClose
    "&Column  " [Format(col_ss:-60):-60]  ,Read(col_ss)     , DontClose
    " Options"  [Format(col_opts:-8):-8]  ,Read(col_opts)   , DontClose
    " Tag    "  [col_tag:-3]              ,Read(col_tag)    , DontClose
    "&Message " [Format(msg_ss:-60):-60]  ,Read(msg_ss)     , DontClose
    " Options"  [Format(msg_opts:-8):-8]  ,Read(msg_opts)   , DontClose
    " Tag    "  [msg_tag:-3]              ,Read(msg_tag)    , DontClose
end

proc ReadExt()
    if Read(ext)
        if ext[1] <> '.'
            ext = '.' + ext
        endif
    endif
end

menu CompilerSetupMenu()
    "E&xtension  " [ext:-EXT_LEN]                                   ,ReadExt()      , DontClose
    "&Description" [Format(desc:-SizeOf(desc)):-SizeOf(desc)]       ,Read(desc)     , DontClose
    "&Command    " [Format(cmd:-52):-52]                            ,Read(cmd)      , DontClose
    "&Output     " [Format(MenuStr(OutputMenu, output_no):-52):-52] ,OutputMenu()   , DontClose
    "&Rules      "                                                  ,RulesMenu()    , DontClose
    "&User Macro " [Format(rule_mac:-52):-52]                       ,Read(rule_mac) , DontClose
end

constant ADD_RECORD     =   1,
         COPY_RECORD    =   2,
         DELETE_RECORD  =   3

// Read record as specified in list file and leave data file current
proc ReadRecordFromList(integer flag)
    integer n = Val(GetText(LIST_WIDTH + 1,8))

    GotoBufferId(data_id)
    GotoLine(iif(flag == ADD_RECORD, NumLines(), n))
    ReadRecord()
end

proc EditCompilerCommon(integer flag)
    integer id

    id = GetBufferId()

    ReadRecordFromList(flag)

    SetXY()
    if flag == DELETE_RECORD
        if YesNo("Delete "+desc+"?") == 1
            PushBlock()
            MarkLine(compiler_start, compiler_end)
            KillBlock()
            PopBlock()
        endif
    else
        CompilerSetupMenu("Add Compiler")
        if Length(ext)
            if flag == COPY_RECORD
                GotoLine(NumLines())
            endif
            WriteRecord()
        endif
    endif

    SaveCompileData()

    GotoBufferId(id)

    EndProcess()
end

proc EditCompiler()
    EditCompilerCommon(0)
end

public proc AddCompiler()
    EditCompilerCommon(ADD_RECORD)
end

proc DelCompiler()
    if num_compilers
        EditCompilerCommon(DELETE_RECORD)
    endif
end

proc CopyCompiler()
    EditCompilerCommon(COPY_RECORD)
end

integer exec_compiler_done

KeyDef ExecCompilerKeys
    <Escape>        exec_compiler_done = -1       EndProcess(TRUE)
    <Ctrl Enter>    exec_compiler_done = -2       EndProcess(TRUE)
    <Enter>         exec_compiler_done = 1        EndProcess(TRUE)
    <Ins>           AddCompiler()
    <GreyIns>       AddCompiler()
    <Del>           DelCompiler()
    <GreyDel>       DelCompiler()
    <Alt E>         EditCompiler()
    <Alt C>         CopyCompiler()
    <F1>            Help("Compile from list")
end

proc ExecCompilerHook()
    if Enable(ExecCompilerKeys)
        ListFooter("{F1}-Help")
    endif
end

proc MakePrimary()
    string ext[EXT_LEN]
    integer above

    GotoLine(compiler_start)
    ext = GetText(2,SizeOf(ext))
    PushBlock()
    MarkLine(compiler_start, compiler_end)
    BegLine()
    if lFind(''+ext,"bi^$")
        while lRepeatFind()
        endwhile
        above = Set(InsertLineBlocksAbove, TRUE)
        MoveBlock()
        Set(InsertLineBlocksAbove, above)
        SaveCompileData()
    endif
    PopBlock()
end

integer proc GetCompiler(string ext, integer invoke_list)
    integer id, display_full_list, exec = FALSE
    string keyword[48] = ext

    id = GotoBufferId(data_id)

    PushBlock()

    if not invoke_list and lFind(''+ext,"^i$g")
        ReadRecord()
        exec = TRUE
    else
        exec_compiler_done = 0
        display_full_list = FALSE
        repeat
            GenerateCompilerList(iif(display_full_list, '', ext))
            BegFile()
            lFind(keyword,"")
            Hook(_LIST_STARTUP_,ExecCompilerHook)
            Set(Y1, 3)
            lList("Select compiler", LIST_WIDTH, Query(ScreenRows), _ENABLE_SEARCH_ | _BLOCK_SEARCH_)
            UnHook(ExecCompilerHook)
            keyword = desc
            if exec_compiler_done == -2
                display_full_list = not display_full_list
                exec_compiler_done = 0
            endif
        until exec_compiler_done
        if exec_compiler_done == 1
            ReadRecordFromList(0)
            MakePrimary()
            exec = TRUE
        endif
    endif

    PopBlock()

    GotoBufferId(id)
    return (exec)
end

string proc JustDrive(string fn)
    return (Chr(Asc(SplitPath(fn, _DRIVE_))))
end

string proc JustPath(string fn)
    string s[MAXPATH] = SplitPath(fn, _PATH_)
    integer len

    len = Length(s)
    return (iif(len > 1, SubStr(s,1,len - 1), s))
end

string proc JustExt(string fn)
    string ext[EXT_LEN]

    ext = SplitPath(fn, _EXT_)
    return (ext[2:Length(ext) - 1])
end

integer proc _FileExists(string fn)
    return (FileExists(fn) & ~(_VOLUME_ | _DIRECTORY_))
end

/**************************************************************************
  Get the next token from s.  Delimited by spaces, ignore quoted spaces.
 **************************************************************************/
string proc mGetToken(string s, var integer p)
    integer in_quote, startp

    in_quote = FALSE
    startp = p
    // remove leading spaces
    while p <= Length(s) and s[p] == ' '
        p = p + 1
    endwhile

    while p <= Length(s)
        case s[p]
            when '"'
                if in_quote
                    in_quote = FALSE
                else
                    in_quote = TRUE
                endif
            when ' '
                if not in_quote
                    break
                endif
        endcase
        p = p + 1
    endwhile
    return (Trim(s[startp..p]))
end

/**************************************************************************
  Try to change "c:\foo bar\"e32.exe to "c:\foo bar\e32.exe"
 **************************************************************************/
proc mTempFixQuoteStuff()
    string s[MAXPATH], s2[MAXPATH], tok[MAXPATH]
    integer p

    PushPosition()
    s = GetText(1, CurrLineLen())
    s2 = ""
    p = 1
    while p <= Length(s)
        tok = mQuotePath(mRemoveQuotesFrom(mGetToken(s, p)))
        if tok == ""
            break
        endif
        s2 = s2 + " " + tok
    endwhile
    BegLine()
    KillToEol()
    InsertText(Trim(s2))
    PopPosition()
end

integer proc TranslateCommands(string fn0)
    string s[255], fn[MAXPATH]
    integer id, ok

    ok = TRUE
    fn = fn0

    id = GotoBufferId(batch_id)

    EmptyBuffer()

    AddLine(cmd)
    BegFile()
    while lFind('[~\\]\c;','x')
        DelChar()
        SplitLine()
    endwhile

    lReplace("\;",";","gn")

    // set the err filename, in temp dir, or dir of file to be compiled
    s = GetEnvStr("TEMP")
    if s == ""
        s = GetEnvStr("TMP")
        if s == ""
            s = SplitPath(fn, _DRIVE_|_PATH_)
        endif
    endif
    full_output_fn = MakeTempName(s, ".err")

    // make replacements within command(s)
    BegFile()
    lReplace('&Fn&'     ,QuotePath(fn), 'gin')
    lReplace('&Drive&'  ,JustDrive(fn), 'gin')
    lReplace('&Path&'   ,QuotePath(JustPath(fn)), 'gin')
    lReplace('&Name&'   ,QuotePath(SplitPath(fn,_NAME_)), 'gin')
    lReplace('&Ext&'    ,JustExt(fn)          , 'gin')
    lReplace('&Output&' ,QuotePath(full_output_fn), 'gin')
    lReplace('&CmdLine&',Query(MacroCmdLine)  , 'gin')

    // special handling for sc32
    if lFind("&SCPath&","gi")
        find_editor_program(SCEXE, s)
        if not _FileExists(s)
            s = SearchPath(SCEXE, Query(TSEPath))
            if s == ''
                s = SearchPath(SCEXE, GetEnvStr("PATH"))
                if s == ''
                    ok = Warn("Semware's SAL compiler '", SCEXE, "' not found")
                endif
            endif
        endif
        // On windows, we really need the ".exe".  Add it here if not specified
        lReplace("&SCPath&sc32 ", QuotePath(SplitPath(s,_DRIVE_ | _PATH_)) + SCEXE + " ","gin")
        lReplace("&SCPath&",QuotePath(SplitPath(s,_DRIVE_ | _PATH_)),"gin")
        lReplace("&SCFn&",SCEXE,"gin")
        // And translate -b (burnin) option to -s (external)
        lReplace(" -b&TSE", " -s&TSE", "gin")
    endif

    lReplace("&TSEPath&",QuotePath(LoadDir()),"gin")
    lReplace("&TSEFn&",QuotePath(SplitPath(LoadDir(1),_NAME_ | _EXT_)),"gin")

    if ok
        // user variable replacement
        PushBlock()
        while lFind('&{[~ \t]#}&','x')
            MarkFoundText()
            s = GetMarkedText()
            s = SubStr(s,2,Length(s) - 2)
            KillBlock()
            InsertText(GetGlobalStr(s),_INSERT_)
        endwhile
        PopBlock()
    endif
    mTempFixQuoteStuff()

    GotoBufferId(id)
    return (ok)
end

integer proc _EraseDiskFile(string fn)
    do 4 times
        if EraseDiskFile(fn)
            return (True)
        endif
        Delay(9)
    enddo
    return (False)
end

integer proc DeleteOutputFile()
    if not errors_in_source and _FileExists(full_output_fn) and not _EraseDiskFile(full_output_fn)
        return (Warn("Cannot delete output file '",full_output_fn,"'"))
    endif
    return (TRUE)
end

integer macro_debugging

integer proc ExecuteCommands(string fn, integer background)
    integer id, success, level, save_it
    string  exec[255]
#ifdef LOCKFIX
    integer locked
#endif

    success = TRUE

    id = GetBufferId()

    PushPosition()
    if GetBufferId(fn)
        GotoBufferId(GetBufferId(fn))
        if FileChanged() or not FileExists(CurrFilename())
            save_it = GetBufferId() == id
            if not save_it
                case YesNo(Format("Save changes to '",fn,"'"))
                    when 1  // yes
                        save_it = TRUE
                    when 2  // no... do nothing
                        if macro_debugging
                            success = FALSE
                        endif
                    otherwise
                        success = FALSE
                endcase
            endif
            if save_it
                if not SaveFile()
                    success = Warn("Cannot save file")
                endif
            endif
        endif
    else
        // may just be compiling file which is not loaded
        //success = FALSE
    endif
    PopPosition()

    if success
        Msg(Format("Compiling '",desc,"'..."))

        id = GotoBufferId(batch_id)
        BegFile()
        if NumLines() > 1
            exec = batch_fn
            if not SaveAs(batch_fn,_OVERWRITE_)
                success = Warn("Error saving batch file '",batch_fn,"'")
            else
                DbgMsg(Format("Executing: batch file ",batch_fn))
            endif
        else
            exec = GetText(1,SizeOf(exec))
            DbgMsg(Format("Executing: ",exec))
        endif
        GotoBufferId(id)

        if success and exec <> ""
            success = DeleteOutputFile()
            if success
                if exec[1] == '@'
                    success = ExecMacro(exec[2..SizeOf(exec)])
                else
                    level = Set(MsgLevel, _WARNINGS_ONLY_)

#ifdef LOCKFIX
                    locked = 0
                    if Query(BufferFlags) & _READONLY_LOCKING_
                        locked = _READONLY_LOCKING_
                    elseif Query(BufferFlags) & _HANDLE_LOCKING_
                        locked = _HANDLE_LOCKING_
                    endif
                    if locked
                        UnLockCurrentFile()
//                      Warn("Unlocking current file... ", iif((Query(BufferFlags) & (_READONLY_LOCKING_ | _HANDLE_LOCKING_)) == 0, "Success!", "Failure"))
                    endif
#endif

//                    success = Dos(exec, iif(dbg_enabled and output_no == 4, _TEE_OUTPUT_, output_type))
                    success = mDos(exec, iif(dbg_enabled and output_no == 4, _TEE_OUTPUT_, output_type), background)
                    SetGlobalStr(compile_error_fn, full_output_fn)

#ifdef LOCKFIX
                    if locked
                        LockCurrentFile(locked)
//                      Warn("Locking current file... ", iif((Query(BufferFlags) & (_READONLY_LOCKING_ | _HANDLE_LOCKING_)) <> 0, "Success!", "Failure"))
                    endif
#endif

                    Set(MsgLevel, level)
                    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
                endif
                if success and full_output_fn <> '' and not background and not _FileExists(full_output_fn)
                    Msg(Format("Error executing ", cmd))
                    Msg(Format("Output file not found: ", full_output_fn))
                    mAlarm()
                    success = FALSE
                endif
            endif
        endif
    endif

    return (success)
end

integer abort

// Mark error information
integer proc Search(string description, string search, string options, string tag, var string found_text)
    integer found, x, y, y1, attr

    found_text = ""
    if abort
        return (FALSE)
    endif

    found = TRUE
    if search <> ''
        if not lFind(search, options + 'x')
            UnMarkBlock()
            found = FALSE
        endif
    endif

    if found and tag <> "" and Val(tag) >= 0
        MarkFoundText(Val(tag))
        found_text = GetMarkedText()
    endif

    if dbg_enabled
        UpdateDisplay()
        x = WhereX()
        y = WhereY()

        y1 = y - Query(WindowY1) + 8
        if y1 + 4 > Query(ScreenRows)
            y1 = Query(ScreenRows) - 4
        endif

        if not found
            mAlarm()
        endif

        if PopWinOpen(10, y1, 70, y1 + 4, 1, description + iif(found, " found"," not found"), Query(MenuBorderAttr))
            attr = Set(Attr, Query(MenuTextAttr))
            ClrScr()
            VHomeCursor()
            WriteLine("Search for: ",search)
            WriteLine("Options:    ",options)
            Write    ("Tag:        ",tag)

            VGotoXY(5, 4)
            PutHelpLine("{Enter}-Single Step  {C}-Continuous    {Escape}-Abort")

            GotoXY(x - 10, y - y1)

            loop
                case GetKey()
                    when <Enter>
                        break
                    when <Escape>
                        abort = TRUE
                        break
                    when <c>, <Shift C>
                        dbg_enabled = FALSE
                        break
                endcase
            endloop

            Set(Attr, attr)
            PopWinClose()
        endif
    endif

    return (found)
end

/**************************************************************************
  Specil handling for Python.
  Python displays:

  bbbbProgram text
      ^

  The 'bbbb' represents spaces.  Python indents the program
  text.  So, we can't just use the position of the "^", we
  have to offset it by how many spaces procede the program
  text.
 **************************************************************************/
integer proc LineColFixup(string tag, integer n)
    integer result

    if Val(tag) == -2
        PushPosition()
        Up()
        GotoPos(PosFirstNonWhite())
        result = n - CurrCol() + 1
        PopPosition()
    else
        result = iif((tag == "" or Val(tag) == -1), n, Val(GetMarkedText()))
    endif
    return (result)
end

integer proc mSaveFile()
    string fn[MAXPATH] = CurrFilename()

    if AskFilename("New name for '"+CurrFilename()+"'", fn, _WRITE_ACCESS_)
        return (SaveAs(fn))
    endif
    return (FALSE)
end

menu ReloadMenu()
    "&Abandon Changes and Reload"
    "&Save Changes and Reload"
    "&Ignore Reload"
    "&Cancel Macro"
end

// Keep EditFile from showing errors or picklists
integer proc _EditFile(string fn)
    return (_FileExists(fn) and EditFile(QuotePath(fn), _DONT_PROMPT_))
end

integer proc LoadFile(string afn, integer force_reload)
    string fn[MAXPATH]

    fn = mGetLongPath(afn)

    if _EditFile(fn)
        if FileChanged()
            case ReloadMenu(fn + " has changed")
                when 1
                    AbandonFile()
                when 2
                    if mSaveFile()
                        AbandonFile()
                    else
                        return (FALSE)
                    endif
                when 3
                    // do nothing
                when 4
                    return (FALSE)
            endcase
        elseif force_reload
            AbandonFile()
        else
            return (TRUE)
        endif
        if _EditFile(fn)
            return (TRUE)
        endif
    endif
    return (Warn("Cannot load file '",fn,"'"))
end

proc SetupErrorWindow()
    integer n = WindowId()

    if not GotoWindow(2) or Query(WindowX1) > 2
        GotoWindow(n)
        OneWindow()
        HWindow()
        GotoBufferId(err_id)
        ResizeWindow(_UP_, -Query(WindowRows) + Min(4, NumLines()))
    endif

    // make sure the error file is in window #2
    GotoBufferId(err_id)
end

string macro_fn[128]
string burnin_fn[128]
integer fatal_errors
integer is_macro_file
integer loadable_macro
integer first_time

integer proc DisplayError()
    integer attr, n, tmp
    integer win1_x1, win1_y1, win1_len
    integer win2_x1, win2_y1, win2_len

    n = WindowId()

    // get error number and message
    GotoWindow(2)

    BegLine()
    // avoid showing <*** End of File ***> if possible
    tmp = CurrLine() - NumLines() + Query(WindowRows)
    if CurrRow() < tmp
        ScrollToRow(tmp)
    endif

    if err_no <> CurrLine() or err_row <> CurrRow() or CurrCol() <> err_col
        err_no  = CurrLine()
        err_row = CurrRow()
        err_col = CurrCol()
        err_msg = GetText(1,SizeOf(err_msg))

        win2_x1  = Query(WindowX1)
        win2_y1  = Query(WindowY1) + CurrRow() - 1
        win2_len = Query(WindowCols)

        // position at proper location in editing window
        GotoWindow(1)
        GotoMark(Str(err_no))
        ScrollToCenter()

        // simple hack for now to set the update flag for this file.
        // Basically, toggle the display mode to hex and back to
        // its original setting
        DisplayMode(DisplayMode(_DISPLAY_HEX_))

        win1_x1  = Query(WindowX1)
        win1_y1  = Query(WindowY1) + CurrRow() - 1
        win1_len = Query(WindowCols)

        // restore window and update display with appropriate changes
        GotoWindow(n)
        UpdateDisplay(_WINDOW_REFRESH_)

        // highlight cursor line in window 1
        VGotoXY(win1_x1, win1_y1)
        PutAttr(Query(HiLiteAttr), win1_len)

        // highlight cursor line in window 2
        VGotoXY(win2_x1, win2_y1)
        PutAttr(Query(HiLiteAttr), win2_len)

        // see if we're in or going to window #2
        if n == 2
            attr = Query(Attr)
            if loadable_macro and first_time
                UpdateDisplay(_STATUSLINE_REFRESH_)     // clear message
                first_time = FALSE
                BufferVideo()
                if PopWinOpen(17,6,63,17,2,"",Query(MenuBorderAttr))
                    Set(Attr, Query(MenuTextAttr))
                    ClrScr()
                    WriteLine("Macro compiled with only warnings and notes.")
                    Write("That is, no ")
                    Set(Attr, Query(MenuTextLtrAttr))
                    Write    ("fatal")
                    Set(Attr, Query(MenuTextAttr))
                    WriteLine(" errors were found.")
                    WriteLine("")
                    WriteLine("If, upon inspection of the given warnings, it")
                    WriteLine("is determined that the macro will perform")
                    WriteLine("properly in spite of these warnings, you may")
                    WriteLine("load and/or execute the macro by pressing")
                    Set(Attr, Query(MenuTextLtrAttr))
                    if WhichOS() == _LINUX_
                      Write    ("<Ctrl E>")
                    else
                      Write    ("<Ctrl Enter>")
                    endif
                    Set(Attr, Query(MenuTextAttr))
                    WriteLine(" while in the Error window.")
                    WriteLine("")
                    mAlarm()
                    Write("         Press any key to continue")
                    UnBufferVideo()
                    GetKey()
                    PopWinClose()
                    Set(Attr, attr)
                else
                    UnBufferVideo()
                endif
            endif
            Set(Attr, Query(CurrWinBorderAttr))
            VGotoXY(Query(WindowX1) + 10, Query(WindowY1) - 1)
            PutStr("<Enter> to select, Cursor keys to move, or <F1> for Help")
            Set(Attr, attr)
        endif
        return (TRUE)
    endif

    GotoWindow(n)
    return (FALSE)
end

constant NO_MSG         = 0x00,
         NO_PREV_ERR    = 0x01,
         NO_MORE_ERR    = 0x02

proc LeftBtn()
    integer cursorattr, cursorinblockattr, blockattr

    case MouseHotSpot()
        when _NONE_         ,
             _MOUSE_ZOOM_   ,
             _MOUSE_VWINDOW_,
             _MOUSE_HWINDOW_

        when _MOUSE_CLOSE_
             ProcessHotSpot()
             EndProcess(FALSE)

        when _MOUSE_MARKING_
            if MouseWindowId() == 2
                goto process_mouse
            elseif GotoWindow(MouseWindowId())
                EndProcess(FALSE)
            endif

        otherwise
            process_mouse:

            PushBlock()
            cursorattr        = Set(CursorAttr,         Query(MsgAttr))
            cursorinblockattr = Set(CursorInBlockAttr,  Query(MsgAttr))
            blockattr         = Set(BlockAttr,          Query(TextAttr))
            ProcessHotSpot()
            Set(CursorAttr          , cursorattr)
            Set(CursorInBlockAttr   , cursorinblockattr)
            Set(BlockAttr           , blockattr)
            PopBlock()
            DisplayError()
    endcase
end

proc Move(integer success, integer flag)
    if not DisplayError() and not success
        case flag
            when NO_PREV_ERR
                mAlarm()
                Message("No previous errors")
            when NO_MORE_ERR
                mAlarm()
                Message("No more errors")
        endcase
    endif
end

/**************************************************************************
  Hilite the current line in the specified color
 **************************************************************************/
proc mHiliteCurrLine(integer colour)
    // update display with appropriate changes
    UpdateDisplay(_WINDOW_REFRESH_)

    // highlight cursor line in window 2
    PutAttrXY(Query(WindowX1), Query(WindowY1) + CurrRow() - 1, colour, Query(WindowCols))
end

/**************************************************************************
  Find the length of the longest line in the current window
 **************************************************************************/
integer proc mLongestLineOnScreen()
    integer longest_line = 0

    PushPosition()
    BegWindow()
    do Query(WindowRows) times
        longest_line = Max(CurrLineLen(), longest_line)
        Down()
    enddo
    PopPosition()
    return (longest_line)
end

/**************************************************************************
  Roll right, but only if the specified line_length is off-screen
 **************************************************************************/
proc mRollRight(integer linelen)
    integer d = linelen - Query(ScreenCols)
    if d > 0 and CurrXOffset() < d
        RollRight()
    endif
end

KeyDef ErrorKeys
    <PREV_ERROR_KEY>    Move(Up()       , NO_PREV_ERR)
    <CursorUp>          Move(Up()       , NO_PREV_ERR)

    <NEXT_ERROR_KEY>    Move(Down()     , NO_MORE_ERR)
    <CursorDown>        Move(Down()     , NO_MORE_ERR)

    <Home>              Move(BegFile()  , NO_MSG     )
    <Ctrl PgUp>         Move(BegFile()  , NO_MSG     )

    <End>               Move(EndFile()  , NO_MSG     )
    <Ctrl PgDn>         Move(EndFile()  , NO_MSG     )

    <PgUp>              Move(PageUp()   , NO_PREV_ERR)
    <PgDn>              Move(PageDown() , NO_MORE_ERR)

    <CursorRight>       mRollRight(mLongestLineOnScreen()) mHiliteCurrLine(Query(HiLiteAttr))
    <CursorLeft>        RollLeft()                         mHiliteCurrLine(Query(HiLiteAttr))

    <F1>                Help("Compiler Key Assignments")

    <LeftBtn>           LeftBtn()
    <RightBtn>          EndProcess(FALSE)

    <Enter>             EndProcess(TRUE)
    // accept either ctrl-e or ctrl-enter
    <Ctrl E>            if loadable_macro
                            EndProcess(2)
                        endif
    <Ctrl Enter>        if loadable_macro
                            EndProcess(2)
                        endif
    <Escape>            EndProcess(3)
end

forward proc ProcessErrorWindow()

// Determine when the user has switched to the Error window.
proc AfterEditingCommand()
    if WindowId() == 2 and GetBufferId() == err_id
        ProcessErrorWindow()
    endif
end
proc CloseErrorWindow()
    integer n = WindowId(), id = GetBufferId()

    if GotoWindow(2) and GetBufferId() == err_id
        CloseWindow()
    endif

    GotoWindow(n)
    GotoBufferId(id)
    UpdateDisplay(_WINDOW_REFRESH_)
end

proc LoadCurrMacro(integer execute_it)
    if LoadMacro(QuotePath(macro_fn)) and execute_it and isMacroLoaded(macro_fn)
        ExecMacro(QuotePath(macro_fn))
    endif
end

menu ExecLoadMenu()
    "&Execute Macro"  ,   LoadCurrMacro(TRUE)
    "&Load Macro"     ,   LoadCurrMacro(FALSE)
    "&Cancel"
end

proc ProcessErrorWindow()
    integer n, enhanced_keys

    err_col = -1    // force update
    DisplayError()
    if Enable(ErrorKeys, _EXCLUSIVE_)
        UnHook(AfterEditingCommand)

        enhanced_keys = Set(EquateEnhancedKbd, TRUE)
        n = Process()
        Set(EquateEnhancedKbd, enhanced_keys)
        Disable(ErrorKeys)

        GotoWindow(1)
        if n == 2 and loadable_macro
            CloseErrorWindow()
            if macro_debugging
                ExecMacro(Format("Debug ",compile_fn))
            else
                ExecLoadMenu("No fatal errors...")
            endif
        elseif n == 3
            CloseErrorWindow()
            num_errors = 0
        else
            DisplayError()
            Message(err_msg)
            Hook(_AFTER_COMMAND_,AfterEditingCommand)
        endif

    endif
end

integer proc isMacroFile(string ext)
    integer id

    macro_fn = ''
    burnin_fn = ''
    fatal_errors = 0
    loadable_macro = 0
    InstallSuccessfulMsg = ""
    LoadableUI = ""
    LoadableCfg = ""

    if ext in ".s",".ui",".cfg",".si"
        PushBlock()
        id = GotoBufferId(work_id)
        BegFile()
        if lFind("SAL Compiler","^c") or lFind("SAL GUI Compiler","^c")
            if lFind("^Writing output to file '{.*}'$","xg")
                MarkFoundText(1)
                macro_fn = GetMarkedText()
                macro_fn = SplitPath(macro_fn, _DRIVE_ | _NAME_)
                macro_fn = QuotePath(macro_fn)
            endif

            if lFind("^Burning configuration into file '{.*}'$","xg")
                MarkFoundText(1)
                burnin_fn = GetMarkedText()
            endif

            PushBlock()
            UnMarkBlock()
            MarkLine()

            // see if we eventually succeeded.  If so, remove related error messages
            if lFind("^Installation successful!","x")
                InstallSuccessfulMsg = GetText(1,CurrLineLen())
                Up()
                MarkLine()
                if lFind("Creating separate user-interface file '{.*}'","xlg")
                    LoadableUI = GetFoundText(1)
                endif
                if lFind("Creating separate settings files '{.*}'","xlg")
                    LoadableCfg = GetFoundText(1)
                endif
                KillBlock()
            endif

            PopBlock()
            fatal_errors = lFind("^Error  ","xg")
        endif

        GotoBufferId(id)
        PopBlock()
        loadable_macro = fatal_errors == 0 and burnin_fn == ''
        first_time = TRUE
        return (TRUE)
    endif
    return (FALSE)
end

/*************************************************************************
  error file format is:
  fn
  line col
  message
 *************************************************************************/
proc mark_user_errors(integer err_id)
    string fn[_MAX_PATH_], line[10], col[10]

    GotoBufferId(err_id)
    if dbg_enabled UpdateDisplay() DbgMsg("Marking errors (file from helper macro displayed)") endif
    if NumLines() > 0
        BegFile()

        // line col "fn" error message
        repeat
            fn = GetText(1, sizeof(fn))
            KillLine()
            line = GetToken(GetText(1, MAXSTRINGLEN), ' ', 1)
            col  = GetToken(GetText(1, MAXSTRINGLEN), ' ', 2)
            KillLine()
            // Now should be on the message line

            EditThisFile(fn)
            GotoLine(val(line))
            GotoColumn(iif(col == "", 1, val(col)))
            if not PlaceMark(Str(num_errors + 1))
                Warn("Max errors reached... Only first ",num_errors," are marked")
                break
            else
                num_errors = num_errors + 1
            endif
            GotoBufferId(err_id)
        until not Down()
    endif
    DbgMsg(Format(num_errors; "errors marked"))
end

integer proc call_user_rule_macro(string cur_fn, integer err_id)
    string helper[30] = "compiler-helper"
    string debug[10] = iif(dbg_enabled, "-debug", "")

    if rule_mac == ""
        return (False)
    endif

    if not ExecMacro(Format(helper; cur_fn; err_id; debug), rule_mac)
        return (Warn("Error macro"; rule_mac; "not found"))
    endif
    PurgeMacro(helper)

    mark_user_errors(err_id)

    return (True)
end

integer proc MarkErrors(integer delete_error_file)
    integer id, success, buffer_type, tab_width
    integer line, col, work, n, ok
    string fn[MAXPATH], curr_fn[MAXPATH], curr_ext[MAXPATH], s[MAXPATH]

    // Clear error buffer
    EmptyBuffer(err_id)

    num_errors =  0
    success = TRUE
    errors_in_source = FALSE

    PushBlock()

    curr_ext = mCurrExt()
    curr_fn = CurrFilename()
    id = GetBufferId()

    if rule_mac == "" and Trim(fn_tag) == ''   // errors in source!
        work = id
        errors_in_source = TRUE
        success = LoadFile(curr_fn, TRUE)
        if success and dbg_enabled
            DbgMsg(Format("Checking errors in '",curr_fn,"'"))
        endif
    else
        work = work_id
        GotoBufferId(work_id)
        EmptyBuffer()
        success = InsertFile(full_output_fn, _DONT_PROMPT_)
        if success and dbg_enabled
            DbgMsg(Format("Checking errors in '",full_output_fn,"'"))
        endif
    endif

    if delete_error_file and not dbg_enabled
        DeleteOutputFile()  // for now, always delete error output file
    endif

    is_macro_file = isMacroFile(curr_ext)

    if success and NumLines()
        GotoBufferId(work_id)
        BegFile()
        Msg("Checking for errors")

        abort = FALSE

        ok = call_user_rule_macro(curr_fn, err_id)
        if not ok and Search("Error", err_ss, err_opts, '', s)
            ok = True
            repeat
                PushPosition()
                fn = ""
                Search("Filename", fn_ss,  fn_opts,  fn_tag, fn )
                if fn == '' or Trim(fn_tag) == ''
                    fn = curr_fn
                endif
                PopPosition()

                PushPosition()
                Search("Line number", ln_ss,  ln_opts,  ln_tag, s)
                line = LineColFixup(ln_tag, CurrLine())
                PopPosition()

                PushPosition()
                Search("Column number", col_ss, col_opts, col_tag, s)
                buffer_type = BufferType(_HIDDEN_)
                tab_width = Set(TabWidth, 8)
                col = LineColFixup(col_tag, 1)
                BufferType(buffer_type)
                Set(TabWidth, tab_width)
                PopPosition()

                PushPosition()
                s = ""
                Search("Error message", msg_ss, msg_opts, msg_tag, s)
                PopPosition()

                if abort
                    success = FALSE
                    break
                else
                    if fn == '' or not LoadFile(fn, FALSE)
                        break
                    else
                        PushPosition()  // in case errors in source file
                        GotoLine(line)
                        GotoPos(col)
                        n = PlaceMark(Str(num_errors + 1))
                        PopPosition()
                        if n == 0
                            Warn("Max errors reached... Only first ",num_errors," are marked")
                            break
                        else
                            num_errors = num_errors + 1
                            AddLine(s, err_id)
                        endif
                    endif
                    GotoBufferId(work)
                endif
            until not Search("Error",err_ss,err_opts+'+','', s)
        endif

        if ok
            GotoBufferId(err_id)

            if InstallSuccessfulMsg <> ""
                EndFile()
                AddLine(InstallSuccessfulMsg)
            endif

            FileChanged(FALSE)
        endif
    endif

    PopBlock()
    GotoBufferId(id)
    return (success)
end

proc ProcessErrors()
    integer attr

    if num_errors > 0
        SetupErrorWindow()
        BegFile()
    else
        CloseErrorWindow()
    endif

    if is_macro_file and fatal_errors == 0 and (burnin_fn <> '' or  (Length(LoadableUI) <> 0 or Length(LoadableCfg) <> 0))
        UpdateDisplay()

        if PopWinOpen(13,6,67,18,2,"",Query(MenuBorderAttr))
            attr = Set(Attr, Query(MenuTextAttr))
            ClrScr()
            WriteLine("")
            if (Length(LoadableUI) == 0 and Length(LoadableCfg) == 0)
                WriteLine("New interface has been burned into the editor")
                WriteLine("")
                PutStr   (SqueezePath(burnin_fn, Query(PopWinCols)), Query(MenuTextLtrAttr))
                WriteLine("")
            else
                WriteLine("New interface file(s) installed:")
                WriteLine("")
                if Length(LoadableUI)
                    PutStr(SqueezePath(LoadableUI, Query(PopWinCols)), Query(MenuTextLtrAttr))
                    WriteLine("")
                endif
                if Length(LoadableCfg)
                    PutStr(SqueezePath(LoadableCfg, Query(PopWinCols)), Query(MenuTextLtrAttr))
                    WriteLine("")
                endif
            endif
            WriteLine("")
            WriteLine("The new configuration will not take effect until")
            WriteLine("the editor is restarted")
            WriteLine("")
            Write("              Press any key to continue")
            GetKey()
            PopWinClose()
            Set(Attr, attr)
            loadable_macro = FALSE
            //LoadUserInterface() - don't do this now
        endif
    endif

    if num_errors > 0
        ProcessErrorWindow()
    elseif is_macro_file and loadable_macro
        if not macro_debugging
            ExecLoadMenu("Compilation Successful... No Errors.")
        endif
    else
        Message("Compilation Successful... No Errors.")
    endif
end

proc ExecCompiler(string fn, string ext, integer flags)
    integer background = flags & BACKGROUND_FLAG,
        process_error_file = flags & PROCESS_ERROR_FILE_FLAG,
        invoke_list = flags & INVOKE_LIST_FLAG

    if ChangedFilesExist()
        if query_save_changed_files
            case MsgBox("", "Save Changed Files?", _YES_NO_CANCEL_)
                when 0 return ()            // cancel
                when 1 SaveAllFiles()       // yes
                when 2                      // no
            endcase
        elseif save_changed_files
            SaveAllFiles()
        endif
    endif

    // get the *Exact* case of the filename for compilers which require it such as Javac
    compile_fn = QuotePath(ExpandPath(fn, TRUE))
    SetGlobalStr(compile_source_fn, compile_fn)
    if GetCompiler(ext, invoke_list) and (process_error_file or TranslateCommands(compile_fn))
        if process_error_file or ExecuteCommands(compile_fn, background)
            if not background
                if MarkErrors(TRUE)
                    ProcessErrors()
                endif
                EmptyBuffer(work_id)
            endif
        endif
    endif
end

proc NextPrevErr(integer next)
    integer moved, line

    SetupErrorWindow()
    moved = iif(next, Down(), Up())
    GotoWindow(1)

    if num_errors == 1
        err_col = -1    // force update
        line = CurrLine()
        DisplayError()
        if CurrLine() <> line
            return()
        endif
    endif
    Move(moved, iif(next, NO_MORE_ERR, NO_PREV_ERR))
end

public proc PrevErr()
    NextPrevErr(FALSE)
end

public proc NextErr()
    NextPrevErr(TRUE)
end

proc PrevErrKey()
    iif(num_errors, PrevErr(), ChainCmd())
end

proc NextErrKey()
    iif(num_errors, NextErr(), ChainCmd())
end

/*
public proc CompileExt()
    string ext[4] = ''

    if Query(MacroCmdLine) == "" and Ask("Specify extension to compile", ext)
        Set(MacroCmdLine, ext)
    endif
    if Query(MacroCmdLine) <> ""
        ext = GetToken(Query(MacroCmdLine),' ',1)
        Set(MacroCmdLine, SubStr(Query(MacroCmdLine),Length(ext),255))
        ExecCompiler(ext, FALSE)
    endif
end
*/

public proc CompileCurrentFile()
    ExecCompiler(CurrFilename(), mCurrExt(), 0)
end

public proc DebugCompilerSetup()
    dbg_enabled = TRUE
    ExecCompiler(CurrFilename(), mCurrExt(), INVOKE_LIST_FLAG)
    dbg_enabled = FALSE
end

proc SwitchWindow()
    if WindowId() == 2
        GotoWindow(1)
    else
        SetupErrorWindow()
    endif
end

integer proc InsertCompileData(string fn, integer error)
    integer success

    PushBlock()
    success = InsertFile(fn, _DONT_PROMPT_)
    PopBlock()
    BegFile()
    if GetText(1,CurrLineLen()) <> EyeCatcher
        success = FALSE
        if error
            Warn("Invalid data file '",fn,"'")
        endif
    endif
    if lFind("^DBG={.*}$","xg")
        dbg_ext = GetFoundText(1)
    endif
    return (success)
end

integer proc LoadPrevDataFile(var string prev_data_fn)
    integer id, ok

    PushPosition()
    if CreateTempBuffer()
        id = GetBufferId()
        ok = InsertCompileData(prev_data_fn, FALSE)
        if not ok
            prev_data_fn = ExpandPath(prev_data_fn)
            if SplitPath(prev_data_fn, _NAME_) == "*"
                prev_data_fn = SplitPath(prev_data_fn, _DRIVE_ | _PATH_) + SplitPath(data_fn,_NAME_) + SplitPath(prev_data_fn, _EXT_)
            endif
            if SplitPath(prev_data_fn, _EXT_) in ".*", ""
                prev_data_fn = SplitPath(prev_data_fn, _DRIVE_ | _PATH_ | _NAME_) + SplitPath(data_fn, _EXT_)
            endif
            ok = InsertCompileData(prev_data_fn, FALSE)
        endif

        if ok
            KillPosition()
        else
            PopPosition()
            AbandonFile(id)
            id = 0
        endif
    else
        PopPosition()
    endif
    return (id)
end

integer proc isSetupSame(integer id)
    string data[255], merge[255]

    do 20 times
        GotoBufferId(data_id)
        data  = GetText(1,SizeOf(data))
        Down()
        GotoBufferId(id)
        merge = GetText(1,SizeOf(merge))
        Down()
        if data <> merge
            return (FALSE)
        endif
    enddo
    return (TRUE)
end

proc RemoveDuplicateSetups(integer merge_id)
    integer data_line, merge_line

    GotoBufferId(data_id)
    BegFile()
    if lFind('',"^")
        repeat
            data_line = CurrLine()
            Down()
            Message("Checking: ",GetText(1,132))
            Up()
            GotoBufferId(merge_id)
            BegFile()
            while lRepeatFind()
                merge_line = CurrLine()
                GotoBufferId(data_id)
                GotoLine(data_line)
                if isSetupSame(merge_id)
                    GotoLine(merge_line)
                    KillLine(20)
                    Up()
                endif
            endwhile
            GotoBufferId(data_id)
        until not lRepeatFind()
    endif
    GotoBufferId(merge_id)
end

proc MergeDataFile()
    integer attr, id, merge_id
    string path[MAXPATH] = '', prev_data_fn[MAXPATH]

    id = GetBufferId()

    if PopWinOpen(8, 4, 72, 9, 1, "Merge compiler data", Query(MenuBorderAttr))
        attr = Set(Attr, Query(MenuTextAttr))
        ClrScr()
        WriteLine("To merge compiler setups from a previous version of TSE,")
        WriteLine("Enter the path to the previous version of the compile macro")
        WriteLine("")
        if Read(path)
            prev_data_fn = path
            merge_id = LoadPrevDataFile(prev_data_fn)
            if merge_id
                RemoveDuplicateSetups(merge_id)
                BegFile()
                if lFind('',"^g")
                    PushBlock()
                    UnMarkBlock()
                    MarkLine()
                    EndFile()
                    GotoBufferId(data_id)
                    EndFile()
                    AddLine()
                    CopyBlock()
                    SaveCompileData()
                    Message("Compile data file updated")
                else
                    Message("Compile data already up-to-date")
                endif
                AbandonFile(merge_id)
            else
                Warn("Cannot locate 'compile.dat' in '",path,"'")
            endif
        endif
        PopWinClose()
        Set(Attr, attr)
    endif
    GotoBufferId(id)

end

proc SetDbgExt()
    if Read(dbg_ext)
        PushPosition()
        if GotoBufferId(data_id)
            if not lReplace("{^DBG=}{.*}$","\1"+dbg_ext,"xg1")
                BegFile()
                AddLine("DBG="+dbg_ext)
            endif
            SaveCompileData()
        endif
        PopPosition()
    endif
end

menu AdvancedSetupMenu()
    "TSE Debugger E&xtension" [dbg_ext:4] ,SetDbgExt()    ,CloseAfter ,"Specify extension to use for compiling TSE macros for macro_debugging"
    "&Merge Data File"                    ,MergeDataFile(),CloseBefore,"Merge compiler data file with previous data file"
end

menu TheCompileMenu()
    title = "Compile Menu"

    "&Compile Current File" ,CompileCurrentFile()         ,CloseAllBefore
    "Compile from &List"    ,ExecCompiler(CurrFilename(), mCurrExt(), INVOKE_LIST_FLAG),CloseAllBefore
    "&Previous Error"       ,PrevErr()                    ,CloseBefore
    "&Next Error"           ,NextErr()                    ,CloseBefore
    "&Switch Window"        ,SwitchWindow()               ,CloseBefore
    "Close Error &Window"   ,CloseErrorWindow()           ,CloseAllBefore
    "&Help on Keys"         ,Help("Compiler Key Assignments"), DontClose
    ""                      ,                             ,Divide
    "&Advanced Setup"       ,AdvancedSetupMenu()          ,DontClose
    "&Debug Compiler Setup" ,DebugCompilerSetup()         ,CloseAllBefore
    "&Overview Help"        ,Help("tsehelp|Compiling Programs from Within the Editor"),  DontClose
end

public proc CompileMenu()
    TheCompileMenu()
end

public proc ParseErrorFile()
    full_output_fn = CurrFilename()
    if GetCompiler(CurrExt(), TRUE)
        if MarkErrors(FALSE)
            ProcessErrors()
        endif
        EmptyBuffer(work_id)
    endif
end

integer proc SetupCompileFiles()
    integer id, ok

    ok = FALSE

    find_editor_program(DATA_FN, full_data_fn)
    if not _FileExists(full_data_fn)
        full_data_fn = SplitPath(CurrMacroFilename(), _DRIVE_ | _PATH_) + DATA_FN
    endif
    if not _FileExists(full_data_fn)
        full_data_fn = SearchPath(DATA_FN, Query(TSEPath), "mac")
    endif
    if full_data_fn == ""
        new_file = TRUE
        full_data_fn = DATA_FN
    endif

    id = GetBufferId()

    work_id  = CreateTempBuffer()
    err_id   = CreateTempBuffer()
    batch_id = CreateTempBuffer()
    data_id  = CreateTempBuffer()

    if data_id and not new_file
        ok = InsertCompileData(full_data_fn, TRUE)
        dbg_ext = iif(lFind("^DBG={.*}$","xgi"),GetFoundText(1),".dbg")
        BegFile()
    endif
    EndFile()
    if CurrLineLen()
        AddLine()   // make sure there's at least 1 blank line at end
    endif

    FileChanged(FALSE)
    GotoBufferId(id)
    return (ok)
end

string cmdline[255]

integer whenloaded_executed

proc Main()
    integer extension_specified, n, p, flags = 0, background = FALSE

    string ext[EXT_LEN]
    string fn[MAXPATH], s[MAXPATH]

    // This is ONLY required of the compile macro since it can
    // be called from DEBUG macro.
    if not whenloaded_executed
        WhenLoaded()
        if not whenloaded_executed
            return()
        endif
    endif

    fn = ""
    extension_specified = FALSE
    num_errors = -1

    cmdline = Query(MacroCmdLine)
    if Trim(cmdline) == ''
        cmdline = QuotePath(CurrFilename())
        if not AskFilename("File to compile", cmdline, 0, GetFreeHistory("Compile:cmdline"))
            return ()
        elseif Pos(' ', cmdline)            // SEM 07/13/98 AskFilename returns a single, non-quoted fn
            cmdline = QuotePath(cmdline)
        endif
    endif

    cmdline = Trim(cmdline) // CHO 08/04/19 Removed Lower() here, compensated for it in the following lines

    case Lower(cmdline[1:2])
        when '-c'
            CompileCurrentFile()
        when '-m'
            TheCompileMenu()
            return ()
        when '-p'   // -p[error_fn [source]]
            flags = flags | PROCESS_ERROR_FILE_FLAG
            s = cmdline[3:SizeOf(cmdline)]
            n = NumFileTokens(s)
            if n > 0
                full_output_fn = GetFileToken(s, 1)
                if n > 1
                    fn = GetFileToken(s, 2)
                endif
            endif

            if full_output_fn == ""
                full_output_fn = GetGlobalStr(compile_error_fn)
            endif
            if fn == ""
                fn = GetGlobalStr(compile_source_fn)
            endif
            ExecCompiler(fn, SplitPath(fn, _EXT_), flags)
        otherwise
            if Lower(cmdline[1:2]) == '-x'
                extension_specified = TRUE
                p = Pos(' ', cmdline[3:Length(cmdline)])
                ext = cmdline[3..iif(p, p, Length(cmdline))]
                if ext[1] <> '.'
                    ext = '.' + ext
                endif
                cmdline = iif(p, Trim(cmdline[p:Length(cmdline)]), "")
            endif
            case Lower(cmdline[1:2])
                when "-b"
                    background = TRUE
                    flags = flags | BACKGROUND_FLAG
                    fn = Trim(cmdline[3:Length(cmdline)])
                when "-d"
                    extension_specified = TRUE
                    macro_debugging = TRUE
                    ext = dbg_ext
                    fn = Trim(cmdline[3:Length(cmdline)])
                otherwise
                    fn = cmdline
            endcase

            if background and fn == ""
                fn = CurrFilename()
            endif
            if not extension_specified
                ext = SplitPath(fn, _EXT_)
            endif
            ExecCompiler(fn, ext, flags)
    endcase

    Set(MacroCmdLine, Str(num_errors))
    macro_debugging = FALSE
end

proc DefineKey(string name, string Key)
    SetGlobalStr("Compile->" + name, '<'+iif(Key <> '',
                iif(Key == KeyName(NOT_DEFINED), "Undefined",Key),
                "Undefined")+'>')
end

proc WhenLoaded()
    SCEXE = "sc32.exe"
    separator = GetDirSeparator()
    if WhichOS() == _LINUX_
        SCEXE = "sc32"
    endif
    DefineKey("PrevErrorKey",    KeyName(<PREV_ERROR_KEY>))
    DefineKey("NextErrorKey",    KeyName(<NEXT_ERROR_KEY>))
    DefineKey("SwitchWindowKey", KeyName(<SWITCH_WINDOW_KEY>))
    DefineKey("CompileMenuKey",  KeyName(<COMPILE_MENU_KEY>))
    DefineKey("CompileFileKey",  KeyName(<COMPILE_FILE_KEY>))
    LocalHelp(TRUE)

    find_editor_program("tee32.exe", tee32)
    find_editor_program("tee16.exe", tee16)

    if not SetupCompileFiles()
        Warn(data_fn, " not found.")
        PurgeMacro(CurrMacroFilename())
    else
        whenloaded_executed = TRUE
    endif
end

proc WhenPurged()
    DefineKey("PrevErrorKey",    "")
    DefineKey("NextErrorKey",    "")
    DefineKey("SwitchWindowKey", "")
    DefineKey("CompileMenuKey",  "")
    DefineKey("CompileFileKey",  "")

    AbandonFile(batch_id)
    AbandonFile(work_id)
    AbandonFile(data_id)
    AbandonFile(err_id)
end

// key assignments

<COMPILE_MENU_KEY>      CompileMenu()
<COMPILE_FILE_KEY>      CompileCurrentFile()
<NEXT_ERROR_KEY>        NextErrKey()
<PREV_ERROR_KEY>        PrevErrKey()
<SWITCH_WINDOW_KEY>     SwitchWindow()
