/*************************************************************************
  Clock       StatusLine clock for TSE

  Author:     Ian Campbell (Contributing User)

  Date:

  Version:    1.02

  Overview:

  This clock uses binary support to help determine whether or not data
  is overlaying the spot where the time would go.  It also retrieves
  its color attribute byte from the first byte on the screen, so it
  always knows what color to set the clock to.

  In addition, if "Eliminate Snow" is set within TSE, snow checking
  will also be performed, if the card is determined to be a CGA card.

  Keys:       (none)

  Usage notes:

  Simply load or execute this macro to display a simple clock on the
  StatusLine.

*************************************************************************/

// global values
integer DISPLAY_SECONDS = TRUE  // set this to FALSE to omit seconds.
integer SavedSeconds            // update screen if current seconds do not match
string LastTime[13] = ""        // last time string output to the screen

/*
Check a string for spaces.
Return TRUE if the string is composed completely of spaces.
Otherwise return FALSE.
*/
integer proc CheckForAllSpaces(string s)
    integer i = 1

    while i <= Length(s)
        if s[i] <> Chr(Query(StatusLineFillChar))
            return(FALSE)
        endif
        i = i + 1
    endwhile
    return(TRUE)
end CheckForAllSpaces

/*
This routine interfaces with the binary to read text from the physical screen.
It takes the column, row, and the string length, and translates them into
the appropriate screen offset.

On return, it fills in the string text (via var string s), and then
returns the color attribute of the first byte in the string.
*/
integer proc mGetScreenText(var string s, integer column, integer row, integer StringLength)
    string s1[1], s2[1]

    s1 = ""
    s2 = ""
    GetStrXY(column, row, s, StringLength)
    GetStrAttrXY(column, row, s1, s2, 1)
    return (Asc(s2[1]))
end mGetScreenText

#if 0
proc Show(string msg)
    vGotoXYAbs(80, 1)
    PutStr(msg, Query(StatusLineAttr))
end
#endif

/*
Main clock entry point hooked to _IDLE_
*/
proc clock()
    string ThisTime[13]
    string ThisScreenText[13] = ""
    string FillChar[1] = Chr(Query(StatusLineFillChar))
    string TimeStringPad[4] = FillChar
    string sep[1] = Chr(Query(TimeSeparator))
    string s[1] = ""
    integer hours, minutes, seconds, hundreds
    integer ClockRow = Query(StatusLineRow)
    integer ShowSeconds = DISPLAY_SECONDS
    integer OriginalAttr

    // vote to see which color attribute is used for the clock!
    if (mGetScreenText(s, 1, ClockRow, 0) == mGetScreenText(s, 2, ClockRow, 0))
        OriginalAttr = Set(Attr, mGetScreenText(s, 1, ClockRow, 0))
    else
        OriginalAttr = Set(Attr, mGetScreenText(s, 3, ClockRow, 0))
    endif

    if Query(ShowStatusLine) == off
        return()
    endif

//    Show("CLOCK")

    GetTime(hours, minutes, seconds, hundreds)
    if SavedSeconds <> seconds
        SavedSeconds = seconds
        if Query(TimeFormat) == 2           // 12 hour clock?
            TimeStringPad = FillChar + iif(hours < 12, 'am ', 'pm ')
            hours = iif(hours == 0, 12, hours)
            hours = iif(hours > 12, hours - 12, hours)
        endif
        loop
            ThisTime = format(hours:3:FillChar, sep, minutes:2:"0")
                + iif(ShowSeconds, format(sep, seconds:2:"0"), "")
                    + TimeStringPad
            mGetScreenText(ThisScreenText, Query(ScreenCols)
                - Length(ThisTime) + 1, ClockRow, Length(ThisTime))
            if ThisScreenText == LastTime or CheckForAllSpaces(ThisScreenText)
                VGoToXYAbs(Query(ScreenCols) - Length(ThisTime) + 1, ClockRow)
                PutLine(ThisTime, Length(ThisTime))
                LastTime = ThisTime         // track the screen time updates
                break                       // exit the loop
            else                        // clock won't fit on the status line!
                // first, dump the "am/pm indicators"
                if Length(TimeStringPad) > 1
                    TimeStringPad = FillChar
                // next, dump the "seconds"
                elseif ShowSeconds
                    ShowSeconds = FALSE
                // finally, dump the right space
                elseif Length(TimeStringPad)
                    TimeStringPad = ""
                else
                    break
                endif
            endif
        endloop
    endif
    Set(Attr, OriginalAttr)             // put the original attribute back

//    Show("     ")
end

// force the clock to be updated after every updatedisplay.
proc ForceClock()
    SavedSeconds = 99               // invalid value --> immediate update
//    Clock()                         // update this immediately
end ForceClock

proc WhenLoaded()
    Hook(_IDLE_, clock)
    Hook(_AFTER_UPDATE_STATUSLINE_, ForceClock)
    Hook(_AFTER_UPDATEDISPLAY_, ForceClock)
end

