/*************************************************************************
  CD          A "chdir" command for TSE.

  Author:     Ian Campbell

  Date:       Aug 22, 1994 (v0.95)
              Oct 09, 1996 - Take advantage of TSE Pro 2.5 features.

  Update:     Mar 9,1998  PHL - replaced binary functions with calls to
                          equivalents native to Tse Pro/32 (#ifdef WIN32)
                          resized file name strings to _MAXPATH_ chars.

              Mar 27, 1998 SEM - make it work for DOS and win32, and merge in
                          additional in-house changes.

              Jun  1, 1998 SEM - Drive comparision was not case insensitive,
                            causing things like "cd .." to fail.

              Aug 17, 1999 SEM - Remove autoload stuff, since "Change Dir" is
                            not on the file menus.

              Mar  9, 2001 SEM - Commented out <CtrlAlt D> key assignment,
                            since cd is now on the file menu. Suggested by
                            Bill Stewart.

              Aug 31, 2001 SEM - Logic error fix. When the macro was
                            converted from DOS to Win32, the DOS global
                            DTA was converted to a Windows FindFile
                            handle.  However, this handle was made
                            local, thus breaking the logic of the
                            original version, and causing the macro to
                            not run properly.  The fix was to make the
                            FindFile handle global (yuck!).

              Feb  9, 2002 JHB - Made helpdef and WindowFooter non-OEM
                           font compliant.

  Overview:

  Provides a facility to change the current directory within TSE.
  Either full or partial pathnames may be entered.  Limited to drives
  identified by alphabetic letters followed by a colon (e.g., "c:").

  Detailed help is available by invoking the macro and following the
  prompts.

*************************************************************************/

/*
Constants.

*/

constant FN_COMP = 0x4
constant
    PATHLEN = _MAXPATH_,
    TMPCOL2 = PATHLEN + 1       // start of 2nd column in ID3 buffer -PHL

constant ffDIRECTORY = 10h      // the directory's attribute
constant DELAYINTICKS = 9       // delay in clock ticks when a key is pressed
constant FILEOFFSETS = 1        // accounts for the drive letter database line
constant AUTOSCAN = FALSE       // automatically rebuild database on startup
constant ALLOWSOUND = TRUE      // allow a beep when database update completed

/*
Global Strings.

*/
string mpath[PATHLEN]         // complete pathname for the search
string DirFile[] = "TSECD.DAT"
                                // directory database filename
string ListName[] = "TSE Change Directory"
string DriveLtrs[30] = "cdefghijklmnopqrstuvwxyz"
                                // drive letters to scan
/*
Global Integers.

*/
integer dir_history             // history number
integer drive_history           // history number
integer DriveLtr = 1            // indexes to the appropriate drive letter
integer ID1 = 0                 // the buffer used for "Change to directory:"
integer ID2                     // scratch buffer used to mark directory names
integer ID3                     // used to search for directories - becomes ID1
integer QuietMode = FALSE       // stop directory display on status line
integer InLoop = FALSE          // 0, 1, 2 - reentrancy level, 0 = none
integer ListBuffer              // temporary buffer for selection picklist
integer ScanDelay = DELAYINTICKS// search delay when user strikes a key
integer DatSaveNeeded = FALSE   // flags that database needs writing to disk
integer DirScan = FALSE         // flags that a database scan in progress
integer Entries                 // count of the number of database entries
integer NoDatabase              // flags whether a database currently exists
integer IdleDly = 0             // idle ticks before assertion of ask prompt
integer FirstUp = 0             // set if ask prompt via IdleDly
integer ticks                   // to release time during backgnd database scan

/*
This help information is displayed when the user selects "Help" from
the "Change to directory:" prompt.

*/
helpdef PopupHelp
    TITLE = "Change Dir Help"

    " <Alt U>    Update Database "
    ""
    ' While at the "Change to directory:" prompt, press <Alt U> to update the '
    ' directory database.  You should periodically do this after you add or '
    ' remove subdirectories, and you must do it the first time you run CD. '
    ''
    " Once the update begins, a list of directories will be displayed on the "
    " editor's StatusLine.  This update may take from 15 seconds to several "
    ' minutes, depending on the size of your disk and speed of your system. '
    ''
    ' However, you can still use the editor while it is occurring.  Simply press '
    ' any key, and then use the editor in the normal way.  Updating will continue '
    ' in the background, and a short beep will advise you when it is complete. '
    ' Presenting the "Change to directory:" prompt will display a "Directory '
    ' database is updating..." message while the update is occurring.  At this '
    ' point, you may press <Alt U> again to view the progress of the search. '
    ''
    " If the update is allowed to occur in the background, the database update "
    ' will not actually be written to the disk until either the "Change to '
    ' directory:" prompt is displayed, or until the editor is exited.  Note that '
    ' no searching will be done unless the editor is truly idle.  Any active menu '
    ' or any prompt constitutes a "busy" condition. '
    "______________________________________________________________________________"
    ''
    " <Alt E>    Edit Drive Letters "
    ""
    ' While at the "Change to directory:" prompt, press <Alt E> to reconfigure '
    " the way CD works.  Any changes will be preserved across editing sessions. "
    ""
    " - Edit Drive Letters "
    ""
    "   Enter a list of drive letters in the order that you want directories to "
    "   appear in the directory database.  This list will be preserved across "
    "   editing sessions.  Drives that do not exist will be ignored.  By default, "
    "   all drives will be checked in alphabetical order. "
    " "
    "______________________________________________________________________________"
    ''
    ' General Usage'
    ''
    ' This macro may be popped up either by executing it, or by selecting '
    ' File, then Change Dir.'
    ''
    ' When the "Change to directory:" prompt is presented, it will display the '
    " current directory on the StatusLine, and the directory of the current file "
    " in the prompt box.  If you wish, simply press <Enter>, and the current "
    " file's directory will become your current directory. "
    ""
    " Otherwise, enter either a partial directory name, or the entire directory "
    " name to select a new current directory.  If your entry is ambiguous, a "
    " PickList will display, allowing you to select the appropriate directory. "
    ' Note that drive letters are acceptable (e.g., "C:TSE"). '
    " "
    " If a partial entry is made, an attempt will be made to first match the name "
    " exactly.  Failing that, an attempt is made to match the name using the "
    " partial entry as the first letters of a directory.  Failing this, an attempt "
    " is then made to match any directory containing the characters somewhere in "
    " its name. "
    ""
    ' An "*" may be used as a wildcard.  In addition to the standard usage, it may '
    ' be entered at the beginning of the name, if you wish to view all directories '
    " containing the characters somewhere in the name -- without regard to first "
    " matching a complete directory name or the beginning of a directory name. "
    ''
    " (eg) Suppose you had three subdirectories: "
    ""
    "   c:\TSE "
    "   c:\TSEDIR "
    "   c:\TSEBEST "
    ""
    '   If you entered the letters "TSE", only one match would be made with "TSE", '
    "   and the other two directories would be ignored.  However, if you entered "
    '   "*TSE", then all three subdirectories would be presented in a picklist. '
    ''
end PopupHelp

integer findfile_handle                     // SEM 08/31/2001 - needs to be global to faithfully emulate DOS to Win32 conversion

integer proc mKeyPressed()
    return(KeyPressed())
end

/*
Add all directory names to two system buffers -- ID2 and ID3.
ID2's format will be the full directory pathname -- one entry per
line.  This buffer will be a scratch buffer -- its purpose is to
collect all of the directory names as they are located on the disk.

ID3's format is "NAME.EXT", placed at the beginning of the line
in the first thirteen columns.  It will be closely followed by the
full pathname starting at column TMPCOL2.

ID3 will later be assigned to ID1 where it will represent the
actual database file which will be written to disk as
"TSECHDIR.DAT".

This macro also handles the InLoop background reentry condition.

In addition, it checks for either a keystroke, or if running in the
background, a timeout of about one second.  At this point the
macro is suspended and the user's file made activate.  If the pause
was due to a keystroke, a delay of about 1/2 second is imposed
before the routine will pick up and continue.  During this time,
other macros or additional keystroke entries are allowed to occur
without interference from CD.

Called by:  mScanAllDirs()

*/

proc MaybeLower(var string s)
    integer i

    for i = 1 to Length(s)
        if s[i] in 'a'..'z'
            return ()
        endif
    endfor

    s = Lower(s)

end

integer proc mAddAllDirnames(integer ID0, integer LoopID)
    string s[PATHLEN]
    integer OldID = GetBufferID()

    if InLoop                           // running in background and interrupted?
        if findfile_handle == 0
            return (FALSE)
        endif
        Goto Restart                    // background reentry point
    endif
    findfile_handle = FindFirstFile(mpath,-1)             // find the first name
    if findfile_handle <> -1                              // did we get one?
        repeat                              // loop until all names found
            s = FFName()
            while not (FFAttribute() & _DIRECTORY_) or (s == ".")
                                                    or (s == "..")
                if not FindNextFile(findfile_handle,-1)
                    Goto Restart
                else
                    s = FFName()
                endif
            endwhile
            // save just the name and extension
            GotoBufferID(ID3)       // new database buffer ID
            MaybeLower(s)
            AddLine(s)              // AddLine(s)
            // place the directory name in string s
            s = SplitPath(mpath, _DRIVE_ | _PATH_) + s
            GotoPos(TMPCOL2)
            // save the entire pathname starting at column-TMPCOL2
            InsertText(s)
            GotoBufferID(ID2)   // scratch buffer ID
            EndFile()           // position to the end
            AddLine(s)          // add the pathname directory entry

            // suspend it if a key is pressed or if held for one second
            if mKeyPressed()
                    or (QuietMode and (GetClockTicks() - ticks >= 18))
                GotoBufferID(ID0)       // back to the user's buffer
                if not QuietMode        // directory names on msg line?
                    UpdateDisplay()
                endif
                InLoop = LoopID         // preserve spot for re-entrancy
                if mKeyPressed()
                    ScanDelay = DELAYINTICKS// wait for DELAYINTICKS after a key
                    if not QuietMode    // directory names on msg line?
                        GetKey()        // any key switches to bkground
                        QuietMode = TRUE// flag background "quiet" mode
                    endif
                endif
                return(FALSE)           // say that a re-entrancy is needed
                break
            endif
    Restart:
        until not FindNextFile(findfile_handle,-1)    // repeat until all files scanned
        FindFileClose(findfile_handle)
        findfile_handle = 0
    endif
    GotoBufferID(OldID)
    return(TRUE)
end mAddAllDirnames

/*
This macro will write the directory database file located in
buffer ID1 to a disk file "TSECD.DAT" in the editors load
directory, provided that the DatSaveNeeded flag is set.
The drive letters located on the first line of the file are
updated to the current values before the file is saved.

If the directory scan is being done in the foreground (the
StatusLine is displaying the directory names) this macro will be
called when the scan is completed.  Otherwise, it will be called the
next time the "Change to directory:" prompt is presented, or failing
that, when the editor exits.

The number of directory entries are saved under integer "Entries".

The macro returns TRUE if the disk file was updated, false
otherwise.  In addition, the _ON_ABANDON_EDITOR_ hook is
disconnected once the save has been done.

Called by:  _ON_ABANDON_EDITOR_ hook, mScanAllDirs(),
            mEditDriveLtrs(), mChangeDir()

*/
integer proc UpdateDiskDat()
    integer ReturnCode = FALSE
    integer OldID = GetBufferID()

                                    // get CD.MAC subdirectory
    if DatSaveNeeded                // save to disk flag set?
        DatSaveNeeded = FALSE
        Unhook(UpdateDiskDat)       // cancel any possible hook
        GotoBufferID(ID1)           // goto the database directory system buffer
        BegFile()                   // access the drive letters
        KillToEOL()                 // delete the existing drive letters
        InsertText(DriveLtrs)       // insert new drive letters
        ReturnCode = SaveAs(LoadDir() + DirFile, _OVERWRITE_)
                                    // save the directory database file now
        Entries = NumLines() - FILEOFFSETS
                                    // save a count of the database entries
        GotoBufferID(OldID)         // restore the entry buffer ID
    endif
    return(ReturnCode)
end UpdateDiskDat

/*
Create two new system buffers -- ID2 and ID3.  On entry, note the
time in clock ticks.  This will be used by mAddAllDirnames() to
release time to other applications on a regular basis.

This macro checks for a ScanDelay value before activating.  If
this value is not zero, it is decremented, and the macro is exited
until the next clock tick, where the check is repeated.  The
purpose of this is to allow other macros access to the CPU.

Once the scan begins it is flagged via DirScan.  This macro also
supports reentrancy while running in the background.  This is
accomplished via the InLoop flag.

A loop is then entered where all of the subdirectories are
sequentially scanned.  System buffer ID2 is used as a scratch
buffer during this process.  All subdirectories are displayed on
the StatusLine and placed into this buffer, one per line, as they
are found.  These subdirectories, in turn, are scanned to see if
they contain subdirectories of their own and they are again added
to the end of ID2.  Once a subdirectory has been fully scanned for
other subdirectories, it is deleted from system scratch buffer
ID2.

Simultaneously each new subdirectory is split into a "NAME.EXT"
component, and this is added to system buffer ID3, at the
beginning of a new line.  Thirteen bytes is reserved for each
entry.  Following it on the same line, beginning at column-TMPCOL2,
is the complete pathname.

This entire process is repeated for each disk drive letter, and
continues until all drive letters have been scanned and ID2 is
empty.

Once all of the directory names have been placed into ID3, it is
then alphabetically sorted.

At this point, ID2 is abandoned.  If ID1 exists (the old directory
database buffer), it is also abandoned, and the contents of ID3
are placed in ID2.  ID3 is then set to 0 (available for next
time).

If the scan was going in the background, a short beep is
heard, indicating that the scan has completed, and a flag is set
indicating that a new directory database exists that needs to be
written to disk.  Also, routine "UpdateDiskDat()" is hooked to
_ON_ABANDON_EDITOR_.  The disk write will occur the next time the
"Change to directory:" prompt is displayed, or alternatively when
the editor is exited.

If the scan was going in the foreground, the new directory
database is immediately written to disk.

Called by:  mUpdateDataBase(), WhenLoaded()

*/
proc mScanAllDirs()
    string s[PATHLEN]
    integer ID0 = GetBufferID()
    integer level
    integer TmpBufferID

    ticks = GetClockTicks()     // record the start time of this routine
    if ScanDelay
        ScanDelay = ScanDelay - 1
        return()                // give other applications some time
    endif
    DirScan = TRUE
    if not ID3
        ID3 = CreateTempBuffer()
    endif
    if not ID2
        ID2 = CreateTempBuffer()
    endif
    if (not ID2) or (not ID3)
        warn("Out of Memory -- Cannot create buffer -- ABORTING!")
        GotoBufferID(ID0)
        PurgeMacro(CurrMacroFilename())
        return()
    endif
    GotoBufferID(ID2)
    case InLoop                         // check for background activity
        when 0                          // not background re-entry
            AddLine(DriveLtrs, ID3)
        when 1                          // background re-entry-1
            Goto Retry1
        when 2                          // background re-entry-2
            Goto Retry2
    endcase
    loop
        mpath = DriveLtrs[DriveLtr] + ":\*.*"
    retry1:
        // add all directory names in the current directory to two buffers
        if not mAddAllDirnames(ID0, 1)
            return()
        endif
        InLoop = 0
        BegFile()
        repeat
            s = GetText(1, CurrLineLen())
            if not Length(s)            // illegal state
                break
            endif
            if not QuietMode
                Message('CD - Adding Directory "' + s + '"')
            endif
            mpath = s + GetDirSeparator() + "*.*"
        retry2:
            // add all directory names in the current directory to two buffers
            if not mAddAllDirnames(ID0, 2)
                return()
            endif
            InLoop = 0
            BegFile()
            KillLine()                  // kill each line as it's used
        until not NumLines()            // loop until all lines killed
        if FileExists(DriveLtrs[DriveLtr] + ":\") & _DIRECTORY_
            TmpBufferID = GetBufferID() // PHL 03/09/98 - begin change
            GotoBufferID(ID3)
            AddLine(DriveLtrs[DriveLtr] + ":\")
            BegLine()
            InsertText(Format(" ":TMPCOL2-1),_INSERT_)
            GotoBufferID(TmpBufferID)   // PHL 03/09/98 - end change
        endif                           // add the drive letter to new buffer
        if DriveLtr >= Length(DriveLtrs)// searched all drive letters?
            DriveLtr = 1                // reset for next time
            break                       // all done, exit the loop
        endif
        DriveLtr = DriveLtr + 1         // advance to the next disk drive letter
    endloop
    GotoBufferID(ID3)                   // go to the new buffer just created
    level = Set(MsgLevel, _NONE_)       // no messages
    PushBlock()                         // save any previously marked block
    UnMarkBlock()                       // start new block fresh
    // mark the full pathname
    MarkColumn(2, TMPCOL2, NumLines(), 2*TMPCOL2)   // PHL - 03/09/98
    Sort(_IGNORE_CASE_)                 // sort via the full pathname
    PopBlock()                          // restore block state
    AbandonFile(ID2)                    // dump the temporary buffer
    ID2 = 0                             // designate it as nonexistent
    if ID1
        AbandonFile(ID1)                // dump any previous database buffer
    endif
    ID1 = ID3                           // switch to newly created buffer
    ID3 = 0                             // done with ID3 designator, zero it
    Set(MsgLevel, level)
    DatSaveNeeded = TRUE                // flag that a disk write needs doing
    Hook(_ON_ABANDON_EDITOR_, UpdateDiskDat)// ensure new database saved on disk
    GotobufferID(ID0)                   // back to original buffer
    if QuietMode
        if ALLOWSOUND
            Sound(600, 2 * 10000 / 182)                      // beep to say we're done
        endif
    else
        if UpdateDiskDat()              // save database file on disk
            Message(Upper(DirFile), ' created -- ', Entries,
                " directories")
            UpdateDisplay()             // move cursor off status line
        endif
    endif
    QuietMode = FALSE                   // all done, reset some flags
    DriveLtr = 1
    DirScan = FALSE
    NoDatabase = FALSE
    Unhook(mScanAllDirs)                //  and disconnect itself
end mScanAllDirs

/*
Enter with the current directory set to the directory database
system buffer -- ID1.  Goto the line where the database
information starts -- line 2.  Check for a drive letter and a
colon, and seperate them if they exist.

Depending on the search parameters, this routine may either search
the _NAME_/_EXT_ component, or the entire pathname.  Note that if
an extension is given, then the _NAME_ and the _EXT_ component
will be individually searched.

This routine may be called several times if a search completely
fails -- first as an attempt to exactly match the _NAME_
component, then to match the first few letters of the _NAME_
component, and next to match any part of the _NAME_ component.  If
a match is still not found then the entire pathname is then
searched -- once for an exact match and then again for any match.

As the names are found, add them to the BufferList buffer, for
later display.

Called by:  mChangeDir()

*/
integer proc AddWordsToList(string s4, string s, var integer len)
    string s0[20] = s               // the search options
    string s1[PATHLEN]
    string s2[PATHLEN] = s4       // the search string
    string s3[PATHLEN]
    string Drive[2] = ""
    integer Found = FALSE, FlgFound, i

    GotoLine(2)
    BegLine()
    if (s2[2] == ":") and (Length(s2) > 2) and (not Pos("\", s2))
        Drive = SubStr(s2, 1, 2)
        s2 = SubStr(s2, 3, Length(s2))  // dump the drive letter
    endif
    i = Pos(".", s2)                    // is there an extension?
    if i and Length(s)      // if both an extension and search options exist...
        s3 = SubStr(s2, 1, i - 1)       // place _NAME_ component into s3
        while lFind(s3, s + 'i') or not Length(s3)    // search for the _NAME_
            FlgFound = FALSE
            if s2[Length(s2)] == "."    // terminating dot
                if not lFind(".", "iclg")
                    if Length(Drive)    // was there a drive letter?
                        if lFind(Drive, "icg")   // then check the drive letter
                            FlgFound = TRUE
                        endif
                    else
                        FlgFound = TRUE
                    endif
                endif
            else
                // look for ".EXT" now on the entire current line
                if lFind(SubStr(s2, i, Length(s2)), "iclg")
                    if Length(Drive)    // was there a drive letter?
                        if lFind(Drive, "icg")   // then check the drive letter
                            FlgFound = TRUE
                        endif
                    else                    // no drive -- then true
                        FlgFound = TRUE
                    endif
                endif
            endif
            if FlgFound
                Found = TRUE
                // read the entire pathname
                s1 = GetText(TMPCOL2,CurrLineLen()) // PHL 03/09/98
                s1 = Trim(s1)
                // record the longest pathname
                len = iif(Length(s1) + 1 > len, Length(s1) + 1, len)
                // add the pathname to the list buffer
                AddLine(s1, ListBuffer)
            endif

            BegLine()
            if not Down()   // continue scanning beginning of the next line
                break
            endif
        endwhile
        return(Found)
    endif
    // enter here if no ".EXT" OR no "_NAME_" OR no "Search Options"
    while lFind(s2, s0 + 'i')
        FlgFound = FALSE
        if Length(Drive)        // if a drive exists, make sure it matches
            if lFind(Drive, "icg")
                FlgFound = TRUE
            endif
        else
            FlgFound = TRUE
        endif
        if FlgFound
            Found = TRUE
            // read the entire pathname
            s1 = GetText(TMPCOL2, CurrLineLen())    // PHL 03/09/98
            s1 = Trim(s1)
            // track the longest pathname
            len = iif(Length(s1) + 1 > len, Length(s1) + 1, len)
            // add the pathname to the ListBuffer
            AddLine(s1, ListBuffer)
        endif
        BegLine()
        if not Down()       // continue scanning beginning of the next line
            break
        endif
    endwhile
    return(Found)
end AddWordsToList

/*
This routine combines TSE's current WordSet with a second WordSet,
and returns the combined wordsets.

*/
string proc mAddTwoWordSets(string ws2)
    string ws1[32]
    string ws3[32] = ""
    integer i = 1

    // Get the current wordset
    ws1 = Query(WordSet)
    // merge the two WordSets together
    while i <= 32
        ws3 = ws3 + Chr(Asc(ws1[i]) | Asc(ws2[i]))
        i = i + 1
    endwhile
    return(ws3)
end mAddTwoWordSets

/*
Returns the current directory, minus the trailing slash (unless root).

*/
string proc mCurrDir()
    string s[PATHLEN]

    s = CurrDir()
    if Length(s) > 3
        s = s[1: Length(s) - 1]
    endif
    return(s)
end mCurrDir

/*
Add RepeatFind() to the spacebar for the buffer list display.

Called by:  mEnableListKeys

*/
keydef ListKeys
    <SpaceBar> RepeatFind()
end ListKeys

/*
Attach the spacebar to RepeatFind().  Add a matching directory
count to the bottom of the buffer list window.

Called by:  Hooked to _LIST_STARTUP_.

*/
proc mEnableListKeys()
    Enable(ListKeys)
    WindowFooter(" " + Str(NumLines()) + " matching directories ")
end mEnableListKeys

/*
Try to change to a directory.  Check a number of conditions before
attempting it, and fail if any of these conditions is met.

Called by:  mChangeDir()

*/
integer proc mDirChange(string s1)
    string s2[PATHLEN]
    string s3[PATHLEN]

    if Length(s1) == 0 or       // abort if nothing in s1
            s1 == "." or        // abort if the "." is entered
            Pos("*", s1) > 0 or // abort if a wildcard is entered
            s1[2:2] == ":." or  // abort if drive letter followed by a ":."
            ExpandPath(s1) == ""
        return (FALSE)
    endif

    s2 = ExpandPath(s1)
    // did the path expand to "*.*"?
    if SplitPath(s2, _NAME_ | _EXT_) == "*.*"
        // then remove the "*.*"
        s2 = SplitPath(s2, _DRIVE_ | _PATH_)
    endif
    s3 = GetDrive()             // save the current drive
    LogDrive(s2)                // switch to the new drive letter
    // did the switch fail?
    if Lower(GetDrive()) + ":" <> Lower(SplitPath(s2, _DRIVE_))
        // put the drive back to the original and exit
        LogDrive(s3)
        return(FALSE)
    endif
    if not ChDir(s2)            // try to change to the directory now
        LogDrive(s3)            // put the drive letter back and exit
        return(FALSE)           //  if the change failed.
    endif
    return(TRUE)                // successful exit -- directory changed
end mDirChange

/*
Entered when the user selects the "Update Database" command from within
the "Change to directory:" prompt.

*/
proc mUpdateDataBase()
    ScanDelay = 0           // cancel any initial delay before scanning drives
    if QuietMode
        QuietMode = FALSE   // begin by displaying all drives on the statusline
    else
        Hook(_IDLE_, mScanAllDirs)  // start scanning on next _IDLE_ clock tick
    endif
end mUpdateDataBase

/*
Restrict the prompt keydef to the following alphabetic keys:

Called by:  mRestrictLtrs()
*/
keydef RestrictLtrs
    <a> SelfInsert()
    <Shift a> SelfInsert()
    <b> SelfInsert()
    <Shift b> SelfInsert()
    <c> SelfInsert()
    <Shift c> SelfInsert()
    <d> SelfInsert()
    <Shift d> SelfInsert()
    <e> SelfInsert()
    <Shift e> SelfInsert()
    <f> SelfInsert()
    <Shift f> SelfInsert()
    <g> SelfInsert()
    <Shift g> SelfInsert()
    <h> SelfInsert()
    <Shift h> SelfInsert()
    <i> SelfInsert()
    <Shift i> SelfInsert()
    <j> SelfInsert()
    <Shift j> SelfInsert()
    <k> SelfInsert()
    <Shift k> SelfInsert()
    <l> SelfInsert()
    <Shift l> SelfInsert()
    <m> SelfInsert()
    <Shift m> SelfInsert()
    <n> SelfInsert()
    <Shift n> SelfInsert()
    <o> SelfInsert()
    <Shift o> SelfInsert()
    <p> SelfInsert()
    <Shift p> SelfInsert()
    <q> SelfInsert()
    <Shift q> SelfInsert()
    <r> SelfInsert()
    <Shift r> SelfInsert()
    <s> SelfInsert()
    <Shift s> SelfInsert()
    <t> SelfInsert()
    <Shift t> SelfInsert()
    <u> SelfInsert()
    <Shift u> SelfInsert()
    <v> SelfInsert()
    <Shift v> SelfInsert()
    <w> SelfInsert()
    <Shift w> SelfInsert()
    <x> SelfInsert()
    <Shift x> SelfInsert()
    <y> SelfInsert()
    <Shift y> SelfInsert()
    <z> SelfInsert()
    <Shift z> SelfInsert()
    <CursorRight> Right()
    <CursorLeft> Left()
    <CursorUp> Up()
    <CursorDown> Down()
    <Home> BegLine()
    <End> EndLine()
    <Escape> EndProcess(0)
    <BackSpace> BackSpace()
    <Del> DelChar()
    <Enter> EndProcess(1)
end RestrictLtrs

/*
Restrict the prompt keydef to the following alphabetic keys:

Called by:  Hooked to _PROMPT_STARTUP_ in routine mEditDriveLtrs().
*/
proc mRestrictLtrs()
    enable(RestrictLtrs, _EXCLUSIVE_)
end mRestrictLtrs

/*
Remove duplicate drive letters, and ensure that only alphabetics
are allowed.

Called by:  mEditDriveLtrs(), WhenLoaded()

*/
string proc FilterDriveLtrs(string s0)
    integer i
    string s2[30] = ""

    for i = 1 to Length(s0)
        if not Pos(s0[i], SubStr(s0, i + 1, Length(s0)))
            case s0[i]
                when "a" .. "z", "A" .. "Z"
                    s2 = s2 + s0[i]
            endcase
        endif
    endfor
    return(Lower(s2))
end FilterDriveLtrs

/*
Center a prompt box when it is presented.  Hook a restriction to
the allowed prompt letters,  a - z only.  If the drive letters are
changed via this prompt, then remove any impossible non-alphabetic
characters, and any drive duplicates.  If no drive letters, then
add the default drive letters "c - z".

Save the directory database with the new drive letters on disk.
Advise the user of the new drive letters on the StatusLine.

Called by:  mChangeDir()
*/
proc mEditDriveLtrs()
    string s0[27] = DriveLtrs
    integer AskReturn, OldY1, OldX1, Cursor

    OldY1 = Set(Y1, (Query(ScreenRows)) / 2)
    OldX1 = Set(X1, (Query(ScreenCols) - 27) / 2)
    Hook (_PROMPT_STARTUP_, mRestrictLtrs)
    AskReturn = Ask("Drive letters:", s0, drive_history)
    UnHook(mRestrictLtrs)
    if Lower(s0) <> Lower(DriveLtrs)            // any change?
        if AskReturn
            if Length(s0)
                // remove any duplicate drive letters
                DriveLtrs = FilterDriveLtrs(s0)
            else
                // no letters -- go with the defaults
                DriveLtrs = "cdefghijklmnopqrstuvwxyz"
            endif
            // turn off the cursor
            Cursor = Set(Cursor, OFF)
            DatSaveNeeded = TRUE
            // save the directory database with new drive letters on disk
            UpdateDiskDat()
            Message('Drive letters changed to "' + DriveLtrs + '"')
            // wait for a keypress and then blank the statusline
            while not KeyPressed()
            endwhile
            Message("")
            // restore the cursor
            Set(Cursor, Cursor)
        endif
    endif
    Set(Y1, OldY1)
    Set(X1, OldX1)
end mEditDriveLtrs

/*
These keys close the "Change to directory:" prompt via the
EndProcess(-x) command, and return a value indicating which key
was pressed.

Called by: mPromptStartup()

*/
keydef PromptKeys
    <Alt H> EndProcess(-1)
    <Alt U> EndProcess(-2)
    <Alt E> EndProcess(-3)
end PromptKeys

/*
This macro adds headers and footers to the "Change to directory:"
prompt.

Called by:  Hooked to _PROMPT_STARTUP_ in mChangeDir().

*/
proc mPromptStartup()
    string TitleMessage[80] = ""

    Enable(PromptKeys)
    if NoDatabase
        TitleMessage = ' {TSECD.DAT database does NOT exist.  Press <Alt U> to create a new one.} '
    endif
    if DirScan
        TitleMessage = " {Directory database is updating...} "
        WindowFooter(" {<Alt H>} {H}elp     {<Alt U>} View {U}pdates     {<Alt E>} {E}dit Drives ")
    else
        WindowFooter(" {<Alt H>} {H}elp     {<Alt U>} {U}pdate Database     {<Alt E>} {E}dit Drives ")
    endif
    VGotoXY(Query(WindowX1) + (Query(WindowCols) / 2)
        - Length(TitleMessage) / 2, Query(WindowY1) - 2)
    PutHelpLine(TitleMessage)
end mPromptStartup

/*
Wait until IdleDly is zero before popping up.  This allows other
macros some initial cpu time when this routine is hooked to the
_IDLE_ hook via WhenLoaded().

If necessary, save a directory database to disk when this activates.

Display the "Change to directory:" prompt, complete with the
header and footer information (via hooks), and place the current
files directory as the default entry in the prompt box.

Do a fair amount of wildcard filtering in here.  Search the
directory database entries for matches on ONLY ONE of the
following:

    1.  A perfect match with _NAME_ field.
    2.  Match the first few letters in _NAME_.
    3.  Match any letters in _NAME_ or _EXT_.
    4.  Match the entire directory pathname completely.
    5.  Match any letters in the directory pathname.

Wildcards may vary the entry point for the above list.

Called by:  Main(), Hooked to _IDLE_ by WhenLoaded(), assigned by
default to the <CtrlAlt D> key.

*/
proc mChangeDir()
    string SaveString[PATHLEN]                    // PHL 03/09/98
    string s1[PATHLEN]                            // PHL 03/09/98
    string s2[PATHLEN]                            // PHL 03/09/98
    string SavedWordSet[32]
    integer ID0 = GetBufferID()
    integer len
    integer i
    integer ListAbandoned
    integer FlagFullChk
    integer StarDot, WildCardLeader, AskReturn, WildCardTrailer
    integer OldX1 = Query(X1), OldY1 = Query(Y1)
    integer OldMenuTextAttr, OldMenuBorderAttr
    integer save_readflags

    if (IdleDly)
        IdleDly = IdleDly - 1
        return()
    endif
    UnHook(mChangeDir)      // hooked during WhenLoaded()
    UpdateDiskDat()         // write the directory database to disk if needed
    // Get the current file's directory (trailing slash removed).
    s2 = CurrFilename()
    s2 = SplitPath(s2, _DRIVE_ | _PATH_)
    if Length(s2) > 3
        s2 = SubStr(s2, 1, Length(s2) - 1)
    endif
    SaveString = s2         // preserve this directory for later
RepeatChdir:
    Set(Y1, (Query(ScreenRows) - 4) / 2)
    s1 = ""
    len = Length(ListName) + 10
    ListAbandoned = FALSE
    FlagFullChk = FALSE
    Message('Current Directory: "' + mCurrDir() + '"')
    Hook(_PROMPT_STARTUP_, mPromptStartup)
    save_readflags = Set(ReadFlags, Query(ReadFlags) | FN_COMP)
    AskReturn = Ask("Change to directory:", s2, dir_history)
    Set(ReadFlags, save_readflags)

    if FirstUp              // up via the _IDLE_ prompt?
        FirstUp = FALSE
        UpdateDisplay()     // then get the cursor off the statusline
    endif
    UnHook(mPromptStartup)
    case AskReturn
        when 0                      // escape pressed -- do nothing

        when -1                     // <Alt H>
            OldMenuTextAttr = Set(MenuTextAttr, Color(BRIGHT WHITE ON BLACK))
            OldMenuBorderAttr = Set(MenuBorderAttr, Color(BRIGHT RED ON BLACK))
            QuickHelp(PopupHelp)
            Set(MenuTextAttr, OldMenuTextAttr)
            Set(MenuBorderAttr, OldMenuBorderAttr)
            s2 = SaveString         // restore s2
            goto RepeatChdir        // spin back to the main prompt
        when -2                     // <Alt U>
            mUpdateDataBase()
        when -3                     // <Alt E>
            Set(Y1, (Query(ScreenRows) - 6) / 2)
            Set(X1, (Query(ScreenCols) - 38) / 2)
            mEditDriveLtrs()
            s2 = SaveString     // restore s2
            goto RepeatChdir    // spin back to the main prompt
        otherwise                   // new entry
            Message("")             // blank the message line
            AddHistoryStr(s2, dir_history)
            SaveString = s2         // save s2
            s2 = Trim(s2)
            s2 = Lower(s2)
            if mDirChange(s2)
                // in case drive-a or drive-b produce an error message
                UpdateDisplay(_ALL_WINDOWS_REFRESH_ | _HELPLINE_REFRESH_)
                Message('Current Directory: "' + mCurrDir() + '"')
            else
                case s2
                    when "*.*", ".", ".*"
                        s2 = ""
                endcase
                case SubStr(s2, 2, Length(s2))
                    when ":\*.*", ":*.*", ":\*", ":*"
                        // isolate s2 to the drive letter and colon only
                        s2 = SubStr(s2, 1, 2)
                        // check the entire database file
                        FlagFullChk = TRUE
                endcase
                StarDot = FALSE         // assume FALSE
                if (Pos("*.", s2)) and (s2[Length(s2)] <> ".")
                    StarDot = TRUE
                endif
                if ((s2[Length(s2)] == "*") and (not Pos(".", s2)))
                    StarDot = TRUE
                endif
                WildCardLeader = FALSE
                WildCardTrailer = FALSE
                i = Pos("*", s2)
                if i
                    if i < Length(s2)
                        if s2[i + 1] <> "."
                            WildCardLeader = TRUE
                        endif
                    else
                        WildCardTrailer = TRUE
                    endif
                endif
                i = Pos("*.*", s2)
                if i > 1
                    s2 = SubStr(s2, 1, i - 1)   // strip the ".*"
                    WildCardTrailer = TRUE
                endif
                i = 1
                while i <= Length(s2)
                    if s2[i] <> "*" and s2[i] <> "?"
                        s1 = s1 + s2[i]
                    endif
                    i = i + 1
                endwhile
                ListBuffer = CreateTempBuffer()
                if not ListBuffer
                    warn("Out of Memory -- Cannot create buffer -- ABORTING!")
                    Set(X1, OldX1)
                    Set(Y1, OldY1)
                    GotoBufferID(ID0)
                    PurgeMacro(CurrMacroFilename())
                    return()
                endif
                GotoBufferID(ID1)
                s2 = s1
                if not Length(s2)
                    s2 = ":\"                       // match everything
                endif
                case Lower(s2)
                    when "\", "/"                   // root request?
                        s2 = GetDrive() + ":\"      // go to the root
                        AddLine(s2, ListBuffer)
                    when ":"                        // just eat a ":" by itself

                    otherwise
                        PushPosition()
                        GotoLine(2)
                        BegLine()
                        PushBlock()
                        UnMarkblock()
                        // mark the _NAME_ | _EXT_ at beginning of lines
                        MarkColumn(2, 1, NumLines(), TMPCOL2-1) // PHL 03/09/98
                        SavedWordSet = ChrSet(".\\")
                        SavedWordSet = mAddTwoWordSets(SavedWordSet)
                        SavedWordSet = Set(WordSet, SavedWordSet)
                        // check the entire pathname?
                        if FlagFullChk
                            FlagFullChk = FALSE
                            Goto FullCheck
                        endif
                        if Pos("\", s2)
                            // check the full pathname
                            Goto WordCheck
                        endif
                        if WildCardLeader
                            // don't limit search to beginning of _NAME_
                            Goto WildCardEntry
                        endif
                        if WildCardTrailer
                            // match beginning of _NAME_
                            Goto WildTrailEntry
                        endif
                        if not AddWordsToList(s2, "LW^", len)
                    WildTrailEntry:
                            if not AddWordsToList(s2, "L^", len)
                                if StarDot
                                    Goto AllDone
                                endif
                    WildCardEntry:
                                if not AddWordsToList(s2, "L", len)
                    WordCheck:      if not AddWordsToList(s2, "W", len)
                    FullCheck:          AddWordsToList(s2, "", len)
                                    endif
                                endif
                            AllDone:
                            endif
                        endif
                        Set(WordSet, SavedWordSet)
                        PopBlock()
                        PopPosition()
                endcase
                GotoBufferID(ListBuffer)
                BegFile()
                case NumLines()
                    when 0          // do nothing
                    when 1
                        s2 = GetText(1, CurrLineLen())
                    otherwise
                        // display the list buffer
                        Hook(_LIST_STARTUP_, mEnableListKeys)
                        if lList(ListName,
                        iif(len < Query(ScreenCols),len,Query(ScreenCols)),
                        iif(NumLines() < Query(ScreenRows),NumLines(),
                        Query(ScreenRows)),_ENABLE_SEARCH_|_ENABLE_HSCROLL_)
                            s2 = GetText(1, CurrLineLen())
                            SaveString = s2 // save s2
                        else
                            ListAbandoned = TRUE
                        endif
                        UnHook(mEnableListKeys)
                endcase
                AbandonFile(ListBuffer)
                if ListAbandoned
                    s2 = SaveString     // restore s2
                    goto RepeatChdir    // spin back to the main prompt
                else
                    if not mDirChange(s2)
                        Message('Directory "' + Upper(SaveString) + '" not found...')
                    else
                        UpdateDisplay(_ALL_WINDOWS_REFRESH_ | _HELPLINE_REFRESH_)
                        Message('Current Directory: "' + mCurrDir() + '"')
                    endif
                endif
            endif
    endcase
    Set(X1, OldX1)
    Set(Y1, OldY1)
    GotoBufferID(ID0)
end mChangeDir

/*
Stop any assigned hooks from chaining forward.

Called by:  Hooked to _ON_FIRST_EDIT_ and _ON_CHANGING_FILES_ by WhenLoaded()

*/
proc MHookKiller()
    BreakHookChain()
end MHookKiller

/*
Load the directory database file "TSECD.DAT" into system buffer
ID1.  Access the Drive Letters and update string DriveLtrs.

If TSECD.DAT does not exist, then hook mChangeDir() to the
_IDLE_ hook.  mChangeDir() will activate and advise the user that
a directory database must be created.  In general, this situation
only happens once.

Called by:  TSE when the editor starts

*/
proc WhenLoaded()
    integer ID0 = GetBufferID()

    dir_history = GetFreeHistory("CD:Dir")
    drive_history = GetFreeHistory("CD:Drive")

    if FileExists(LoadDir() + DirFile)
        NoDatabase = FALSE
        Hook(_ON_FIRST_EDIT_, mHookKiller)
        Hook(_ON_CHANGING_FILES_, mHookKiller)
        ID1 = CreateTempBuffer()
        if not ID1
            warn("Out of Memory -- Cannot create buffer -- ABORTING!")
            PurgeMacro(CurrMacroFilename())
            return()
        endif
        PushBlock()
        InsertFile(LoadDir() + DirFile)
        Begfile()
        DriveLtrs = FilterDriveLtrs(GetText(1, CurrLineLen()))
        PopBlock()
        UnHook(MHookKiller)
        if AUTOSCAN
            QuietMode = TRUE
            Hook(_IDLE_, mScanAllDirs)
        endif
    else
        ID1 = CreateTempBuffer()
        if not ID1
            warn("Out of Memory -- Cannot create buffer -- ABORTING!")
            PurgeMacro(CurrMacroFilename())
            return()
        endif
        NoDatabase = TRUE
        IdleDly = 2             // clock ticks before asserting ask prompt
        FirstUp = TRUE
        Hook(_IDLE_, mchangeDir)
    endif
    GotoBufferID(ID0)
end WhenLoaded

/*
Present the "Change to directory:" prompt.  Equivalent to pressing
the <CtrlAlt D> popup key.  This allows CD to be tied to other
macros (including TSE.UI), and activated by merely executing the
macro.

Called by:  TSE when the CD.MAC is executed.

*/
proc Main()
    IdleDly = 0
    mChangeDir()
end Main

/*
Dump the directory database file represented by bufferID ID1.

Called by TSE when CD is purged.

*/

proc WhenPurged()
    if ID1
        AbandonFile(ID1)
    endif
end WhenPurged

//<CtrlAlt D> mChangeDir()

