#!/usr/bin/env python3

#!/usr/bin/env python3
"""
bso.py — BSO Outbound Directory Reporter
Version: 1.0

Copyright (c) 2026 Sean Rima and Murphy

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
"""

import os
import re
import argparse

# -----------------------------------
# CONFIG SECTION — DEFAULT VALUES
# -----------------------------------
CONFIG = {
    "bso_path": "/home/fmail/ftn/spool/out",     # Path to the main out directory
    "default_zone": 2,               # Zone for the plain "out" folder
    "summary_per_address": True     # True = one line per address with totals
}
# -----------------------------------

# ANSI colours for terminal output (may be disabled later)
RESET = "\033[0m"
BOLD = "\033[1m"
CYAN = "\033[36m"
YELLOW = "\033[33m"
DIM = "\033[2m"

def disable_colours():
    """Turn off all colour escape sequences."""
    global RESET, BOLD, CYAN, YELLOW, DIM
    RESET = BOLD = CYAN = YELLOW = DIM = ""


def format_size(num_bytes: int) -> str:
    """Human-friendly size. Uses K/M/G/T with one decimal where useful."""
    try:
        n = float(num_bytes)
    except Exception:
        return str(num_bytes)

    units = ["B", "K", "M", "G", "T", "P"]
    u = 0
    while n >= 1024.0 and u < len(units) - 1:
        n /= 1024.0
        u += 1

    if u == 0:
        return f"{int(n)}{units[u]}"
    if n >= 10:
        return f"{n:.0f}{units[u]}"
    return f"{n:.1f}{units[u]}"


def ellipsize(s: str, width: int) -> str:
    """Trim string to width with an ellipsis if needed."""
    if width <= 0:
        return ""
    if s is None:
        s = ""
    s = str(s)
    if len(s) <= width:
        return s
    if width == 1:
        return "…"
    return s[: width - 1] + "…"


class BSO:
    def __init__(self, bso_path, default_zone):
        self.bso_path = os.path.abspath(bso_path)
        self.default_zone = int(default_zone)
        self.files = {}

        if not os.path.isdir(self.bso_path):
            raise NotADirectoryError(f"{self.bso_path} is not a directory")

    def read(self):
        """
        Look in the parent directory for:
          out
          out.<HEX>
        and work out the zone for each.
        """
        base_dir = os.path.dirname(self.bso_path)
        base_name = os.path.basename(self.bso_path)

        pattern = re.compile(rf"^{re.escape(base_name)}(?:\.([0-9a-fA-F]+))?$")

        for entry in sorted(os.listdir(base_dir)):
            m = pattern.match(entry)
            if not m:
                continue

            full_path = os.path.join(base_dir, entry)
            if not os.path.isdir(full_path):
                continue

            hex_zone = m.group(1)
            if hex_zone is None:
                zone = self.default_zone
            else:
                try:
                    zone = int(hex_zone, 16)
                except ValueError:
                    continue

            self.read_bso(full_path, zone)

        return self.files

    def read_bso(self, dir_path, zone, net=None, node=None):
        """
        Recursively read BSO directory structure.
        """
        for item in os.listdir(dir_path):
            full = os.path.join(dir_path, item)
            base, ext = os.path.splitext(item)

            if os.path.isfile(full):
                size = os.path.getsize(full)
                if size == 0:
                    continue

                prio = None
                kknd = None
                mnet = net
                mnode = node
                mpoint = 0

                # .cut / .dut / .hut / .nut / .fut
                m = re.match(r'\.([cdhnf])ut$', ext)
                if m:
                    prio = m.group(1)
                    kknd = "^"

                    if mnet is None:
                        try:
                            mnet = int(base[0:4], 16)
                            mnode = int(base[4:8], 16)
                        except ValueError:
                            continue
                    else:
                        try:
                            mpoint = int(base[4:8], 16)
                        except ValueError:
                            pass

                    addr = f"{zone}:{mnet}/{mnode}.{mpoint}"
                    self.add(full, addr, None, kknd, prio)

                # .clo/.dlo/.hlo/.flo/.nlo
                m = re.match(r'\.([cdhnf])lo$', ext)
                if m:
                    prio = m.group(1)

                    if mnet is None:
                        try:
                            mnet = int(base[0:4], 16)
                            mnode = int(base[4:8], 16)
                        except ValueError:
                            continue
                    else:
                        try:
                            mpoint = int(base[4:8], 16)
                        except ValueError:
                            pass

                    addr = f"{zone}:{mnet}/{mnode}.{mpoint}"

                    with open(full, "r", encoding="utf-8", errors="ignore") as f:
                        for line in f.read().splitlines():
                            if not line:
                                continue
                            kknd = line[0]
                            filename = line[1:]
                            self.add(filename, addr, full, kknd, prio)

            elif os.path.isdir(full):
                # *.pnt directories
                if ext == ".pnt":
                    try:
                        pnet = int(base[0:4], 16)
                        pnode = int(base[4:8], 16)
                    except ValueError:
                        continue

                    self.read_bso(full, zone, pnet, pnode)

    def add(self, file, addr, bundle, kknd, prio):
        if addr not in self.files:
            self.files[addr] = []

        size = os.path.getsize(file) if os.path.isfile(file) else 0
        zone_str = addr.split(":")[0]

        self.files[addr].append({
            "file": file,
            "size": size,
            "bundle": bundle,
            "kknd": kknd,
            "prio": prio,
            "zone": zone_str,
            "address": addr
        })


def parse_address(addr):
    """Split address zone:net/node.point → tuple of ints."""
    try:
        zone_part, rest = addr.split(":", 1)
        net_part, node_point = rest.split("/", 1)
        node_part, point_part = node_point.split(".", 1)
        return int(zone_part), int(net_part), int(node_part), int(point_part)
    except Exception:
        return 0, 0, 0, 0


def print_table(entries, summary=False, no_blank_lines=False):
    if summary:
        # Tight summary table (no ZONE column)
        headers = ["ADDRESS", "FILES", "TOTAL"]
        col_widths = [18, 5, 10]

        def fmt_row(row, colour_prefix=""):
            return (
                f"{colour_prefix}{row[0]:<{col_widths[0]}} "
                f"{row[1]:>{col_widths[1]}} "
                f"{row[2]:>{col_widths[2]}}{RESET}"
            )

        summary_map = {}
        for addr, items in entries.items():
            z, n, nd, p = parse_address(addr)
            summary_map[addr] = {
                "zone_int": z,
                "count": len(items),
                "total_size": sum(i["size"] for i in items),
            }

        sorted_addrs = sorted(
            summary_map.items(),
            key=lambda kv: (kv[1]["zone_int"],) + parse_address(kv[0])[1:]
        )

        print()
        print(fmt_row(headers, BOLD + CYAN))
        print("-" * (sum(col_widths) + 2))

        last_zone = None
        for addr, data in sorted_addrs:
            # Keep zone-based grouping, optionally suppress blank lines
            if (not no_blank_lines) and last_zone is not None and data["zone_int"] != last_zone:
                print()

            row = [
                addr,
                data["count"],
                format_size(data["total_size"]),
            ]
            print(fmt_row(row, YELLOW))

            last_zone = data["zone_int"]

    else:
        # Tight detailed table (no ZONE, no KKND)
        headers = ["ADDRESS", "FILE", "SIZE", "P", "BUNDLE"]
        col_widths = [18, 20, 8, 1, 20]

        def fmt_row(row, colour_prefix=""):
            return (
                f"{colour_prefix}{row[0]:<{col_widths[0]}} "
                f"{row[1]:<{col_widths[1]}} "
                f"{row[2]:>{col_widths[2]}} "
                f"{row[3]:<{col_widths[3]}} "
                f"{row[4]:<{col_widths[4]}}{RESET}"
            )

        flat = []
        for addr, items in entries.items():
            z, n, nd, p = parse_address(addr)
            for item in items:
                flat.append((z, n, nd, p, addr, item))

        flat.sort(key=lambda t: (t[0], t[1], t[2], t[3], os.path.basename(t[5]["file"])))

        print()
        print(fmt_row(headers, BOLD + CYAN))
        print("-" * (sum(col_widths) + 4))

        last_zone = None
        last_addr = None

        for z, n, nd, p, addr, item in flat:
            filename = os.path.basename(item["file"])
            bundle_name = os.path.basename(item["bundle"]) if item["bundle"] else ""
            bundle_name = ellipsize(bundle_name, col_widths[4])

            row = [
                addr,
                ellipsize(filename, col_widths[1]),
                format_size(item["size"]),
                item["prio"] or "",
                bundle_name,
            ]

            if (not no_blank_lines) and last_zone is not None and z != last_zone:
                print()

            colour = YELLOW if addr != last_addr else ""
            print(fmt_row(row, colour))

            last_zone = z
            last_addr = addr


# -----------------------
# MAIN WITH ARGPARSE
# -----------------------
def main():
    parser = argparse.ArgumentParser(description="BSO out directory inspector")

    parser.add_argument("-p", "--path", help="Override BSO path", default=CONFIG["bso_path"])
    parser.add_argument("-z", "--zone", help="Override default zone", type=int, default=CONFIG["default_zone"])
    parser.add_argument("-s", "--summary", help="Summary output (one line per address)", action="store_true")
    parser.add_argument("-d", "--details", help="Detailed output (one line per file)", action="store_true")
    parser.add_argument("-n", "--nocolour", help="Disable colour output", action="store_true")
    parser.add_argument("--no-blank-lines", help="Do not print blank lines between zone groups", action="store_true")

    args = parser.parse_args()

    if args.nocolour:
        disable_colours()

    # If both -s and -d missing → use default from CONFIG
    if args.summary:
        summary = True
    elif args.details:
        summary = False
    else:
        summary = CONFIG["summary_per_address"]

    bso = BSO(args.path, args.zone)
    results = bso.read()

    print_table(results, summary=summary, no_blank_lines=args.no_blank_lines)


if __name__ == "__main__":
    main()
